// token/token.go

package token

type TokenType string

type TToken struct {
	Type		TokenType
	Literal	string
}


const (
	UNKNOWN				= "UNKNOWN"
	EOF						= "EOF"
	
	DATATYPE			= "DATATYPE"
	IDENTIFIER		= "IDENTIFIER"
	INTEGER				= "INTEGER"
	FLOAT					= "FLOAT"
	STRING				= "STRING"
	
	ASSIGN			= "="
	PLUS				= "+"
	MINUS				= "-"
	ASTERISK		= "*"
	SLASH				= "/"
	PERCENT			= "%"
	CARET				= "^"
	
	LESS				= "<"
	GREATER			= ">"
	
	EQUAL				= "=="
	NOT_EQUAL		= "<>"
	LESS_EQ			= "<="
	GREATER_EQ	= ">="
	
	IS					= "is"
	NOT					= "not"
	
	OR					= "or"
	XOR					= "xor"
	NOR					= "nor"
	AND					=	"and"
	
	COMMA				= ","
	COLON				= ":"
	SEMICOLON		= ";"
	
	LPAREN			= "("
	RPAREN			= ")"
	LBRACE			= "{"
	RBRACE			= "}"
	LBRACKET		= "["
	RBRACKET		= "]"
	
	UNDERSCORE	= "_"
	DOT					= "."
	
	LLIST				= "[<"
	RLIST				= ">]"
	
	REF					= "REF"
	CASE				= "CASE"
	CONST				= "CONST"
	CONTINUE		= "CONTINUE"
	VAR					= "VAR"
	DEFAULT			= "DEFAULT"
	DO					= "DO"
	ELSE				= "ELSE"
	ELSEIF			= "ELSEIF"
	ENUM				= "ENUM"
	EXIT				= "EXIT"
	END					= "END"
	FALSE				= "FALSE"
	FOR					=	"FOR"
	FUNCTION		= "FUNCTION"
	IF					= "IF"
	OF					= "OF"
	RETURN			= "RETURN"
	SCOPE				= "SCOPE"
	SELECT			= "SELECT"
	STEP				= "STEP"
	SUB					= "SUB"
	TO					= "TO"
	TRUE				= "TRUE"
	TYPE				= "TYPE"
	WHILE				= "WHILE"
	
)


var keywords = map[string] TokenType {
	"and":			AND,
	"ref":			REF,
	"case":			CASE,
	"const":		CONST,
	"continue":	CONTINUE,
	"default":	DEFAULT,
	"do":				DO,
	"else":			ELSE,
	"elseif":		ELSEIF,
	"end":			END,
	"enum":			ENUM,
	"exit":			EXIT,
	"false":		FALSE,
	"for":			FOR,
	"function": FUNCTION,
	"if":				IF,
	"is":				IS,
	"nor":			NOR,
	"not":			NOT,
	"of":				OF,
	"or":				OR,
	"return":		RETURN,
	"scope":		SCOPE,
	"select":		SELECT,
	"step":			STEP,
	"sub":			SUB,
	"to":				TO,
	"type":			TYPE,
	"true":			TRUE,
	"var":			VAR,
	"while":		WHILE,
	"xor":			XOR,
}


/* data types map to identifiers in the parser and subsequently to
 * builtin functions in the evaluator. See the parser for details.
 */ 

var datatypes = map[string] TokenType {
	"array":		DATATYPE,
	"boolean":	DATATYPE,
	"byte":			DATATYPE,
	"double": 	DATATYPE,
	"func":			DATATYPE,
	"integer":	DATATYPE,
	"list":			DATATYPE,
	"long": 		DATATYPE,
	"map":			DATATYPE,
	"nil":			DATATYPE,
	"short":		DATATYPE,
	"single": 	DATATYPE,
	"string":		DATATYPE,
	"ubyte":		DATATYPE,
	"uinteger":	DATATYPE,
	"ulong":		DATATYPE,
	"ushort":		DATATYPE,
}


func LookupIdent(ident string) TokenType {
	if tok, ok := keywords[ident]; ok {
		return tok
	}
	
	if tok, ok := datatypes[ident]; ok {
		return tok
	}
	
	return IDENTIFIER
}

