// parser/parser.go

package parser

import (
	"fmt"
	"strconv"
	"SB/ast"
	"SB/lexer"
	"SB/token"
)

const (
	_ int = iota
	LOWEST
	ASSIGN
	BOOLXOR
	BOOLANDNOR
	BOOLISNOT
	COMPARE
	SUM
	PRODUCT
	POWER
	PREFIX
	CALL
	INDEX
	REFERENCE
)

var precedences = map[token.TokenType] int {
	token.ASSIGN:			ASSIGN,
	token.TO:					ASSIGN,
	token.XOR:				BOOLXOR,
	token.OR:         BOOLANDNOR,
	token.NOR:				BOOLANDNOR,
	token.AND:        BOOLANDNOR,
	token.IS:					BOOLISNOT,
	token.NOT:				BOOLISNOT,
	token.EQUAL:			COMPARE,
	token.NOT_EQUAL:	COMPARE,
	token.LESS:				COMPARE,
	token.LESS_EQ:		COMPARE,
	token.GREATER:		COMPARE,
	token.GREATER_EQ:	COMPARE,
	token.PLUS:				SUM,
	token.MINUS:			SUM,
	token.PERCENT:		PRODUCT,
	token.SLASH:			PRODUCT,
	token.ASTERISK:		PRODUCT,
	token.CARET:			POWER,
	token.LPAREN:			CALL,
	token.LBRACKET:		INDEX,
	token.LLIST:			INDEX,
	token.DOT:				REFERENCE,
}

type (
	prefixParseFn			func() ast.IExpression
	infixParseFn			func(ast.IExpression) ast.IExpression
)

type TBlockType struct {
	blockType			token.TokenType
	startToken		token.TokenType
	endTokens			[]token.TokenType
}

type TParser struct {
	l											*lexer.TLexer
	errors 								[]string
	curToken							token.TToken
	peekToken							token.TToken
	prefixParseFns				map[token.TokenType]prefixParseFn
	infixParseFns					map[token.TokenType]infixParseFn
	
	tokenStack						[]token.TToken
	tokenStackCount				int
	
	caseBlock							TBlockType
	defaultBlock					TBlockType
	ifBlock								TBlockType
	forBlock							TBlockType
	functionBlock					TBlockType
	scopeBlock						TBlockType
	subBlock							TBlockType
	whileBlock						TBlockType
}


func New(l *lexer.TLexer) *TParser {
	p := &TParser { l: l, errors: []string{}, }
	
	p.prefixParseFns = make(map[token.TokenType]prefixParseFn)
	p.registerPrefix(token.CONTINUE, p.parseContinueExpressionInvalid)
	p.registerPrefix(token.DATATYPE, p.parseIdentifier)
	p.registerPrefix(token.EXIT, p.parseExitExpressionInvalid)
	p.registerPrefix(token.FALSE, p.parseBoolean)
	p.registerPrefix(token.FLOAT, p.parseFloatLiteral)
	p.registerPrefix(token.FUNCTION, p.parseFunctionLiteral)
	p.registerPrefix(token.IDENTIFIER, p.parseIdentifier)
	p.registerPrefix(token.INTEGER, p.parseIntegerLiteral)
	p.registerPrefix(token.IS, p.parsePrefixExpression)
	p.registerPrefix(token.LBRACE, p.parseMapLiteral)
	p.registerPrefix(token.LPAREN, p.parseGroupedExpression)
	p.registerPrefix(token.MINUS, p.parsePrefixExpression)
	p.registerPrefix(token.NOT, p.parsePrefixExpression)
	p.registerPrefix(token.STEP, p.parseStepExpressionInvalid)
	p.registerPrefix(token.STRING, p.parseStringLiteral)
	p.registerPrefix(token.SUB, p.parseSubLiteral)
	p.registerPrefix(token.TO, p.parseFromToExpressionInvalid)
	p.registerPrefix(token.TRUE, p.parseBoolean)
	
	p.infixParseFns = make(map[token.TokenType]infixParseFn)
	p.registerInfix(token.AND, p.parseInfixExpression)
	p.registerInfix(token.ASSIGN, p.parseAssignExpression)
	p.registerInfix(token.ASTERISK, p.parseInfixExpression)
	p.registerInfix(token.CARET, p.parseInfixExpression)
	p.registerInfix(token.DOT, p.parseReferenceExpression)
	p.registerInfix(token.GREATER, p.parseInfixExpression)
	p.registerInfix(token.GREATER_EQ, p.parseInfixExpression)
	p.registerInfix(token.EQUAL, p.parseInfixExpression)
	p.registerInfix(token.LBRACKET, p.parseArrayIndexExpression)
	p.registerInfix(token.LESS, p.parseInfixExpression)
	p.registerInfix(token.LESS_EQ, p.parseInfixExpression)
	p.registerInfix(token.LLIST, p.parseListIndexExpression)
	p.registerInfix(token.LPAREN, p.parseCallExpression)
	p.registerInfix(token.MINUS, p.parseInfixExpression)
	p.registerInfix(token.NOR, p.parseInfixExpression)
	p.registerInfix(token.NOT_EQUAL, p.parseInfixExpression)
	p.registerInfix(token.OR, p.parseInfixExpression)
	p.registerInfix(token.PERCENT, p.parseInfixExpression)
	p.registerInfix(token.PLUS, p.parseInfixExpression)
	p.registerInfix(token.SLASH, p.parseInfixExpression)
	p.registerInfix(token.XOR, p.parseInfixExpression)
	
	// case block
	p.caseBlock.blockType = token.CASE
	p.caseBlock.startToken = token.COLON
	p.caseBlock.endTokens = append(p.caseBlock.endTokens, token.CASE)
	p.caseBlock.endTokens = append(p.caseBlock.endTokens, token.DEFAULT)
	p.caseBlock.endTokens = append(p.caseBlock.endTokens, token.END)
	
	// default case block
	p.defaultBlock.blockType = token.DEFAULT
	p.defaultBlock.startToken = token.COLON
	p.defaultBlock.endTokens = append(p.defaultBlock.endTokens, token.CASE)
	p.defaultBlock.endTokens = append(p.defaultBlock.endTokens, token.END)
	
	// if block
	p.ifBlock.blockType = token.IF
	p.ifBlock.startToken = token.DO
	p.ifBlock.endTokens = append(p.ifBlock.endTokens, token.END)
	
	// for block
	p.forBlock.blockType = token.FOR
	p.forBlock.startToken = token.DO
	p.forBlock.endTokens = append(p.forBlock.endTokens, token.END)
	
	// function block
	p.functionBlock.blockType = token.FUNCTION
	p.functionBlock.startToken = token.DO
	p.functionBlock.endTokens = append(p.functionBlock.endTokens, token.END)
	
	// scope block
	p.scopeBlock.blockType = token.SCOPE
	p.scopeBlock.startToken = token.DO
	p.scopeBlock.endTokens = append(p.scopeBlock.endTokens, token.END)
	
	// sub block
	p.subBlock.blockType = token.SUB
	p.subBlock.startToken = token.DO
	p.subBlock.endTokens = append(p.subBlock.endTokens, token.END)
	
	// while block
	p.whileBlock.blockType = token.WHILE
	p.whileBlock.startToken = token.DO
	p.whileBlock.endTokens = append(p.whileBlock.endTokens, token.END)
	
	p.nextToken()
	p.nextToken()
	
	return p
}


func (p *TParser) nextToken() {
	p.curToken = p.peekToken
	
	if p.tokenStackCount > 0 {
		p.peekToken = p.popToken()
		
	} else {
		p.peekToken = p.l.NextToken()
	}
}

func (p *TParser) ParseProgram() *ast.TProgram {
	program := &ast.TProgram{}
	program.Statements = []ast.IStatement{}
	
	if p.curTokenIs(token.SEMICOLON) && p.peekTokenIs(token.EOF) {
		return program
	}
	
	for p.curToken.Type != token.EOF {
		stmt := p.parseStatement()
		if stmt != nil {
			program.Statements = append(program.Statements, stmt)
		}
		p.nextToken()
	}
	return program
}

func (p *TParser) parseSelect() ast.IStatement {
	p.registerInfix(token.TO, p.parseFromToExpression)
	
	exp := p.parseSelectStatement()
	
	p.registerPrefix(token.TO, p.parseFromToExpressionInvalid)
	
	return exp
}

func (p *TParser) parseSelectStatement() ast.IStatement {
	ss := &ast.TSelectStatement { Token: p.curToken }
	ss.Cases = []ast.IStatement{}
	
	p.nextToken()
	
	ss.Expression = p.parseExpression(LOWEST)
	
	if !p.expectPeek(token.DO) {
		return nil
	}
	
	p.nextToken()
	
	var def *ast.TDefaultBlock
	
	for !p.curTokenIs(token.END) {
		switch p.curToken.Type {
		case token.CASE:
			cas := &ast.TCaseBlock { Token: p.curToken }
			
			p.nextToken()
			
			for {
				exp := p.parseExpression(LOWEST)
				
				if exp == nil {
					break
				}
				
				cas.Expressions = append(cas.Expressions, exp)
				
				if p.peekToken.Type == token.COMMA {
					p.nextToken()
					p.nextToken()
				} else {
					break
				}
			}
			
			cas.Block = p.parseBlockStatement(p.caseBlock)
			if cas.Block != nil {
				ss.Cases = append(ss.Cases, cas)
			}
			
		case token.DEFAULT:
			// default:
			if def == nil {
				def = &ast.TDefaultBlock { Token: p.curToken }
				ss.Cases = append(ss.Cases, def)
				def.Block = p.parseBlockStatement(p.defaultBlock)
			} else {
				msg := fmt.Sprintf("Syntax Error: duplicate %s in %s block.", token.DEFAULT, token.SELECT)
				p.errors = append(p.errors, msg)
				return nil
			}
		default:
			msg := fmt.Sprintf("Syntax Error: expected %s or %s, got %s instead.", token.CASE, token.END, p.curToken.Type)
			p.errors = append(p.errors, msg)
			return nil
		}
	}
	
	return ss
}


func (p *TParser) parseStatement() ast.IStatement {
	var result ast.IStatement
	
	switch p.curToken.Type {
	case token.SEMICOLON:
		return nil
	case token.CONST:
		result = p.parseConstStatement()
		
	case token.ENUM:
		result = p.parseEnumStatement()
	
	case token.FOR:
		result = p.parseForLoop()
	
	case token.FUNCTION:
		result = p.parseFunctionStatement()
	
	case token.IF:
		result = p.parseIfStatement()
	
	case token.RETURN:
		result = p.parseReturnStatement()
	
	case token.SELECT:
		result = p.parseSelect()
	
	case token.SCOPE:
		result = p.parseScopeStatement()
		
	case token.TYPE:
		result = p.parseTypeStatement()
		
	case token.VAR:
		result = p.parseVarStatement()
	
	case token.WHILE:
		result = p.parseWhileLoop()
		
	default:
		result = p.parseOtherStatement()
		
	}
	
	return result
}


func (p *TParser) pushToken(tok token.TToken) {
	p.tokenStack = append(p.tokenStack, tok)
	p.tokenStackCount += 1
}

func (p *TParser) popToken() token.TToken {
	tok := p.tokenStack[0]
	
	p.tokenStack = p.tokenStack[1:]
	p.tokenStackCount -= 1
	
	return tok
}

func (p *TParser) parseOtherStatement() *ast.TExpressionStatement {
	stmt := p.parseExpressionStatement()
	
	switch exp := stmt.Expression.(type) {
	case *ast.TCallExpression:
		// make it a sub routine call
		stmt.Expression = p.makeCallStatement(exp)
		
	case *ast.TAssignExpression:
		// todo: remove later when this switch is complete
		//fmt.Println("asgnexp")
	
	case *ast.TSubLiteral:
		// is ok
		
	case *ast.TIdentifier:
		//fmt.Println("identif")
		msg := fmt.Sprintf("Syntax Error: incomplete identifier statement.")
		p.errors = append(p.errors, msg)
			
	case *ast.TInfixExpression:
		//fmt.Println("infxexp")
		msg := fmt.Sprintf("Syntax Error: incomplete expression statement.")
		p.errors = append(p.errors, msg)
		
	case *ast.TPrefixExpression:
		//fmt.Println("prfxexp")
		msg := fmt.Sprintf("Syntax Error: incomplete expression statement.")
		p.errors = append(p.errors, msg)
	
	case *ast.TIntegerLiteral:
		msg := fmt.Sprintf("Syntax Error: incomplete integer statement.")
		p.errors = append(p.errors, msg)
	
	case *ast.TFloatLiteral:
		msg := fmt.Sprintf("Syntax Error: incomplete float statement.")
		p.errors = append(p.errors, msg)
		
	case *ast.TStringLiteral:
		msg := fmt.Sprintf("Syntax Error: incomplete string statement.")
		p.errors = append(p.errors, msg)
	
	case *ast.TBoolean:
		msg := fmt.Sprintf("Syntax Error: incomplete boolean statement.")
		p.errors = append(p.errors, msg)
		
	default:
		fmt.Println("unknown statement: " + stmt.TokenLiteral())
		
	}
	
	return stmt
}

func (p *TParser) parseExpression(precedence int) ast.IExpression {
	prefix := p.prefixParseFns[p.curToken.Type]
	
	if prefix == nil {
		p.noPrefixParseFnError(p.curToken.Type)
		return nil
	}
	
	leftExp := prefix()
	
	for !p.peekTokenIs(token.SEMICOLON) && precedence < p.peekPrecedence() {
		infix := p.infixParseFns[p.peekToken.Type]
		
		if infix == nil {
			return leftExp
		}
		
		p.nextToken()
		
		leftExp = infix(leftExp)
	}
	
	return leftExp
}


func (p *TParser) parseVarStatement() *ast.TVarStatement {
	stmt := &ast.TVarStatement { Token: p.curToken }
	
	for {
		p.nextToken()
		
		if !p.curTokenIs(token.IDENTIFIER) {
			msg := fmt.Sprintf("Syntax Error: expected identifier, got %s instead.", p.curToken.Type)
			p.errors = append(p.errors, msg)
			return stmt
		}
		
		name := &ast.TIdentifier{Token: p.curToken, Value: p.curToken.Literal}
		stmt.Names = append(stmt.Names, name)

		p.nextToken()
		
		if p.curTokenIs(token.COLON) || p.curTokenIs(token.SEMICOLON) {
			break
		}
		if !p.curTokenIs(token.COMMA) {
			msg := fmt.Sprintf("Syntax Error: expected COMMA, got %s instead.", p.curToken.Type)
			p.errors = append(p.errors, msg)
			return stmt
		}
	}

	if p.curTokenIs(token.SEMICOLON) { return stmt }

	p.nextToken()
	
	switch p.curToken.Type {
	case token.SEMICOLON, token.FUNCTION:
		msg := fmt.Sprintf("Syntax Error: expected expression, got %s instead.", p.curToken.Type)
		p.errors = append(p.errors, msg)
		return stmt
	}
	
	for {
		v := p.parseExpressionStatement().Expression
		//v := p.parseExpression(LOWEST)
		stmt.Values = append(stmt.Values, v)

		if !p.peekTokenIs(token.COMMA) {
			break
		}
		
		p.nextToken()
		p.nextToken()
	}
	
	if len(stmt.Values) > len(stmt.Names) {
		msg := fmt.Sprintf("Syntax Error: too many expressions.")
		p.errors = append(p.errors, msg)
	}
	
	return stmt
}


func (p *TParser) parseReferenceExpression(left ast.IExpression) ast.IExpression {
	refs := &ast.TReferenceExpression { Token: p.curToken }
	refs.Root = left
	
	for {
		p.nextToken()
		if p.curTokenIs(token.IDENTIFIER) {
			left := &ast.TIdentifier { Token: p.curToken, Value: p.curToken.Literal }
			refs.Names = append(refs.Names, p.curToken.Literal)
			switch p.peekToken.Type {
			case token.LBRACKET:
				p.nextToken()
				refs.Members = append(refs.Members, p.parseArrayIndexExpression(left))
			case token.LLIST:
				refs.Members = append(refs.Members, p.parseListIndexExpression(left))
			default:
				refs.Members = append(refs.Members, left)
			}
		}
		if p.peekToken.Type == token.DOT {
			p.nextToken()
		} else {
			break
		}
	}
	
	return refs
}


func (p *TParser) parseReturnStatement() *ast.TReturnStatement {
	stmt := &ast.TReturnStatement { Token: p.curToken }
	
	p.nextToken()
	
	stmt.ReturnValue = p.parseExpression(LOWEST)
	
	if p.peekTokenIs(token.SEMICOLON) {
		p.nextToken()
	}
	
	return stmt
}


func (p *TParser) parseExpressionStatement() *ast.TExpressionStatement {
	stmt := &ast.TExpressionStatement { Token: p.curToken }
	stmt.Expression = p.parseExpression(LOWEST)
	
	if p.peekTokenIs(token.SEMICOLON) {
		p.nextToken()
	}
	
	return stmt
}


func (p *TParser) parseIdentifier() ast.IExpression {
	
	/* Data types are treated as normal identifiers because they are 
	 * actually builtin functions which are recognized and called by
	 * the evaluator. However, the required parentheses can be omitted
	 * in source code in which case they will be inserted here. ;-)
	 */
	 
	if p.curToken.Type == token.DATATYPE && p.peekToken.Type != token.LPAREN {
		
		// insert '()'
		p.pushToken(token.TToken{Type: token.LPAREN, Literal: token.LPAREN})
		p.pushToken(token.TToken{Type: token.RPAREN, Literal: token.RPAREN})
		
		// save the current peekToken so it won't get lost
		p.pushToken(p.peekToken)
		
		// set peekToken to the first token to be inserted
		p.peekToken = p.popToken()
		
		/* tokens to be inserted are taken from the stack in nextToken()
		 * until the stack is empty.
		 */
	}
	exp := &ast.TIdentifier { Token: p.curToken, Value: p.curToken.Literal }
	
	return exp
}

func (p *TParser) parseIntegerLiteral() ast.IExpression {
	lit := &ast.TIntegerLiteral { Token: p.curToken }
	
	value, err := strconv.ParseInt(p.curToken.Literal, 0, 64)
	if err != nil {
		msg := fmt.Sprintf("could not parse %q as integer", p.curToken.Literal)
		p.errors = append(p.errors, msg)
		return nil
	}
	
	lit.Value = value
	
	return lit
}


func (p *TParser) parseFloatLiteral() ast.IExpression {
	lit := &ast.TFloatLiteral { Token: p.curToken }
	
	value, err := strconv.ParseFloat(p.curToken.Literal, 64)
	if err != nil {
		msg := fmt.Sprintf("could not parse %q as float", p.curToken.Literal)
		p.errors = append(p.errors, msg)
		return nil
	}
	
	lit.Value = value
	
	return lit
}


func (p *TParser) parseStringLiteral() ast.IExpression {
	return &ast.TStringLiteral { Token: p.curToken, Value: p.curToken.Literal }
}


func (p *TParser) parseSubLiteral() ast.IExpression {
	stmt := &ast.TSubLiteral { Token: p.curToken }
	
	p.nextToken()
	
	stmt.Name = &ast.TIdentifier{ Token: p.curToken, Value: p.curToken.Literal }
	
	if !p.expectPeek(token.LPAREN) {
		return nil
	}
	
	stmt.Parameters = p.parseParameters()	
	stmt.Body = p.parseBlockStatement(p.subBlock)
	
	if p.peekTokenIs(token.SEMICOLON) {
		p.nextToken()
	}

	return stmt
}


func (p *TParser) parsePrefixExpression() ast.IExpression {
	exp := &ast.TPrefixExpression { Token: p.curToken, Operator: p.curToken.Literal }
	
	p.nextToken()
	
	exp.Right = p.parseExpression(PREFIX)
	return exp
}

func (p *TParser) parseInfixExpression(left ast.IExpression) ast.IExpression {
	exp := &ast.TInfixExpression { Token: p.curToken, Operator: p.curToken.Literal, Left: left }
	
	pre := p.curPrecedence()
	
	p.nextToken()
	
	exp.Right = p.parseExpression(pre)
	
	return exp
}

func (p *TParser) parseBoolean() ast.IExpression {
	return &ast.TBoolean { Token: p.curToken, Value: p.curTokenIs(token.TRUE) }
}


func (p *TParser) parseGroupedExpression() ast.IExpression {
	p.nextToken()
	
	exp := p.parseExpression(LOWEST)
	
	if !p.expectPeek(token.RPAREN) {
		return nil
	}
	return exp
}



func (p *TParser) parseIfStatement() ast.IStatement {
	stmt := &ast.TIfStatement{Token: p.curToken}
	stmt.Conditions = p.parseElseIfConditions(stmt)
	
	return stmt
}


func (p *TParser) parseElseIfConditions(stmt *ast.TIfStatement) []*ast.TIfCondition {
	exp := []*ast.TIfCondition { p.parseIfCondition() }
	
	for p.peekTokenIs(token.ELSEIF) || p.peekTokenIs(token.ELSE) {
				
		p.nextToken()
		
		if p.curTokenIs(token.ELSE) {
			
			// else
			stmt.Default = p.parseBlockStatement(p.ifBlock)
			break
			
		} else { // elseif
			
			exp = append(exp, p.parseIfCondition())
		}
	}

	return exp
}

func (p *TParser) parseIfCondition() *ast.TIfCondition {
	exp := &ast.TIfCondition { Token: p.curToken }
	
	p.nextToken()
	exp.Condition = p.parseExpression(LOWEST)
	exp.Body = p.parseBlockStatement(p.ifBlock)
	
	return exp
}


func (p *TParser) parseBlockStatement(bt TBlockType) *ast.TBlockStatement {
	block := &ast.TBlockStatement { Token: p.curToken }
	block.Statements = []ast.IStatement{}
	
	if !p.expectPeek(bt.startToken) {
		return nil
	}
	
	p.nextToken()
	
	blockLoop:
	for {
		
		for et := range bt.endTokens {
			if p.curTokenIs(bt.endTokens[et]) || p.curTokenIs(token.EOF) {
				break blockLoop
			}
		}
		
		// invalid tokens?
		switch p.curToken.Type {
		case token.ELSEIF, token.ELSE:
			if bt.blockType == token.IF {
				p.tokenError(token.END, p.curToken.Type)
			} else {
				p.tokenError(token.IF, p.curToken.Type)
			}
			return nil
		case token.CASE:
			if bt.blockType != token.CASE && bt.blockType != token.DEFAULT {
				p.tokenError(token.SELECT, p.curToken.Type)
			}
			return nil
		case token.DEFAULT:
			if bt.blockType != token.CASE {
				p.tokenError(token.SELECT, p.curToken.Type)
			}
			return nil
		}
		
		// raise error if "exit("
		if p.curTokenIs(token.EXIT) && p.peekTokenIs(token.LPAREN) {
			msg := fmt.Sprintf("Invalid function call '%s'", p.curToken.Literal)
			p.errors = append(p.errors, msg)
			return nil
		}
		
		// raise error if "continue("
		if p.curTokenIs(token.CONTINUE) && p.peekTokenIs(token.LPAREN) {
			msg := fmt.Sprintf("Invalid function call '%s'", p.curToken.Literal)
			p.errors = append(p.errors, msg)
			return nil
		}
		
		stmt := p.parseStatement()
		if stmt != nil {
			block.Statements = append(block.Statements, stmt)
		}
		
		p.nextToken()
	}
	
	return block
}


func (p *TParser) parseFunctionLiteral() ast.IExpression {
	exp := &ast.TFunctionLiteral {Token: p.curToken }
	
	if !p.expectPeek(token.LPAREN) {
		return nil
	}
	
	exp.Parameters = p.parseParameters()
	
	// function datatype?
	if !p.expectPeek(token.COLON) {
		msg := fmt.Sprintf("Function's return type not specified.")
		p.errors = append(p.errors, msg)
		fmt.Println(len(p.Errors()))
		return nil
	}
	
	p.nextToken()
	
	exp.ReturnType = p.parseExpression(LOWEST)
	exp.Body = p.parseBlockStatement(p.functionBlock)
	
	return exp
}


func (p *TParser) parseParameters() map[ast.IExpression]*ast.TParameters {
	params := make(map[ast.IExpression]*ast.TParameters)
	value := &ast.TParameters{}
	
	p.nextToken()
	
	for !p.curTokenIs(token.RPAREN) {
		value.Ref = p.curTokenIs(token.REF)
		if value.Ref {
			p.nextToken()
		}
		
		key := p.parseExpression(LOWEST)
		if !p.expectPeek(token.COLON) {
			return nil
		}
		
		p.nextToken()
		
		value.Parameter = p.parseExpression(LOWEST)
		params[key] = value
		
		p.nextToken()
		
		if p.curTokenIs(token.COMMA) {
			p.nextToken()
		}
	}
	
	return params
}


func (p *TParser) parseCallExpression(rt ast.IExpression) ast.IExpression {
	exp := &ast.TCallExpression { 
		Token: p.curToken,
		Statement: false,
		Routine: rt,
		Arguments: p.parseExpressionList(token.RPAREN),
	}
	
	return exp
}


func (p *TParser) makeCallStatement(ce *ast.TCallExpression) ast.IExpression {
	// set statement flag
	
	ce.Statement = true
	
	return ce
}


func (p *TParser) noPrefixParseFnError(t token.TokenType) {
	msg := fmt.Sprintf("no prefix parse function for %s found", t)
	p.errors = append(p.errors, msg)
}

func (p *TParser) curTokenIs(t token.TokenType) bool {
	return p.curToken.Type == t
}

func (p *TParser) peekTokenIs(t token.TokenType) bool {
	return p.peekToken.Type == t
}

func (p *TParser) expectPeek(t token.TokenType) bool {
	if p.peekTokenIs(t) {
		p.nextToken()
		return true
	}
	p.tokenError(t, p.peekToken.Type)
	return false
}

func (p *TParser) Errors() []string {
	return p.errors
}

func (p *TParser) tokenError(exp token.TokenType, got token.TokenType) {
	msg := fmt.Sprintf("expected %s, got %s instead", exp, got)
	p.errors = append(p.errors, msg)
}

func (p *TParser) registerPrefix(t token.TokenType, fn prefixParseFn) {
	p.prefixParseFns[t] = fn
}

func (p *TParser) registerInfix(t token.TokenType, fn infixParseFn) {
	p.infixParseFns[t] = fn
}

func (p *TParser) peekPrecedence() int {
	if p, ok := precedences[p.peekToken.Type]; ok {
		return p
	}
	
	return LOWEST
}

func (p *TParser) curPrecedence() int {
	if p, ok := precedences[p.curToken.Type]; ok {
		return p
	}
	return LOWEST
}


func (p *TParser) parseFunctionStatement() ast.IStatement {
	FnStmt := &ast.TFunctionStatement{ Token: p.curToken }
	
	p.nextToken()
	
	FnStmt.Name = &ast.TIdentifier{ Token: p.curToken, Value: p.curToken.Literal }
	
	lit := p.parseFunctionLiteral()
	if lit == nil {
		msg := fmt.Sprintf("Syntax Error: function without function body")
		p.errors = append(p.errors, msg)
		return FnStmt
	}
	
	FnStmt.FunctionLiteral = lit.(*ast.TFunctionLiteral)

	if p.peekTokenIs(token.SEMICOLON) {
		p.nextToken()
	}

	return FnStmt
}

/*
func (p *TParser) parseArrayExpression() ast.IExpression {
	arr := &ast.TArrayLiteral { Token: p.curToken }
	arr.Elements = p.parseExpressionList(token.RBRACKET)
	
	return arr
}
*/

/*
func (p *TParser) parseListExpression() ast.IExpression {
	lst := &ast.TListLiteral { Token: p.curToken }
	lst.Elements = p.parseExpressionList(token.RLIST)
	
	return lst
}
*/


func (p *TParser) parseAssignExpression(name ast.IExpression) ast.IExpression {
	exp := &ast.TAssignExpression{Token: p.curToken}
	
	if iname, ok := name.(*ast.TIdentifier); ok {
		exp.Name = iname
		
	} else if indexExp, ok := name.(*ast.TIndexExpression); ok {
		
		switch indexExp.Left.(type) {
		case *ast.TIdentifier:
			exp.Name = indexExp
			
		default:
			msg := fmt.Sprintf("Syntax Error: Index assignment expects an identifier")
			p.errors = append(p.errors, msg)
			return exp
		}
	} else if rname, ok := name.(*ast.TReferenceExpression); ok {
			exp.Name = rname
	} else {
		msg := fmt.Sprintf("Syntax Error: %s is not an identifier", name.TokenLiteral())
		p.errors = append(p.errors, msg)
		
		return exp
	}

	p.nextToken()
	exp.Value = p.parseExpression(LOWEST)

	return exp
}


func (p *TParser) parseExpressionList(end token.TokenType) []ast.IExpression {
	list := []ast.IExpression{}
	
	if p.peekTokenIs(end) {
		p.nextToken()
		return list
	}
	
	p.nextToken()
		
	list = append(list, p.parseExpression(LOWEST))
	
	for p.peekTokenIs(token.COMMA) {
		p.nextToken()
		p.nextToken()
		list = append(list, p.parseExpression(LOWEST))
	}
	
	if !p.expectPeek(end) {
		return nil
	}
	
	return list
}

func (p *TParser) parseArrayIndexExpression(left ast.IExpression) ast.IExpression {
	exp := &ast.TIndexExpression { Token: p.curToken, Left: left }
	
	p.nextToken()
	
	exp.Index = p.parseExpression(LOWEST)
	
	if !p.expectPeek(token.RBRACKET) {
		return nil
	}
		
	return exp
}


func (p *TParser) parseListIndexExpression(left ast.IExpression) ast.IExpression {
	exp := &ast.TIndexExpression { Token: p.curToken, Left: left }
	
	p.nextToken()
	exp.Index = p.parseExpression(LOWEST)
	
	if !p.expectPeek(token.RLIST) {
		return nil
	}
	
	return exp
}


func (p *TParser) parseMapLiteral() ast.IExpression {
	hash := &ast.TMapLiteral { Token: p.curToken }
	hash.Pairs = make(map[ast.IExpression]ast.IExpression)
	
	for !p.peekTokenIs(token.RBRACE) {
		p.nextToken()
		key := p.parseExpression(LOWEST)
		
		if !p.expectPeek(token.COLON) {
			return nil
		}
		
		p.nextToken()
		value := p.parseExpression(LOWEST)
		
		hash.Pairs[key] = value
		
		if !p.peekTokenIs(token.RBRACE) && !p.expectPeek(token.COMMA) {
			return nil
		}
	}
	if !p.expectPeek(token.RBRACE) {
		return nil
	}
	return hash
}

func (p *TParser) parseForLoopStatement() ast.IStatement {
	var iter *ast.TIdentifier
	
	p.nextToken()
	
	/*
	if p.curToken.Type != token.IDENTIFIER {
		p.tokenError(token.IDENTIFIER, p.curToken.Type)
		return nil
	}
	*/
	
	name := p.parseIdentifier()
	if Name, ok := name.(*ast.TIdentifier); !ok {
		msg := fmt.Sprintf("Syntax Error: invalid for-loop identifier.")
		p.errors = append(p.errors, msg)
		return nil
	} else {
		iter = Name
	}
	
	p.nextToken()
	
	if p.curToken.Type != token.ASSIGN {
		p.tokenError(token.ASSIGN, p.curToken.Type)
		return nil
	}
	
	p.nextToken()
		
	rang := p.parseExpression(LOWEST)

	if _, ok := rang.(*ast.TFromToExpression); !ok {
		msg := fmt.Sprintf("Syntax Error: invalid range expression in for-loop.")
		p.errors = append(p.errors, msg)
		return nil
	}
	
	var step ast.IExpression
	
	if p.peekToken.Type == token.STEP {
		p.nextToken()
		p.nextToken()
		step = p.parseExpression(LOWEST)
	}
	
	block := p.parseBlockStatement(p.forBlock)
	if block == nil {
		return nil
	}
	
	floop := &ast.TForLoopStatement { 
		Token: p.curToken,
		Iterator: iter,
		Range: rang,
		Step: step,
		Block: block,
	}
	
	return floop
}


func (p *TParser) parseForLoop() ast.IStatement {
	p.registerInfix(token.TO, p.parseFromToExpression)
	p.registerPrefix(token.STEP, p.parseStepExpression)
	p.registerPrefix(token.EXIT, p.parseExitExpression)
	p.registerPrefix(token.CONTINUE, p.parseContinueExpression)
	
	exp := p.parseForLoopStatement()
	
	p.registerPrefix(token.CONTINUE, p.parseContinueExpressionInvalid)
	p.registerPrefix(token.EXIT, p.parseExitExpressionInvalid)
	p.registerPrefix(token.STEP, p.parseStepExpressionInvalid)
	p.registerPrefix(token.TO, p.parseFromToExpressionInvalid)
	
	return exp
}


func (p *TParser) parseStepExpression() ast.IExpression {
	return &ast.TStepExpression { Token: p.curToken }
}


func (p *TParser) parseStepExpressionInvalid() ast.IExpression {
	msg := fmt.Sprintf("Syntax Error: 'step' without 'for'")
	p.errors = append(p.errors, msg)

	return &ast.TStepExpression { Token: p.curToken }
}


func (p *TParser) parseFromToExpression(left ast.IExpression) ast.IExpression {
	exp := &ast.TFromToExpression { Token: p.curToken }
	
	exp.Expression = p.parseInfixExpression(left)
	
	return exp
}


func (p *TParser) parseFromToExpressionInvalid() ast.IExpression {
	msg := fmt.Sprintf("Syntax Error: 'to' without 'for' or 'case'")
	p.errors = append(p.errors, msg)

	return &ast.TFromToExpression { Token: p.curToken }
}


func (p *TParser) parseScopeStatement() ast.IStatement {
	p.registerPrefix(token.EXIT, p.parseExitExpression)
	
	stmt := &ast.TScopeStatement{ Token: p.curToken }
	stmt.Block = p.parseBlockStatement(p.scopeBlock)
	
	p.registerPrefix(token.EXIT, p.parseExitExpressionInvalid)
	
	if p.peekTokenIs(token.SEMICOLON) {
		p.nextToken()
	}
	
	return stmt
}


func (p *TParser) parseWhileLoopStatement() ast.IStatement {
	stmt := &ast.TWhileLoopStatement{ Token: p.curToken }
	
	p.nextToken()
	
	stmt.Condition = p.parseExpression(LOWEST)
	stmt.Block = p.parseBlockStatement(p.whileBlock)
	
	if p.peekTokenIs(token.SEMICOLON) {
		p.nextToken()
	}
	
	return stmt
}


func (p *TParser) parseWhileLoop() ast.IStatement {
	p.registerPrefix(token.EXIT, p.parseExitExpression)
	p.registerPrefix(token.CONTINUE, p.parseContinueExpression)

	wl := p.parseWhileLoopStatement()
	
	p.registerPrefix(token.EXIT, p.parseExitExpressionInvalid)
	p.registerPrefix(token.CONTINUE, p.parseContinueExpressionInvalid)

	return wl
}


func (p *TParser) parseExitExpressionInvalid() ast.IExpression {
	msg := fmt.Sprintf("'exit' outside of loop context")
	p.errors = append(p.errors, msg)

	return &ast.TExitExpression{Token: p.curToken}
}


func (p *TParser) parseExitExpression() ast.IExpression {
	return &ast.TExitExpression{Token: p.curToken}
}


func (p *TParser) parseContinueExpressionInvalid() ast.IExpression {
	msg := fmt.Sprintf("'continue' outside of loop context")
	p.errors = append(p.errors, msg)

	return &ast.TContinueExpression{Token: p.curToken}
}


func (p *TParser) parseContinueExpression() ast.IExpression {
	return &ast.TContinueExpression{Token: p.curToken}
}


func (p *TParser) parseTypeStatement() ast.IStatement {
	t := &ast.TTypeStatement{Token: p.curToken}
	t.Members = make(map[ast.IExpression]ast.IExpression)
	
	p.nextToken()
		
	exp := p.parseIdentifier()
	if name, ok := exp.(*ast.TIdentifier); !ok {
		msg := fmt.Sprintf("Syntax Error: invalid type identifier.")
		p.errors = append(p.errors, msg)
		return nil
	} else {
		t.Name = name
	}
		
	if !p.expectPeek(token.OF) {
		return nil
	}
	
	p.nextToken()
	
	for !p.curTokenIs(token.END){
		key := p.parseExpression(LOWEST)
		if !p.expectPeek(token.COLON) {
			return nil
		}
		
		p.nextToken()
		
		t.Members[key] = p.parseExpression(LOWEST)
		
		p.nextToken()
		
		if p.curTokenIs(token.SEMICOLON) {
			p.nextToken()
		}
	}
	
	if len(t.Members) == 0 {
		msg := fmt.Sprintf("Syntax Error: type without member(s)")
		p.errors = append(p.errors, msg)
	}
	
	if p.peekTokenIs(token.SEMICOLON) {
		p.nextToken()
	}
	
	return t
}


func (p *TParser) parseConstStatement() *ast.TConstStatement {
	stmt := &ast.TConstStatement { Token: p.curToken, Grouped: false, Named: false }
	
	p.nextToken()
	
	// unnamed const block
	if p.curTokenIs(token.OF) {
		stmt.Grouped = true
		
	// named const block
	} else if p.peekTokenIs(token.OF) {
		if !p.curTokenIs(token.IDENTIFIER) {
			msg := fmt.Sprintf("Syntax Error: expected identifier, got %s instead.", p.curToken.Type)
			p.errors = append(p.errors, msg)
			return stmt
		}
		stmt.Name = &ast.TIdentifier{Token: p.curToken, Value: p.curToken.Literal}
		stmt.Grouped = true
		stmt.Named = true
		
		p.nextToken()
	
	// simple const declaration
	} else if p.peekTokenIs(token.ASSIGN) {
		if !p.curTokenIs(token.IDENTIFIER) {
			msg := fmt.Sprintf("Syntax Error: expected identifier, got %s instead.", p.curToken.Type)
			p.errors = append(p.errors, msg)
			return stmt
		}
		stmt.Names = append(stmt.Names, &ast.TIdentifier{Token: p.curToken, Value: p.curToken.Literal})
		
		p.nextToken()
		p.nextToken()
		
		stmt.Values = append(stmt.Values, p.parseExpressionStatement().Expression)
		
		return stmt
		
	// syntaxt error
	} else {
		msg := fmt.Sprintf("Syntax Error: invalid const declaration")
		p.errors = append(p.errors, msg)
		return stmt
	}
	
	p.nextToken()
	
	// process const block until token.END
	for !p.curTokenIs(token.END) {
		
		if !p.curTokenIs(token.IDENTIFIER) {
			msg := fmt.Sprintf("Syntax Error: expected identifier, got %s instead.", p.curToken.Type)
			p.errors = append(p.errors, msg)
			return stmt
		}
		
		name := &ast.TIdentifier{Token: p.curToken, Value: p.curToken.Literal}
		stmt.Names = append(stmt.Names, name)

		p.nextToken()		
		
		if !p.curTokenIs(token.ASSIGN) {
			msg := fmt.Sprintf("Syntax Error: expected assign operator, got %s instead.", p.curToken.Type)
			p.errors = append(p.errors, msg)
			return stmt
		}
		
		p.nextToken()
		
		v := p.parseExpressionStatement().Expression
		stmt.Values = append(stmt.Values, v)
		
		p.nextToken()
		
	}
	
	if len(stmt.Names) == 0 {
		msg := fmt.Sprintf("Syntax Error: constant block without definition(s)")
		p.errors = append(p.errors, msg)
	}
	
	if p.peekTokenIs(token.SEMICOLON) {
		p.nextToken()
	}
	
	return stmt
}


func (p *TParser) parseEnumStatement() *ast.TEnumStatement {
	stmt := &ast.TEnumStatement { Token: p.curToken, Named: false }
	
	p.nextToken()
	
	if !p.curTokenIs(token.OF) {
		if !p.curTokenIs(token.IDENTIFIER) {
			msg := fmt.Sprintf("Syntax Error: expected identifier, got %s instead.", p.curToken.Type)
			p.errors = append(p.errors, msg)
			return stmt
		} else if !p.peekTokenIs(token.OF) {
			msg := fmt.Sprintf("Syntax Error: invalid enum declaration")
			p.errors = append(p.errors, msg)
			return stmt
		}		
		stmt.Name = &ast.TIdentifier{Token: p.curToken, Value: p.curToken.Literal}
		stmt.Named = true
		
		p.nextToken()	
	}
	
	p.nextToken()
	
	// process enum block until token.END
	for !p.curTokenIs(token.END) {
		
		if !p.curTokenIs(token.IDENTIFIER) {
			msg := fmt.Sprintf("Syntax Error: expected identifier, got %s instead.", p.curToken.Type)
			p.errors = append(p.errors, msg)
			return stmt
		}
		
		name := &ast.TIdentifier{Token: p.curToken, Value: p.curToken.Literal}
		stmt.Names = append(stmt.Names, name)

		p.nextToken()		
		
		if p.curTokenIs(token.ASSIGN) {
			p.nextToken()
			v := p.parseExpressionStatement().Expression
			stmt.Values = append(stmt.Values, v)
			p.nextToken()
		} else {
			stmt.Values = append(stmt.Values, nil)
			if p.curTokenIs(token.SEMICOLON) {
				p.nextToken()
			}
		}		
	}
	
	if len(stmt.Names) == 0 {
		msg := fmt.Sprintf("Syntax Error: enumeration block without definition(s)")
		p.errors = append(p.errors, msg)
	}
	
	if p.peekTokenIs(token.SEMICOLON) {
		p.nextToken()
	}
	
	return stmt
}
