// object/error.go

package object

import (
	"fmt"
)

type NullActionType uint8

var RaiseNULL, AllowNULL NullActionType = 1, 0

type TError struct {
	Message string
}

func (e *TError) Type() ObjectType { return ERROR_OBJ }
func (e *TError) Inspect() string { return "ERROR: " + e.Message }

func NewError(format string, a ...interface{}) *TError {
	return &TError { Message: fmt.Sprintf(format, a...) }
}

func ObjectState(obj IObject, nullAction NullActionType) (IObject, bool) {
	if obj == nil {
		return obj, false
	}
	
	if obj == NULL {
		if nullAction == RaiseNULL {
			obj = ERR_NULL_OBJECT_EXCEPTION()
		}
	}
	
	return obj, (obj.Type() != ERROR_OBJ)
}


func ERR_CONST_ASSIGNMENT() *TError {
	return NewError("constants cannot be assigned a value")
}


func ERR_ENUM_ASSIGNMENT() *TError {
	return NewError("enumerations cannot be assigned a value")
}


func ERR_DUPLICATE_DEFINITION(name string) *TError {
	return NewError("duplicate definition %s", name)
}

func ERR_INVALID_OPERATOR(p string) *TError {
	msg := "invalid operator: %s "
	
	/*
	if lh != nil && rh != nil {
		return NewError(msg + "%s %s", lh.Type(), p, rh.Type())
	} else if rh != nil {
		return NewError(msg + "%s", p, rh.Type())
	}
	*/
	
	return NewError(msg, p)
}


func ERR_INVALID_ENUM_VALUE() *TError {
	return NewError("invalid enumeration value")
}


func ERR_INVALID_EXPRESSION(p string, lh, rh IObject) *TError {
	msg := "invalid expression: %s "
	
	if lh != nil && rh != nil {
		return NewError(msg + "%s %s", lh.Type(), p, rh.Type())
	} else if rh != nil {
		return NewError(msg + "%s", p, rh.Type())
	}
	
	return NewError(msg, p)
}


func ERR_INVALID_BOOLEAN_EXPRESSION(p string, lh, rh IObject) *TError {
	msg := "invalid boolean expression: %s "
	
	if lh != nil && rh != nil {
		return NewError(msg + "%s %s", lh.Type(), p, rh.Type())
	} else if rh != nil {
		return NewError(msg + "%s", p, rh.Type())
	}
	
	return NewError(msg, p)
}


func ERR_EXPRESSION_TYPE_MISMATCH(p string, lh, rh IObject) *TError {
	msg := "type mismatch: %s "
	
	if lh != nil && rh != nil {
		return NewError(msg + "%s %s", lh.Type(), p, rh.Type())
	} else if rh != nil {
		return NewError(msg + "%s", p, rh.Type())
	}
	
	return NewError(msg, p)
}


func ERR_FUNCTION_NOT_DEFINED() *TError {
	return NewError("function not defined")
}


func ERR_NOT_A_FUNCTION() *TError {
	return NewError("not a function")
}


func ERR_NOT_A_SUB() *TError {
	
	return NewError("not a sub-routine")
}


func ERR_ARG_COUNT_MISMATCH() *TError {
	return NewError("argument count mismatch")
}


func ERR_INVALID_ARG_TYPE(o IObject) *TError {
	return NewError("invalid argument type %s", o.Type())
}


func ERR_INVALID_ARG_VALUE(o IObject) *TError {
	return NewError("invalid argument value %d", o.Inspect())
}


func ERR_INVALID_DATA_TYPE(s string) *TError {
	return NewError("invalid datatype: %s", s)
}


func ERR_INVALID_FUNCTION_DATA_TYPE() *TError {
	return NewError("invalid function datatype")
}


func ERR_INVALID_IDENTIFIER(s string) *TError {
	return NewError("invalid identifier: %s", s)
}


func ERR_INVALID_INDEX(o IObject) *TError {
	return NewError("invalid index %s", o.Inspect())
}


func ERR_INVALID_INDEX_OPERATOR(o IObject) *TError {
	return NewError("invalid index operator %s", o.Type())
}


func ERR_INVALID_HASH_KEY(o IObject) *TError {
	return NewError("invalid hash key %s", o.Type())
}


func ERR_TYPE_MISMATCH(left, right IObject) *TError {
	return NewError("type mismatch %s versus %s", left.Type(), right.Type())
}


func ERR_SUB_NOT_DEFINED() *TError {
	return NewError("sub-routine not defined")
}


func ERR_SUBSCRIPT_OUT_OF_RANGE(o IObject) *TError {
	return NewError("subscript out of range %s", o.Inspect())
}


func ERR_INVALID_ASSIGNMENT(val IObject) *TError {
	return NewError("invalid assignment %s", val.Inspect())
}


func ERR_NULL_OBJECT_EXCEPTION() *TError {
	return NewError("NULL object exception.")
}


func ERR_DEFINITION_COUNT_MISMATCH() *TError {
	return NewError("Definition count mismatch.")
}


func ERR_NOT_ENOUGH_MEMORY(o IObject) *TError {
	return NewError("not enough memory %s", o.Inspect())
}


func ERR_INVALID_MAP_KEY(o IObject) *TError {
	return NewError("invalid map key %s", o.Inspect())
}


func ERR_INVALID_DATA_TYPE_COMPARISON(value, low, high IObject) *TError {
	msg := "invalid data type comparison %s to %s"
	if high != nil {
		msg += " and %s"
		return NewError(msg, value.Type(), low.Type(), high.Type())
	}
	
	return NewError(msg, value.Type(), low.Type())
}


func ERR_INVALID_ARITHMETIC(value1, value2 IObject, operator string) *TError {
	return NewError("invalid arithmetic %s %t %s", value1.Type(), operator, value2.Type())
}


func ERR_INVALID_TYPE() *TError {
	return NewError("invalid type")
}


func ERR_INVALID_TYPE_MEMBER(m string) *TError {
	return NewError("invalid type member %s", m)
}


func ERR_FUNCTION_WITHOUT_RETURN() *TError {
	return NewError("function does not return a value")
}


func ERR_RETURN_TYPE_UNDEFINED() *TError {
	return NewError("return type not defined")
}


func ERR_RETURN_WITHOUT_FUNCTION() *TError {
	return NewError("return without function")
}


func ERR_UNKNOWN_IDENTIFIER(name string) *TError {
	return NewError("unknown identifier: " + name)
}


func ERR_TYPES_DO_NOT_MATCH(lname, rname string) *TError {
	return NewError("types do not match %s versus %s", lname, rname)
}


func ERR_INVALID_STATEMENT() *TError {
	return NewError("invalid statement")
}


func ERR_STACK_OVERFLOW() *TError {
	return NewError("stack overflow")
}


/*
func ERR_ARRAY_TYPES_DO_NOT_MATCH(left, right IObject) *TError {
	return NewError("types do not match %s versus %s", left.Type(), right.Type())
}
*/

