// object/builtin.go

package object

import (
	"fmt"
	"math"
	"strings"
)

const (
	// misc. functions
	BUILTIN_ABS				= "abs"
	BUILTIN_APPEND		= "append"
	BUILTIN_ARRAY			= "array"
	BUILTIN_ASC				= "asc"
	BUILTIN_CHR				= "chr"
	BUILTIN_INSTR			= "instr"
	BUILTIN_LCASE			= "lcase"
	BUILTIN_LEFT			= "left"
	BUILTIN_LEN				= "len"
	BUILTIN_LIST			= "list"
	BUILTIN_MAP				= "map"
	BUILTIN_MID				= "mid"
	BUILTIN_NIL				= "nil"
	BUILTIN_PCASE			= "pcase"
	BUILTIN_PREPEND		= "prepend"
	BUILTIN_PRINT			= "print"
	BUILTIN_RIGHT			= "right"
	BUILTIN_SQR				= "sqr"
	BUILTIN_TYPOF			= "typof"
	BUILTIN_UCASE			= "ucase"
	
	// data types
	BUILTIN_BOOLEAN		= "boolean"
	BUILTIN_BYTE			= "byte"
	BUILTIN_DOUBLE		= "double"
	BUILTIN_SINGLE		= "single"
	BUILTIN_INTEGER		= "integer"
	BUILTIN_LONG			= "long"
	BUILTIN_SHORT			= "short"
	BUILTIN_STRING		= "string"
	BUILTIN_UBYTE			= "ubyte"
	BUILTIN_UINTEGER	= "uinteger"
	BUILTIN_USHORT		= "ushort"
	BUILTIN_ULONG			= "ulong"
	BUILTIN_FUNC			= "func"
)

type FnBuiltinFunction func(line string, args ...IObject) IObject

type TBuiltin struct {
	Fn 				FnBuiltinFunction
	Statement	bool
}

func (b *TBuiltin) Type() ObjectType { return BUILTIN_OBJ }
func (b *TBuiltin) Inspect() string { return "builtin function" }

func BuiltinByName(name string) (*TBuiltin, bool) {
	for _, def := range Builtins {
		if def.Name == name {
			return def.Builtin, true
		}
	}
	return nil, false
}


var Builtins = []struct {
	Name string
	Builtin *TBuiltin
}{
		
	// ========================== DATA TYPES ===========================
	
	{
		// 0.NIL
		BUILTIN_NIL, &TBuiltin { Statement: false, 
			Fn: func(line string, args ...IObject) IObject {
				arc := len(args)
				if arc != 0 {
					return ERR_ARG_COUNT_MISMATCH()
				}
				return NULL
			},
		},
	},
	
	{
		// 1.BOOLEAN
		BUILTIN_BOOLEAN, &TBuiltin { Statement: false, 
			Fn: func(line string, args ...IObject) IObject {
				arc := len(args)
				if arc > 1 {
					return ERR_ARG_COUNT_MISMATCH()
				}
				
				if arc == 1 {
					arg, ok := ObjectState(args[0], RaiseNULL)
					if !ok { return arg }
					
					if arg, ok = BooleanObject(arg, EXPLICIT); !ok {
						return ERR_INVALID_ARG_TYPE(arg)
					}
					
					return arg
				}
				
				return FALSE //BooleanValue(false)
			},
		},
	},
	
	{
		// 2.BYTE (int8)
		BUILTIN_BYTE, &TBuiltin { Statement: false, 
			Fn: func(line string, args ...IObject) IObject {
				arc := len(args)
				if arc > 1 {
					return ERR_ARG_COUNT_MISMATCH()
				}
				
				if arc == 1 {
					arg, ok := ObjectState(args[0], RaiseNULL)
					if !ok { return arg }
					
					if arg, ok = ByteObject(arg, EXPLICIT); !ok {				
						return ERR_INVALID_ARG_TYPE(arg)
					}
					
					return arg
				}
				
				return BYTE // ByteValue(0)
			},
		},
	},
	
	{
		// 3.DOUBLE (float64)
		BUILTIN_DOUBLE, &TBuiltin { Statement: false, 
			Fn: func(line string, args ...IObject) IObject {
				arc := len(args)
				if arc > 1 {
					return ERR_ARG_COUNT_MISMATCH()
				}
				
				if arc == 1 {
					arg, ok := ObjectState(args[0], RaiseNULL)
					if !ok { return arg }
					
					if arg, ok = DoubleObject(arg, EXPLICIT); !ok {				
						return ERR_INVALID_ARG_TYPE(arg)
					}
					
					return arg
				}
				
				return DOUBLE // DoubleValue(0)
			},
		},
	},
	
	{
		// 4.SINGLE (float32)
		BUILTIN_SINGLE, &TBuiltin { Statement: false, 
			Fn: func(line string, args ...IObject) IObject {
				arc := len(args)
				if arc > 1 {
					return ERR_ARG_COUNT_MISMATCH()
				}
				
				if arc == 1 {
					arg, ok := ObjectState(args[0], RaiseNULL)
					if !ok { return arg }
					
					if arg, ok = SingleObject(arg, EXPLICIT); !ok {				
						return ERR_INVALID_ARG_TYPE(arg)
					}
					
					return arg
				}
				
				return SINGLE //SingleValue(0)
			},
		},
	},
	
	{
		// 5.INTEGER (int64) default
		BUILTIN_INTEGER, &TBuiltin { Statement: false, 
			Fn: func(line string, args ...IObject) IObject {
				arc := len(args)
				if arc > 1 {
					return ERR_ARG_COUNT_MISMATCH()
				}
				
				if arc == 1 {
					arg, ok := ObjectState(args[0], RaiseNULL)
					if !ok { return arg }
					
					if arg, ok = IntegerObject(arg, EXPLICIT); !ok {				
						return ERR_INVALID_ARG_TYPE(arg)
					}
					
					return arg
				}
				
				return INTEGER //IntegerValue(0)
			},
		},
	},
	
	{
		// 6.LONG (int32)
		BUILTIN_LONG, &TBuiltin { Statement: false, 
			Fn: func(line string, args ...IObject) IObject {
				arc := len(args)
				if arc > 1 {
					return ERR_ARG_COUNT_MISMATCH()
				}
				
				if arc == 1 {
					arg, ok := ObjectState(args[0], RaiseNULL)
					if !ok { return arg }
					
					if arg, ok = LongObject(arg, EXPLICIT); !ok {				
						return ERR_INVALID_ARG_TYPE(arg)
					}
					
					return arg
				}
				
				return LONG //LongValue(0)
			},
		},
	},
	
	{
		// 7.SHORT (int16)
		BUILTIN_SHORT, &TBuiltin { Statement: false, 
			Fn: func(line string, args ...IObject) IObject {
				arc := len(args)
				if arc > 1 {
					return ERR_ARG_COUNT_MISMATCH()
				}
				
				if arc == 1 {
					arg, ok := ObjectState(args[0], RaiseNULL)
					if !ok { return arg }
					
					if arg, ok = ShortObject(arg, EXPLICIT); !ok {
						return ERR_INVALID_ARG_TYPE(arg)
					}
					
					return arg
				}
				
				return SHORT //ShortValue(0)
			},
		},
	},
	
	{
		// 8.STRING
		BUILTIN_STRING, &TBuiltin { Statement: false, 
			Fn: func(line string, args ...IObject) IObject {
				arc := len(args)
				if arc > 1 {
					return ERR_ARG_COUNT_MISMATCH()
				}
				
				if arc == 1 {
					arg, ok := ObjectState(args[0], RaiseNULL)
					if !ok { return arg }
					
					if arg, ok = AssignString(arg, EXPLICIT); !ok {
						return ERR_INVALID_ARG_TYPE(arg)
					}
					
					return arg
				}
				
				return STRING //StringValue("")
			},
		},
	},
	
	{
		// 9.UBYTE (uint8)
		BUILTIN_UBYTE, &TBuiltin { Statement: false, 
			Fn: func(line string, args ...IObject) IObject {
				arc := len(args)
				if arc > 1 {
					return ERR_ARG_COUNT_MISMATCH()
				}
				
				if arc == 1 {
					arg, ok := ObjectState(args[0], RaiseNULL)
					if !ok { return arg }
					
					if arg, ok = UByteObject(arg, EXPLICIT); !ok {
						return ERR_INVALID_ARG_TYPE(arg)
					}
					
					return arg
				}
				
				return UBYTE //UByteValue(0)
			},
		},
	},
	
	{
		// 10.UINTEGER (uint64)
		BUILTIN_UINTEGER, &TBuiltin { Statement: false, 
			Fn: func(line string, args ...IObject) IObject {
				arc := len(args)
				if arc > 1 {
					return ERR_ARG_COUNT_MISMATCH()
				}
				
				if arc == 1 {
					arg, ok := ObjectState(args[0], RaiseNULL)
					if !ok { return arg }
					
					if arg, ok = UIntegerObject(arg, EXPLICIT); !ok {
						return ERR_INVALID_ARG_TYPE(arg)
					}
					
					return arg
				}
				
				return UINTEGER //UIntegerValue(0)
			},
		},
	},
	
	{
		// 11.ULONG (uint32)
		BUILTIN_ULONG, &TBuiltin { Statement: false, 
			Fn: func(line string, args ...IObject) IObject {
				arc := len(args)
				if arc > 1 {
					return ERR_ARG_COUNT_MISMATCH()
				}
				
				if arc == 1 {
					arg, ok := ObjectState(args[0], RaiseNULL)
					if !ok { return arg }
					
					if arg, ok = ULongObject(arg, EXPLICIT); !ok {
						return ERR_INVALID_ARG_TYPE(arg)
					}
					
					return arg
				}
				
				return ULONG //ULongValue(0)
			},
		},
	},
	
	{
		// 12.USHORT (uint16)
		BUILTIN_USHORT, &TBuiltin { Statement: false, 
			Fn: func(line string, args ...IObject) IObject {
				arc := len(args)
				
				if arc > 1 {
					return ERR_ARG_COUNT_MISMATCH()
				}
				
				if arc == 1 {
					arg, ok := ObjectState(args[0], RaiseNULL)
					if !ok { return arg }
					
					if arg, ok = UShortObject(arg, EXPLICIT); !ok {
						return ERR_INVALID_ARG_TYPE(arg)
					}
					
					return arg
				}
				
				return USHORT //UShortValue(0)
			},
		},
	},
	
	{
		// 13.FUNC: DATA TYPE
		BUILTIN_FUNC, &TBuiltin { Statement: false, 
			Fn: func(line string, args ...IObject) IObject {
				if len(args) != 0 {
					return ERR_ARG_COUNT_MISMATCH()
				}
				
				return FUNCTION
			},
		},
	},
	
	{
		// 14.ARRAY
		BUILTIN_ARRAY, &TBuiltin { Statement: false, 
			Fn: func(line string, args ...IObject) IObject {
				arc := len(args)
				if arc < 1 || arc > 2 {
					return ERR_ARG_COUNT_MISMATCH()
				}
				
				var count IObject
				
				if arc == 1 { // no elements, just the type
					count = UIntegerValue(0)
				} else { // predefined number of elements
					arg, ok := ObjectState(args[0], RaiseNULL)
					if !ok { return arg }
					
					// count = Deref(arg)
					count = arg
				}
				
				elems, ok := evalNumIndex(count)
				if !ok { return ERR_INVALID_INDEX(count) }
				
				var objType IObject
				
				array := make([]IObject, elems)
				if array == nil {
					return ERR_NOT_ENOUGH_MEMORY(count)
				}
				
				arg, ok := ObjectState(args[arc - 1], RaiseNULL)
				if !ok { return arg }
				
				arg, _ = Deref(arg)
				
				if obj, ok := arg.(*TType); ok {
					objType = CopyType(obj)
				} else {
					if arg == nil {
						return ERR_INVALID_TYPE()
					}
					objType = arg
				}
				
				for i := uint64(0); i < elems; i++ {
					array[i] = objType
				}
				
				return &TArray { DataType: objType, Elements: array }
			},
		},
	},
	
	{
		// 15.LIST
		BUILTIN_LIST, &TBuiltin { Statement: false, 
			Fn: func(line string, args ...IObject) IObject {
				arc := len(args)
				if arc > 1 {
					return ERR_ARG_COUNT_MISMATCH()
				}
				
				var arg IObject
				var ok bool
				
				if arc == 1 {
					arg, ok = ObjectState(args[0], RaiseNULL)
					if !ok { return arg }
					
					/*
					if arg, ok = IsNumberType(arg); !ok {
						return ERR_INVALID_ARG_TYPE(arg)
					}
					*/
				} else {	
					arg = UIntegerValue(0)
				}
				
				elems, ok := evalNumIndex(arg)
				if !ok { return ERR_INVALID_INDEX(arg) }
				
				list := make([]IObject, elems)
				if list == nil {
					return ERR_NOT_ENOUGH_MEMORY(arg)
				}
				
				for i := uint64(0); i < elems; i++ {
					list[i] = StringValue("")
				}
				
				return &TList { Elements: list }
			},
		},
	},
	
	{
		// 16.MAP
		BUILTIN_MAP, &TBuiltin { Statement: false, 
			Fn: func(line string, args ...IObject) IObject {
				arc := len(args)
				if arc > 0 {
					return ERR_ARG_COUNT_MISMATCH()
				}
				return &TMap { Order:[]TMapKey{}, Pairs: make(map[TMapKey]TMapPair) }
			},
		},
	},
	
	// =========================== BUILTINS =============================
	
	{	
		// ABS: NUMERIC
		BUILTIN_ABS, &TBuiltin { Statement: false, 	
			Fn: func(line string, args ...IObject) IObject {
				arc := len(args)
				if arc != 1 {
					return ERR_ARG_COUNT_MISMATCH()
				}
				
				arg, ok := ObjectState(args[0], RaiseNULL)
				if !ok { return arg }
				
				if arg, ok = IsNumberType(arg); !ok {
					return ERR_INVALID_ARG_TYPE(arg)
				}
								
				switch arg.(type) {
				case *TByte:
					n := arg.(*TByte)
					if n.Value > -1 {
						return n
					} else {
						return ByteValue(n.Value * -1)
					}
					
				case *TDouble:
					n := arg.(*TDouble)
					if n.Value > -1 {
						return n
					} else {
						return DoubleValue(n.Value * -1)
					}
				
				case *TSingle:
					n := arg.(*TSingle)
					if n.Value > -1 {
						return n
					} else {
						return SingleValue(n.Value * -1)
					}
				
				case *TInteger:
					n := arg.(*TInteger)
					if n.Value > -1 {
						return n
					} else {
						return IntegerValue(n.Value * -1)
					}
				
				case *TLong:
					n := arg.(*TLong)
					if n.Value > -1 {
						return n
					} else {
						return LongValue(n.Value * -1)
					}
				
				case *TShort:
					n := arg.(*TShort)
					if n.Value > -1 {
						return n
					} else {
						return ShortValue(n.Value * -1)
					}
				
				case *TUByte:
					return arg.(*TUByte)
				
				case *TUInteger:
					return arg.(*TUInteger)
				
				case *TULong:
					return arg.(*TULong)
				
				case *TUShort:
					return arg.(*TUShort)
				}
				
				return ERR_INVALID_ARG_TYPE(arg)
			},
		},
	},
	
	{
		// APPEND: ARRAY, LIST, MAP
		BUILTIN_APPEND, &TBuiltin { Statement: false, 
			Fn: func(line string, args ...IObject) IObject {
				arc := len(args)
				if 2 > arc || arc > 3 {
					return ERR_ARG_COUNT_MISMATCH()
				}
				
				arg1, ok := ObjectState(args[0], RaiseNULL)
				if !ok { return arg1 }
				
				arg1, _ = Deref(arg1)
				
				switch arg1.(type) {
				case *TArray:
					if arc != 2 {
						return ERR_ARG_COUNT_MISMATCH()
					}
					arr := arg1.(*TArray)
					
					arg2, ok := ObjectState(args[1], RaiseNULL)
					if !ok { return arg2 }
					
					arg2, _ = Deref(arg2)
					
					if arg2.Type() != arr.DataType.Type() {
						return ERR_TYPE_MISMATCH(arr.DataType, arg2)
					}
					
					arrl := len(arr.Elements)
					newElements := make([]IObject, arrl + 1)
					copy(newElements, arr.Elements)
					newElements[arrl] = arg2
					
					return &TArray { DataType: arr.DataType, Elements: newElements }
					
				case *TList:
					if arc != 2 {
						return ERR_ARG_COUNT_MISMATCH()
					}
					
					lst := arg1.(*TList)
					lstl := len(lst.Elements)
					newElements := make([]IObject, lstl + 1)
					copy(newElements, lst.Elements)
					
					arg2, ok := ObjectState(args[1], RaiseNULL)
					if !ok { return arg2 }
					
					arg2, _ = Deref(arg2)
					newElements[lstl] = arg2	
							
					return &TList { Elements: newElements }
				}
				
				return ERR_INVALID_ARG_TYPE(arg1)
			},
		},
	},
	
	{
		// ASC: STRING
		BUILTIN_ASC, &TBuiltin { Statement: false, 
			Fn: func(line string, args ...IObject) IObject {
				arc := len(args)
				if arc != 1 {
					return ERR_ARG_COUNT_MISMATCH()
				}
				
				arg, ok := ObjectState(args[0], RaiseNULL)
				if !ok { return arg }
					
				arg, ok = IsStringType(arg)
				if !ok { return ERR_INVALID_ARG_TYPE(arg) }
				
				str := arg.(*TString).Value
				if len(str) == 0 {
					return UByteValue(0)
				}
				
				return UByteValue(str[0])
			},
		},
	},
	
	{
		// CHR: STRING
		BUILTIN_CHR, &TBuiltin { Statement: false, 
			Fn: func(line string, args ...IObject) IObject {
				arc := len(args)
				if arc == 0 {
					return ERR_ARG_COUNT_MISMATCH()
				}
				
				var result string
				
				for _, obj := range args {
					arg, ok := ObjectState(obj, RaiseNULL)
					if !ok { return arg }
					
					val, ok := evalNumIndex(arg)
					if !ok { return ERR_INVALID_INDEX(arg) }
					
					if val > 255 {
						return ERR_INVALID_ARG_VALUE(arg)
					}
					result += string(val)
				}

				return StringValue(result)
			},
		},
	},
	
	{
		// INSTR: STRING
		BUILTIN_INSTR, &TBuiltin { Statement: false, 
			Fn: func(line string, args ...IObject) IObject {
				arc := len(args)
				if arc < 2 || arc > 3 {
					return ERR_ARG_COUNT_MISMATCH()
				}
				
				arg, ok := ObjectState(args[arc - 2], RaiseNULL)
				if !ok { return arg }
				
				if arg, ok = IsStringType(arg); !ok {
					return ERR_INVALID_ARG_TYPE(arg)
				}
							
				str1 := arg.(*TString).Value
				
				arg, ok = ObjectState(args[arc - 1], RaiseNULL)
				if !ok { return arg }
				
				if arg, ok = IsStringType(arg); !ok {
					return ERR_INVALID_ARG_TYPE(arg)
				}
				
				str2 := arg.(*TString).Value

				if arc == 2 {
					return UIntegerValue(uint64(strings.Index(str1, str2)) + 1)
				}
				
				arg, ok = ObjectState(args[0], RaiseNULL)
				if !ok { return arg }
				
				/*
				if arg, ok = IsNumberType(arg); !ok {
					return ERR_INVALID_ARG_TYPE(arg)
				}
				*/
				
				pos, ok := evalNumIndex(arg)
				if !ok { return ERR_INVALID_INDEX(arg) }
				
				if pos > uint64(len(str1)) {
					return ERR_INVALID_INDEX(arg)
				}
				
				return UIntegerValue(uint64(strings.Index(str1[pos:], str2)) + pos + 1)
			},
		},
	},
	
	{
		// LCASE: STRING
		BUILTIN_LCASE, &TBuiltin { Statement: false, 
			Fn: func(line string, args ...IObject) IObject {
				arc := len(args)
				if arc != 1 {
					return ERR_ARG_COUNT_MISMATCH()
				}
				
				arg, ok := ObjectState(args[0], RaiseNULL)
				if !ok { return arg }
				
				if arg, ok = IsStringType(arg); !ok {
					return ERR_INVALID_ARG_TYPE(arg)
				}
				
				str1 := arg.(*TString).Value
				
				return StringValue(strings.ToLower(str1))
			},
		},
	},
	
	{
		// LEFT: STRING
		BUILTIN_LEFT, &TBuiltin { Statement: false, 
			Fn: func(line string, args ...IObject) IObject {
				arc := len(args)
				if arc != 2 {
					return ERR_ARG_COUNT_MISMATCH()
				}
				
				arg, ok := ObjectState(args[1], RaiseNULL)
				if !ok { return arg }
				
				/*
				if arg, ok = IsNumberType(arg); !ok {
					return ERR_INVALID_ARG_TYPE(arg)
				}
				*/
				
				cutl, ok := evalNumIndex(arg)
				if !ok { return ERR_INVALID_INDEX(arg) }
				
				if cutl < 1 {
					return StringValue("")
				}
				
				arg, ok = ObjectState(args[0], RaiseNULL)
				if !ok { return arg }
				
				if arg, ok = IsStringType(arg); !ok {
					return ERR_INVALID_ARG_TYPE(arg)
				}
				
				str1 := arg.(*TString).Value
				
				strl := uint64(len(str1))
				if cutl > strl {
					cutl = strl
				}
				
				return StringValue(str1[:cutl])
			},
		},
	},
	
	{
		// LEN: STRING, ARRAY, LIST, MAP
		BUILTIN_LEN, &TBuiltin { Statement: false, 
			Fn: func(line string, args ...IObject) IObject {
				arc := len(args)
				if arc != 1 {
					return ERR_ARG_COUNT_MISMATCH()
				}
				
				arg, ok := ObjectState(args[0], RaiseNULL)
				if !ok { return arg }
				
				arg, _ = Deref(arg)
				
				switch a := arg.(type) {
				case *TString:
					return UIntegerValue(uint64(len(a.Value)))
				
				case *TArray:
					return UIntegerValue(uint64(len(a.Elements)))
					
				case *TList:
					return UIntegerValue(uint64(len(a.Elements)))
				
				case *TMap:
					return UIntegerValue(uint64(len(a.Pairs)))
				
				default:
					return ERR_INVALID_ARG_TYPE(arg)
				}
			},
		},
	},
	
	{		
		// MID: STRING
		BUILTIN_MID, &TBuiltin { Statement: false, 
			Fn: func(line string, args ...IObject) IObject {
				arc := len(args)
				if arc < 2 || arc > 3 {
					return ERR_ARG_COUNT_MISMATCH()
				}
				
				arg, ok := ObjectState(args[1], RaiseNULL)
				if !ok { return arg }
				
				/*
				if arg, ok = IsNumberType(arg); !ok {
					return ERR_INVALID_ARG_TYPE(arg)
				}
				*/
				
				pos, ok := evalNumIndex(arg)
				if !ok { return ERR_INVALID_INDEX(arg) }
				
				if pos < 1 {
					return StringValue("")
				}
				
				arg, ok = ObjectState(args[0], RaiseNULL)
				if !ok { return arg }
				
				if arg, ok = IsStringType(arg); !ok { 
					return ERR_INVALID_ARG_TYPE(arg)
				}
				
				str1 := arg.(*TString).Value
				
				strl := uint64(len(str1))
				if pos > strl {
					return StringValue("")
				}
				
				pos -= 1
				if arc == 2 {
					return StringValue(str1[pos:])
				}
				
				// three arguments
				
				arg, ok = ObjectState(args[2], RaiseNULL)
				if !ok { return arg }
				
				/*
				if arg, ok = IsNumberType(arg); !ok {
					return ERR_INVALID_ARG_TYPE(arg)
				}
				*/
				
				cutl, ok := evalNumIndex(arg)
				if !ok { return ERR_INVALID_INDEX(arg) }
				
				if cutl < 1 {
					return StringValue("")
				}
				
				lpos := uint64(pos + cutl)
				if lpos > strl {
					lpos = strl
				}
				
				return StringValue(str1[pos:lpos])
			},
		},
	},
	
	{
		// PCASE: STRING
		BUILTIN_PCASE, &TBuiltin { Statement: false, 
			Fn: func(line string, args ...IObject) IObject {
				arc := len(args)
				if arc != 1 {
					return ERR_ARG_COUNT_MISMATCH()
				}
				
				arg, ok := ObjectState(args[0], RaiseNULL)
				if !ok { return arg }
				
				if arg, ok = IsStringType(arg); !ok {
					return ERR_INVALID_ARG_TYPE(arg)
				}
				
				str1 := arg.(*TString).Value
				
				return StringValue(strings.Title(str1))
			},
		},
	},
	
	{
		// PREPEND: ARRAY
		BUILTIN_PREPEND, &TBuiltin { Statement: false, 
			Fn: func(line string, args ...IObject) IObject {
				arc := len(args)
				if arc != 2 {
					return ERR_ARG_COUNT_MISMATCH()
				}
				
				arg, ok := ObjectState(args[0], RaiseNULL)
				if !ok { return arg }
				
				arg, _ = Deref(arg)
				
				if arg.Type() != ARRAY_OBJ {
					return ERR_INVALID_ARG_TYPE(arg)
				}
				
				arr := arg.(*TArray)
				arrl := len(arr.Elements)
				
				newElements := make([]IObject, arrl, arrl)
				copy(newElements, arr.Elements)
				
				arg, ok = ObjectState(args[1], RaiseNULL)
				if !ok { return arg }
				
				arg, _ = Deref(arg)
				
				newElements = append([]IObject{arg}, newElements...)
				
				return &TArray { Elements: newElements }
			},
		},
	},
	
	{
		// PRINT
		BUILTIN_PRINT, &TBuiltin { Statement: true, 
			Fn: func(line string, args ...IObject) IObject {
				for _, arg := range args {
					if obj, ok := ObjectState(arg, RaiseNULL); !ok {
						return obj
					}
					
					arg, _ = Deref(arg)
					
					fmt.Println(arg.Inspect())
				}
				return nil
			},
		},
	},
	
	{
		// RIGHT: STRING
		BUILTIN_RIGHT, &TBuiltin { Statement: false, 
			Fn: func(line string, args ...IObject) IObject {
				arc := len(args)
				
				if arc != 2 {
					return ERR_ARG_COUNT_MISMATCH()
				}
				
				arg, ok := ObjectState(args[1], RaiseNULL)
				if !ok { return arg }
				
				/*
				if arg, ok = IsNumberType(arg); !ok {
					return ERR_INVALID_ARG_TYPE(arg)
				}
				*/
				
				cutl, ok := evalNumIndex(arg)
				if !ok { return ERR_INVALID_INDEX(arg) }
				
				if cutl < 1 {
					return StringValue("")
				}
				
				arg, ok = ObjectState(args[0], RaiseNULL)
				if !ok { return arg }
				
				if arg, ok = IsStringType(arg); !ok {
					return ERR_INVALID_ARG_TYPE(arg)
				}
				
				str1 := arg.(*TString).Value
				
				strl := uint64(len(str1))
				if cutl > strl {
					cutl = strl
				}
				
				return StringValue(str1[strl - cutl:])
			},
		},
	},
	
	{
		// SQR: NUMERIC
		BUILTIN_SQR, &TBuiltin { Statement: false, 
			Fn: func(line string, args ...IObject) IObject {
				arc := len(args)
				
				if arc != 1 {
					return ERR_ARG_COUNT_MISMATCH()
				}
				
				arg, ok := ObjectState(args[0], RaiseNULL)
				if !ok { return arg }
				
				/*
				if arg, ok = IsNumberType(arg); !ok {
					return ERR_INVALID_ARG_TYPE(arg)
				}
				*/
				
				val,_ := DoubleObject(arg, EXPLICIT)
				return DoubleValue(math.Sqrt(val.(*TDouble).Value))
			},
		},
	},
	
	{
		// UCASE: STRING
		BUILTIN_UCASE, &TBuiltin { Statement: false, 
			Fn: func(line string, args ...IObject) IObject {
				arc := len(args)
				
				if arc != 1 {
					return ERR_ARG_COUNT_MISMATCH()
				}
				
				arg, ok := ObjectState(args[0], RaiseNULL)
				if !ok { return arg }
				
				if arg, ok = IsStringType(arg); !ok {
					return ERR_INVALID_ARG_TYPE(arg)
				}
				
				str1 := arg.(*TString).Value
				
				return StringValue(strings.ToUpper(str1))
			},
		},
	},
	
	{
		// TYPEOF: OBJECT
		BUILTIN_TYPOF, &TBuiltin { Statement: false, 
			Fn: func(line string, args ...IObject) IObject {
				arc := len(args)
				if arc != 1 {
					return ERR_ARG_COUNT_MISMATCH()
				}
				
				arg, ok := ObjectState(args[0], AllowNULL)
				if !ok { return arg }
				
				if _, ok := arg.(*TReference); ok {
					arg = arg.(*TReference).Value
				}
				
				typ := arg.Type()
				
				if _, ok := arg.(*TConst); ok {
					arg = arg.(*TConst).Value
					typ = typ + " " + arg.Type()
				}
				
				return StringValue(fmt.Sprintf("%s", typ))
			},
		},
	},
	
}

