// lexer/lexer.go

package lexer

import (
	"SB/token"
)

type TLexer struct {
	input					string
	position			int
	readPosition	int
	ch						byte
}


func (l *TLexer) NextToken() token.TToken {
	var tok token.TToken
	
	l.skipSpace()
	
	for (l.ch == '/' && l.peekChar() == '/') || l.ch == 39 {
		l.skipComment()
	}
	
	switch l.ch {
	case '=':
		if l.peekChar() == '=' {
			ch := l.ch
			l.readChar()
			tok = token.TToken { Type: token.EQUAL, Literal: string(ch) + string(l.ch) }
		} else {
			tok = CreateToken(token.ASSIGN, l.ch)
		}
	case '+':
		tok = CreateToken(token.PLUS, l.ch)
	case '-':
		tok = CreateToken(token.MINUS, l.ch)
	case '*':
		tok = CreateToken(token.ASTERISK, l.ch)
	case '/':
		tok = CreateToken(token.SLASH, l.ch)
	case '%':
		tok = CreateToken(token.PERCENT, l.ch)
	case '^':
		tok = CreateToken(token.CARET, l.ch)
	case '<':
		switch l.peekChar() {
		case '=':
			ch := l.ch
			l.readChar()
			tok = token.TToken { Type: token.LESS_EQ, Literal: string(ch) + string(l.ch) }
		case '>':
			ch := l.ch
			l.readChar()
			tok = token.TToken { Type: token.NOT_EQUAL, Literal: string(ch) + string(l.ch) }
		default:
			tok = CreateToken(token.LESS, l.ch)
		}
	case '>':
		switch l.peekChar() {
		case '=':
			ch := l.ch
			l.readChar()
			tok = token.TToken { Type: token.GREATER_EQ, Literal: string(ch) + string(l.ch) }
		case ']':
			ch := l.ch
			l.readChar()
			tok = token.TToken { Type: token.RLIST, Literal: string(ch) + string(l.ch) }
		default:
			tok = CreateToken(token.GREATER, l.ch)
		}
	case ',':
		tok = CreateToken(token.COMMA, l.ch)
	case ':':
		tok = CreateToken(token.COLON, l.ch)
	case ';':
		tok = CreateToken(token.SEMICOLON, l.ch)
	case '(':
		tok = CreateToken(token.LPAREN, l.ch)
	case ')':
		tok = CreateToken(token.RPAREN, l.ch)
	case '{':
		tok = CreateToken(token.LBRACE, l.ch)
	case '}':
		tok = CreateToken(token.RBRACE, l.ch)
	case '[':
		if l.peekChar() == '<' {
			ch := l.ch
			l.readChar()
			tok = token.TToken { Type: token.LLIST, Literal: string(ch) + string(l.ch) }
		} else {
			tok = CreateToken(token.LBRACKET, l.ch)
		}
	case ']':
		tok = CreateToken(token.RBRACKET, l.ch)
	case '"':
		tok.Type = token.STRING
		tok.Literal = l.readString()
	case '.':
		tok = CreateToken(token.DOT, l.ch)
	case 0:
		tok.Literal = ""
		tok.Type = token.EOF
	default:
		if isLetter(l.ch) {
			tok.Literal = l.readIdentifier()
			tok.Type = token.LookupIdent(tok.Literal)
			return tok
		}
		if isDigit(l.ch) {
			return l.readNumber()
		}
		tok = CreateToken(token.UNKNOWN, l.ch)
	}
	l.readChar()
	return tok
}

func (l *TLexer) readIdentifier() string {
	p := l.position
	for isLetter(l.ch) || isDigit(l.ch) {
		l.readChar()
	}
	return l.input[p:l.position]
}

func (l *TLexer) readNumber() token.TToken {
	var tok token.TToken
	
	p := l.position
	
	for isDigit(l.ch) {
		l.readChar()
		if l.ch == '.' {
			tok.Type = token.FLOAT
			l.readChar()
		}
	}
	
	if tok.Type != token.FLOAT {
		tok.Type = token.INTEGER
	}
	
	tok.Literal = l.input[p:l.position]
	
	return tok
}

func (l *TLexer) skipComment() {
	for l.ch != '\n' && l.ch != 0 {
		l.readChar()
	}
	l.skipSpace()
}

func (l *TLexer) skipSpace() {
	for l.ch == ' ' || l.ch == '\t' || l.ch == '\n' || l.ch == '\r' {
		l.readChar()
	}
}

func (l *TLexer) readChar() {
	
	if l.readPosition >= len(l.input) {
		l.ch = 0
	} else {
		l.ch = l.input[l.readPosition]
	}
	l.position = l.readPosition
	l.readPosition += 1
}

func (l *TLexer) peekChar() byte {
	if l.readPosition >= len(l.input) {
		return 0
	}
	return l.input[l.readPosition]
}

func isLetter(ch byte) bool {
	return 'a' <= ch && ch <= 'z' || 'A' <= ch && ch <= 'Z' || ch == '-'
}

func isDigit(ch byte) bool {
	return '0' <= ch && ch <= '9'
}

func CreateToken(tokenType token.TokenType, ch byte) token.TToken {
	return token.TToken {Type: tokenType, Literal: string(ch) }
}

func New(input string) *TLexer {
	l := &TLexer { input: input }
	l.readChar()
	return l
}

func (l *TLexer) readString() string {
	p := l.position + 1
	for {
		l.readChar()
		if l.ch == '"' {
			break
		}
	}
	return l.input[p:l.position]
}

