// compiler/symbol_table.go

package compiler


type SymbolScope string


const(
	GlobalScope SymbolScope = "GLOBAL"
	LocalScope SymbolScope = "LOCAL"
	BuiltinScope SymbolScope = "BUILTIN"
)


type TSymbol struct {
	Name string
	Scope SymbolScope
	Index int
}


type TSymbolTable struct{
	Outer *TSymbolTable
	store map[string]TSymbol
	numDefinitions int
}


func NewSymbolTable() *TSymbolTable {
	s := make(map[string]TSymbol)
	return &TSymbolTable { store: s }
}


func NewEnclosedSymbolTable(outer *TSymbolTable) *TSymbolTable {
	s := NewSymbolTable()
	s.Outer = outer
	return s
}


func (s *TSymbolTable) Define(name string) TSymbol {
	symbol := TSymbol { Name: name, Index: s.numDefinitions } //, Scope: GlobalScope}
	if s.Outer == nil {
		symbol.Scope = GlobalScope
	} else {
		symbol.Scope = LocalScope
	}

	s.store[name] = symbol
	s.numDefinitions++
	return symbol
}


func (s *TSymbolTable) Resolve(name string) (TSymbol, bool) {
	obj, ok := s.store[name]
	if !ok && s.Outer != nil {
		obj, ok = s.Outer.Resolve(name)
		return obj, ok
	}
	
	return obj, ok
}


func (s *TSymbolTable) DefineBuiltin(index int, name string) TSymbol {
	symbol := TSymbol { Name: name, Index: index, Scope: BuiltinScope}
	s.store[name] = symbol
	return symbol
}

