// compiler/compiler_test.go

package compiler

import(
	"fmt"
	"testing"
	"SB/ast"
	"SB/lexer"
	"SB/object"
	"SB/parser"
	"SB/CMP/code"
)

type compilerTestCase struct {
	input string
	expectedConstants []interface{}
	expectedInstructions []code.Instructions
}


func parse(input string) *ast.TProgram {
	l := lexer.New(input)
	p := parser.New(l)
	return p.ParseProgram()
}

func runCompilerTests(t *testing.T, tests []compilerTestCase) {
	t.Helper()

	for _, tt := range tests {
		program := parse(tt.input)
		compiler := New()
		eobj:= compiler.Compile(program)
		if eobj != nil {
			t.Fatalf(eobj.(*object.TError).Message)
		}

		bytecode := compiler.Bytecode()

		err := testInstructions(tt.expectedInstructions, bytecode.Instructions)
		if err != nil {
			t.Fatalf("testInstructions failed: %s", err)
		}
		err = testConstants(t, tt.expectedConstants, bytecode.Constants)
		if	err != nil {
			t.Fatalf("testConstants failed: %s", err)
		}
	}
}

func testInstructions(expected []code.Instructions, actual code.Instructions,) error {
	concatted := concatInstructions(expected)
	if len(actual) != len(concatted) {
		return fmt.Errorf("wrong instructions length \nexpected %q, got %q", concatted, actual)
	}
	
	for i, ins := range concatted {
		if actual[i] != ins {
			return fmt.Errorf("wrong instruction at %d \nexpected %q, got %q", i, concatted, actual)
		}
	}
	
	return nil
}


func concatInstructions(s []code.Instructions) code.Instructions {
	out := code.Instructions{}
	
	for _, ins := range s {
		out = append(out, ins...)
	}
	
	return out
}


func testConstants(t *testing.T, expected []interface{}, actual []object.IObject,) error {
	if len(expected) != len(actual) {
		return fmt.Errorf("wrong number of constants, got %d, expected %d", len(actual), len(expected))
	}
	
	for i, constant := range expected {
		switch constant := constant.(type) {
		case int:
			err := testIntegerObject(int64(constant), actual[i])
			if err != nil {
				return fmt.Errorf("constant %d - testIntegerObject failed: %s", i, err)
			}
		case string:
			err := testStringObject(constant, actual[i])
			if err != nil {
				return fmt.Errorf("constant %d - testStringObject failed: %s", i, err)
			}
		case []code.Instructions:
			fn, ok := actual[i].(*object.TCompiledFunction)
			if !ok {
				return fmt.Errorf("constant %d - not a function: %T", i, actual[i])
			}
			err := testInstructions(constant, fn.Instructions)
			if err != nil {
				return fmt.Errorf("constant %d - testInstructions failed: %s", i, err)
			}
		}
	}
	
	return nil
}


func testIntegerObject(expected int64, actual object.IObject) error {
	result, ok := actual.(*object.TInteger)
	if !ok {
		return fmt.Errorf("object is not Integer. got %T (%+v)", actual, actual)
	}
	
	if result.Value != expected {
		return fmt.Errorf("object has wrong value. got %d, expected %d", result.Value, expected)
	}

	return nil
}


func testStringObject(expected string, actual object.IObject) error {
	result, ok := actual.(*object.TString)
	if !ok {
		return fmt.Errorf("object is not String. got=%T (%+v)", actual, actual)
	}

	if result.Value != expected {
		return fmt.Errorf("object has wrong value. got=%q, want=%q", result.Value, expected)
	}

	return nil
}


func TestNestedFunctions(t *testing.T) {
	tests := []compilerTestCase {
		{
			input: `
				function Test1(): integer do
					var x: integer;
					
					function Test2(x: integer): integer do
						return x;
					end;
					
					x = 10;
					
					return Test2(x);
				end;

				print(Test1())
			`,
			expectedConstants: []interface{} {
				
				[]code.Instructions { //Test2
					code.Make(code.OP_GETLC, 0),	// x
					code.Make(code.OP_RETV),
				},
				10,
				[]code.Instructions { //Test1
					code.Make(code.OP_GETBU, 5), // integer
					code.Make(code.OP_CALL, 0),
					code.Make(code.OP_DEFLC, 0),	// x
					code.Make(code.OP_CONST, 0),  // Test2
					code.Make(code.OP_DEFLC, 1),  
					code.Make(code.OP_CONST, 1),	// 10
					code.Make(code.OP_MOVLC, 0),	// x = 10
					code.Make(code.OP_GETLC, 1),	// Test2
					code.Make(code.OP_GETLC, 0),	// x
					code.Make(code.OP_CALL, 1),
					code.Make(code.OP_RETV),
				},
			},
			expectedInstructions: []code.Instructions {
				code.Make(code.OP_CONST, 2),
				code.Make(code.OP_DEFGL, 0),
				code.Make(code.OP_GETBU, 28),
				code.Make(code.OP_GETGL, 0),
				code.Make(code.OP_CALL, 0),
				code.Make(code.OP_CALL, 1),
			},
		},
		
	}

	runCompilerTests(t, tests)
}


func TestAssignment(t *testing.T) {
	tests := []compilerTestCase {
		{
			input: `
				var a: integer;
				a = 1;
				a = 2;
			`,
			expectedConstants: []interface{} {
				1,
				2,
			},
			expectedInstructions: []code.Instructions {
				code.Make(code.OP_GETBU, 5),	// integer
				code.Make(code.OP_CALL, 0),		// integer()
				code.Make(code.OP_DEFGL, 0),	// var a
				code.Make(code.OP_CONST, 0),	// 1
				code.Make(code.OP_MOVGL, 0),	// a = 1
				code.Make(code.OP_CONST, 1),	// 2
				code.Make(code.OP_MOVGL, 0),	// a = 2
			},
		},
		
		{
			input: `
				var A: integer(9);
				function test(): integer do
					var a: integer;
					a = A;
					return a;
				end;
				print(test());
			`,
			expectedConstants: []interface{} {
					9,
				[]code.Instructions {
					code.Make(code.OP_GETBU, 5), // integer
					code.Make(code.OP_CALL, 0),
					code.Make(code.OP_DEFLC, 0), // var a = integer
					code.Make(code.OP_GETGL, 0), // A
					code.Make(code.OP_MOVLC, 0), // a = A
					code.Make(code.OP_GETLC, 0), // a
					code.Make(code.OP_RETV),     // return a
				},
			},
			expectedInstructions: []code.Instructions {
				code.Make(code.OP_GETBU, 5),  // integer
				code.Make(code.OP_CONST, 0),  // 9
				code.Make(code.OP_CALL, 1),   // integer(9)
				code.Make(code.OP_DEFGL, 0),  // var A
				code.Make(code.OP_CONST, 1),  // function
				code.Make(code.OP_DEFGL, 1),  // test()
				code.Make(code.OP_GETBU, 28), // print()
				code.Make(code.OP_GETGL, 1),  // test()
				code.Make(code.OP_CALL, 0),   // call test()
				code.Make(code.OP_CALL, 1),   // call print()
			},
		},
		
		
	}

	runCompilerTests(t, tests)
}
