// code/code_test.go

package code

import "testing"

func TestMake(t *testing.T) {
	tests := []struct {
		op Opcode
		operands []int
		expected []byte
	}{
		{OP_CONSTANT, []int{65534}, []byte{byte(OP_CONSTANT), 255, 254}},
		{OP_ADD, []int{}, []byte{byte(OP_ADD)}},
		{OP_GET_LOCAL, []int{255}, []byte{byte(OP_GET_LOCAL), 255}},
	}
	for _, tt := range tests {
		instruction := Make(tt.op, tt.operands...)
		if len(instruction) != len(tt.expected) {
			t.Errorf("instruction has wrong length, expected %d, got %d", len(tt.expected), len(instruction))
		}
		for i, b := range tt.expected {
			if instruction[i] != tt.expected[i] {
				t.Errorf("wrong byte at pos %d, expected %d, got %d", i, b, instruction[i])
			}
		}
	}
}

func TestInstructionsString(t *testing.T) {
	instructions := []Instructions {
		Make(OP_ADD),
		Make(OP_GET_LOCAL, 1),
		Make(OP_CONSTANT, 2),
		Make(OP_CONSTANT, 65535),
	}
	expected := `0000 OP_ADD
0001 OP_GET_LOCAL 1
0003 OP_CONSTANT 2
0006 OP_CONSTANT 65535
`

	concatted := Instructions{}
	for _, ins := range instructions {
		concatted = append(concatted, ins...)
	}
	if concatted.String() != expected {
		t.Errorf("instructions wrongly formatted.\nexpected %q\ngot %q", expected, concatted.String())
	}
}

func TestReadOperands(t *testing.T) {
	tests := []struct {
		op Opcode
		operands []int
		bytesRead int
	}{
		{	OP_CONSTANT, []int{65535}, 2 },
		{ OP_GET_LOCAL, []int{255}, 1},
	}

	for _, tt := range tests {
		instruction := Make(tt.op, tt.operands...)
		def, err := Lookup(byte(tt.op))
		if err != nil {
			t.Fatalf("definition not found: %q\n", err)
		}
		
		operandsRead, n := ReadOperands(def, instruction[1:])
		if n != tt.bytesRead {
			t.Fatalf("n wrong. expected %d, got %d", tt.bytesRead, n)
		}
		
		for i, want := range tt.operands {
			if operandsRead[i] != want {
				t.Errorf("operand wrong. expected %d, got %d", want, operandsRead[i])
			}
		}
	}
}

