// object/scope.go

package object

import (
	"fmt"
	"sync"
	"SB/ast"
)

var GlobalScopes map[string]IObject = make(map[string]IObject)
var GlobalMutex sync.RWMutex

type TCallFrame struct {
	FuncScope   *TScope
	CurrentCall *ast.TCallExpression
	defers      []func()
}

func (frame *TCallFrame) runDefers(s *TScope) {
	// execute defers last-to-first
	defers := frame.defers
	for i := len(defers) - 1; i >= 0; i-- {
		defers[i]()
	}
}


type TCallStack struct {
	Frames []TCallFrame
}

type TScope struct {
	store       	map[string]IObject
	parentScope 	*TScope
	CallStack   	*TCallStack
	sync.RWMutex
}

func (s *TScope) Exists(name string) bool {
	s.RLock()
	defer s.RUnlock()
	
	_, flag := s.store[name]
	return flag
}

func (s *TScope) Get(name string) (IObject, bool) {
	s.RLock()
	defer s.RUnlock()
	
	obj, ok := s.store[name]
	if !ok && s.parentScope != nil {
		obj, ok = s.parentScope.Get(name)
	}
	return obj, ok
}

func (s *TScope) GetKeys() []string {
	keys := make([]string, 0, len(s.store))
	for k := range s.store {
		keys = append(keys, k)
	}
	return keys
}

func (s *TScope) DebugPrint(indent string) {
	s.Lock()
	defer s.Unlock()

	for k, v := range s.store {
		fmt.Printf("%s<%s> = <%s>  value.Type: %T\n", indent, k, v.Inspect(), v)
	}

	if s.parentScope != nil {
		fmt.Printf("\n%sParentScope:\n", indent)
		s.parentScope.DebugPrint(indent + "  ")
	}

}

func (s *TScope) Set(name string, val IObject) IObject {
	s.Lock()
	defer s.Unlock()

	s.store[name] = val
	return val
}

func (s *TScope) Reset(name string, val IObject) (IObject, bool) {
	s.Lock()
	defer s.Unlock()

	var ok bool
	_, ok = s.store[name]
	if ok {
		s.store[name] = val
	}

	if !ok && s.parentScope != nil {
		_, ok = s.parentScope.Reset(name, val)
	}

	if !ok {
		s.store[name] = val
		ok = true
	}
	return val, ok
}

func (s *TScope) CurrentFrame() *TCallFrame {
	s.RLock()
	s.RUnlock()

	if s != nil {
		frames := s.CallStack.Frames
		if n := len(frames); n > 0 {
			return &frames[n-1] //return top
		}
	}
	return nil
}

// CallerFrame return caller's CallFrame
func (s *TScope) CallerFrame() *TCallFrame {
	s.RLock()
	s.RUnlock()

	if s != nil {
		frames := s.CallStack.Frames
		if n := len(frames); n > 1 {
			return &frames[n-2] //return caller's frame
		}
	}
	return nil
}


func GetGlobalObj(name string) (IObject, bool) {
	GlobalMutex.Lock()
	defer GlobalMutex.Unlock()

	obj, ok := GlobalScopes[name]
	return obj, ok
}


func SetGlobalObj(name string, Obj IObject) {
	GlobalMutex.Lock()
	defer GlobalMutex.Unlock()

	GlobalScopes[name] = Obj
}


func NewScope(p *TScope) *TScope {
	s := make(map[string]IObject)
	ret := &TScope { store: s, parentScope: p }
	if p == nil {
		ret.CallStack = &TCallStack { Frames: []TCallFrame { TCallFrame{} } }
	} else {
		ret.CallStack = p.CallStack
	}

	return ret
}
