// object/object.go

package object

import (
	"bytes"
	"fmt"
	"strings"
	"hash/fnv"
	"SB/ast"
	"SB/token"
)

const (
	BOOLEAN_OBJ 			= "BOOLEAN"
	BYTE_OBJ					= "BYTE"
	DOUBLE_OBJ				= "DOUBLE"
	SINGLE_OBJ				= "SINGLE"
	LONG_OBJ					= "LONG"
	INTEGER_OBJ 			= "INTEGER"
	SHORT_OBJ					= "SHORT"
	STRING_OBJ				= "STRING"
	UBYTE_OBJ					= "UBYTE"
	ULONG_OBJ					= "ULONG"
	UINTEGER_OBJ			= "UINTEGER"
	USHORT_OBJ				= "USHORT"
	
	CONST_OBJ					= "CONST"
	CONST_GROUP_OBJ		= "CONST_GROUP"
	
	ENUM_OBJ					= "ENUM"
	ENUM_GROUP_OBJ		= "ENUM_GROUP"
	
	TYPE_OBJ					= "TYPE"
	TYPE_REF_OBJ			= "TYPE_REF"
	
	REF_OBJ						= "REF"
	
	NULL_OBJ					= "NULL"
	RETURN_VALUE_OBJ	= "RETURN_VALUE"
	ERROR_OBJ					= "ERROR"
	FUNCTION_OBJ			= "FUNCTION"
	SUB_OBJ						= "SUB"
	BUILTIN_OBJ				= "BUILTIN"
	ARRAY_OBJ					= "ARRAY"
	LIST_OBJ					= "LIST"
	MAP_OBJ						= "MAP"
	EXIT_OBJ					= "EXIT"
	CONTINUE_OBJ			= "CONTINUE"
)


var (
	NULL				= &TNull {}
	TRUE				= &TBoolean { Value: true }
	FALSE				= &TBoolean { Value: false }
	BYTE				= &TByte { Value: 0 }
	DOUBLE			= &TDouble { Value: 0 }
	SINGLE			= &TSingle { Value: 0 }
	INTEGER			= &TInteger { Value: 0 }
	LONG				= &TLong { Value: 0 }
	SHORT				= &TShort { Value: 0 }
	STRING			= &TString { Value: "" }
	UBYTE				= &TUByte { Value: 0 }
	UINTEGER		= &TUInteger { Value: 0 }
	ULONG				= &TULong { Value: 0 }
	USHORT			= &TUShort { Value: 0 }
	FUNCTION		= &TFunction {}
	
	EXIT				= &TExit {}
	CONTINUE		= &TContinue {}
	
)

type ObjectType string


type IObject interface {
	Type()		ObjectType
	Inspect()	string
}


type TNull struct {}
func (n *TNull) Type() ObjectType { return NULL_OBJ }
func (n *TNull) Inspect() string { return "nil" }


type TBoolean struct {
	Value bool
}
func (b *TBoolean) Type() ObjectType { return BOOLEAN_OBJ }
func (b *TBoolean) Inspect() string { return fmt.Sprintf("%t", b.Value) }


type TByte struct {
	Value int8
}
func (b *TByte) Type() ObjectType { return BYTE_OBJ }
func (b *TByte) Inspect() string { return fmt.Sprintf("%d", b.Value) }


type TDouble struct {
	Value float64
}
func (d *TDouble) Type() ObjectType { return DOUBLE_OBJ }
func (d *TDouble) Inspect() string { return fmt.Sprintf("%g", d.Value) }


type TSingle struct {
	Value float32
}
func (s *TSingle) Type() ObjectType { return SINGLE_OBJ }
func (s *TSingle) Inspect() string { return fmt.Sprintf("%f", s.Value) }


type TInteger struct {
	Value int64
}
func (i *TInteger) Type() ObjectType { return INTEGER_OBJ }
func (i *TInteger) Inspect() string { return fmt.Sprintf("%d", i.Value) }


type TLong struct {
	Value int32
}
func (l *TLong) Type() ObjectType { return LONG_OBJ }
func (l *TLong) Inspect() string { return fmt.Sprintf("%d", l.Value) }


type TShort struct {
	Value int16
}
func (s *TShort) Type() ObjectType { return SHORT_OBJ }
func (s *TShort) Inspect() string { return fmt.Sprintf("%d", s.Value) }


type TString struct {
	Value string
}
func (s *TString) Type() ObjectType { return STRING_OBJ }
func (s *TString) Inspect() string { return s.Value }


type TUByte struct {
	Value uint8
}
func (u *TUByte) Type() ObjectType { return UBYTE_OBJ }
func (u *TUByte) Inspect() string { return fmt.Sprintf("%d", u.Value) }


type TUInteger struct {
	Value uint64
}
func (i *TUInteger) Type() ObjectType { return UINTEGER_OBJ }
func (i *TUInteger) Inspect() string { return fmt.Sprintf("%d", i.Value) }


type TULong struct{
	Value uint32
}
func (l *TULong) Type() ObjectType { return ULONG_OBJ }
func (l *TULong) Inspect() string { return fmt.Sprintf("%d", l.Value) }


type TUShort struct {
	Value uint16
}
func (s *TUShort) Type() ObjectType { return USHORT_OBJ }
func (s *TUShort) Inspect() string { return fmt.Sprintf("%d", s.Value) }


type TReturnValue struct {
	Value IObject
}
func (rv *TReturnValue) Type() ObjectType { return RETURN_VALUE_OBJ }
func (rv *TReturnValue) Inspect() string { return rv.Value.Inspect() }


type TFunction struct {
	Scope				*TScope
	Literal			*ast.TFunctionLiteral
	ReturnType	IObject
}
func (f *TFunction) Type() ObjectType { return FUNCTION_OBJ }
func (f *TFunction) Inspect() string { return f.Literal.String() }


type TSub struct {
	Scope *TScope
	Literal *ast.TSubLiteral
}
func (s *TSub) Type() ObjectType { return SUB_OBJ }
func (s *TSub) Inspect() string { return s.Literal.String() }


type TArray struct {
	DataType	IObject
	Elements	[]IObject
}
func (a *TArray) Type() ObjectType { return ARRAY_OBJ }
func (a *TArray) Inspect() string {
	var out bytes.Buffer
	
	elems := []string{}
	for _, el := range a.Elements {
		elems = append(elems, el.Inspect())
	}
	
	out.WriteString(token.LBRACKET)
	out.WriteString(strings.Join(elems, ", "))
	out.WriteString(token.RBRACKET)
	
	return out.String()
}


type TList struct {
	Elements	[]IObject
}
func (l *TList) Type() ObjectType { return LIST_OBJ }
func (l *TList) Inspect() string {
	var out bytes.Buffer
	
	elems := []string{}
	for _, el := range l.Elements {
		elems = append(elems, el.Inspect())
	}
	
	out.WriteString(token.LLIST)
	out.WriteString(strings.Join(elems, ", "))
	out.WriteString(token.RLIST)
	
	return out.String()
}


type TMapKey struct {
	Type	ObjectType
	Value	uint64
}

func (b *TBoolean) MapKey() TMapKey {
	var value uint64
	
	if b.Value {
		value = 1
	} else {
		value = 0
	}
	
	return TMapKey { Type: b.Type(), Value: value }
}

func (i *TInteger) MapKey() TMapKey {
	return TMapKey { Type: i.Type(), Value: uint64(i.Value) }
}

func (s *TString) MapKey() TMapKey {
	h := fnv.New64a()
	h.Write([]byte(s.Value))
	
	return TMapKey { Type: s.Type(), Value: h.Sum64() }
}


type TMapPair struct {
	Key		IObject
	Value IObject
}


type TMap struct {
	Pairs map[TMapKey]TMapPair
	Order  []TMapKey
}
func (h *TMap) Type() ObjectType { return MAP_OBJ }
func (h *TMap) Inspect() string {
	var out bytes.Buffer
	
	pairs := []string{}
	for _, pair := range h.Pairs {
		pairs = append(pairs, fmt.Sprintf("%s: %s", pair.Key.Inspect(), pair.Value.Inspect()))
	}
	
	out.WriteString("{")
	out.WriteString(strings.Join(pairs, ", "))
	out.WriteString("}")
	
	return out.String()
}

func (h *TMap) Push(args ...IObject) IObject {
	if hashable, ok := args[0].(IMapable); ok {
		hk := hashable.MapKey()
		_, exists := h.Pairs[hk]
		
		if !exists {
			h.Order = append(h.Order, hk)
		}

		h.Pairs[hk] = TMapPair{Key: args[0], Value: args[1]}
	} else {
		return ERR_INVALID_MAP_KEY(args[0])
	}
	return h
}


type IMapable interface {
	MapKey() TMapKey
}


type TExit struct{}

func (b *TExit) Type() ObjectType { return EXIT_OBJ }
func (b *TExit) Inspect() string  { return "exit" }


type TContinue struct{}

func (c *TContinue) Type() ObjectType { return CONTINUE_OBJ }
func (c *TContinue) Inspect() string  { return "continue" }


type TReference struct {
	Value		IObject
	Name		string
	Scope		*TScope
}
func (r *TReference) Type() ObjectType { return REF_OBJ }
func (r *TReference) Inspect() string { return r.Value.Inspect() }


type TType struct {
	Scope   	*TScope
	Name			string
	typeName	ObjectType
}
func (t *TType) Type() ObjectType {
	if t.typeName == "" {
		t.typeName = TYPE_OBJ
	}
	return t.typeName
}

func (t *TType) Inspect() string {
	var out bytes.Buffer
	out.WriteString(" type ")
	out.WriteString(t.Name)
	out.WriteString(" of ")
	for k, v := range t.Scope.store {
		out.WriteString(k)
		out.WriteString(".")
		out.WriteString(v.Inspect())
		out.WriteString(";")
	}
	out.WriteString(" end ")

	return out.String()
}


type TConst struct {
	Value IObject
}
func (c *TConst) Type() ObjectType { return CONST_OBJ }
func (c *TConst) Inspect() string { return c.Value.Inspect() }


type TConstGroup struct {
	Scope   	*TScope
	Name			string
}
func (cg *TConstGroup) Type() ObjectType { return CONST_GROUP_OBJ }
func (cg *TConstGroup) Inspect() string {
	var out bytes.Buffer
	out.WriteString(" const ")
	out.WriteString(cg.Name)
	out.WriteString(" of ")
	for k, v := range cg.Scope.store {
		out.WriteString(k)
		out.WriteString(".")
		out.WriteString(v.Inspect())
		out.WriteString(";")
	}
	out.WriteString(" end ")

	return out.String()
}


type TEnum struct {
	Value IObject
}
func (e *TEnum) Type() ObjectType { return ENUM_OBJ }
func (e *TEnum) Inspect() string { return e.Value.Inspect() }


type TEnumGroup struct {
	Scope   	*TScope
	Name			string
}
func (eg *TEnumGroup) Type() ObjectType { return ENUM_GROUP_OBJ }
func (eg *TEnumGroup) Inspect() string {
	var out bytes.Buffer
	out.WriteString(" const ")
	out.WriteString(eg.Name)
	out.WriteString(" of ")
	for k, v := range eg.Scope.store {
		out.WriteString(k)
		out.WriteString(".")
		out.WriteString(v.Inspect())
		out.WriteString(";")
	}
	out.WriteString(" end ")

	return out.String()
}
