// object/eval.go

package object

import (
	"fmt"
	"math"
	//"strings"
	"SB/token"
	"SB/ast"
)


func Eval(node ast.INode, scope *TScope) IObject {
	//fmt.Println(node.String())
	
	switch node := node.(type) {
	
	// statements
	case *ast.TProgram:
		return evalProgram(node, scope)
	
	case *ast.TBlockStatement:
		return evalBlockStatements(node.Statements, scope)
	
	case *ast.TConstStatement:
		return evalConstStatement(node, scope)
	
	case *ast.TEnumStatement:
		return evalEnumStatement(node, scope)
	
	case *ast.TExpressionStatement:
		return Eval(node.Expression, scope)
	
	case *ast.TForLoopStatement:
		return evalForLoopStatement(node, scope)
	
	case *ast.TFunctionStatement:
		obj := evalFunctionStatement(node, scope)
		if eobj, ok := ObjectState(obj, AllowNULL); !ok {
			return eobj
		}
	
	case *ast.TIfStatement:
		return evalIfStatement(node, scope)
		
	case *ast.TReturnStatement:
		return evalReturnStatement(node, scope)
	
	case *ast.TSelectStatement:
		return evalSelectStatement(node, scope)
	
	case *ast.TSubLiteral:
	obj := evalSubLiteral(node, scope)
		if eobj, ok := ObjectState(obj, RaiseNULL); !ok {
			return eobj
		}
		
	case *ast.TVarStatement:
		obj := evalVarStatement(node, scope)
		if eobj, ok := ObjectState(obj, AllowNULL); !ok {
			return eobj
		}
	
	case *ast.TScopeStatement:
		return evalScopeStatement(node, scope)
	
	
	
	case *ast.TWhileLoopStatement:
		return evalWhileLoopStatement(node, scope)
	
	
	// expressions
	
	case *ast.TAssignExpression:
		obj := evalAssignExpression(node, scope)
		if eobj, ok := ObjectState(obj, AllowNULL); !ok {
			return eobj
		}
	
	case *ast.TBoolean:
		return nativeBoolToBooleanObject(node.Value)
	
	case *ast.TCallExpression:
		if node.Statement {
			// sub
			return evalCallStatement(node, scope)
		} else {
			// function
			return evalCallExpression(node, scope)
		}
	
	case *ast.TContinueExpression:
		return CONTINUE
	
	case *ast.TExitExpression:
		return EXIT
		
	case *ast.TIntegerLiteral:
		return &TInteger { Value: node.Value }
	
	case *ast.TFloatLiteral:
		return &TDouble { Value: node.Value }
		
	case *ast.TStringLiteral:
		return &TString { Value: node.Value }
		
	case *ast.TPrefixExpression:
		right := Eval(node.Right, scope)
		if obj, ok := ObjectState(right, AllowNULL); !ok {
			return obj
		}
		return evalPrefixExpression(node.Operator, right)
	
	case *ast.TInfixExpression:
		if left, right, op, errObj, ok := evalLeftRight(node, scope, AllowNULL); ok {
			return evalInfixExpression(op, left, right)
		} else {
			return errObj
		}
		
	case *ast.TIdentifier:
		return evalIdentifier(node, scope)
	
	case *ast.TFunctionLiteral:
		return evalFunctionLiteral(node, scope)
	
	case *ast.TIndexExpression:
		left := Eval(node.Left, scope)
		if obj, ok := ObjectState(left, RaiseNULL); !ok {
			return obj
		}
		index := Eval(node.Index, scope)
		if obj, ok := ObjectState(index, RaiseNULL); !ok {
			return obj
		}
		return evalIndexExpression(left, index)
	
	case *ast.TMapLiteral:
		return evalMapLiteral(node, scope)
	
	case *ast.TTypeStatement:
		obj := evalTypeStatement(node, scope)
		if eobj, ok := ObjectState(obj, RaiseNULL); !ok {
			return eobj
		}
	
	case *ast.TReferenceExpression:
		return evalReferenceExpression(node, scope)
	}
	
	return nil
}


func evalExpressions(exps []ast.IExpression, scope *TScope) []IObject {
	var result []IObject
	
	for _, e := range exps {
		ev := Eval(e, scope)
		if obj, ok := ObjectState(ev, AllowNULL); !ok {
			return []IObject { obj }
		}
		
		result = append(result, ev)
	}
	
	return result
}


func evalIdentifier(node *ast.TIdentifier, scope *TScope) IObject {
	if obj, ok := scope.Get(node.Value); ok {
		return obj
	}
	if builtin, ok := BuiltinByName(node.Value); ok {
		return builtin
	}
	return ERR_UNKNOWN_IDENTIFIER(node.Value)
}


func nativeBoolToBooleanObject(input bool) *TBoolean {
	if input {
		return TRUE
	}
	return FALSE
}


func evalPrefixExpression(operator string, right IObject) IObject {
	right, _ = Deref(right)
	
	switch operator {
	case token.IS:
		return evalIsOperatorExpression(right)
		
	case token.NOT:
		return evalNotOperatorExpression(right)
		
	case token.MINUS:
		return evalMinusPrefixOperatorExpression(right)	
	}
	
	return ERR_INVALID_EXPRESSION(operator, nil, right)
}


func evalProgram(program *ast.TProgram, scope *TScope) IObject {
	var result IObject
	
	for _, stmt := range program.Statements {
		result = Eval(stmt, scope)
		if result != nil {
			switch result := result.(type) {
			case *TReturnValue:
				return ERR_RETURN_WITHOUT_FUNCTION()
				// return result.Value
			case *TError:
				return result
			default:
				if result != nil {
					fmt.Println(result.Inspect())
				}
			}
		}
	}
	
	return nil //result
}

func evalVarStatement(vs *ast.TVarStatement, scope *TScope) IObject {
	var val IObject
	
	values := []IObject{}
	valuesCount := 0
	
	for _, value := range vs.Values {
		v := Eval(value, scope)
		// allow NULL object assignment
		if obj, ok := ObjectState(v, AllowNULL); !ok {
			return obj
		}
		
		if obj, ok := v.(*TArray); ok {
			// definition by [] is invalid (no data type)
			if obj.DataType == nil {
				return ERR_INVALID_ASSIGNMENT(v)
			}
		} else if _, ok := v.(*TBuiltin); ok {
			return ERR_INVALID_ASSIGNMENT(v)
		}
		
		valuesCount += 1
		values = append(values, v)
	}
	
	/* // checked by parser
	if valuesCount > len(vs.Names) {
		return ERR_DEFINITION_COUNT_MISMATCH()
	}
	*/
	
	if valuesCount == 1 {
		for _, item := range vs.Names {
			name := item.String()
			if scope.Exists(name) {
				return ERR_DUPLICATE_DEFINITION(name)
			}
			val = values[0]
			if _, ok := val.(*TType); ok {
				cpy := CopyType(val.(*TType))
				scope.Set(name, cpy)
			} else {
				dobj, _ := Deref(val)
				scope.Set(name, dobj)
			}
		}
		
		return nil
	}
	
	// null object
	if valuesCount == 0 {
		val = NULL
	}
	
	for names, item := range vs.Names {
		name := item.String()
		if scope.Exists(name) {
			return ERR_DUPLICATE_DEFINITION(name)
		}
		if names < valuesCount {
			val = values[names]
		}
		if _, ok := val.(*TType); ok {
			cpy := CopyType(val.(*TType))
			scope.Set(name, cpy)
		} else {
			dobj, _ := Deref(val)
			scope.Set(name, dobj)
		}
	}
	
	return nil
}


func evalIsOperatorExpression(right IObject) IObject {
	if val, ok := BooleanObject(right, EXPLICIT); ok {
		switch val.(*TBoolean).Value {
		case true:
			return TRUE
		case false:
			return FALSE
		default:
			return FALSE
		}
	}
	
	return nil
}


func evalNumIndex(index IObject) (uint64, bool) {
	var result uint64
	var number int64
	
	number = 0
	result = 0
	
	switch obj := index.(type) {
	case *TByte:
		number = int64(obj.Value)
	case *TInteger:
		number = int64(obj.Value)
	case *TLong:
		number = int64(obj.Value)
	case *TShort:
		number = int64(obj.Value)
	case *TUByte:
		result = uint64(obj.Value)
	case *TUInteger:
		result = obj.Value
	case *TULong:
		result = uint64(obj.Value)
	case *TUShort:
		result = uint64(obj.Value)
	default:
		if dobj, ok := Deref(index); ok {
			return evalNumIndex(dobj)
		}		
		return result, false
	}
	
	if number < 0 {
		return result, false
	} else if number > 0 {
		result = uint64(number)
	}
	
	return result, true
}


func evalNotOperatorExpression(right IObject) IObject {
	if val, ok := BooleanObject(right, EXPLICIT); ok {
		switch val.(*TBoolean).Value {
		case true:
			return FALSE
		case false:
			return TRUE
		default:
			return FALSE
		}
	}
	return nil
}


func evalMinusPrefixOperatorExpression(right IObject) IObject {
	switch right.(type) {
	case *TByte:
		value := right.(*TByte).Value
		return &TByte { Value: -value }
	case *TDouble:
		value := right.(*TDouble).Value
		return &TDouble { Value: -value }
	case *TSingle:
		value := right.(*TSingle).Value
		return &TSingle { Value: -value }
	case *TInteger:
		value := right.(*TInteger).Value
		return &TInteger { Value: -value }
	case *TLong:
		value := right.(*TLong).Value
		return &TLong { Value: -value }
	case *TShort:
		value := right.(*TShort).Value
		return &TShort { Value: -value }
	case *TUByte:
		value := right.(*TUByte).Value
		return &TUByte { Value: -value }
	case *TUInteger:
		value := right.(*TUInteger).Value
		return &TUInteger { Value: -value }
	case *TULong:
		value := right.(*TULong).Value
		return &TULong { Value: -value }
	case *TUShort:
		value := right.(*TUShort).Value
		return &TUShort { Value: -value }
	}
	
	return ERR_INVALID_EXPRESSION(token.MINUS, nil, right)
}


func evalToArray(a *ast.TAssignExpression, name string, left IObject, scope *TScope, right IObject) IObject {
	leftVals := left.(*TArray).Elements
	elemCount := uint64(len(leftVals))
	
	if robj, ok := right.(*TReference); ok {
		return evalToArray(a, name, left, scope, robj.Value)
	}
	
	switch nodeType := a.Name.(type) {
	case *ast.TIndexExpression:
		// assign to array element
		index := Eval(nodeType.Index, scope)	
		if obj, ok := ObjectState(index, RaiseNULL); !ok {
			return obj
		}
		
		idx, ok := evalNumIndex(index)
		if !ok { return ERR_INVALID_INDEX(index) }
		
		if idx >= elemCount {
			return ERR_SUBSCRIPT_OUT_OF_RANGE(index)
		}
		
		switch right.(type){
		case *TArray, *TMap:
			return ERR_INVALID_ASSIGNMENT(right)
		}
		//fmt.Println(right.Type())
		//fmt.Println(left.(*TArray).DataType.Type())
		if right.Type() != left.(*TArray).DataType.Type() {
			return ERR_TYPE_MISMATCH(left.(*TArray).DataType, right)
		}
		
		if robj, ok := right.(*TType); ok {
			// type identifiers must match
			if lobj, ok := leftVals[idx].(*TType); ok {
				if robj.Name != lobj.Name {
					return ERR_TYPES_DO_NOT_MATCH(lobj.Name, robj.Name)
				}
			}
		}
		
		leftVals[idx] = right
		
		if obj, ok := scope.Reset(name, &TArray { DataType: left.(*TArray).DataType, Elements: leftVals }); !ok {
			return ERR_INVALID_ASSIGNMENT(obj)
		}
	default:
		// assign to array
		if obj, ok := scope.Reset(name, right); !ok {
			return ERR_INVALID_ASSIGNMENT(obj)
		}
	}
	return nil
}


func evalToList(a *ast.TAssignExpression, name string, left IObject, scope *TScope, right IObject) IObject {
	leftVals := left.(*TList).Elements
	elemCount := uint64(len(leftVals))
	
	if robj, ok := right.(*TReference); ok {
		return evalToList(a, name, left, scope, robj.Value)
	}
	
	switch nodeType := a.Name.(type) {
	case *ast.TIndexExpression:
		// assign to list element
		index := Eval(nodeType.Index, scope)	
		if obj, ok := ObjectState(index, RaiseNULL); !ok {
			return obj
		}
		
		idx, ok := evalNumIndex(index)
		if !ok { return ERR_INVALID_INDEX(index) }
		
		if idx >= elemCount {
			return ERR_SUBSCRIPT_OUT_OF_RANGE(index)
		}
		
		switch right.(type){
		case *TArray, *TList, *TMap:
			return ERR_INVALID_ASSIGNMENT(right)
		}
		
		leftVals[idx] = right
		
		if obj, ok := scope.Reset(name, &TList { Elements: leftVals }); !ok {
			return ERR_INVALID_ASSIGNMENT(obj)
		}
	default:
		// assign to list
		if obj, ok := scope.Reset(name, right); !ok {
			return ERR_INVALID_ASSIGNMENT(obj)
		}
	}
	return nil
}


func evalToMap(a *ast.TAssignExpression, name string, left IObject, scope *TScope, val IObject) IObject {
	leftHash := left.(*TMap)
	
	if robj, ok := val.(*TReference); ok {
		return evalToMap(a, name, left, scope, robj.Value)
	}
	
	if a.Token.Literal == token.ASSIGN {
		
		switch nodeType := a.Name.(type) {
		case *ast.TIndexExpression:
			key := Eval(nodeType.Index, scope)
			leftHash.Push(key, val)
			
			return nil // leftHash
			
		default:
			// map = map
			if _, ok := val.(*TMap); ok {
				if obj, ok := scope.Reset(name, val); ok {
					return nil
				} else {
					return ERR_INVALID_ASSIGNMENT(obj)
				}
			}
		}
	}
	
	return ERR_TYPE_MISMATCH(left, val)
}


func evalToType(left IObject, name string, scope *TScope, right IObject) IObject {
	lobj := left.(*TType)
	robj, ok := right.(*TType)
	
	if !ok {
		return ERR_INVALID_ASSIGNMENT(robj)
	}
	
	if robj.Name != lobj.Name {
		return ERR_TYPES_DO_NOT_MATCH(lobj.Name, robj.Name)
	}
	
	val := CopyType(robj)
	if _, ok := scope.Reset(name, val); !ok {
		return ERR_INVALID_ASSIGNMENT(robj)
	}
	
	return val
}


func evalInfixExpression(operator string, left, right IObject) IObject {
	if obj, ok := ObjectState(left, AllowNULL); !ok {
		return obj
	}
	
	if obj, ok := ObjectState(right, AllowNULL); !ok {
		return obj
	}
	
	left, _ = Deref(left)
	right, _ = Deref(right)
	
	if left.Type() == STRING_OBJ && right.Type() == STRING_OBJ {
		return evalStringInfixExpression(operator, left, right)
	}
	
	if left.Type() == DOUBLE_OBJ || right.Type() == DOUBLE_OBJ {
		l, ok := DoubleObject(left, IMPLICIT)
		if !ok { return ERR_TYPE_MISMATCH(left, right) }
		r, ok := DoubleObject(right, IMPLICIT)
		if !ok { return ERR_TYPE_MISMATCH(left, right) }
		return evalDoubleInfixExpression(operator, l, r)
	}
	
	if left.Type() == SINGLE_OBJ || right.Type() == SINGLE_OBJ {
		l, ok := SingleObject(left, IMPLICIT)
		if !ok { return ERR_TYPE_MISMATCH(left, right) }
		r, ok := SingleObject(right, IMPLICIT)
		if !ok { return ERR_TYPE_MISMATCH(left, right) }
		return evalSingleInfixExpression(operator, l, r)
	}
	
	if left.Type() == INTEGER_OBJ || right.Type() == INTEGER_OBJ {
		l, ok := IntegerObject(left, IMPLICIT)
		if !ok { return ERR_TYPE_MISMATCH(left, right) }
		r, ok := IntegerObject(right, IMPLICIT)
		if !ok { return ERR_TYPE_MISMATCH(left, right) }
		return evalIntegerInfixExpression(operator, l, r)
	}
	
	if left.Type() == LONG_OBJ || right.Type() == LONG_OBJ {
		l, ok := LongObject(left, IMPLICIT)
		if !ok { return ERR_TYPE_MISMATCH(left, right) }
		r, ok := LongObject(right, IMPLICIT)
		if !ok { return ERR_TYPE_MISMATCH(left, right) }
		return evalLongInfixExpression(operator, l, r)
	}
	
	if left.Type() == SHORT_OBJ || right.Type() == SHORT_OBJ {
		l,ok := ShortObject(left, IMPLICIT)
		if !ok { return ERR_TYPE_MISMATCH(left, right) }
		r, ok := ShortObject(right, IMPLICIT)
		if !ok { return ERR_TYPE_MISMATCH(left, right) }
		return evalShortInfixExpression(operator, l, r)
	}
	
	if left.Type() == BYTE_OBJ || right.Type() == BYTE_OBJ {
		l, ok := ByteObject(left, IMPLICIT)
		if !ok { return ERR_TYPE_MISMATCH(left, right) }
		r, ok := ByteObject(right, IMPLICIT)
		if !ok { return ERR_TYPE_MISMATCH(left, right) }
		return evalByteInfixExpression(operator, l, r)
	}
	
	if left.Type() == UINTEGER_OBJ || right.Type() == UINTEGER_OBJ {
		l, ok := UIntegerObject(left, IMPLICIT)
		if !ok { return ERR_TYPE_MISMATCH(left, right) }
		r, ok := UIntegerObject(right, IMPLICIT)
		if !ok { return ERR_TYPE_MISMATCH(left, right) }
		return evalUIntegerInfixExpression(operator, l, r)
	}
	
	if left.Type() == ULONG_OBJ || right.Type() == ULONG_OBJ {
		l, ok := ULongObject(left, IMPLICIT)
		if !ok { return ERR_TYPE_MISMATCH(left, right) }
		r, ok := ULongObject(right, IMPLICIT)
		if !ok { return ERR_TYPE_MISMATCH(left, right) }
		return evalULongInfixExpression(operator, l, r)
	}
	
	if left.Type() == USHORT_OBJ || right.Type() == USHORT_OBJ {
		l, ok := UShortObject(left, IMPLICIT)
		if !ok { return ERR_TYPE_MISMATCH(left, right) }
		r, ok := UShortObject(right, IMPLICIT)
		if !ok { return ERR_TYPE_MISMATCH(left, right) }
		return evalUShortInfixExpression(operator, l, r)
	}
	
	if left.Type() == UBYTE_OBJ || right.Type() == UBYTE_OBJ {
		l, ok := UByteObject(left, IMPLICIT)
		if !ok { return ERR_TYPE_MISMATCH(left, right) }
		r, ok := UByteObject(right, IMPLICIT)
		if !ok { return ERR_TYPE_MISMATCH(left, right) }
		return evalUByteInfixExpression(operator, l, r)
	}
	
	if operator == token.EQUAL {
		return nativeBoolToBooleanObject(left == right)
	}
	
	if operator == token.NOT_EQUAL {
		return nativeBoolToBooleanObject(left != right)
	}
	
	if operator == token.XOR {
		return nativeBoolToBooleanObject((objectToBool(left) == false) || (objectToBool(right) == false))
	}
	
	if operator == token.OR {
		return nativeBoolToBooleanObject(objectToBool(left) || objectToBool(right))
	}
	
	if operator == token.AND {
		return nativeBoolToBooleanObject(objectToBool(left) && objectToBool(right))
	}
	
	if operator == token.NOR {
		return nativeBoolToBooleanObject((objectToBool(left) == false) && (objectToBool(right) == false))
	}
		
	if left.Type() != right.Type() {
		return ERR_EXPRESSION_TYPE_MISMATCH(operator, left, right)
	}
			
	return ERR_INVALID_EXPRESSION(operator, left, right)
}


func IsNumberType(obj IObject) (IObject, bool) {
	
	switch obj.(type) {
	case *TBoolean, *TByte, *TDouble, *TSingle, *TInteger, *TLong, *TShort, *TUByte, *TUInteger, *TULong, *TUShort:
		return obj, true
	default:
		if dobj, ok := Deref(obj); ok {
			return IsNumberType(dobj)
		}
	}
	
	return obj, false
}


func IsStringType(obj IObject) (IObject, bool) {
	
	switch obj.(type) {
	case *TString:
		return obj, true
	default:
		if dobj, ok := Deref(obj); ok {
			return IsStringType(dobj)
		}
	}
	
	return obj, false
}

func evalStringInfixExpression(operator string, left, right IObject) IObject {
	lv := left.(*TString).Value
	rv := right.(*TString).Value
		
	switch operator {
	case token.PLUS:
		return &TString { Value: lv + rv }
	case token.EQUAL:
		return &TBoolean { Value: lv == rv }
	case token.NOT_EQUAL:
		return &TBoolean { Value: (lv != rv) }
	case token.XOR:
		return &TBoolean { Value: (lv == "") || (rv == "") }
	case token.OR:
		return &TBoolean { Value: (lv != "") || (rv != "") }
	case token.AND:
		return &TBoolean { Value: (lv != "") && (rv != "") }
	case token.NOR:
		return &TBoolean { Value: (lv == "") && (rv == "") }
	}
	
	return ERR_INVALID_EXPRESSION(operator, left, right)
}


func evalSubLiteral(ss *ast.TSubLiteral, scope *TScope) IObject {
	sb := &TSub { Literal: ss, Scope: scope }
	
	for key, value := range ss.Parameters {
		if name, ok := key.(*ast.TIdentifier); !ok {
			return ERR_INVALID_IDENTIFIER(name.Value)
		} else {
			obj := Eval(value.Parameter, sb.Scope)
			if _, ok := obj.(*TError); ok {
				return obj
			}
			sb.Scope.Set(name.String(), obj)
		}
	}
	
	scope.Set(ss.Name.String(), sb)
	
	return sb
}


func evalDoubleInfixExpression(operator string, left, right IObject) IObject {
	lv := left.(*TDouble).Value
	rv := right.(*TDouble).Value
	
	switch operator {
	case token.PLUS:
		return &TDouble { Value: lv + rv }
	case token.MINUS:
		return &TDouble { Value: lv - rv }
	case token.ASTERISK:
		return &TDouble { Value: lv * rv }
	case token.SLASH:
		return &TDouble { Value: lv / rv }
	case token.CARET:
		return &TDouble { Value: math.Pow(lv, rv) }
	case token.LESS:
		return nativeBoolToBooleanObject(lv < rv)
	case token.LESS_EQ:
		return nativeBoolToBooleanObject(lv <= rv)
	case token.GREATER:
		return nativeBoolToBooleanObject(lv > rv)
	case token.GREATER_EQ:
		return nativeBoolToBooleanObject(lv >= rv)
	case token.EQUAL:
		return nativeBoolToBooleanObject(lv == rv)
	case token.NOT_EQUAL:
		return nativeBoolToBooleanObject(lv != rv)
	case token.OR:
		return nativeBoolToBooleanObject((lv != 0) || (rv != 0))
	case token.XOR:
		return nativeBoolToBooleanObject((lv == 0) || (rv == 0))
	case token.AND:
		return nativeBoolToBooleanObject((lv != 0) && (rv != 0))
	case token.NOR:
		return nativeBoolToBooleanObject((lv == 0) && (rv == 0))
	
	default:
		return ERR_INVALID_OPERATOR(operator)
	}
}


func evalSingleInfixExpression(operator string, left, right IObject) IObject {
	lv := left.(*TSingle).Value
	rv := right.(*TSingle).Value
	
	switch operator {
	case token.PLUS:
		return &TSingle { Value: lv + rv }
	case token.MINUS:
		return &TSingle { Value: lv - rv }
	case token.ASTERISK:
		return &TSingle { Value: lv * rv }
	case token.SLASH:
		return &TDouble { Value: float64(lv) / float64(rv) }
	case token.CARET:
		return &TSingle { Value: float32(math.Pow(float64(lv), float64(rv))) }
	case token.LESS:
		return nativeBoolToBooleanObject(lv < rv)
	case token.LESS_EQ:
		return nativeBoolToBooleanObject(lv <= rv)
	case token.GREATER:
		return nativeBoolToBooleanObject(lv > rv)
	case token.GREATER_EQ:
		return nativeBoolToBooleanObject(lv >= rv)
	case token.EQUAL:
		return nativeBoolToBooleanObject(lv == rv)
	case token.NOT_EQUAL:
		return nativeBoolToBooleanObject(lv != rv)
	case token.OR:
		return nativeBoolToBooleanObject((lv != 0) || (rv != 0))
	case token.XOR:
		return nativeBoolToBooleanObject((lv == 0) || (rv == 0))
	case token.AND:
		return nativeBoolToBooleanObject((lv != 0) && (rv != 0))
	case token.NOR:
		return nativeBoolToBooleanObject((lv == 0) && (rv == 0))
	default:
		return ERR_INVALID_OPERATOR(operator)
	}
}


func evalByteInfixExpression(operator string, left, right IObject) IObject {
	lv := left.(*TByte).Value
	rv := right.(*TByte).Value
	
	switch operator {
	case token.PLUS:
		return &TByte { Value: lv + rv }
	case token.MINUS:
		return &TByte { Value: lv - rv }
	case token.ASTERISK:
		return &TByte { Value: lv * rv }
	case token.SLASH:
		return &TDouble { Value: float64(lv) / float64(rv) }
	case token.PERCENT:
		return &TByte { Value: lv % rv }
	case token.CARET:
		return &TByte { Value: int8(math.Pow(float64(lv), float64(rv))) }
	case token.LESS:
		return nativeBoolToBooleanObject(lv < rv)
	case token.LESS_EQ:
		return nativeBoolToBooleanObject(lv <= rv)
	case token.GREATER:
		return nativeBoolToBooleanObject(lv > rv)
	case token.GREATER_EQ:
		return nativeBoolToBooleanObject(lv >= rv)
	case token.EQUAL:
		return nativeBoolToBooleanObject(lv == rv)
	case token.NOT_EQUAL:
		return nativeBoolToBooleanObject(lv != rv)
	case token.OR:
		return nativeBoolToBooleanObject((lv != 0) || (rv != 0))
	case token.XOR:
		return nativeBoolToBooleanObject((lv == 0) || (rv == 0))
	case token.AND:
		return nativeBoolToBooleanObject((lv != 0) && (rv != 0))
	case token.NOR:
		return nativeBoolToBooleanObject((lv == 0) && (rv == 0))
	default:
		return ERR_INVALID_OPERATOR(operator)
	}
}


func evalIntegerInfixExpression(operator string, left, right IObject) IObject {
	lv := left.(*TInteger).Value
	rv := right.(*TInteger).Value
	
	switch operator {
	case token.PLUS:
		return &TInteger { Value: lv + rv }
	case token.MINUS:
		return &TInteger { Value: lv - rv }
	case token.ASTERISK:
		return &TInteger { Value: lv * rv }
	case token.SLASH:
		return &TDouble { Value: float64(lv) / float64(rv) }
	case token.PERCENT:
		return &TInteger { Value: lv % rv }
	case token.CARET:
		return &TInteger { Value: int64(math.Pow(float64(lv), float64(rv))) }
	case token.LESS:
		return nativeBoolToBooleanObject(lv < rv)
	case token.LESS_EQ:
		return nativeBoolToBooleanObject(lv <= rv)
	case token.GREATER:
		return nativeBoolToBooleanObject(lv > rv)
	case token.GREATER_EQ:
		return nativeBoolToBooleanObject(lv >= rv)
	case token.EQUAL:
		return nativeBoolToBooleanObject(lv == rv)
	case token.NOT_EQUAL:
		return nativeBoolToBooleanObject(lv != rv)
	case token.OR:
		return nativeBoolToBooleanObject((lv != 0) || (rv != 0))
	case token.XOR:
		return nativeBoolToBooleanObject((lv == 0) || (rv == 0))
	case token.AND:
		return nativeBoolToBooleanObject((lv != 0) && (rv != 0))
	case token.NOR:
		return nativeBoolToBooleanObject((lv == 0) && (rv == 0))
	default:
		return ERR_INVALID_OPERATOR(operator)
	}
}

func evalLeftRight(node ast.IExpression, scope *TScope, nullAction NullActionType) (IObject, IObject, string, IObject, bool) {
	
	switch node := node.(type) {
	case *ast.TInfixExpression:
		left := Eval(node.Left, scope)
		if obj, ok := ObjectState(left, nullAction); !ok {
			return nil, nil, "", obj, false
		}
		
		right := Eval(node.Right, scope)
		if obj, ok := ObjectState(right, nullAction); !ok {
			return nil, nil, "", obj, false
		}
		
		return left, right, node.Operator, nil, true
	}
	
	return nil, nil, "", nil, false
}


func evalLongInfixExpression(operator string, left, right IObject) IObject {
	lv := left.(*TLong).Value
	rv := right.(*TLong).Value
	
	switch operator {
	case token.PLUS:
		return &TLong { Value: lv + rv }
	case token.MINUS:
		return &TLong { Value: lv - rv }
	case token.ASTERISK:
		return &TLong { Value: lv * rv }
	case token.SLASH:
		return &TDouble { Value: float64(lv) / float64(rv) }
	case token.PERCENT:
		return &TLong { Value: lv % rv }
	case token.CARET:
		return &TLong { Value: int32(math.Pow(float64(lv), float64(rv))) }
	case token.LESS:
		return nativeBoolToBooleanObject(lv < rv)
	case token.LESS_EQ:
		return nativeBoolToBooleanObject(lv <= rv)
	case token.GREATER:
		return nativeBoolToBooleanObject(lv > rv)
	case token.GREATER_EQ:
		return nativeBoolToBooleanObject(lv >= rv)
	case token.EQUAL:
		return nativeBoolToBooleanObject(lv == rv)
	case token.NOT_EQUAL:
		return nativeBoolToBooleanObject(lv != rv)
	case token.OR:
		return nativeBoolToBooleanObject((lv != 0) || (rv != 0))
	case token.XOR:
		return nativeBoolToBooleanObject((lv == 0) || (rv == 0))
	case token.AND:
		return nativeBoolToBooleanObject((lv != 0) && (rv != 0))
	case token.NOR:
		return nativeBoolToBooleanObject((lv == 0) && (rv == 0))
	default:
		return ERR_INVALID_OPERATOR(operator)
	}
}


func evalReferenceExpression(re *ast.TReferenceExpression, scope *TScope) IObject {
	var refObj IObject
	var refName string
	
	refScope := scope
	refCount := len(re.Members)
	
	mainObj := Eval(re.Root, refScope)
	if nobj, ok := ObjectState(mainObj, RaiseNULL); !ok {
		return nobj // error
	}
	
	switch mainObj.(type) {
	case *TType:
		obj := mainObj.(*TType)
		refScope = obj.Scope
		
	case *TConstGroup:
		obj := mainObj.(*TConstGroup)
		refScope = obj.Scope
	
	case *TEnumGroup:
		obj := mainObj.(*TEnumGroup)
		refScope = obj.Scope
		
	// case class, namespace, module etc...
	}
	
	// members
	for i := 0; i < refCount; i++ {
		ref := Eval(re.Members[i], refScope)
		if nobj, ok := ObjectState(ref, AllowNULL); !ok {
			return nobj // error
		}
		switch ref.(type) {
		case *TType:
			obj := ref.(*TType)
			refScope = obj.Scope
		case *TConstGroup:
			obj := ref.(*TConstGroup)
			refScope = obj.Scope
		case *TEnumGroup:
			obj := ref.(*TEnumGroup)
			refScope = obj.Scope
		case *TNull:
			if i < refCount - 1 {
				return ERR_NULL_OBJECT_EXCEPTION()
			}
		// classes, objects...
		}
		refObj = ref
		refName = re.Names[i]
	}
	
	return &TReference { Value: refObj, Name: refName, Scope: refScope }
}


func evalReturnStatement(rs *ast.TReturnStatement, scope *TScope) IObject {
	stmt := Eval(rs.ReturnValue, scope)
	
	if obj, ok := ObjectState(stmt, AllowNULL); !ok {
		return obj
	}
	
	return &TReturnValue { Value: stmt }
}


func evalSelectStatement(ss *ast.TSelectStatement, scope *TScope) IObject {
	sexp := Eval(ss.Expression, scope)
	if obj, ok := ObjectState(sexp, AllowNULL); !ok {
		return obj;
	}
	
	var def				*ast.TDefaultBlock
	var execBlock bool
	
	for _, cases := range ss.Cases {
		execBlock = false
		
		if obj, ok := cases.(*ast.TDefaultBlock); ok {
			def = obj
			continue
		}
		
		for _, cexps := range cases.(*ast.TCaseBlock).Expressions {
			
			switch cexps.(type) {
			case *ast.TFromToExpression:
				cexp := cexps.(*ast.TFromToExpression).Expression
				if left, right, _, errObj, ok := evalLeftRight(cexp, NewScope(scope), RaiseNULL); ok {
					if exec, errObj, ok := isFromTo(sexp, left, right); ok {
						execBlock = exec
					} else {
						return errObj
					}
				} else {
					return errObj
				}
			default:
				cexp := Eval(cexps, NewScope(scope))
				if obj, ok := ObjectState(cexp, AllowNULL); !ok {
					return obj;
				}
				if _, ok := cexp.(*TBoolean); ok {
					execBlock = cexp.(*TBoolean).Value
				} else {
					if exec, errObj, ok := isEqual(cexp, sexp); ok {
						execBlock = exec
					} else {
						return errObj
					}
				}
			}
			
			if execBlock {
				block := Eval(cases.(*ast.TCaseBlock).Block, NewScope(scope))
				
				if block != nil {
					if obj, ok := block.(*TError); ok {
						return obj
					} else if obj, ok := block.(*TExit); ok {
						return obj
					} else if obj, ok := block.(*TContinue); ok {
						return obj
					} else if ret, ok := block.(*TReturnValue); ok {
						if ret.Value != nil {
							return ret
						}
					}
				}
				return nil
			}
		}
	}
	
	if def != nil {
		sexp = Eval(def.Block, NewScope(scope))
	}
	
	return nil
}


func evalShortInfixExpression(operator string, left, right IObject) IObject {
	lv := left.(*TShort).Value
	rv := right.(*TShort).Value
	
	switch operator {
	case token.PLUS:
		return &TShort { Value: lv + rv }
	case token.MINUS:
		return &TShort { Value: lv - rv }
	case token.ASTERISK:
		return &TShort { Value: lv * rv }
	case token.SLASH:
		return &TDouble { Value: float64(lv) / float64(rv) }
	case token.PERCENT:
		return &TShort { Value: lv % rv }
	case token.CARET:
		return &TShort { Value: int16(math.Pow(float64(lv), float64(rv))) }
	case token.LESS:
		return nativeBoolToBooleanObject(lv < rv)
	case token.LESS_EQ:
		return nativeBoolToBooleanObject(lv <= rv)
	case token.GREATER:
		return nativeBoolToBooleanObject(lv > rv)
	case token.GREATER_EQ:
		return nativeBoolToBooleanObject(lv >= rv)
	case token.EQUAL:
		return nativeBoolToBooleanObject(lv == rv)
	case token.NOT_EQUAL:
		return nativeBoolToBooleanObject(lv != rv)
	case token.OR:
		return nativeBoolToBooleanObject((lv != 0) || (rv != 0))
	case token.XOR:
		return nativeBoolToBooleanObject((lv == 0) || (rv == 0))
	case token.AND:
		return nativeBoolToBooleanObject((lv != 0) && (rv != 0))
	case token.NOR:
		return nativeBoolToBooleanObject((lv == 0) && (rv == 0))
	default:
		return ERR_INVALID_OPERATOR(operator)
	}
}


func evalUByteInfixExpression(operator string, left, right IObject) IObject {
	lv := left.(*TUByte).Value
	rv := right.(*TUByte).Value
	
	switch operator {
	case token.PLUS:
		return &TUByte { Value: lv + rv }
	case token.MINUS:
		return &TUByte { Value: lv - rv }
	case token.ASTERISK:
		return &TUByte { Value: lv * rv }
	case token.SLASH:
		return &TDouble { Value: float64(lv) / float64(rv) }
	case token.PERCENT:
		return &TUByte { Value: lv % rv }
	case token.CARET:
		return &TUByte { Value: uint8(math.Pow(float64(lv), float64(rv))) }
	case token.LESS:
		return nativeBoolToBooleanObject(lv < rv)
	case token.LESS_EQ:
		return nativeBoolToBooleanObject(lv <= rv)
	case token.GREATER:
		return nativeBoolToBooleanObject(lv > rv)
	case token.GREATER_EQ:
		return nativeBoolToBooleanObject(lv >= rv)
	case token.EQUAL:
		return nativeBoolToBooleanObject(lv == rv)
	case token.NOT_EQUAL:
		return nativeBoolToBooleanObject(lv != rv)
	case token.OR:
		return nativeBoolToBooleanObject((lv != 0) || (rv != 0))
	case token.XOR:
		return nativeBoolToBooleanObject((lv == 0) || (rv == 0))
	case token.AND:
		return nativeBoolToBooleanObject((lv != 0) && (rv != 0))
	case token.NOR:
		return nativeBoolToBooleanObject((lv == 0) && (rv == 0))
	default:
		return ERR_INVALID_OPERATOR(operator)
	}
}


func evalUIntegerInfixExpression(operator string, left, right IObject) IObject {
	lv := left.(*TUInteger).Value
	rv := right.(*TUInteger).Value
	
	switch operator {
	case token.PLUS:
		return &TUInteger { Value: lv + rv }
	case token.MINUS:
		return &TUInteger { Value: lv - rv }
	case token.ASTERISK:
		return &TUInteger { Value: lv * rv }
	case token.SLASH:
		return &TDouble { Value: float64(lv) / float64(rv) }
	case token.PERCENT:
		return &TUInteger { Value: lv % rv }
	case token.CARET:
		return &TUInteger { Value: uint64(math.Pow(float64(lv), float64(rv))) }
	case token.LESS:
		return nativeBoolToBooleanObject(lv < rv)
	case token.LESS_EQ:
		return nativeBoolToBooleanObject(lv <= rv)
	case token.GREATER:
		return nativeBoolToBooleanObject(lv > rv)
	case token.GREATER_EQ:
		return nativeBoolToBooleanObject(lv >= rv)
	case token.EQUAL:
		return nativeBoolToBooleanObject(lv == rv)
	case token.NOT_EQUAL:
		return nativeBoolToBooleanObject(lv != rv)
	case token.OR:
		return nativeBoolToBooleanObject((lv != 0) || (rv != 0))
	case token.XOR:
		return nativeBoolToBooleanObject((lv == 0) || (rv == 0))
	case token.AND:
		return nativeBoolToBooleanObject((lv != 0) && (rv != 0))
	case token.NOR:
		return nativeBoolToBooleanObject((lv == 0) && (rv == 0))
	default:
		return ERR_INVALID_OPERATOR(operator)
	}
}


func evalULongInfixExpression(operator string, left, right IObject) IObject {
	lv := left.(*TULong).Value
	rv := right.(*TULong).Value
	
	switch operator {
	case token.PLUS:
		return &TULong { Value: lv + rv }
	case token.MINUS:
		return &TULong { Value: lv - rv }
	case token.ASTERISK:
		return &TULong { Value: lv * rv }
	case token.SLASH:
		return &TDouble { Value: float64(lv) / float64(rv) }
	case token.PERCENT:
		return &TULong { Value: lv % rv }
	case token.CARET:
		return &TULong { Value: uint32(math.Pow(float64(lv), float64(rv))) }
	case token.LESS:
		return nativeBoolToBooleanObject(lv < rv)
	case token.LESS_EQ:
		return nativeBoolToBooleanObject(lv <= rv)
	case token.GREATER:
		return nativeBoolToBooleanObject(lv > rv)
	case token.GREATER_EQ:
		return nativeBoolToBooleanObject(lv >= rv)
	case token.EQUAL:
		return nativeBoolToBooleanObject(lv == rv)
	case token.NOT_EQUAL:
		return nativeBoolToBooleanObject(lv != rv)
	case token.OR:
		return nativeBoolToBooleanObject((lv != 0) || (rv != 0))
	case token.XOR:
		return nativeBoolToBooleanObject((lv == 0) || (rv == 0))
	case token.AND:
		return nativeBoolToBooleanObject((lv != 0) && (rv != 0))
	case token.NOR:
		return nativeBoolToBooleanObject((lv == 0) && (rv == 0))
	default:
		return ERR_INVALID_OPERATOR(operator)
	}
}


func evalUShortInfixExpression(operator string, left, right IObject) IObject {
	lv := left.(*TUShort).Value
	rv := right.(*TUShort).Value
	
	switch operator {
	case token.PLUS:
		return &TUShort { Value: lv + rv }
	case token.MINUS:
		return &TUShort { Value: lv - rv }
	case token.ASTERISK:
		return &TUShort { Value: lv * rv }
	case token.SLASH:
		return &TDouble { Value: float64(lv) / float64(rv) }
	case token.PERCENT:
		return &TUShort { Value: lv % rv }
	case token.CARET:
		return &TUShort { Value: uint16(math.Pow(float64(lv), float64(rv))) }
	case token.LESS:
		return nativeBoolToBooleanObject(lv < rv)
	case token.LESS_EQ:
		return nativeBoolToBooleanObject(lv <= rv)
	case token.GREATER:
		return nativeBoolToBooleanObject(lv > rv)
	case token.GREATER_EQ:
		return nativeBoolToBooleanObject(lv >= rv)
	case token.EQUAL:
		return nativeBoolToBooleanObject(lv == rv)
	case token.NOT_EQUAL:
		return nativeBoolToBooleanObject(lv != rv)
	case token.OR:
		return nativeBoolToBooleanObject((lv != 0) || (rv != 0))
	case token.XOR:
		return nativeBoolToBooleanObject((lv == 0) || (rv == 0))
	case token.AND:
		return nativeBoolToBooleanObject((lv != 0) && (rv != 0))
	case token.NOR:
		return nativeBoolToBooleanObject((lv == 0) && (rv == 0))
	default:
		return ERR_INVALID_OPERATOR(operator)
	}
}


func evalIfStatement(is *ast.TIfStatement, scope *TScope) IObject {
	for _, c := range is.Conditions {
		cd := Eval(c.Condition, scope)
		
		if _, ok := cd.(*TError); ok {
			return cd
		}
		
		if isTrue(cd) {
			return evalBlockStatements(c.Body.Statements, scope)
			/*
			switch o := c.Body.(type) {
			case *ast.TBlockStatement:
				return evalBlockStatements(o.Statements, scope)
			}
			return Eval(c.Body, scope)
			*/
		}
	}

	if is.Default != nil {
		return evalBlockStatements(is.Default.Statements, scope)
		/*
		switch o := ie.Default.(type) {
		case *ast.TBlockStatement:
			return evalBlockStatements(o.Statements, scope)
		}
		return Eval(ie.Default, scope)
		*/
	}

	return nil
}


func evalBlockStatements(block []ast.IStatement, scope *TScope) (result IObject) {
	for _, stmt := range block {
		result = Eval(stmt, scope)
		if result != nil {
			switch result.Type() {
			case ERROR_OBJ:
				return
			case RETURN_VALUE_OBJ:
				return
			}
			if _, ok := result.(*TExit); ok {
				return
			}
			if _, ok := result.(*TContinue); ok {
				return
			}
		}		
	}
	return
}


func unwrapReturnValue(obj IObject) IObject {
	if ret, ok := obj.(*TReturnValue); ok {
		return ret.Value
	}
	return obj
}


func evalForLoopStatement(fl *ast.TForLoopStatement, scope *TScope) IObject {
	loopScope := NewScope(scope)
	iter := Eval(fl.Iterator, loopScope)
	
	if objs, ok := ObjectState(iter, RaiseNULL); !ok {
		return objs
	}
	
	rng := fl.Range.(*ast.TFromToExpression).Expression
	
	if left, right, _, obj, ok := evalLeftRight(rng, loopScope, RaiseNULL); !ok {
		return obj
		
	} else {
		
		step := Eval(fl.Step, loopScope)
		
		if step != nil {
			if objs, ok := ObjectState(step, RaiseNULL); !ok {
				return objs
			}
		}
		
		var stepDir int64
		
		if obj, dir, ok := setFromTo(iter, left, right, step); !ok {
			return obj
		} else {
			iter = obj
			stepDir = dir
			
			if obj, ok := loopScope.Reset(fl.Iterator.Value, iter); !ok {
				return ERR_INVALID_ASSIGNMENT(obj)
			}
		}
		
		blockScope := NewScope(loopScope)
		
		for {
			block := Eval(fl.Block, blockScope)
			
			if block != nil {
				
				if obj, ok := block.(*TError); ok {
					return obj
				}
			
				if _, ok := block.(*TExit); ok {
					break
				}
				
				if ret, ok := block.(*TReturnValue); ok {
					return ret
				}
			}
			
			val, exec := loopFromTo(iter, left, right, step, stepDir)
			if exec {
				iter = val
				if obj, ok := loopScope.Reset(fl.Iterator.Value, iter); !ok {
					return ERR_INVALID_ASSIGNMENT(obj)
				}
			} else {
				break
			}
		}
	}
	
	return nil
}

func evalCallExpression(exp *ast.TCallExpression, scope *TScope) IObject {
	fn, ok := scope.Get(exp.Routine.String())
	if !ok {
		if f, ok := exp.Routine.(*ast.TFunctionLiteral); ok {
			fn = &TFunction { Literal: f, Scope: scope }
			scope.Set(exp.Routine.String(), fn)
		/*
		} else if idxExpr, ok := exp.Routine.(*ast.TIndexExpression); ok {
			aValue := Eval(idxExpr, scope)
			if _, ok := aValue.(*TError); ok {
				return aValue
			}

			if aFn, ok := aValue.(*TFunction); ok {
				fn = aFn
				
			} else {
				return ERR_UNKNOWN_IDENTIFIER(exp.Routine.(*ast.TIdentifier).Value)
			}
		*/
			
		} else if builtin, ok := BuiltinByName(exp.Routine.String()); ok {
			if builtin.Statement {
				return ERR_NOT_A_FUNCTION()
			}
			args := evalArgs(exp.Arguments, scope)
			return builtin.Fn(exp.Routine.String(), args...)
		/*
		} else if callExpr, ok := exp.Routine.(*ast.TCallExpression); ok {
			aValue := Eval(callExpr, scope)
			
			if _, ok := aValue.(*TError); ok {
				return aValue
			}
			fn = aValue
		*/
			
		} else if refExpr, ok := exp.Routine.(*ast.TReferenceExpression); ok {
			// in case a type member is a function
			aValue := evalReferenceExpression(refExpr, scope)
			if _, ok := aValue.(*TError); ok {
				return aValue
			}
			fn = aValue.(*TReference).Value
			
		} else {
			// TODO: can be reference expression (bad syntax)
			return ERR_UNKNOWN_IDENTIFIER(exp.Routine.(*ast.TIdentifier).Value)
		}
	}
	
	if _, ok := fn.(*TFunction); !ok {
		return ERR_NOT_A_FUNCTION()
	}
	
	f := fn.(*TFunction)
	
	if f.Literal == nil {
		return ERR_FUNCTION_NOT_DEFINED()
	}
	
	// check number of args...
	arc := len(exp.Arguments)
	if arc != len(f.Literal.Parameters) {
		return ERR_ARG_COUNT_MISMATCH()
	}
	
	newScope := NewScope(f.Scope)
	newScope.CallStack.Frames = append(newScope.CallStack.Frames, TCallFrame { FuncScope: newScope, CurrentCall: exp })
	
	defer func() {
		frame := newScope.CurrentFrame()
		
		if len(frame.defers) != 0 {
			frame.runDefers(newScope)
		}
		stack := newScope.CallStack
		stack.Frames = stack.Frames[0 : len(stack.Frames) - 1]
	}()
	
	// handle arguments to function parameters
	args := evalArgs(exp.Arguments, scope)
	
	if arc != 0 {
		n := 0
		
		for key, value := range f.Literal.Parameters {
			param := Eval(value.Parameter, f.Scope)
			if eobj, ok := evalArgument(args[n], param); !ok {
				return eobj
			}
			name , _ := key.(*ast.TIdentifier)
			newScope.Set(name.String(), args[n])
			n += 1
		}
	}
	
	f.Scope.Set("@_", &TInteger { Value: int64(arc) })
	result := Eval(f.Literal.Body, newScope)
	
	/*
	if objs, ok := ObjectState(result, AllowNULL); !ok {
		return objs
	}
	*/
	
	// robj.Value.Type() must match function type
	if robj, ok := result.(*TReturnValue); ok {
		if aobj, ok := evalArgument(robj.Value, f.ReturnType); !ok {
			return aobj
		} else {
			result = aobj
		}
	} else {
		result = f.ReturnType
	}
	
	// update ref parameters
	if arc != 0 {
		n := 0
		
		for key, value := range f.Literal.Parameters {
			if value.Ref {
				
				if arg, ok := exp.Arguments[n].(*ast.TIdentifier); ok {
					name , _ := key.(*ast.TIdentifier)
					
					if pobj, ok := newScope.Get(name.String()); ok {
						if IsDefaultType(pobj) {
							scope.Reset(arg.Value, pobj)
						}
					}
				}
			}
			n += 1
		}
	}
	
	return result
}


func evalCallStatement(stmt *ast.TCallExpression, scope *TScope) IObject {
	sb, ok := scope.Get(stmt.Routine.String())
	if !ok {
		if s, ok := stmt.Routine.(*ast.TSubLiteral); ok {
			sb = &TSub { Literal: s, Scope: scope }
			scope.Set(stmt.Routine.String(), sb)
			
		} else if builtin, ok := BuiltinByName(stmt.Routine.String()); ok {
			if !builtin.Statement {
				return ERR_NOT_A_SUB()
			}
			args := evalArgs(stmt.Arguments, scope)
			return builtin.Fn(stmt.Routine.String(), args...)
			
		} else if refExpr, ok := stmt.Routine.(*ast.TReferenceExpression); ok {
			// in case a type member is a sub
			aValue := evalReferenceExpression(refExpr, scope)
			if _, ok := aValue.(*TError); ok {
				return aValue
			}
			sb = aValue.(*TReference).Value
			
		} else {
			// TODO: can be reference expression (bad syntax)
			return ERR_UNKNOWN_IDENTIFIER(stmt.Routine.(*ast.TIdentifier).Value)
		}
	}
	
	if _, ok := sb.(*TSub); !ok {
		return ERR_NOT_A_SUB()
	}
	
	s := sb.(*TSub)
	
	if s.Literal == nil {
		return ERR_SUB_NOT_DEFINED()
	}
	
	// check number of args...
	arc := len(stmt.Arguments)
	if arc != len(s.Literal.Parameters) {
		return ERR_ARG_COUNT_MISMATCH()
	}
	
	newScope := NewScope(s.Scope)
	newScope.CallStack.Frames = append(newScope.CallStack.Frames, TCallFrame { FuncScope: newScope, CurrentCall: stmt })
	
	defer func() {
		frame := newScope.CurrentFrame()
		
		if len(frame.defers) != 0 {
			frame.runDefers(newScope)
		}
		stack := newScope.CallStack
		stack.Frames = stack.Frames[0 : len(stack.Frames) - 1]
	}()
	
	// handle arguments to sub parameters
	args := evalArgs(stmt.Arguments, scope)
	
	if arc != 0 {
		n := 0
		for key, value := range s.Literal.Parameters {
			param := Eval(value.Parameter, s.Scope)
			if eobj, ok := evalArgument(args[n], param); !ok {
				return eobj
			}
			name , _ := key.(*ast.TIdentifier)
			newScope.Set(name.String(), args[n])
			n += 1
		}
	}
	
	s.Scope.Set("@_", &TInteger { Value: int64(arc) })
	result := Eval(s.Literal.Body, newScope)
	
	/*
	if objs, ok := ObjectState(result, RaiseNULL); !ok {
		return objs
	}
	*/
	
	// update ref parameters
	if arc != 0 {
		n := 0
		
		for key, value := range s.Literal.Parameters {
			if value.Ref {
				
				if arg, ok := stmt.Arguments[n].(*ast.TIdentifier); ok {
					name , _ := key.(*ast.TIdentifier)
					
					if pobj, ok := newScope.Get(name.String()); ok {
						if IsDefaultType(pobj) {
							scope.Reset(arg.Value, pobj)
						}
					}
				}
			}
			n += 1
		}
	}
	
	return result
}


func evalFunctionStatement(FnStmt *ast.TFunctionStatement, scope *TScope) IObject {
	fnObj := evalFunctionLiteral(FnStmt.FunctionLiteral, scope)
	
	scope.Set(FnStmt.Name.String(), fnObj)

	return fnObj
}


func evalFunctionLiteral(fl *ast.TFunctionLiteral, scope *TScope) IObject {
	fn := &TFunction { Literal: fl, Scope: scope }
	
	//if fl.Values != nil {
	for key, value := range fl.Parameters {
		if name, ok := key.(*ast.TIdentifier); !ok {
			return ERR_INVALID_IDENTIFIER(name.Value)
		} else {
			obj := Eval(value.Parameter, fn.Scope)
			if _, ok := obj.(*TError); ok {
				return obj
			}
			
			fn.Scope.Set(name.String(), obj)
		}
	}
	// }
	
	obj := Eval(fl.ReturnType, scope)
	if sobj, ok := ObjectState(obj, AllowNULL); !ok {
		return sobj
	}
	
	fn.ReturnType = obj	
	
	return fn
}


func evalIndexExpression(left, index IObject) IObject {	
	var idx IObject
	
	if _, ok := index.(*TString); !ok {
		if val, ok := evalNumIndex(index); !ok {
			return ERR_INVALID_INDEX(index)
			
		} else {
			idx = &TUInteger { Value: val }
		}
		
	} else if _, ok := left.(*TMap); ok {
		idx = index
		
	} else {
		return ERR_INVALID_INDEX(index)
	}
	
	switch left.(type) {
	case *TArray:
		return evalArrayIndexExpression(left, idx)
		
	case *TList:
		return evalListIndexExpression(left, idx)
		
	case *TMap:
		return evalMapIndexExpression(left, idx)
		
	default:
		return ERR_INVALID_INDEX_OPERATOR(left)
	}
}


func evalAssignExpression(a *ast.TAssignExpression, scope *TScope) IObject {
	val := Eval(a.Value, scope)
	
	if obj, ok := ObjectState(val, AllowNULL); !ok {
		return obj
	}
	
	var name string
	var lhs IObject
	
	lhsScope := scope
		
	switch nodeType := a.Name.(type) {
	case *ast.TIdentifier:
		name = nodeType.Value
		
	case *ast.TIndexExpression:
		name = nodeType.Left.(*ast.TIdentifier).Value
		
	case *ast.TReferenceExpression:
		robj := evalReferenceExpression(nodeType, scope)
		if obj, ok := ObjectState(robj, AllowNULL); !ok {
			return obj
		}
		lhs = robj.(*TReference).Value
		name = robj.(*TReference).Name
		lhsScope = robj.(*TReference).Scope
	}
	
	if obj, ok := lhsScope.Get(name); !ok {
		return ERR_UNKNOWN_IDENTIFIER(name)
	} else if lhs == nil {
		lhs = obj
	}
	
	// read-only types
	switch lhs.(type) {
	case *TConst, *TConstGroup:
		return ERR_CONST_ASSIGNMENT()
	case *TEnum, *TEnumGroup:
		return ERR_ENUM_ASSIGNMENT()
	}
	
	val, _ = Deref(val)
	
	// try default (simple) data types
	if right, ok := AssignDefault(lhs, val); ok {
		if eobj, ok := lhsScope.Reset(name, right); !ok {
			return ERR_INVALID_ASSIGNMENT(eobj)
		}
		return nil
	}
		
	// other (complex) types
	switch lhs.(type) {
	case *TArray:
		return evalToArray(a, name, lhs, lhsScope, val)
	case *TList:
		return evalToList(a, name, lhs, lhsScope, val)
	case *TMap:
		return evalToMap(a, name, lhs, lhsScope, val)
	case *TType:
		return evalToType(lhs, name, lhsScope, val)
	}
	
	return ERR_TYPE_MISMATCH(lhs, val)
}


func Deref(obj IObject) (IObject, bool) {
	flag := false
	
	// resolve reference
	if _, ok := obj.(*TReference); ok {
		obj = obj.(*TReference).Value
		flag = true
	}
	
	switch obj.(type) {
	case *TConst:
		obj = obj.(*TConst).Value
		flag = true
	case *TEnum:
		obj = obj.(*TEnum).Value
		flag = true
	}
	
	return obj, flag
}


func evalArgs(args []ast.IExpression, scope *TScope) []IObject {
	//TODO: Refactor this to accept the params and args, go ahead and
	// update scope while looping and return the Scope object.
	e := []IObject{}
	for _, v := range args {
		item := Eval(v, scope)
		e = append(e, item)
	}
	return e
}


func evalArgument(arg, param IObject) (IObject, bool) {
	if eobj, ok := ObjectState(arg, AllowNULL); !ok {
		return eobj, false
	}
	
	if eobj, ok := ObjectState(param, AllowNULL); !ok {
		return eobj, false
	}
	
	arg, _ = Deref(arg)
	
	if dobj, ok := AssignDefault(param, arg); ok {
		if param.Type() == NULL_OBJ || dobj.Type() == param.Type(){
			return dobj, true
		}
		return ERR_TYPE_MISMATCH(arg, param), false
	}
	
	if param.Type() == TYPE_OBJ || param.Type() == TYPE_REF_OBJ {
		if arg.Type() != TYPE_REF_OBJ {
			return ERR_TYPE_MISMATCH(arg, param), false
		}
		if arg.(*TType).Name != param.(*TType).Name {
			return ERR_TYPES_DO_NOT_MATCH(arg.(*TType).Name, param.(*TType).Name), false
		}
		return arg, true
	}
	
	if arg.Type() != param.Type() {
		return ERR_TYPE_MISMATCH(arg, param), false
	}
	
	switch param.Type() {
	case CONST_GROUP_OBJ:
		if arg.(*TConstGroup).Name != param.(*TConstGroup).Name {
			return ERR_TYPES_DO_NOT_MATCH(arg.(*TConstGroup).Name, param.(*TConstGroup).Name), false
		}
	
	case ENUM_GROUP_OBJ:
		if arg.(*TEnumGroup).Name != param.(*TEnumGroup).Name {
			return ERR_TYPES_DO_NOT_MATCH(arg.(*TEnumGroup).Name, param.(*TEnumGroup).Name), false
		}
	
	case ARRAY_OBJ:
		return evalArgument(arg.(*TArray).DataType, param.(*TArray).DataType)
	}
	
	return arg, true
}


func evalConstStatement(cs *ast.TConstStatement, scope *TScope) IObject {
	var cobj IObject
	var constScope *TScope
	
	if cs.Named {
		constScope = NewScope(nil)
	} else {
		constScope = scope
	}
	
	for i, name := range cs.Names {
		value := Eval(cs.Values[i], scope)
		if eobj, ok := value.(*TError); ok {
			return eobj
		}
		cobj = &TConst { Value: value }
		constScope.Set(name.String(), cobj)
	}
	
	if cs.Named {
		cobj = &TConstGroup { Scope: constScope, Name: cs.Name.Value }
		scope.Set(cs.Name.Value, cobj)
	}
		
	return nil
}


func evalEnumStatement(es *ast.TEnumStatement, scope *TScope) IObject {
	var nobj, value IObject
	var enumScope *TScope
	
	if es.Named {
		enumScope = NewScope(nil)
	} else {
		enumScope = scope
	}
	
	// data type uinteger (index compatible), no negative values allowed
	// default base value is 0
	
	baseValue := UIntegerValue(0)
	incrValue := UIntegerValue(1)
	
	for i, name := range es.Names {
		if es.Values[i] == nil {
			value = baseValue
			baseValue, _ = AddNumber(baseValue, incrValue)
		} else {
			value = Eval(es.Values[i], scope)
			if eobj, ok := value.(*TError); ok {
				return eobj
			}
			if val, ok := evalNumIndex(value); !ok {
				return ERR_INVALID_ENUM_VALUE()
			} else {
				value = UIntegerValue(val)
			}
			baseValue, _ = AddNumber(value, incrValue)
		}
		
		nobj = &TEnum { Value: value }
		enumScope.Set(name.String(), nobj)
	}
	
	if es.Named {
		nobj = &TEnumGroup { Scope: enumScope, Name: es.Name.Value }
		scope.Set(es.Name.Value, nobj)
	}
		
	return nil
}


func evalArrayIndexExpression(array, index IObject) IObject {
	arrObj := array.(*TArray)
	idx := index.(*TUInteger).Value
	max := uint64(len(arrObj.Elements) - 1)
	
	if idx < 0 || idx > max {
		return ERR_SUBSCRIPT_OUT_OF_RANGE(index)
	}
	
	return arrObj.Elements[idx]
}

/*
func evalArrayLiteral(node *ast.TArrayLiteral, scope *TScope) IObject {
	elems := evalExpressions(node.Elements, scope)
	if len(elems) == 1 {
		if obj, ok := ObjectState(elems[0], RaiseNULL); !ok {
			return obj
		}
	}
	fmt.Println("AL")
	return &TArray { Elements: elems }
}
*/

func evalListIndexExpression(list, index IObject) IObject {
	lstObj := list.(*TList)
	idx := index.(*TUInteger).Value
	max := uint64(len(lstObj.Elements) - 1)
	
	if idx < 0 || idx > max {
		return ERR_SUBSCRIPT_OUT_OF_RANGE(index)
	}
	
	return lstObj.Elements[idx]
}

/*
func evalListLiteral(node *ast.TListLiteral, scope *TScope) IObject {
	elems := evalExpressions(node.Elements, scope)
	if len(elems) == 1 {
		if obj, ok := ObjectState(elems[0], RaiseNULL); !ok {
			return obj
		}
	}
	return &TList { Elements: elems }
}
*/


func evalMapLiteral(node *ast.TMapLiteral, scope *TScope) IObject {
	pairs := make(map[TMapKey]TMapPair)
	
	for keyNode, valueNode := range node.Pairs {
		key := Eval(keyNode, scope)
		if obj, ok := ObjectState(key, RaiseNULL); !ok {
			return obj
		}
		
		hashKey, ok := key.(IMapable)
		if !ok {
			return ERR_INVALID_HASH_KEY(key)
		}
		
		value := Eval(valueNode, scope)
		if obj, ok := ObjectState(value, AllowNULL); !ok {
			return obj
		}
		
		hashed := hashKey.MapKey()
		pairs[hashed] = TMapPair { Key: key, Value: value }
	}
	return &TMap {Pairs: pairs }
}


func evalMapIndexExpression(hash, index IObject) IObject {
	hashObj := hash.(*TMap)
	
	key, ok := index.(IMapable)
	if !ok {
		return ERR_INVALID_HASH_KEY(index)
	}
	
	pair, ok := hashObj.Pairs[key.MapKey()]
	if !ok {
		return NULL
	}
	
	return pair.Value
}


func evalScopeStatement(ss *ast.TScopeStatement, scope *TScope) IObject {
	stmt := Eval(ss.Block, NewScope(scope))
		
	if stmt != nil {
		if eobj, ok := stmt.(*TError); ok {
			return eobj
		}
		
		if ret, ok := stmt.(*TReturnValue); ok {
			if ret.Value != nil {
				return ret
			}
		}
	}
	
	return nil
}


func evalWhileLoopStatement(wl *ast.TWhileLoopStatement, scope *TScope) IObject {
	
	condition := Eval(wl.Condition, scope)
	if obj, ok := ObjectState(condition, RaiseNULL); !ok {
		return obj
	}

	for isTrue(condition) {
		block := Eval(wl.Block, scope)
		
		if block != nil {
			
			if obj, ok := block.(*TError); ok {
				return obj
			}
		
			if _, ok := block.(*TExit); ok {
				break
			}
			
			if ret, ok := block.(*TReturnValue); ok {
				if ret.Value != nil {
					return ret
				}
			}
		}
		
		condition = Eval(wl.Condition, scope)
		if _,ok := condition.(*TError); ok {
			return condition
		}
	}
	
	return nil
}


func isBooleanExpression(exp string) bool {
	switch exp {
	case token.EQUAL, token.NOT_EQUAL, token.LESS, token.LESS_EQ, 
	token.GREATER, token.GREATER_EQ, token.IS, token.NOT, token.AND, 
	token.OR, token.XOR, token.NOR:
		return true	
	}
	
	return false
}


func evalTypeStatement(t *ast.TTypeStatement, scope *TScope) IObject {
	typeScope := NewScope(nil)
	var obj IObject
	
	for key, value := range t.Members {
		if name, ok := key.(*ast.TIdentifier); !ok {
			return ERR_INVALID_IDENTIFIER(name.Value)
		} else {
			obj = Eval(value, scope)
			if _, ok := obj.(*TError); ok {
				return obj
			}
			if tobj, ok := obj.(*TType); ok {
				obj = CopyType(tobj)
			}
			typeScope.Set(name.String(), obj)
		}
	}
	
	obj = &TType { Scope: typeScope, Name: t.Name.Value }
	
	scope.Set(t.Name.Value, obj)
	
	return obj
}


func objectToBool(o IObject) bool {
	// implicit allows numbers but not strings
	if obj, ok := BooleanObject(o, IMPLICIT); ok {
		return obj.(*TBoolean).Value
	} else {
		return false
	}
}


func IsDefaultType(obj IObject) bool {
	switch obj.(type) {
	case *TBoolean, *TByte, *TDouble, *TSingle, *TInteger, *TLong, *TShort, *TString, *TUByte, *TUInteger, *TULong, *TUShort:
		return true
	default:
		if dobj, ok := Deref(obj); ok {
			return IsDefaultType(dobj)
		}
	}
	
	return false
}


func isEqual(value, other IObject) (bool, IObject, bool) {
	/*
	if left == nil && right == nil {
		return true
	}
	*/
	
	if value.Type() == NULL_OBJ && other.Type() == NULL_OBJ {
		return true, nil, true
	}
	
	// string?
	if value.Type() == STRING_OBJ || other.Type() == STRING_OBJ {
		if value.Type() == STRING_OBJ && other.Type() == STRING_OBJ {
			if value.Inspect() == other.Inspect() {
				return true, nil, true
			}
		} else {
			errObj := ERR_INVALID_DATA_TYPE_COMPARISON(value, other, nil)
			return false, errObj, false
		}
	}
	
	// number ?
	result := CompareNumber(value, other)
	if result == CmpEQUAL {
		return true, nil, true
	}
	
	errObj := ERR_INVALID_DATA_TYPE_COMPARISON(value, other, nil)
	return false, errObj, true
}


func isFromTo(value, low, high IObject) (bool, IObject, bool) {	
	// string?
	if low.Type() == STRING_OBJ || high.Type() == STRING_OBJ || value.Type() == STRING_OBJ {
		if low.Type() == STRING_OBJ && high.Type() == STRING_OBJ && value.Type() == STRING_OBJ {
			if value.Inspect() >= low.Inspect() && value.Inspect() <= high.Inspect() {
				return true, nil, true
			} else {
				return false, nil, true
			}			
		} else {
			errObj := ERR_INVALID_DATA_TYPE_COMPARISON(value, low, high)
			return false, errObj, false
		}
	}
	
	// should be a number		
	result := CompareNumberWithin(value, low, high)
	
	if result == CmpWITHIN {
		return true, nil, true
		
	} else if result == CmpINVALID {
		errObj := ERR_INVALID_DATA_TYPE_COMPARISON(value, low, high)
		return false, errObj, false
	} else { // no match
		return false, nil, true
	}
	
	err := ERR_INVALID_DATA_TYPE_COMPARISON(value, low, high)
	return false, err, true
}


func isTrue(obj IObject) bool {
	switch obj.(type) {
	case *TBoolean:
		return obj.(*TBoolean).Value
	case *TNull:
		return false
	}
	return true
}

/*
func isTypeRef(names string) bool {
	return strings.Contains(names, token.DOT)
}
*/

func loopFromTo(iter, left, right, step IObject, stepDir int64) (IObject, bool) {
	
	switch stepDir {
		
	case 1:
		iter,_ = AddNumber(iter, step)
		cmp := CompareNumber(iter, right)
		if cmp == CmpGREATER {
			return iter, false
		}
		
	case -1:
		iter,_ = AddNumber(iter, step)
		cmp := CompareNumber(iter, right)
		if cmp == CmpLESS {
			return iter, false
		}
		
	default:
		iter,_ = AddNumber(iter, IntegerValue(1))
		cmp := CompareNumber(iter, right)
		if cmp == CmpGREATER {
			return iter, false
		}
	}
	
	return iter, true
}

func setFromTo(iter, left, right, step IObject) (IObject, int64, bool) {
	/* If left > right without step -x, do not execute loop, otherwise
	 * assign left to iter and iterate -x until iter equals right
	 * 
	 * If left < right assign left to iter and iterate (step x) or 1
	 * unitl iter equals right
	*/
	
	if left.Type() != right.Type() {
		err := ERR_INVALID_DATA_TYPE_COMPARISON(iter, left, right)
		return err, 0, false
	}
		
	var stepDir int64
	
	if step != nil {
		cmp := CompareNumber(step, IntegerValue(0))
		
		switch cmp {
			
		case CmpGREATER:
			// step +N
			stepDir = 1
			
		case CmpLESS:
			// step -N
			stepDir = -1
			
		default:
			// invalid step
			return nil, 0, false
		}
		
	} else {
		// no step, so left should be smaller than right
		cmp := CompareNumber(left, right)
		if cmp == CmpGREATER {
			return nil, 0, false
		}
		stepDir = 0
	}
	
	// set iter
	if obj, ok := SetNumber(iter, left); !ok {
		return obj, 0, false
	} else {
		iter = obj
	}	
	
	return iter, stepDir, true
}


func CopyType(t *TType) IObject {
	scope := NewScope(nil)
	
	// each member must be registered in the new scope
	for k, v := range t.Scope.store {
		scope.Set(k, v)
	}
	
	return &TType { Scope: scope, Name: t.Name, typeName: TYPE_REF_OBJ }
}

