// ast/ast.go

package ast

import (
	"bytes"
	"strings"
	"SB/token"
)


type INode interface {
	TokenLiteral()	string
	String()				string
}


type IStatement interface {
	INode
	statementNode()
}


type IExpression interface {
	INode
	expressionNode()
}


type TProgram struct {
	Statements []IStatement
}
func (p *TProgram) TokenLiteral() string {
	if len(p.Statements) > 0 {
		return p.Statements[0].TokenLiteral()
	}
	return ""
}
func (p *TProgram) String() string {
	var out bytes.Buffer
	
	for _, s := range p.Statements {
		out.WriteString(s.String())
	}
	return out.String()
}


type TIdentifier struct {
	Token token.TToken
	Value string
}
func (i *TIdentifier) expressionNode() {}
func (i *TIdentifier) TokenLiteral() string { return i.Token.Literal }
func (i *TIdentifier) String() string { return i.Value }


type TVarStatement struct {
	Token  token.TToken
	Names  []*TIdentifier
	Values []IExpression
}

func (vs *TVarStatement) statementNode()       {}
func (vs *TVarStatement) TokenLiteral() string { return vs.Token.Literal }
func (vs *TVarStatement) String() string {
	var out bytes.Buffer

	out.WriteString(vs.TokenLiteral() + " ")
	
	names := []string{}
	for _, name := range vs.Names {
		names = append(names, name.String())
	}
	
	out.WriteString(strings.Join(names, ", "))

	if len(vs.Values) == 0 { //e.g. 'let x'
		out.WriteString(";")
		return out.String()
	}

	out.WriteString(": ")

	values := []string{}
	for _, value := range vs.Values {
		values = append(values, value.String())
	}
	out.WriteString(strings.Join(values, ", "))

	return out.String()
}


type TReturnStatement struct {
	Token					token.TToken
	ReturnValue		IExpression
}

func (rs *TReturnStatement) statementNode() {}
func (rs *TReturnStatement) TokenLiteral() string { return rs.Token.Literal }
func (rs *TReturnStatement) String() string {
	var out bytes.Buffer
	
	out.WriteString(rs.TokenLiteral() + " ")
	
	if rs.ReturnValue != nil {
		out.WriteString(rs.ReturnValue.String())
	}
	
	out.WriteString(token.SEMICOLON)
	
	return out.String()
}


type TAssignExpression struct {
	Token token.TToken
	Name  IExpression
	Value IExpression
}

func (ae *TAssignExpression) expressionNode()      {}
func (ae *TAssignExpression) TokenLiteral() string { return ae.Token.Literal }
func (ae *TAssignExpression) String() string {
	var out bytes.Buffer

	out.WriteString(ae.Name.String())
	//out.WriteString(" = ")
	out.WriteString(ae.Token.Literal)
	out.WriteString(ae.Value.String())

	return out.String()
}


type TExpressionStatement struct {
	Token				token.TToken
	Expression	IExpression
}
func (es *TExpressionStatement) statementNode() {}
func (es *TExpressionStatement) TokenLiteral() string { return es.Token.Literal }
func (es *TExpressionStatement) String() string {
	if es.Expression != nil {
		str := es.Expression.String()
		if str[len(str)-1:] !=";" {
			str = str + ";"
		}
		return str
	}
	return ""
}


type TIntegerLiteral struct {
	Token token.TToken
	Value int64
}

func (il *TIntegerLiteral) expressionNode() {}
func (il *TIntegerLiteral) TokenLiteral() string { return il.Token.Literal }
func (il *TIntegerLiteral) String() string { return il.Token.Literal }


type TFloatLiteral struct {
	Token token.TToken
	Value float64
}

func (fl *TFloatLiteral) expressionNode() {}
func (fl *TFloatLiteral) TokenLiteral() string { return fl.Token.Literal }
func (fl *TFloatLiteral) String() string { return fl.Token.Literal }


type TStringLiteral struct {
	Token token.TToken
	Value string
}

func (sl *TStringLiteral) expressionNode() {}
func (sl *TStringLiteral) TokenLiteral() string { return sl.Token.Literal }
func (sl *TStringLiteral) String() string { return sl.Token.Literal }

type TPrefixExpression struct {
	Token			token.TToken
	Operator	string
	Right			IExpression
}

func (pe *TPrefixExpression) expressionNode() {}
func (pe *TPrefixExpression) TokenLiteral() string { return pe.Token.Literal }
func (pe *TPrefixExpression) String() string {
	var out bytes.Buffer
	
	out.WriteString("(")
	out.WriteString(pe.Operator)
	out.WriteString(pe.Right.String())
	out.WriteString(")")
	
	return out.String()
}

type TInfixExpression struct {
	Token			token.TToken
	Left			IExpression
	Operator	string
	Right			IExpression
}

func (oe *TInfixExpression) expressionNode() {}
func (oe *TInfixExpression) TokenLiteral() string { return oe.Token.Literal }
func (oe *TInfixExpression) String() string {
	var out bytes.Buffer
	
	out.WriteString("(")
	out.WriteString(oe.Left.String())
	out.WriteString(" " + oe.Operator + " ")
	out.WriteString(oe.Right.String())
	out.WriteString(")")
	
	return out.String()
}

type TBoolean struct {
	Token token.TToken
	Value bool
}

func (b *TBoolean) expressionNode() {}
func (b *TBoolean) TokenLiteral() string { return b.Token.Literal }
func (b *TBoolean) String() string { return b.Token.Literal }


type TIfStatement struct {
	Token				token.TToken
	Conditions  []*TIfCondition
	Default			*TBlockStatement // INode
}

func (is *TIfStatement) statementNode()      {}
func (is *TIfStatement) TokenLiteral() string { return is.Token.Literal }
func (is *TIfStatement) String() string {
	var out bytes.Buffer

	for i, c := range is.Conditions {
		if i == 0 {
			out.WriteString("if ")
		} else {
			out.WriteString("elseif ")
		}
		out.WriteString(c.String())
	}

	if is.Default != nil {
		out.WriteString(" else")
		out.WriteString(" do ")
		out.WriteString(is.Default.String())
		out.WriteString(" end")
	}

	return out.String()
}


type TIfCondition struct {
	Token token.TToken
	Condition IExpression 
	Body			*TBlockStatement // INode
}

func (ic *TIfCondition) expressionNode()      {}
func (ic *TIfCondition) TokenLiteral() string { return ic.Token.Literal }
func (ic *TIfCondition) String() string {
	var out bytes.Buffer

	out.WriteString(ic.Condition.String())
	out.WriteString(" do ")
	out.WriteString(ic.Body.String())
	out.WriteString(" end")

	return out.String()
}


type TBlockStatement struct {
	Token				token.TToken
	Statements	[]IStatement
}

func (bs *TBlockStatement) statementNode() {}
func (bs *TBlockStatement) TokenLiteral() string { return bs.Token.Literal }
func (bs *TBlockStatement) String() string {
	var out bytes.Buffer
	
	for _, s := range bs.Statements {
		str := s.String()

		out.WriteString(str)
		if str[len(str)-1:] != token.SEMICOLON {
			out.WriteString(token.SEMICOLON)
		}
	}
	
	return out.String()
}

/*
type TArrayLiteral struct {
	Token					token.TToken
	Elements			[]IExpression
}
func (al *TArrayLiteral) expressionNode() {}
func (al *TArrayLiteral) TokenLiteral() string { return al.Token.Literal }
func (al *TArrayLiteral) String() string {
	var out bytes.Buffer
	
	elems := []string{}
	for _, el := range al.Elements {
		elems = append(elems, el.String())
	}
	
	out.WriteString("[")
	out.WriteString(strings.Join(elems, ", "))
	out.WriteString("]")
	
	return out.String()
}
*/

/*
type TListLiteral struct {
	Token					token.TToken
	Elements			[]IExpression
}
func (ll *TListLiteral) expressionNode() {}
func (ll *TListLiteral) TokenLiteral() string { return ll.Token.Literal }
func (ll *TListLiteral) String() string {
	var out bytes.Buffer
	
	elems := []string{}
	for _, el := range ll.Elements {
		elems = append(elems, el.String())
	}
	
	out.WriteString("[")
	out.WriteString(strings.Join(elems, ", "))
	out.WriteString("]")
	
	return out.String()
}
*/


type TCallExpression struct {
	Token			token.TToken
	Statement	bool
	Routine		IExpression
	Arguments	[]IExpression
}

func (ce *TCallExpression) expressionNode() {}
func (ce *TCallExpression) TokenLiteral() string { return ce.Token.Literal }
func (ce *TCallExpression) String() string {
	var out bytes.Buffer
	
	args := []string{}
	for _, a := range ce.Arguments {
		args = append(args, a.String())
	}
	
	out.WriteString(ce.Routine.String())
	out.WriteString("(")
	out.WriteString(strings.Join(args, ", "))
	out.WriteString(")")
	
	return out.String()
}

/*
type TCallStatement struct {
	Token			token.TToken
	Routine		IExpression
	Arguments	[]IExpression
}

func (cs *TCallStatement) expressionNode() {}
func (cs *TCallStatement) TokenLiteral() string { return cs.Token.Literal }
func (cs *TCallStatement) String() string {
	var out bytes.Buffer
	
	args := []string{}
	for _, a := range cs.Arguments {
		args = append(args, a.String())
	}
	
	out.WriteString(cs.Routine.String())
	out.WriteString("(")
	out.WriteString(strings.Join(args, ", "))
	out.WriteString(")")
	
	return out.String()
}
*/


type TParameters struct {
	Ref				bool
	Parameter IExpression
}

type TFunctionLiteral struct {
	Token				token.TToken
	Parameters	map[IExpression]*TParameters
	ReturnType	IExpression
	Body				*TBlockStatement
	//Values 			map[string]IExpression
}
func (fl *TFunctionLiteral) expressionNode() {}
func (fl *TFunctionLiteral) TokenLiteral() string { return fl.Token.Literal }
func (fl *TFunctionLiteral) String() string {
	var out bytes.Buffer
	var ref string
	
	if fl == nil {
		return out.String()
	}
	
	params := []string{}
	
	for key, value := range fl.Parameters {
		if value.Ref {
			ref = "ref "
		} else {
			ref = ""
		}
		params = append(params, ref + key.String() + ": " + value.Parameter.String())
	}
	
	out.WriteString(fl.TokenLiteral())
	out.WriteString("(")
	out.WriteString(strings.Join(params, ", "))
	out.WriteString(")")
	
	if fl.ReturnType != nil {
		out.WriteString(": " + fl.ReturnType.String())
	}
	
	out.WriteString(" do\n")
	out.WriteString(fl.Body.String())
	out.WriteString("\nend;")
	
	return out.String()
}


type TFunctionStatement struct {
	Token           token.TToken
	Name            *TIdentifier
	FunctionLiteral *TFunctionLiteral
}

func (f *TFunctionStatement) statementNode() {}
func (f *TFunctionStatement) TokenLiteral() string { return f.Token.Literal }
func (f *TFunctionStatement) String() string {
	var out bytes.Buffer

	out.WriteString("function ")
	out.WriteString(f.Name.String())

	params := []string{}
	for _, p := range f.FunctionLiteral.Parameters {
		params = append(params, p.Parameter.String())
	}
	out.WriteString(" (")
	out.WriteString(strings.Join(params, ", "))
	out.WriteString(") ")
	out.WriteString("do\n")
	out.WriteString(f.FunctionLiteral.Body.String())
	out.WriteString("\nend;")

	return out.String()
}


type TSubLiteral struct {
	Token				token.TToken
	Name				*TIdentifier
	Parameters	map[IExpression]*TParameters
	Body				*TBlockStatement
}
func (ss *TSubLiteral) expressionNode() {}
func (ss *TSubLiteral) TokenLiteral() string { return ss.Token.Literal }
func (ss *TSubLiteral) String() string {
	var out bytes.Buffer

	out.WriteString("function ")
	out.WriteString(ss.Name.String())

	params := []string{}
	for _, p := range ss.Parameters {
		params = append(params, p.Parameter.String())
	}
	out.WriteString(" (")
	out.WriteString(strings.Join(params, ", "))
	out.WriteString(") ")
	out.WriteString("do\n")
	out.WriteString(ss.Body.String())
	out.WriteString("\nend;")

	return out.String()
}


type TIndexExpression struct {
	Token token.TToken
	Left	IExpression
	Index	IExpression
}

func (ie *TIndexExpression) expressionNode() {}
func (ie *TIndexExpression) TokenLiteral() string { return ie.Token.Literal }
func (ie *TIndexExpression) String() string {
	var out bytes.Buffer
	
	out.WriteString("(")
	out.WriteString(ie.Left.String())
	out.WriteString("[")
	out.WriteString(ie.Index.String())
	out.WriteString("])")
	
	return out.String()
}


type TMapLiteral struct {
	Token token.TToken
	Pairs	map[IExpression]IExpression
}
func (hl *TMapLiteral) expressionNode() {}
func (hl *TMapLiteral) TokenLiteral() string { return hl.Token.Literal }
func (hl *TMapLiteral) String() string {
	var out bytes.Buffer
	
	pairs := []string{}
	for key, value := range hl.Pairs {
		pairs = append(pairs, key.String() + ":" + value.String())
	}
	
	out.WriteString("{")
	out.WriteString(strings.Join(pairs, ", "))
	out.WriteString("}")
	
	return out.String()
}


type TScopeStatement struct {
	Token     token.TToken
	Block			*TBlockStatement
}

func (ss *TScopeStatement) statementNode() {}
func (ss *TScopeStatement) TokenLiteral() string { return ss.Token.Literal }
func (ss *TScopeStatement) String() string {
	var out bytes.Buffer

	out.WriteString("scope ")
	out.WriteString(" do\n")
	out.WriteString(ss.Block.String())
	out.WriteString("\nend;")

	return out.String()
}


type TWhileLoopStatement struct {
	Token     token.TToken
	Condition IExpression
	Block			*TBlockStatement
}

func (wl *TWhileLoopStatement) statementNode() {}
func (wl *TWhileLoopStatement) TokenLiteral() string { return wl.Token.Literal }
func (wl *TWhileLoopStatement) String() string {
	var out bytes.Buffer

	out.WriteString("while ")
	out.WriteString(wl.Condition.String())
	out.WriteString(" do\n")
	out.WriteString(wl.Block.String())
	out.WriteString("\nend;")

	return out.String()
}


type TExitExpression struct {
	Token token.TToken
}
func (bs *TExitExpression) expressionNode() {}
func (bs *TExitExpression) TokenLiteral() string { return bs.Token.Literal }
func (bs *TExitExpression) String() string { return bs.Token.Literal }


type TContinueExpression struct {
	Token token.TToken
}
func (ce *TContinueExpression) expressionNode() {}
func (ce *TContinueExpression) TokenLiteral() string { return ce.Token.Literal }

func (ce *TContinueExpression) String() string { return ce.Token.Literal }


type TSelectStatement struct {
	Token        token.TToken
	Expression	IExpression
	Cases				[]IStatement
}

func (ss *TSelectStatement) statementNode() {}
func (ss *TSelectStatement) TokenLiteral() string { return ss.Token.Literal }
func (ss *TSelectStatement) String() string {
	var out bytes.Buffer

	out.WriteString("select ")
	out.WriteString(ss.Expression.String())
	out.WriteString(" do\n")
	
	cases := []string{}
	
	for _, i := range ss.Cases {
		cases = append(cases, i.String())
	}

	out.WriteString(strings.Join(cases, " "))
	out.WriteString(" end;\n")
	
	return out.String()
}

type TCaseBlock struct {
	Token					token.TToken
	Block					*TBlockStatement
	Expressions		[]IExpression
}

func (cb *TCaseBlock) statementNode() {}
func (cb *TCaseBlock) TokenLiteral() string { return cb.Token.Literal }
func (cb *TCaseBlock) String() string {
	var out bytes.Buffer
	
	out.WriteString(cb.Block.String())
	
	return out.String()
}

type TDefaultBlock struct {
	Token		token.TToken
	Block		*TBlockStatement
}

func (db *TDefaultBlock) statementNode() {}
func (db *TDefaultBlock) TokenLiteral() string { return db.Token.Literal }
func (db *TDefaultBlock) String() string {
	var out bytes.Buffer
	
	out.WriteString(db.Block.String())
	
	return out.String()
}

type TFromToExpression struct {
	Token				token.TToken
	Expression	IExpression
}

func (ft *TFromToExpression) expressionNode() {}
func (ft *TFromToExpression) TokenLiteral() string { return ft.Token.Literal }
func (ft *TFromToExpression) String() string {
	var out bytes.Buffer
	
	out.WriteString(ft.Expression.String())
	
	return out.String()
}

type TForLoopStatement struct {
	Token  			token.TToken
	Iterator		*TIdentifier
	Range				IExpression
	Step 				IExpression
	Block  			*TBlockStatement
}

func (fl *TForLoopStatement) statementNode() {}
func (fl *TForLoopStatement) TokenLiteral() string { return fl.Token.Literal }
func (fl *TForLoopStatement) String() string {
	var out bytes.Buffer

	out.WriteString("for ")
	out.WriteString(fl.Iterator.TokenLiteral())
	out.WriteString(" = ")
	out.WriteString(fl.Range.String())
	
	if fl.Step != nil {
		out.WriteString(" step ")
		out.WriteString(fl.Step.String())
	}
	
	out.WriteString(" do\n")
	out.WriteString(fl.Block.String())
	out.WriteString("\nend;\n")

	return out.String()
}

type TStepExpression struct {
	Token				token.TToken
}

func (se *TStepExpression) expressionNode() {}
func (se *TStepExpression) TokenLiteral() string { return se.Token.Literal }
func (se *TStepExpression) String() string { return se.Token.Literal }


type TTypeStatement struct {
	Token			token.TToken
	Name			*TIdentifier
	Members 	map[IExpression]IExpression
}

func (t *TTypeStatement) statementNode() {}
func (t *TTypeStatement) TokenLiteral() string { return t.Token.Literal }
func (t *TTypeStatement) String() string {
	var out bytes.Buffer

	members := []string{}
	for key, value := range t.Members {
		members = append(members, key.String()+ "." + value.String())
	}
	out.WriteString("type ")
	out.WriteString(t.Name.Value)
	out.WriteString(" of\n")
	out.WriteString(strings.Join(members, ", "))
	out.WriteString("\nend;\n")

	return out.String()
}


type TReferenceExpression struct {
	Token			token.TToken
	Root			IExpression
	Members		[]IExpression
	Names			[]string
}
func (re *TReferenceExpression) expressionNode() {}
func (re *TReferenceExpression) TokenLiteral() string { return re.Token.Literal }
func (re *TReferenceExpression) String() string {
	var out bytes.Buffer
	
	out.WriteString(re.Root.String())
	
	members := []string{}
	
	for _, i := range re.Members {
		members = append(members, i.String())
	}
	
	out.WriteString(strings.Join(members, "."))
		
	return out.String()
}
	
type TConstStatement struct {
	Token		token.TToken
	Grouped	bool
	Named		bool
	Name		*TIdentifier
	Names		[]*TIdentifier
	Values	[]IExpression
}

func (cs *TConstStatement) statementNode()       {}
func (cs *TConstStatement) TokenLiteral() string { return cs.Token.Literal }
func (cs *TConstStatement) String() string {
	var out bytes.Buffer

	out.WriteString(cs.TokenLiteral() + " ")
	
	if cs.Grouped {
		if cs.Named {
			out.WriteString(cs.Name.Value + " ")
		}
		out.WriteString("of\n")
	}
	
	for i := 0; i < len(cs.Names); i++ {
		out.WriteString(cs.Names[i].Value)
		out.WriteString(" = ")
		out.WriteString(cs.Values[i].String())
		out.WriteString(";")
	}
	
	if cs.Grouped {
		out.WriteString("\nend;")
	}

	return out.String()
}


type TEnumStatement struct {
	Token		token.TToken
	Named		bool
	Name		*TIdentifier
	Names		[]*TIdentifier
	Values	[]IExpression
}

func (es *TEnumStatement) statementNode()       {}
func (es *TEnumStatement) TokenLiteral() string { return es.Token.Literal }
func (es *TEnumStatement) String() string {
	var out bytes.Buffer

	out.WriteString(es.TokenLiteral() + " ")
	
	if es.Named {
		out.WriteString(es.Name.Value + " ")
	}
	
	out.WriteString("of\n")
		
	for i := 0; i < len(es.Names); i++ {
		out.WriteString(es.Names[i].Value)
		out.WriteString(" = ")
		out.WriteString(es.Values[i].String())
		out.WriteString(";")
	}
	
	out.WriteString("\nend;")
	
	return out.String()
}
