// vm/vm_test.go

package vm

import(
	"fmt"
	"io"
	"os"
	"testing"
	"SB/ast"
	"SB/lexer"
	"SB/object"
	"SB/parser"
	"SB/CMP/compiler"
)

func printParserErrors(out io.Writer, errors []string) {
	for _, msg := range errors {
		io.WriteString(out, "\t" + msg + "\n")
	}
}

func parse(input string) *ast.TProgram {
	l := lexer.New(input)
	p := parser.New(l)
	
	program := p.ParseProgram()
	
	if len(p.Errors()) != 0 {
		printParserErrors(os.Stdout, p.Errors())
		return nil
	}
	
	return program
}

func testIntegerObject(expected int64, actual object.IObject) error {
	result, ok := actual.(*object.TInteger)
	if !ok {
		return fmt.Errorf("object is not Integer, got %T (%+v)", actual, actual)
	}
	
	if result.Value != expected {
		return fmt.Errorf("object has wrong value, got %d, expected %d", result.Value, expected)
	}

	return nil
}


func testStringObject(expected string, actual object.IObject) error {
	result, ok := actual.(*object.TString)
	if !ok {
		return fmt.Errorf("object is not String. got=%T (%+v)", actual, actual)
	}

	if result.Value != expected {
		return fmt.Errorf("object has wrong value. got=%q, want=%q", result.Value, expected)
	}

	return nil
}


func testBooleanObject(expected bool, actual object.IObject) error {
	result, ok := actual.(*object.TBoolean)
	if !ok {
		return fmt.Errorf("object is not Boolean, got %T (%+v)", actual, actual)
	}

	if result.Value != expected {
		return fmt.Errorf("object has wrong value, got %t, expected %t", result.Value, expected)
	}

	return nil
}


type vmTestCase struct{
	input string
	expected interface{}
}


func runVmTests(t *testing.T, tests []vmTestCase) {
	t.Helper()

	for _, tt := range tests {
		program := parse(tt.input)
		if program != nil {
			comp := compiler.New()
			eobj := comp.Compile(program)
			if eobj != nil {
				t.Fatalf(eobj.(*object.TError).Message)
			}
			vm := New(comp.Bytecode())
			eobj = vm.Run()
			if eobj != nil {
				t.Fatalf("vm error: %s", eobj.(*object.TError).Message)
			}
			stackElem := vm.LastPoppedStackElem()
			testExpectedObject(t, tt.expected, stackElem)
		}
	}
}


func testExpectedObject(t *testing.T, expected interface{}, actual object.IObject,) {
	t.Helper()
	switch expected := expected.(type) {
	case int:
		err := testIntegerObject(int64(expected), actual)
		if err != nil {
			t.Errorf("testIntegerObject failed: %s", err)
		}
	case bool:
		err := testBooleanObject(bool(expected), actual)
		if err != nil {
			t.Errorf("testBooleanObject failed: %s", err)
		}
	case string:
		err := testStringObject(expected, actual)
		if err != nil {
			t.Errorf("testStringObject failed: %s", err)
		}
	case *object.TNull:
		if actual != Null {
			t.Errorf("object is not Null: %T (%+v)", actual, actual)
		}
	}
}

/*
func TestBuiltinFunctions(t *testing.T) {
	tests := []vmTestCase {
		{
			input:
				`len("")`,
			expected: &object.TUInteger { Value: 0 },
		},
		
		{
			input:
				`len("four")`,
			expected: &object.TUInteger { Value: 4 },
		},
		
		{
			input:
				`len("hello world ")`,
			expected: &object.TUInteger { Value: 11 },
		},
		
		{
			input:
				`abs(byte(-1))`,
			expected: &object.TByte { Value: 1 },
		},
	}
	
	runVmTests(t, tests)
}


func TestNestedFunctions(t *testing.T) {
	tests := []vmTestCase {
		{
			input: `
				function One1(): integer do
					function One2(): integer do
						return 1;
					end;
					return One2();
				end;
				print(One1());
			`,
			expected: 1,
		},
	}

	runVmTests(t, tests)
}


func TestCallingFunctionsWithArgumentsAndBindings(t *testing.T) {
	tests := []vmTestCase {
		{
			input: `
				function identity(a: integer): integer do
					return a;
				end;
				print(identity(4));
			`,
			expected: 4,
		},
		
		{
			input: `
				function sum(a:integer, b:integer): integer do
					return (a + b);
				end;
				print(sum(1, 2));
			`,
			expected: 3,
		},
		
		{
			input: `
				function sum(a:integer, b:integer):integer do
					var c: a + b;
					return c;
				end;
				print(sum(1, 2));
			`,
			expected: 3,
		},
		
		{
			input: `
				function sum(a:integer, b:integer):integer do
					var c: a + b;
					return c;
				end;
				print(sum(1, 2) + sum(3, 4));
			`,
			expected: 10,
		},
		
		{
			input: `
				function sum(a:integer, b:integer): integer do
					var c: a + b;
					return c;
				end;
				function outer(): integer do
					return sum(1, 2) + sum(3, 4);
				end;
				print(outer());
			`,
			expected: 10,
		},		
		
		{
			input: `
				var globalNum: 10;

				function sum(a:integer, b:integer): integer do
					var c: a + b;
					return c + globalNum;
				end;

				function outer(): integer do
					return sum(1, 2) + sum(3, 4) + globalNum;
				end;

				print(outer() + globalNum);
			`,
			expected: 50,
		},
		
	}

	runVmTests(t, tests)
}
*/

func TestCallingFunctionsWithoutArguments(t *testing.T) {
	tests := []vmTestCase {
		/*
		{
			input: `
				function fivePlusTen(): integer do
					return 5 + 10;
				end;

				print(fivePlusTen());
			`,
			expected: 15,
		},
		
		{
			input: `
				function NoReturn() do
					var x: 5;
				end;

				print(NoReturn());
			`,
			expected: 0,
		},
		*/
		
		{
			input: `
				function Nothing() do
				end;

				print(Nothing());
			`,
			expected: nil,
		},
		/*
		{
			input: `
				function one(): integer do
					return 1;
				end;

				function two(): integer do
					return 2;
				end;

				print(one() + two())
			`,
			expected: 3,
		},
		
		{
			input: `
				function returnsOne(): integer do
					return 1;
				end;
				function returnsOneReturner(): integer do
					return returnsOne();
				end;
				print(returnsOneReturner());
			`,
			expected: 1,
		},
		*/

		/*
		{
			input: `
				return 1
			`,
			expected: Null,
		},
		*/
		
	}

	runVmTests(t, tests)
}

