// vm/vm.go

package vm

import(
//	"fmt"
	"math"
	"strconv"
	"SB/object"
	"SB/CMP/code"
	"SB/CMP/compiler"
)


const StackSize = 2048
const MaxFrames = 2048
const GlobalsSize = 65536


var True = &object.TBoolean { Value: true }
var False = &object.TBoolean { Value: false }
var Null = &object.TNull {}


// 'sp' always points to the next value; top of stack is stack[sp - 1]
type VM struct {
	constants		[]object.IObject
	stack				[]object.IObject
	sp					int
	globals			[]object.IObject
	frames			[]*TFrame
	framesIndex int
}


func (vm *VM) LastPoppedStackElem() object.IObject {
	return vm.stack[vm.sp]
}

func (vm *VM) push(o object.IObject) object.IObject {
	if vm.sp >= StackSize {
		// return fmt.Errorf("stack overflow")
		return object.ERR_STACK_OVERFLOW()
	}
	
	vm.stack[vm.sp] = o
	vm.sp++
	
	return nil
}

func(vm *VM) pop() object.IObject {
	//fmt.Println("sp: %d", vm.sp)
	o := vm.stack[vm.sp - 1]
	vm.sp--
	return o
}


func New(bytecode *compiler.TBytecode) *VM {
	mainFn := &object.TCompiledFunction{Instructions: bytecode.Instructions}
	mainFrame := NewFrame(mainFn, 0)

	frames := make([]*TFrame, MaxFrames)
	frames[0] = mainFrame

	return &VM {
		constants: bytecode.Constants,
		stack: make([]object.IObject, StackSize),
		sp: 0,
		globals: make([]object.IObject, GlobalsSize),
		frames: frames,
		framesIndex: 1,
	}
}


func (vm *VM) currentFrame() *TFrame {
	return vm.frames[vm.framesIndex - 1]
}


func (vm *VM) pushFrame(f *TFrame) {
	vm.frames[vm.framesIndex] = f
	vm.framesIndex++
}


func (vm *VM) popFrame() *TFrame {
	vm.framesIndex--
	return vm.frames[vm.framesIndex]
}


func (vm *VM) executeInfixString(op code.Opcode, left, right object.IObject,) object.IObject {
	lv := left.(*object.TString).Value
	rv := right.(*object.TString).Value
	
	switch op {
	case code.OP_ADD:
		return vm.push(&object.TString{Value: lv + rv})
	
	case code.OP_EQU:
		return vm.push(nativeBoolToBooleanObject(rv == lv))
		
	case code.OP_NEQ:
		return vm.push(nativeBoolToBooleanObject(rv != lv))
		
	case code.OP_GRE:
		return vm.push(nativeBoolToBooleanObject(lv > rv))
	
	case code.OP_GEQ:
		return vm.push(nativeBoolToBooleanObject(lv >= rv))
	
	case code.OP_AND:
		return vm.push(nativeBoolToBooleanObject((lv != "") && (rv != "")))
	
	case code.OP_NOR:
		return vm.push(nativeBoolToBooleanObject((lv == "") && (rv == "")))
	
	case code.OP_OR:
		return vm.push(nativeBoolToBooleanObject((lv != "") || (rv != "")))
	
	case code.OP_XOR:
		return vm.push(nativeBoolToBooleanObject((lv == "") || (rv == "")))
	}
	
	return object.ERR_INVALID_OPERATOR(strconv.FormatInt(int64(op), 10))
}


func (vm *VM) executeInfixDouble(op code.Opcode, left, right object.IObject,) object.IObject {		
	lv := left.(*object.TDouble).Value
	rv := right.(*object.TDouble).Value
	
	switch op {
	case code.OP_ADD:
		return vm.push(&object.TDouble{Value: lv + rv})
		
	case code.OP_SUB:
		return vm.push(&object.TDouble{Value: lv - rv})
		
	case code.OP_MUL:
		return vm.push(&object.TDouble{Value: lv * rv})
		
	case code.OP_DIV:
		return vm.push(&object.TDouble{Value: lv / rv})
	
	case code.OP_POW:
		return vm.push(&object.TDouble{Value: math.Pow(lv, rv)})
	
	case code.OP_EQU:
		return vm.push(nativeBoolToBooleanObject(rv == lv))
		
	case code.OP_NEQ:
		return vm.push(nativeBoolToBooleanObject(rv != lv))
		
	case code.OP_GRE:
		return vm.push(nativeBoolToBooleanObject(lv > rv))
	
	case code.OP_GEQ:
		return vm.push(nativeBoolToBooleanObject(lv >= rv))
	
	case code.OP_AND:
		return vm.push(nativeBoolToBooleanObject((lv != 0) && (rv != 0)))
	
	case code.OP_NOR:
		return vm.push(nativeBoolToBooleanObject((lv == 0) && (rv == 0)))
	
	case code.OP_OR:
		return vm.push(nativeBoolToBooleanObject((lv != 0) || (rv != 0)))
	
	case code.OP_XOR:
		return vm.push(nativeBoolToBooleanObject((lv == 0) || (rv == 0)))
	}
	
	return object.ERR_INVALID_OPERATOR(strconv.FormatInt(int64(op), 10))
}


func (vm *VM) executeInfixSingle(op code.Opcode, left, right object.IObject,) object.IObject {		
	lv := left.(*object.TSingle).Value
	rv := right.(*object.TSingle).Value
	
	switch op {
	case code.OP_ADD:
		return vm.push(&object.TSingle{Value: lv + rv})
		
	case code.OP_SUB:
		return vm.push(&object.TSingle{Value: lv - rv})
		
	case code.OP_MUL:
		return vm.push(&object.TSingle{Value: lv * rv})
		
	case code.OP_DIV:
		return vm.push(&object.TSingle{Value: lv / rv})
	
	case code.OP_POW:
		return vm.push(&object.TSingle{Value: float32(math.Pow(float64(lv), float64(rv)))})
	
	case code.OP_EQU:
		return vm.push(nativeBoolToBooleanObject(rv == lv))
		
	case code.OP_NEQ:
		return vm.push(nativeBoolToBooleanObject(rv != lv))
		
	case code.OP_GRE:
		return vm.push(nativeBoolToBooleanObject(lv > rv))
	
	case code.OP_GEQ:
		return vm.push(nativeBoolToBooleanObject(lv >= rv))
	
	case code.OP_AND:
		return vm.push(nativeBoolToBooleanObject((lv != 0) && (rv != 0)))
	
	case code.OP_NOR:
		return vm.push(nativeBoolToBooleanObject((lv == 0) && (rv == 0)))
	
	case code.OP_OR:
		return vm.push(nativeBoolToBooleanObject((lv != 0) || (rv != 0)))
	
	case code.OP_XOR:
		return vm.push(nativeBoolToBooleanObject((lv == 0) || (rv == 0)))
	}
	
	return object.ERR_INVALID_OPERATOR(strconv.FormatInt(int64(op), 10))
}


func (vm *VM) executeInfixInteger(op code.Opcode, left, right object.IObject,) object.IObject {
	lv := left.(*object.TInteger).Value
	rv := right.(*object.TInteger).Value
	
	switch op {
	case code.OP_ADD:
		return vm.push(&object.TInteger{Value: lv + rv})
		
	case code.OP_SUB:
		return vm.push(&object.TInteger{Value: lv - rv})
		
	case code.OP_MUL:
		return vm.push(&object.TInteger{Value: lv * rv})
		
	case code.OP_DIV:
		return vm.push(&object.TInteger{Value: lv / rv})
	
	case code.OP_MOD:
		return vm.push(&object.TInteger{Value: lv % rv})
	
	case code.OP_POW:
		return vm.push(&object.TInteger{Value: int64(math.Pow(float64(lv), float64(rv)))})
	
	case code.OP_EQU:
		return vm.push(nativeBoolToBooleanObject(rv == lv))
		
	case code.OP_NEQ:
		return vm.push(nativeBoolToBooleanObject(rv != lv))
		
	case code.OP_GRE:
		return vm.push(nativeBoolToBooleanObject(lv > rv))
	
	case code.OP_GEQ:
		return vm.push(nativeBoolToBooleanObject(lv >= rv))
	
	case code.OP_AND:
		return vm.push(nativeBoolToBooleanObject((lv != 0) && (rv != 0)))
	
	case code.OP_NOR:
		return vm.push(nativeBoolToBooleanObject((lv == 0) && (rv == 0)))
	
	case code.OP_OR:
		return vm.push(nativeBoolToBooleanObject((lv != 0) || (rv != 0)))
	
	case code.OP_XOR:
		return vm.push(nativeBoolToBooleanObject((lv == 0) || (rv == 0)))
	}
	
	return object.ERR_INVALID_OPERATOR(strconv.FormatInt(int64(op), 10))
}


func (vm *VM) executeInfixLong(op code.Opcode, left, right object.IObject,) object.IObject {
	lv := left.(*object.TLong).Value
	rv := right.(*object.TLong).Value
	
	switch op {
	case code.OP_ADD:
		return vm.push(&object.TLong{Value: lv + rv})
	
	case code.OP_SUB:
		return vm.push(&object.TLong{Value: lv - rv})
	
	case code.OP_MUL:
		return vm.push(&object.TLong{Value: lv * rv})
	
	case code.OP_DIV:
		return vm.push(&object.TLong{Value: lv / rv})
	
	case code.OP_MOD:
		return vm.push(&object.TLong{Value: lv % rv})
	
	case code.OP_POW:
		return vm.push(&object.TLong{Value: int32(math.Pow(float64(lv), float64(rv)))})
		
	case code.OP_EQU:
		return vm.push(nativeBoolToBooleanObject(rv == lv))
		
	case code.OP_NEQ:
		return vm.push(nativeBoolToBooleanObject(rv != lv))
		
	case code.OP_GRE:
		return vm.push(nativeBoolToBooleanObject(lv > rv))
	
	case code.OP_GEQ:
		return vm.push(nativeBoolToBooleanObject(lv >= rv))
	
	case code.OP_AND:
		return vm.push(nativeBoolToBooleanObject((lv != 0) && (rv != 0)))
	
	case code.OP_NOR:
		return vm.push(nativeBoolToBooleanObject((lv == 0) && (rv == 0)))
	
	case code.OP_OR:
		return vm.push(nativeBoolToBooleanObject((lv != 0) || (rv != 0)))
	
	case code.OP_XOR:
		return vm.push(nativeBoolToBooleanObject((lv == 0) || (rv == 0)))
	}
	
	return object.ERR_INVALID_OPERATOR(strconv.FormatInt(int64(op), 10))
}


func (vm *VM) executeInfixShort(op code.Opcode, left, right object.IObject,) object.IObject {
	lv := left.(*object.TShort).Value
	rv := right.(*object.TShort).Value
	
	switch op {
	case code.OP_ADD:
		return vm.push(&object.TShort{Value: lv + rv})
		
	case code.OP_SUB:
		return vm.push(&object.TShort{Value: lv - rv})
		
	case code.OP_MUL:
		return vm.push(&object.TShort{Value: lv * rv})
		
	case code.OP_DIV:
		return vm.push(&object.TShort{Value: lv / rv})
	
	case code.OP_MOD:
		return vm.push(&object.TShort{Value: lv % rv})
	
	case code.OP_POW:
		return vm.push(&object.TShort{Value: int16(math.Pow(float64(lv), float64(rv)))})
	
	case code.OP_EQU:
		return vm.push(nativeBoolToBooleanObject(rv == lv))
		
	case code.OP_NEQ:
		return vm.push(nativeBoolToBooleanObject(rv != lv))
		
	case code.OP_GRE:
		return vm.push(nativeBoolToBooleanObject(lv > rv))
	
	case code.OP_GEQ:
		return vm.push(nativeBoolToBooleanObject(lv >= rv))
	
	case code.OP_AND:
		return vm.push(nativeBoolToBooleanObject((lv != 0) && (rv != 0)))
	
	case code.OP_NOR:
		return vm.push(nativeBoolToBooleanObject((lv == 0) && (rv == 0)))
	
	case code.OP_OR:
		return vm.push(nativeBoolToBooleanObject((lv != 0) || (rv != 0)))
	
	case code.OP_XOR:
		return vm.push(nativeBoolToBooleanObject((lv == 0) || (rv == 0)))
	}
	
	return object.ERR_INVALID_OPERATOR(strconv.FormatInt(int64(op), 10))
}


func (vm *VM) executeInfixByte(op code.Opcode, left, right object.IObject,) object.IObject {
	lv := left.(*object.TByte).Value
	rv := right.(*object.TByte).Value
	
	switch op {
	case code.OP_ADD:
		return vm.push(&object.TByte{Value: lv + rv})
	
	case code.OP_SUB:
		return vm.push(&object.TByte{Value: lv - rv})
	
	case code.OP_MUL:
		return vm.push(&object.TByte{Value: lv * rv})
	
	case code.OP_DIV:
		return vm.push(&object.TByte{Value: lv / rv})
	
	case code.OP_MOD:
		return vm.push(&object.TByte{Value: lv % rv})
	
	case code.OP_POW:
		return vm.push(&object.TByte{Value: int8(math.Pow(float64(lv), float64(rv)))})
		
	case code.OP_EQU:
		return vm.push(nativeBoolToBooleanObject(rv == lv))
		
	case code.OP_NEQ:
		return vm.push(nativeBoolToBooleanObject(rv != lv))
		
	case code.OP_GRE:
		return vm.push(nativeBoolToBooleanObject(lv > rv))
	
	case code.OP_GEQ:
		return vm.push(nativeBoolToBooleanObject(lv >= rv))
	
	case code.OP_AND:
		return vm.push(nativeBoolToBooleanObject((lv != 0) && (rv != 0)))
	
	case code.OP_NOR:
		return vm.push(nativeBoolToBooleanObject((lv == 0) && (rv == 0)))
	
	case code.OP_OR:
		return vm.push(nativeBoolToBooleanObject((lv != 0) || (rv != 0)))
	
	case code.OP_XOR:
		return vm.push(nativeBoolToBooleanObject((lv == 0) || (rv == 0)))
	}
	
	return object.ERR_INVALID_OPERATOR(strconv.FormatInt(int64(op), 10))
}


func (vm *VM) executeInfixUInteger(op code.Opcode, left, right object.IObject,) object.IObject {
	lv := left.(*object.TUInteger).Value
	rv := right.(*object.TUInteger).Value
	
	switch op {
	case code.OP_ADD:
		return vm.push(&object.TUInteger{Value: lv + rv})
		
	case code.OP_SUB:
		return vm.push(&object.TUInteger{Value: lv - rv})
		
	case code.OP_MUL:
		return vm.push(&object.TUInteger{Value: lv * rv})
		
	case code.OP_DIV:
		return vm.push(&object.TUInteger{Value: lv / rv})
	
	case code.OP_MOD:
		return vm.push(&object.TUInteger{Value: lv % rv})
	
	case code.OP_POW:
		return vm.push(&object.TUInteger{Value: uint64(math.Pow(float64(lv), float64(rv)))})
	
	case code.OP_EQU:
		return vm.push(nativeBoolToBooleanObject(rv == lv))
		
	case code.OP_NEQ:
		return vm.push(nativeBoolToBooleanObject(rv != lv))
		
	case code.OP_GRE:
		return vm.push(nativeBoolToBooleanObject(lv > rv))
	
	case code.OP_GEQ:
		return vm.push(nativeBoolToBooleanObject(lv >= rv))
	
	case code.OP_AND:
		return vm.push(nativeBoolToBooleanObject((lv != 0) && (rv != 0)))
	
	case code.OP_NOR:
		return vm.push(nativeBoolToBooleanObject((lv == 0) && (rv == 0)))
	
	case code.OP_OR:
		return vm.push(nativeBoolToBooleanObject((lv != 0) || (rv != 0)))
	
	case code.OP_XOR:
		return vm.push(nativeBoolToBooleanObject((lv == 0) || (rv == 0)))
	}
	
	return object.ERR_INVALID_OPERATOR(strconv.FormatInt(int64(op), 10))
}


func (vm *VM) executeInfixULong(op code.Opcode, left, right object.IObject,) object.IObject {
	lv := left.(*object.TULong).Value
	rv := right.(*object.TULong).Value
	
	switch op {
	case code.OP_ADD:
		return vm.push(&object.TULong{Value: lv + rv})
		
	case code.OP_SUB:
		return vm.push(&object.TULong{Value: lv - rv})
		
	case code.OP_MUL:
		return vm.push(&object.TULong{Value: lv * rv})
		
	case code.OP_DIV:
		return vm.push(&object.TULong{Value: lv / rv})
	
	case code.OP_MOD:
		return vm.push(&object.TULong{Value: lv % rv})
	
	case code.OP_POW:
		return vm.push(&object.TULong{Value: uint32(math.Pow(float64(lv), float64(rv)))})
	
	case code.OP_EQU:
		return vm.push(nativeBoolToBooleanObject(rv == lv))
		
	case code.OP_NEQ:
		return vm.push(nativeBoolToBooleanObject(rv != lv))
		
	case code.OP_GRE:
		return vm.push(nativeBoolToBooleanObject(lv > rv))
	
	case code.OP_GEQ:
		return vm.push(nativeBoolToBooleanObject(lv >= rv))
	
	case code.OP_AND:
		return vm.push(nativeBoolToBooleanObject((lv != 0) && (rv != 0)))
	
	case code.OP_NOR:
		return vm.push(nativeBoolToBooleanObject((lv == 0) && (rv == 0)))
	
	case code.OP_OR:
		return vm.push(nativeBoolToBooleanObject((lv != 0) || (rv != 0)))
	
	case code.OP_XOR:
		return vm.push(nativeBoolToBooleanObject((lv == 0) || (rv == 0)))
	}
	
	return object.ERR_INVALID_OPERATOR(strconv.FormatInt(int64(op), 10))
}


func (vm *VM) executeInfixUShort(op code.Opcode, left, right object.IObject,) object.IObject {
	lv := left.(*object.TUShort).Value
	rv := right.(*object.TUShort).Value
	
	switch op {
	case code.OP_ADD:
		return vm.push(&object.TUShort{Value: lv + rv})
		
	case code.OP_SUB:
		return vm.push(&object.TUShort{Value: lv - rv})
		
	case code.OP_MUL:
		return vm.push(&object.TUShort{Value: lv * rv})
		
	case code.OP_DIV:
		return vm.push(&object.TUShort{Value: lv / rv})
	
	case code.OP_MOD:
		return vm.push(&object.TUShort{Value: lv % rv})
	
	case code.OP_POW:
		return vm.push(&object.TUShort{Value: uint16(math.Pow(float64(lv), float64(rv)))})
	
	case code.OP_EQU:
		return vm.push(nativeBoolToBooleanObject(rv == lv))
		
	case code.OP_NEQ:
		return vm.push(nativeBoolToBooleanObject(rv != lv))
		
	case code.OP_GRE:
		return vm.push(nativeBoolToBooleanObject(lv > rv))
	
	case code.OP_GEQ:
		return vm.push(nativeBoolToBooleanObject(lv >= rv))
	
	case code.OP_AND:
		return vm.push(nativeBoolToBooleanObject((lv != 0) && (rv != 0)))
	
	case code.OP_NOR:
		return vm.push(nativeBoolToBooleanObject((lv == 0) && (rv == 0)))
	
	case code.OP_OR:
		return vm.push(nativeBoolToBooleanObject((lv != 0) || (rv != 0)))
	
	case code.OP_XOR:
		return vm.push(nativeBoolToBooleanObject((lv == 0) || (rv == 0)))
	}
	
	return object.ERR_INVALID_OPERATOR(strconv.FormatInt(int64(op), 10))
}


func (vm *VM) executeInfixUByte(op code.Opcode, left, right object.IObject,) object.IObject {
	lv := left.(*object.TUByte).Value
	rv := right.(*object.TUByte).Value
	
	switch op {
	case code.OP_ADD:
		return vm.push(&object.TUByte{Value: lv + rv})
		
	case code.OP_SUB:
		return vm.push(&object.TUByte{Value: lv - rv})
		
	case code.OP_MUL:
		return vm.push(&object.TUByte{Value: lv * rv})
		
	case code.OP_DIV:
		return vm.push(&object.TUByte{Value: lv / rv})
	
	case code.OP_MOD:
		return vm.push(&object.TUByte{Value: lv % rv})
	
	case code.OP_POW:
		return vm.push(&object.TUByte{Value: uint8(math.Pow(float64(lv), float64(rv)))})
	
	case code.OP_EQU:
		return vm.push(nativeBoolToBooleanObject(rv == lv))
		
	case code.OP_NEQ:
		return vm.push(nativeBoolToBooleanObject(rv != lv))
		
	case code.OP_GRE:
		return vm.push(nativeBoolToBooleanObject(lv > rv))
	
	case code.OP_GEQ:
		return vm.push(nativeBoolToBooleanObject(lv >= rv))
	
	case code.OP_AND:
		return vm.push(nativeBoolToBooleanObject((lv != 0) && (rv != 0)))
	
	case code.OP_NOR:
		return vm.push(nativeBoolToBooleanObject((lv == 0) && (rv == 0)))
	
	case code.OP_OR:
		return vm.push(nativeBoolToBooleanObject((lv != 0) || (rv != 0)))
	
	case code.OP_XOR:
		return vm.push(nativeBoolToBooleanObject((lv == 0) || (rv == 0)))
	}
	
	return object.ERR_INVALID_OPERATOR(strconv.FormatInt(int64(op), 10))
}


func (vm *VM) executeInfixBoolean(op code.Opcode, left, right object.IObject,) object.IObject {
	lv := left.(*object.TBoolean).Value
	rv := right.(*object.TBoolean).Value
	
	switch op {
	case code.OP_EQU:
		return vm.push(nativeBoolToBooleanObject(rv == lv))
		
	case code.OP_NEQ:
		return vm.push(nativeBoolToBooleanObject(rv != lv))
	
	case code.OP_AND:
		return vm.push(nativeBoolToBooleanObject((lv != false) && (rv != false)))
	
	case code.OP_NOR:
		return vm.push(nativeBoolToBooleanObject((lv == false) && (rv == false)))
	
	case code.OP_OR:
		return vm.push(nativeBoolToBooleanObject((lv != false) || (rv != false)))
	
	case code.OP_XOR:
		return vm.push(nativeBoolToBooleanObject((lv == false) || (rv == false)))
	}
	
	return object.ERR_INVALID_OPERATOR(strconv.FormatInt(int64(op), 10))
}


func (vm *VM) executeOperatorInfix(op code.Opcode) object.IObject {
	right := vm.pop()
	left := vm.pop()
	
	if eobj, ok := object.ObjectState(left, object.AllowNULL); !ok {
		return eobj
	}
	
	if eobj, ok := object.ObjectState(right, object.AllowNULL); !ok {
		return eobj
	}
	
	left, _ = object.Deref(left)
	right, _ = object.Deref(right)
	
	leftType := left.Type()
	rightType := right.Type()
	
	if leftType == object.BOOLEAN_OBJ && rightType == object.BOOLEAN_OBJ {
		return vm.executeInfixBoolean(op, left, right)
	}
	
	if leftType == object.STRING_OBJ && rightType == object.STRING_OBJ {
		return vm.executeInfixString(op, left, right)
	}
	
	if leftType == object.DOUBLE_OBJ || rightType == object.DOUBLE_OBJ {
		l, ok := object.DoubleObject(left, object.IMPLICIT)
		if !ok { return object.ERR_TYPE_MISMATCH(left, right) }
		r, ok := object.DoubleObject(right, object.IMPLICIT)
		if !ok { return object.ERR_TYPE_MISMATCH(left, right) }
		return vm.executeInfixDouble(op, l, r)
	}
	
	if leftType == object.SINGLE_OBJ || rightType == object.SINGLE_OBJ {
		l, ok := object.SingleObject(left, object.IMPLICIT)
		if !ok { return object.ERR_TYPE_MISMATCH(left, right) }
		r, ok := object.SingleObject(right, object.IMPLICIT)
		if !ok { return object.ERR_TYPE_MISMATCH(left, right) }
		return vm.executeInfixSingle(op, l, r)
	}
	
	if leftType == object.INTEGER_OBJ || rightType == object.INTEGER_OBJ {
		l, ok := object.IntegerObject(left, object.IMPLICIT)
		if !ok { return object.ERR_TYPE_MISMATCH(left, right) }
		r, ok := object.IntegerObject(right, object.IMPLICIT)
		if !ok { return object.ERR_TYPE_MISMATCH(left, right) }
		return vm.executeInfixInteger(op, l, r)
	}
	
	if leftType == object.LONG_OBJ || rightType == object.LONG_OBJ {
		l, ok := object.LongObject(left, object.IMPLICIT)
		if !ok { return object.ERR_TYPE_MISMATCH(left, right) }
		r, ok := object.LongObject(right, object.IMPLICIT)
		if !ok { return object.ERR_TYPE_MISMATCH(left, right) }
		return vm.executeInfixLong(op, l, r)
	}
	
	
	if leftType == object.SHORT_OBJ || rightType == object.SHORT_OBJ {
		l, ok := object.ShortObject(left, object.IMPLICIT)
		if !ok { return object.ERR_TYPE_MISMATCH(left, right) }
		r, ok := object.ShortObject(right, object.IMPLICIT)
		if !ok { return object.ERR_TYPE_MISMATCH(left, right) }
		return vm.executeInfixShort(op, l, r)
	}
	
	
	if leftType == object.BYTE_OBJ || rightType == object.BYTE_OBJ {
		l, ok := object.ByteObject(left, object.IMPLICIT)
		if !ok { return object.ERR_TYPE_MISMATCH(left, right) }
		r, ok := object.ByteObject(right, object.IMPLICIT)
		if !ok { return object.ERR_TYPE_MISMATCH(left, right) }
		return vm.executeInfixByte(op, l, r)
	}
	
	
	if leftType == object.UINTEGER_OBJ || rightType == object.UINTEGER_OBJ {
		l, ok := object.UIntegerObject(left, object.IMPLICIT)
		if !ok { return object.ERR_TYPE_MISMATCH(left, right) }
		r, ok := object.UIntegerObject(right, object.IMPLICIT)
		if !ok { return object.ERR_TYPE_MISMATCH(left, right) }
		return vm.executeInfixUInteger(op, l, r)
	}
	
	if leftType == object.ULONG_OBJ || rightType == object.ULONG_OBJ {
		l, ok := object.ULongObject(left, object.IMPLICIT)
		if !ok { return object.ERR_TYPE_MISMATCH(left, right) }
		r, ok := object.ULongObject(right, object.IMPLICIT)
		if !ok { return object.ERR_TYPE_MISMATCH(left, right) }
		return vm.executeInfixULong(op, l, r)
	}
	
	
	if leftType == object.USHORT_OBJ || rightType == object.USHORT_OBJ {
		l, ok := object.UShortObject(left, object.IMPLICIT)
		if !ok { return object.ERR_TYPE_MISMATCH(left, right) }
		r, ok := object.UShortObject(right, object.IMPLICIT)
		if !ok { return object.ERR_TYPE_MISMATCH(left, right) }
		return vm.executeInfixUShort(op, l, r)
	}
	
	
	if leftType == object.UBYTE_OBJ || rightType == object.UBYTE_OBJ {
		l, ok := object.UByteObject(left, object.IMPLICIT)
		if !ok { return object.ERR_TYPE_MISMATCH(left, right) }
		r, ok := object.UByteObject(right, object.IMPLICIT)
		if !ok { return object.ERR_TYPE_MISMATCH(left, right) }
		return vm.executeInfixUByte(op, l, r)
	}
	
	return object.ERR_INVALID_EXPRESSION(strconv.FormatInt(int64(op), 10), left, right)
}


func isTRUE(obj object.IObject) bool {
	switch obj := obj.(type) {
	case *object.TBoolean:
		return obj.Value
	case *object.TNull:
		return false
	default:
		return true
	}
}


func nativeBoolToBooleanObject(input bool) *object.TBoolean {
	if input {
		return True
	}
	
	return False
}


func (vm *VM) executeOperatorNot() object.IObject {
	operand := vm.pop()
	
	if val, ok := object.BooleanObject(operand, object.EXPLICIT); ok {
		if val.(*object.TBoolean).Value == true {
			return vm.push(False)
		} else {
			return vm.push(True)
		}
	} else {
		return object.ERR_INVALID_EXPRESSION("not", nil, operand)
	}
}


func (vm *VM) executeOperatorIs() object.IObject {
	operand := vm.pop()
	
	if val, ok := object.BooleanObject(operand, object.EXPLICIT); ok {
		if val.(*object.TBoolean).Value == true {
			return vm.push(True)
		} else {
			return vm.push(False)
		}
	} else {
		return object.ERR_INVALID_EXPRESSION("is", nil, operand)
	}
}


func (vm *VM) executeOperatorMinus() object.IObject {
	operand := vm.pop()

	switch operand.(type) {
	case *object.TByte:
		value := operand.(*object.TByte).Value
		return vm.push(&object.TByte { Value: -value })
	
	case *object.TDouble:
		value := operand.(*object.TDouble).Value
		return vm.push(&object.TDouble { Value: -value })
	
	case *object.TSingle:
		value := operand.(*object.TSingle).Value
		return vm.push(&object.TSingle { Value: -value })
	
	case *object.TInteger:
		value := operand.(*object.TInteger).Value
		return vm.push(&object.TInteger{Value: -value})
	
	case *object.TLong:
		value := operand.(*object.TLong).Value
		return vm.push(&object.TLong{Value: -value})
	
	case *object.TShort:
		value := operand.(*object.TShort).Value
		return vm.push(&object.TShort{Value: -value})
	
	case *object.TUByte:
		value := operand.(*object.TUByte).Value
		return vm.push(&object.TUByte{Value: -value})
	
	case *object.TUInteger:
		value := operand.(*object.TUInteger).Value
		return vm.push(&object.TUInteger{Value: -value})
	
	case *object.TULong:
		value := operand.(*object.TULong).Value
		return vm.push(&object.TULong{Value: -value})
	
	case *object.TUShort:
		value := operand.(*object.TUShort).Value
		return vm.push(&object.TUShort{Value: -value})
		
	}
	
	return object.ERR_INVALID_TYPE()
}


func NewWithGlobalsStore(bytecode *compiler.TBytecode, s []object.IObject) *VM {
	vm := New(bytecode)
	vm.globals = s
	return vm
}


func(vm *VM) callFunction(fn *object.TCompiledFunction, numArgs int) object.IObject {
	if numArgs != fn.NumParameters {
		return object.ERR_ARG_COUNT_MISMATCH()
		//returnfmt.Errorf("wrong number of arguments: want=%d, got=%d", fn.NumParameters, numArgs)
	}
	
	frame := NewFrame(fn, vm.sp - numArgs)
	vm.pushFrame(frame)
	vm.sp = frame.bp + fn.NumLocals
	
	return nil
}


func (vm *VM) callBuiltin(builtin *object.TBuiltin, numArgs int) object.IObject {
	args := vm.stack[vm.sp - numArgs : vm.sp]
	result := builtin.Fn("", args...)
	vm.sp = vm.sp - numArgs - 1
	
	if result != nil {
		vm.push(result)
	} else {
		vm.push(Null)
	}

	return nil
}


func (vm *VM) executeCall(numArgs int) object.IObject {
	callee := vm.stack[vm.sp - 1 - numArgs]
	switch callee := callee.(type) {
	case *object.TCompiledFunction:
		return vm.callFunction(callee, numArgs)
	case *object.TBuiltin:
		return vm.callBuiltin(callee, numArgs)
	default:
		return object.ERR_NOT_A_FUNCTION()
	}
}


func (vm *VM) Run() object.IObject {
	var ip int
	var ins code.Instructions
	var op code.Opcode
	
	for vm.currentFrame().ip < len(vm.currentFrame().Instructions()) - 1 {
		vm.currentFrame().ip++
		ip = vm.currentFrame().ip
		ins = vm.currentFrame().Instructions()
		op = code.Opcode(ins[ip])
		
		switch op {
		
		case code.OP_POP:
			vm.pop()
		
		case code.OP_CONST:
			constIndex := code.ReadUint16(ins[ip + 1:])
			vm.currentFrame().ip += 2
			eobj := vm.push(vm.constants[constIndex])
			if eobj != nil {
				return eobj
			}
			
		case code.OP_NULL:
			err := vm.push(Null)
			if err != nil {
				return err
			}
			
		case code.OP_ADD, code.OP_SUB, code.OP_MUL, code.OP_DIV, 
			code.OP_MOD, code.OP_POW, code.OP_EQU, code.OP_NEQ, 
			code.OP_GRE, code.OP_GEQ, code.OP_AND, code.OP_NOR, 
			code.OP_OR, code.OP_XOR:
			eobj := vm.executeOperatorInfix(op)
			if eobj != nil {
				return eobj
			}
		
		case code.OP_TRUE:
			eobj := vm.push(True)
			if eobj != nil {
				return eobj
			}

		case code.OP_FALSE:
			eobj := vm.push(False)
			if eobj != nil {
				return eobj
			}
		
		case code.OP_MIN:
			eobj := vm.executeOperatorMinus()
			if eobj != nil {
				return eobj
			}
		
		case code.OP_NOT:
			eobj := vm.executeOperatorNot()
			if eobj != nil {
				return eobj
			}
		
		case code.OP_IS:
			eobj := vm.executeOperatorIs()
			if eobj != nil {
				return eobj
			}
			
		case code.OP_JMP:
			pos := int(code.ReadUint16(ins[ip + 1:]))
			vm.currentFrame().ip = pos - 1
			
		case code.OP_JNT:
			pos := int(code.ReadUint16(ins[ip + 1:]))
			vm.currentFrame().ip += 2
			condition := vm.pop()
			if !isTRUE(condition) {
				vm.currentFrame().ip = pos - 1
			}
		
		case code.OP_CALL:
			numArgs := code.ReadUint8(ins[ip + 1:])
			vm.currentFrame().ip += 1
			err := vm.executeCall(int(numArgs))
			if err != nil {
				return err
			}
			
		case code.OP_RETV:
			returnValue := vm.pop()
			
			frame := vm.popFrame()
			vm.sp = frame.bp - 1
			
			returnType := frame.fn.ReturnType
			if returnType != nil {
				if robj, ok := object.AssignDefault(returnType, returnValue); !ok {
					return object.ERR_TYPE_MISMATCH(returnType, returnValue)
				} else {
					err := vm.push(robj)
					if err != nil {
						return err
					}
				}
			} else {
				return object.ERR_RETURN_TYPE_UNDEFINED()
			}
		
		case code.OP_RET:
			frame := vm.popFrame()
			vm.sp = frame.bp - 1
			/*
			err := vm.push(object.INTEGER)
			if err != nil {
				return err
			}
			*/
			
		case code.OP_DEFGL:
			globalIndex := code.ReadUint16(ins[ip + 1:])
			vm.currentFrame().ip += 2
			vm.globals[globalIndex] = vm.pop()
			
		case code.OP_GETGL:
			globalIndex := code.ReadUint16(ins[ip + 1:])
			vm.currentFrame().ip += 2
			err := vm.push(vm.globals[globalIndex])
			if err != nil {
				return err
			}
		
		case code.OP_MOVGL:
			val := vm.pop()
			idx := code.ReadUint16(ins[ip + 1:])
			vm.currentFrame().ip += 2
			left := vm.globals[idx]
			if result, ok := object.AssignDefault(left, val); !ok {
				return object.ERR_TYPE_MISMATCH(left, val)
			} else {
				vm.globals[idx] = result	
			}
		
		case code.OP_DEFLC:
			idx := code.ReadUint8(ins[ip + 1:])
			vm.currentFrame().ip += 1
			frame := vm.currentFrame()
			vm.stack[frame.bp + int(idx)] = vm.pop()
		
		case code.OP_GETLC:
			idx := code.ReadUint8(ins[ip + 1:])
			vm.currentFrame().ip += 1
			frame := vm.currentFrame()
			err := vm.push(vm.stack[frame.bp + int(idx)])
			if err != nil {
				return err
			}
		
		case code.OP_MOVLC:
			val := vm.pop()
			idx := code.ReadUint8(ins[ip + 1:])
			vm.currentFrame().ip += 1
			frame := vm.currentFrame()
			left := vm.stack[frame.bp + int(idx)]
			if result, ok := object.AssignDefault(left, val); !ok {
				return object.ERR_TYPE_MISMATCH(left, val)
			} else {
				vm.stack[frame.bp + int(idx)] = result	
			}
			
		case code.OP_GETBU:
			index := code.ReadUint8(ins[ip + 1:])
			vm.currentFrame().ip += 1
			definition := object.Builtins[index]
			err := vm.push(definition.Builtin)
			if err != nil {
				return err
			}
		}
	}

	return nil
}
