// code/code.go

package code

import (
	"bytes"
	"fmt"
	"encoding/binary"
)

type Opcode byte

const UNKNOWN_POS int = 9999

const (
	OP_CONST Opcode = iota
	
	OP_POP
	
	OP_ADD
	OP_SUB
	OP_MUL
	OP_DIV
	OP_MOD
	OP_POW
	
	OP_TRUE
	OP_FALSE
	
	OP_EQU		// equal
	OP_NEQ		// not equal
	OP_GRE		// greater than
	OP_GEQ		// greater than or equal
	
	OP_MIN		// minus
	OP_NOT
	OP_IS
	
	OP_AND
	OP_NOR
	OP_OR
	OP_XOR
	
	OP_JNT		// jump if not true
	OP_JMP		// jump
	
	OP_NULL		// null
	
	OP_CALL
	OP_RETV		// return value
	OP_RET
	
	OP_DEFGL	// define global
	OP_GETGL	// get global
	OP_MOVGL	// move to global
	
	OP_DEFLC	// define local
	OP_GETLC	// get local
	OP_MOVLC	// move to local
	
	OP_GETBU	// get builtin
	
)

type TDefinition struct {
	Name string
	OperandWidths []int
}

var definitions = map[Opcode]*TDefinition {
	OP_CONST:	{ "OP_CONST", []int{2} },
	OP_POP: 	{ "OP_POP", 	[]int{ } },
	OP_ADD:		{ "OP_ADD", 	[]int{ } },
	OP_SUB: 	{ "OP_SUB", 	[]int{ } },
	OP_MUL: 	{ "OP_MUL", 	[]int{ } },
	OP_DIV: 	{ "OP_DIV", 	[]int{ } },
	OP_MOD: 	{ "OP_MOD", 	[]int{ } },
	OP_POW: 	{ "OP_POW", 	[]int{ } },
	OP_TRUE:	{ "OP_TRUE", 	[]int{ } },
	OP_FALSE:	{ "OP_FALSE", []int{ } },
	OP_EQU:		{ "OP_EQU", 	[]int{ } },
	OP_NEQ:		{ "OP_NEQ",		[]int{ } },
	OP_GRE:		{ "OP_GRE", 	[]int{ } },
	OP_GEQ:		{ "OP_GEQ", 	[]int{ } },
	OP_MIN:		{ "OP_MIN", 	[]int{ } },
	OP_NOT:		{ "OP_NOT", 	[]int{ } },
	OP_IS:		{ "OP_IS", 		[]int{ } },
	OP_AND:		{ "OP_AND", 	[]int{ } },
	OP_NOR:		{ "OP_NOR", 	[]int{ } },
	OP_OR:		{ "OP_OR", 		[]int{ } },
	OP_XOR:		{ "OP_XOR", 	[]int{ } },
	OP_JNT:		{ "OP_JNT",		[]int{2} },
	OP_JMP:		{ "OP_JMP",		[]int{2} },
	OP_NULL:	{ "OP_NULL", 	[]int{ } },
	OP_DEFGL:	{ "OP_DEFGL", []int{2} },
	OP_GETGL:	{ "OP_GETGL", []int{2} },
	OP_MOVGL:	{ "OP_MOVGL",	[]int{2} },
	OP_CALL:	{ "OP_CALL", 	[]int{1} },
	OP_RETV:	{ "OP_RETV",	[]int{ } },
	OP_RET:		{ "OP_RET",		[]int{ } },
	OP_DEFLC:	{ "OP_DEFLC", []int{1} },
	OP_GETLC:	{ "OP_GETLC", []int{1} },
	OP_MOVLC:	{ "OP_MOVLC",	[]int{1} },
	OP_GETBU:	{ "OP_GETBU", []int{1} },
}

type Instructions []byte

func(ins Instructions) String() string {
	var out bytes.Buffer
	
	i := 0
	for i < len(ins) {
		def, err := Lookup(ins[i])
		if err != nil {
			fmt.Fprintf(&out, "ERROR: %s\n", err)
			continue
		}
		operands, read := ReadOperands(def, ins[i+1:])
		fmt.Fprintf(&out, "%04d %s\n", i, ins.fmtInstruction(def, operands))
		i += 1 + read
	}
	
	return out.String()
}

func(ins Instructions) fmtInstruction(def *TDefinition, operands []int) string {
	operandCount := len(def.OperandWidths)
	if len(operands) != operandCount {
		return fmt.Sprintf("ERROR: operand len %d does not match defined %d\n", len(operands), operandCount)
	}
	
	switch operandCount {
	case 0:
		return def.Name
	case 1:
		return fmt.Sprintf("%s %d", def.Name, operands[0])
	}
	
	return fmt.Sprintf("ERROR: unhandled operandCount for %s\n", def.Name)
}


func ReadUint8(ins Instructions) uint8 {
	return uint8(ins[0])
}


func ReadUint16(ins Instructions) uint16 {
	return binary.BigEndian.Uint16(ins)
}


func Lookup(op byte) (*TDefinition, error) {
	def, ok := definitions[Opcode(op)]
	if !ok {
		return nil, fmt.Errorf("opcode %d undefined", op)
	}
	
	return def, nil
}

func Make(op Opcode, operands ...int) []byte {
	def, ok := definitions[op]
	
	if !ok {
		return []byte{}
	}
	
	instructionLen := 1
	
	for _, w := range def.OperandWidths {
		instructionLen += w
	}
	
	instruction := make([]byte, instructionLen)
	instruction[0] = byte(op)
	offset := 1
	
	for i, o := range operands {
		width := def.OperandWidths[i]
		switch width {
		case 2:
			binary.BigEndian.PutUint16(instruction[offset:], uint16(o))
		case 1:
			instruction[offset] = byte(o)
		}
		offset += width
	}
	return instruction
}

func ReadOperands(def *TDefinition, ins Instructions) ([]int, int) {
	operands := make([]int, len(def.OperandWidths))
	offset := 0
	for i, width := range def.OperandWidths {
		switch width {
		case 2:
			operands[i] = int(ReadUint16(ins[offset:]))
		case 1:
			operands[i] = int(ReadUint8(ins[offset:]))
		}
		offset += width
	}
	
	return operands, offset
}

