"""
Slice a Menger sponge.

This idea came from

   http://www.johndcook.com/blog/2011/08/30/slice-a-menger-sponge/

   http://www.nytimes.com/2011/06/28/science/28math-menger.html

although the Numpy implementation here is completely different from
what was done in the above blog.

"""

import numpy as np
import matplotlib.pyplot as plt
from numpy import newaxis

#import resource
#resource.setrlimit(resource.RLIMIT_AS, (500e6, 500e6))

N = 81*3 # 3*3*3*3
sponge = np.ones((N, N, N), dtype=np.int8)

# Drill holes to make the sponge

def drill_holes(block, level):
    if level < 1:
        # stop recursion
        return

    ax = block.shape[0]//3
    bx = block.shape[0] - ax
    ay = block.shape[1]//3
    by = block.shape[1] - ay
    az = block.shape[2]//3
    bz = block.shape[2] - az

    # Drill
    xslices = (np.s_[:ax], np.s_[ax:bx], np.s_[bx:])
    yslices = (np.s_[:ay], np.s_[ay:by], np.s_[by:])
    zslices = (np.s_[:az], np.s_[az:bz], np.s_[bz:])

    block[ax:bx,ay:by,:] = 0
    block[ax:bx,:,az:bz] = 0
    block[:,ay:by,az:bz] = 0

    # Recurse
    for xslice in xslices:
        for yslice in yslices:
            for zslice in zslices:
                drill_holes(block[xslice,yslice,zslice], level=level-1)

drill_holes(sponge, level=4) # drill 4 levels

plt.figure()
plt.imshow(sponge[:,:,0])
plt.colorbar()
plt.gray()
plt.title("Sponge from top (3 iterations)")
plt.savefig('sponge_top.png', dpi=50)

# -- 2-D cut through the sponge

# Along these directions
u = np.array([0.0,  1.0, -1.0])
v = np.array([1.0, -0.5, -0.5])

u /= np.linalg.norm(u)
v /= np.linalg.norm(v)

# Generate indices
i, j = np.ogrid[-N:N,-N:N]

coords = (i[:,:,newaxis] * u + j[:,:,newaxis] * v).astype(int)
coords += N//2

mask = (np.all(coords >= 0, axis=2) & np.all(coords < N, axis=2))
coords = coords[mask]

sponge_cut = np.zeros((2*N, 2*N), dtype=np.int8)
sponge_cut[mask] = sponge[coords[:,0], coords[:,1], coords[:,2]]

plt.figure()
plt.imshow(sponge_cut)
plt.colorbar()
plt.gray()
plt.title("Cross-cut from the sponge")
plt.savefig('sponge_cut.png', dpi=50)

plt.show()


# Dump as df3 files (for Povray)

def write_df3(file_name, grid):
    nx, ny, nz = grid.shape
    f = open(file_name, 'wb')
    hdr = np.array([(nx, ny, nz)],
                   dtype=[('nx', '>i2'), ('ny', '>i2'), ('nz', '>i2')])
    hdr.tofile(f)
    data = (grid * 255).astype('>u1')
    data.tofile(f)
    f.close()

x, y, z = np.ogrid[0:N,0:N,0:N]
mask = (x + y + z)/np.sqrt(3) > np.sqrt(3)*N/2.

sponge_1 = sponge * ( mask)  # cut the sponge open
sponge_2 = sponge * (~mask)  # cut the sponge open

write_df3('sponge.df3', sponge)
write_df3('sponge_1.df3', sponge_1)
write_df3('sponge_2.df3', sponge_2)
