package com.trumphurst.utils;

import java.util.*;
import java.io.*;

/**
 * Internal error handler which displays errors on System.out, and ignores error lines.
 * @see trumphurst.utils.TProperties#load
 * @see trumphurst.utils.TProperties#fullload
 */
class DefaultHandler implements ValidationExceptionHandler {
	public String handleException(String setcmd, String key,
		String value, String classname, String arguments, Exception e) {
		System.out.println("Exception '" + e + "' adding property '" + setcmd + "'");
		return ";";
	}
}

/**
 * A hashed property list which contains properties with validation.
 * Separate methods are provided for amending existing properties and adding 
 * new properties. Methods are also provided for amending or adding properties
 * given only a human-readable command string, so that a TProperties list can
 * easily be used with a simple command parser to allow humans to edit the 
 * properties without any danger of the properties containing invalid values.
 * @see trumphurst.utils.TProperty
 */
public class TProperties {
	static RCSVersion version = new RCSVersion("$Id: TProperties.java 1.2 1997/09/19 17:09:21 nikki Exp nikki $");
	/**
	 * Debug flag - should be set to 0 in a production environment.
	 * Setting to >0 does extra String allocation to make the VJ++ show values correctly.
	 */
	final static int debug = 1;

	/**
	 * The name of this package - used in the dynamic class loader.
	 */
	private final static String pkg = "trumphurst.utils.";

	/**
	 * The default (only) constructor is public.
	 */
	public TProperties() {
	}

	/**
	 * Add an arbitrary property to the list.
	 * Note that any existing property with the same key will be replaced.
	 * @param key the name or key of this property
	 * @param p the property to add
	 * @see trumphurst.utils.TStringProperty
	 */
	public void add(String key, TProperty p) {
		table.put(key, p);
	}

	/**
	 * Add a new TStringProperty to the list.
	 * Note that any existing property with the same key will be replaced.
	 * This method is here to provide a complete programmers interface.
	 * As it allows the type of a property to be changed, it would not normally
	 * be used.
	 * @param setcmd user-entered command of the form "key=value"
	 * @return true if the command was syntactically correct (the string was added)
	 */
	public boolean add(String setcmd) {
		if(!parse(setcmd))
			return false;
		table.put(key, new TStringProperty(value));
		return true;
	}

	/**
	 * Add an arbitrary TProperty to the list.
	 * This powerful method allows the user to add any kind of property, using
	 * dynamic class loading. It is mainly for testing the TProperties class, and by fullload.
	 * @param setcmd user-entered command of the form "key=classname(parameters)"
	 * @exception ValidationException if the value argument is invalid for the new property
	 * @exception ClassNotFoundException if the classname could not be loaded
	 * @exception InstantiationException if the classname could not be instantiated
	 * @exception IllegalAccessException if the classname is not accessable from trumphurst.utils
	 * @exception ClassCastException if the classname does not implement TProperty
	 * @return true if the command was valid, classname implements TProperty, and the parameters were valid
	 * @see trumphurst.utils.TProperties#fullload
	 */
	public boolean addnew(String setcmd) 
			throws ValidationException, ClassNotFoundException, 
				InstantiationException, IllegalAccessException,
				ClassCastException {
		if(!parse(setcmd))
			return false;
		if(!parseclass())
			return false;
		try {
			Class cl = Class.forName(classname);
		}
		catch(ClassNotFoundException cnf) {
			classname = pkg + classname;
		}

		Class cl = Class.forName(classname);
		TProperty p = (TProperty)cl.newInstance();

		p.construct(arguments);
		table.put(key, p);
		return true;
	}

	/**
	 * Removes a property.
	 * @param key the key of the property to remove
	 * @return the property removed, or null if there was no property with that key
	 */
	public TProperty remove(String key) {
		return (TProperty)table.remove(key);
	}

	/**
	 * Changes an existing property to an arbitrary new one.
	 * This method is here to help provide a complete programmers interface.
	 * As it allows the type of a property to be changed, it would not normally
	 * be used.
	 * @param key the key of the property to change
	 * @param p the new property to replace the existing one
	 * @return true if the old property exists and was replaced, false if not
	 */
	public boolean set(String key, TProperty p) {
		TProperty old = (TProperty)table.get(key);

		if(old == null)
			return false;
		table.put(key, p);
		return true;
	}

	/**
	 * Sets an existing property to a new value, validating the value.
	 * This is the heart of the functionality provided - it allows a string
	 * value to be validated to ensure it is a correct value for the property.
	 * @param key the key of the existing property
	 * @param value the string representation of the new value
	 * @return true if the key exists and the value is valid
	 * @exception ValidationException value is not a valid value for the existing property
	 */
	public boolean set(String key, String value) throws ValidationException {
		TProperty old = (TProperty)table.get(key);

		if(old == null)
			return false;
		old.fromString(value);
		return true;
	}

	/**
	 * Sets an existing property to a new value, validating the value.
	 * This is the method that would be called from a user command processor.
	 * It splits up the setcmd into key and value, and calls set(key, value).
	 * @param setcmd user-entered command of the form "key=value"
	 * @exception ValidationException value is not a valid value for the existing property
	 * @return true if the setcmd is syntactically correct, the key exists and the value is valid
	 */
	public boolean set(String setcmd) throws ValidationException {
		if(!parse(setcmd))
			throw new ValidationException("no '='");
		return set(key, value);
	}

	/**
	 * Return the property for a particular key.
	 * @param key the key to look up
	 * @return the property, or null if there wasn't one for that key
	 */
	public TProperty get(String key) {
		return (TProperty)table.get(key);
	}

	/**
	 * A convenience method to return the value of the TStringProperty for a particular key.
	 * @param key the key to look up
	 * @return the String value of the property, or null if there wasn't one for that key, or the property wasn't a TStringProperty
	 * @see trumphurst.utils.TStringProperty
	 */
	public String getString(String key) {
		return TStringProperty.get(this, key);
	}

	/**
	 * A convenience method to return the value of the TIntegerProperty for a particular key.
	 * @param key the key to look up
	 * @return the int value of the property, or null if there wasn't one for that key, or the property wasn't a TIntegerProperty
	 * @see trumphurst.utils.TIntegerProperty
	 */
	public int getInt(String key) {
		return TIntegerProperty.get(this, key);
	}

	/**
	 * A convenience method to return the value of the TLongProperty for a particular key.
	 * @param key the key to look up
	 * @return the long value of the property, or null if there wasn't one for that key, or the property wasn't a TLongProperty
	 * @see trumphurst.utils.TLongProperty
	 */
	public long getLong(String key) {
		return TLongProperty.get(this, key);
	}

	/**
	 * A convenience method to return the value of the TDoubleProperty for a particular key.
	 * @param key the key to look up
	 * @return the double value of the property, or null if there wasn't one for that key, or the property wasn't a TDoubleProperty
	 * @see trumphurst.utils.TDoubleProperty
	 */
	public double getDouble(String key) {
		return TDoubleProperty.get(this, key);
	}

	/**
	 * A convenience method to return the value of the TBooleanProperty for a particular key.
	 * @param key the key to look up
	 * @return the boolean value of the property, or null if there wasn't one for that key, or the property wasn't a TBooleanProperty
	 * @see trumphurst.utils.TBooleanProperty
	 */
	public boolean getBoolean(String key) {
		return TBooleanProperty.get(this, key);
	}

	/**
	 * Return an Enumeration which iterates over the keys of the property list.
	 */
	public Enumeration keys() {
		return table.keys();
	}

	/**
	 * Return a SortedEnumeration which iterates over the keys of the property list.
	 */
	public Enumeration sortedKeys() {
		return new SortedEnumeration(table.keys(), new StringComparer());
	}

	/**
	 * Load the list of properties from a stream.
	 * The stream consists of lines of the form "key=value".
	 * Lines starting with ";", or which don't take the correct form, are ignored.
	 * Each line is loaded and parsed, then used as an argument to set.
	 * If a property with that key does not already exist, a TStringProperty
	 * with that value is added.
	 * If set throws a ValidationException (because the value is not valid for that property),
	 * the handler is called. The return value from the handler may be ...<DL>
	 * <DT>Null, or the original line
	 * <DD>The original property is replaced with a TStringProperty containing the value
	 * <DT>A string starting with ";", or an invalid command line
	 * <DD>The original property is left unchanged
	 * <DT>A different valid command line
	 * <DD>The new command line is reparsed, and reprocessed in the same way</DL>
	 * @param in the stream to load
	 * @param handler a handler to deal with exceptions that occur during the load.
	 * @exception IOException if an IOException is thrown by in.readline
	 */
	public void load(DataInputStream in, ValidationExceptionHandler handler) throws IOException {
		String setcmd;
		
		while((setcmd = in.readLine()) != null) {
			while(true) {
				if(setcmd.startsWith(";"))
					break;
				if(!parse(setcmd))
					break;
				try {
					if(set(key, value))
						break;
				}
				catch(ValidationException e) {
					String newcmd = handler.handleException(setcmd, 
							key, value, classname, arguments, e);

					if(newcmd != null && !newcmd.equals(setcmd)) {
						setcmd = newcmd;
						continue;
					}
				}
				set(key, new TStringProperty(value));
				break;
			}
		}
	}

	/**
	 * Load the list of properties from a stream.
	 * The stream consists of lines of the form "key=value".
	 * Lines starting with ";", or which don't take the correct form, are ignored.
	 * Each line is loaded and parsed, then used as an argument to set.
	 * If a property with that key does not already exist, a TStringProperty
	 * with that value is added.
	 * If set throws a ValidationException (because the value is not valid),
	 * the original property is left unchanged.
	 * @param in the stream to load
	 * @exception IOException if an IOException is thrown by in.readline
	 */
	public void load(DataInputStream in) throws IOException {
		load(in, new DefaultHandler());
	}

	/**
	 * Load a list of arbitrary properties from a stream.
	 * The stream consists of lines of the form "key=classname(params)", or "key=value".
	 * Lines starting with ";", or which don't take the correct form, are ignored.
	 * Each line is loaded and parsed. If it takes the second form, a TStringProperty
	 * with that value is added.
	 * Otherwise, the line is used as an argument to addnew.
	 * If addnew throws an Exception (because the value is not valid for the new property,
	 * or some kind of class loading exception occurs), the handler is called. 
	 * The return value from the handler may be ...<DL>
	 * <DT>Null, or the original line
	 * <DD>A new TStringProperty containing everything to the right of the '=' is added
	 * <DT>A string starting with ";", or an invalid command line
	 * <DD>The line is ignored.
	 * <DT>A different valid command line
	 * <DD>The new command line is reparsed, and reprocessed in the same way</DL>
	 * @param in the stream to load
	 * @param handler a handler to deal with exceptions that occur during the load.
	 * @exception IOException if an IOException is thrown by in.readline
	 * @see trumphurst.utils.TProperties#addnew
	 */
	public void fullload(DataInputStream in, ValidationExceptionHandler handler) throws IOException {
		String setcmd;
		
		while((setcmd = in.readLine()) != null) {
			while(true) {
				if(setcmd.startsWith(";"))
					break;
				if(!parse(setcmd))
					break;
				try {
					if(addnew(setcmd))
						break;
				}
				catch(Exception e) {
					String newcmd = handler.handleException(setcmd, 
							key, value, classname, arguments, e);

					if(newcmd != null && !newcmd.equals(setcmd)) {
						setcmd = newcmd;
						continue;
					}
				}
				set(key, new TStringProperty(value));
				break;
			}
		}
	}

	/**
	 * Load a list of arbitrary properties from a stream.
	 * The stream consists of lines of the form "key=classname(params)", or "key=value".
	 * Lines starting with ";", or which don't take the correct form, are ignored.
	 * Each line is loaded and parsed. If it takes the second form, a TStringProperty
	 * with that value is added.
	 * Otherwise, the line is used as an argument to addnew.
	 * If addnew throws an Exception (because the value is not valid for the new property,
	 * or some kind of class loading exception occurs), the line is ignored. 
	 * @param in the stream to load
	 * @exception IOException if an IOException is thrown by in.readline
	 * @see trumphurst.utils.TProperties#addnew
	 */
	public void fullload(DataInputStream in) throws IOException {
		fullload(in, new DefaultHandler());
	}

	/**
	 * Save the property list to a stream.
	 * Saves each property in the form "key=value", one per line, to the stream.
	 * @param out the stream
	 * @param e an Enumeration to iterate over the keys
	 * @see trumphurst.utils.TProperties#load
	 */
	void save(PrintStream out, Enumeration e) {
		try {
			while(e.hasMoreElements()) {
				String key = (String)e.nextElement();
				out.println(key + "=" + get(key));
			}
		}
		catch(NoSuchElementException nse) {
		}
	}

	/**
	 * Save the property list to a stream.
	 * Saves each property in the form "key=value", one per line, to the stream.
	 * @param out the stream
	 * @see trumphurst.utils.TProperties#load
	 */
	public void save(PrintStream out) {
		save(out, keys());
	}

	/**
	 * Save the property list to a stream.
	 * Saves each property in the form "key=value", one per line, to the stream.
	 * Sorts the output.
	 * @param out the stream
	 * @see trumphurst.utils.TProperties#load
	 */
	public void saveSorted(PrintStream out) {
		save(out, sortedKeys());
	}

	/**
	 * Save the property list to a stream.
	 * Saves each property in the form "key=classname(params)", one per line, to the stream.
	 * @param out the stream
	 * @param e an Enumeration to iterate over the keys
	 * @see trumphurst.utils.TProperties#fullload
	 */
	private void fullsave(PrintStream out, Enumeration e) {
		try {
			while(e.hasMoreElements()) {
				String key = (String)e.nextElement();
				TProperty p = get(key);
				String classname = p.getClass().getName();

				if(classname.startsWith(pkg))
					classname = classname.substring(pkg.length());
				out.println(key + "=" + classname + "(" + p.arguments() + ")");
			}
		}
		catch(NoSuchElementException nse) {
		}
	}

	/**
	 * Save the property list to a stream.
	 * Saves each property in the form "key=classname(params)", one per line, to the stream.
	 * @param out the stream
	 * @see trumphurst.utils.TProperties#fullload
	 */
	public void fullsave(PrintStream out) {
		save(out, keys());
	}

	/**
	 * Save the property list to a stream.
	 * Saves each property in the form "key=classname(params)", one per line, to the stream.
	 * Sorts the output.
	 * @param out the stream
	 * @see trumphurst.utils.TProperties#fullload
	 */
	public void fullsaveSorted(PrintStream out) {
		save(out, sortedKeys());
	}

	/** The Hastable containing the properties */
	private Hashtable table = new Hashtable();
	/** Key parsed from "key=value" */
	private String key;
	/** Value parsed from "key=value" */
	private String value;
	/** Classname parsed from "key=classname(arguments)" */
	private String classname;
	/** Arguments parsed from "key=classname(arguments)" */
	private String arguments;
	/** Parse a setcmd into name, value */
	private boolean parse(String setcmd) {
		int pos = setcmd.indexOf('=');

		if(pos < 0)
			return false;
		key = setcmd.substring(0, pos);
		value = setcmd.substring(pos + 1);
		if(debug > 0) {
			key = new String(key);
			value = new String(value);
		}
		return true;
	}
	/** Further parse a value into classname, arguments */
	private boolean parseclass() {
		int pos = value.indexOf('(');

		if(pos < 1 || !value.endsWith(")"))
			return false;
		classname = value.substring(0, pos);
		arguments = value.substring(pos + 1, value.length() - 1);
		if(debug > 0) {
			classname = new String(classname);
			arguments = new String(arguments);
		}
		return true;
	}
}

