package com.trumphurst.utils;

import java.io.*;

/** A TProperty which holds a File value, with optional validation constraints */
public class TFileProperty implements TProperty {
	static RCSVersion version = new RCSVersion("$Id: TFileProperty.java 1.2 1997/09/19 17:09:18 nikki Exp nikki $");
	/** The value */
	protected File f;
	/** The validation constraints */
	protected int flags;

	/** Validation constraint - must be a file, not a directory */
	public final static int ISFILE = 1;
	/** Validation constraint - must be a directory, not a file */
	public final static int ISDIR = 2;
	/** Validation constraint - must exist */
	public final static int EXISTS = 4;
	/** Validation constraint - the parent directory must exist */
	public final static int PATHEXISTS = 8;
	/** Validation constraint - must be readable (if it exists) */
	public final static int READABLE = 16;
	/** Validation constraint - must be a writeable (if it exists) */
	public final static int WRITEABLE = 32;
	/** Validation constraint - an empty filename is allowed */
	public final static int NULLOK = 64;

	/**
	 * Construct with a value
	 * @param name the file/path name
	 * @param flags any number of validation constraints or'ed together
	 * @exception ValidationException if the name does not satisfy the constraints
	 */
	public TFileProperty(String name, int flags) throws ValidationException {
		this.flags = flags;
		fromString(name);
	}

	/** Return the value */
	public File get() {
		return f;
	}

	/**
	 * Return the File value of the specified property in the properties list.
	 * Verifies that the key exists, and refers to a TFileProperty (or a
	 * subclass).
	 * @param properties the properties list
	 * @key the key to look up
	 * @return the File value of the property, or null
	 */
	public static File get(TProperties properties, String key) {
		TProperty p = properties.get(key);
		if(p != null && p instanceof TFileProperty)
			return ((TFileProperty)p).get();
		return null;
	}

	/** 
	 * Set the value 
	 * @param f the value to set
	 * @exception ValidationException the File value does not satisfy all the constraints.
	 */
	public void set(File f) throws ValidationException {
		if((flags & NULLOK) == 0 || f.getPath().length() > 0) {
			if((flags & PATHEXISTS) != 0) {
				String s = f.getAbsolutePath();
				String parent = new File(s).getParent();

				if(parent != null) {
					if(TProperties.debug > 0)
						parent = new String(parent);
					// Work round bug with root directories
					if(parent.length() == 2 && parent.charAt(1) == ':') {
						if(s.length() > 3)
							parent = s.substring(0,3);
						else
							parent = null;
					}
				}
				if(parent != null) {
					File p = new File(parent);
					
					if(!p.isDirectory())
						throw new ValidationException("path does not exist");
				}
				else
					throw new ValidationException("value is the root directory");
			}
			if((flags & EXISTS) != 0) {
				if((flags & ISFILE) != 0)
					if(!f.isFile())
						throw new ValidationException("not an existing file");
				if((flags & ISDIR) != 0)
					if(!f.isDirectory())
						throw new ValidationException("not an existing directory");
			}
			else {
				if((flags & ISDIR) != 0)
					if(f.isFile())
						throw new ValidationException("file of that name already exists");
				if((flags & ISFILE) != 0)
					if(f.isDirectory())
						throw new ValidationException("directory of that name already exists");
			}
			if(f.exists()) {
				if((flags & READABLE) != 0)
					if(!f.canRead())
						throw new ValidationException("not readable");
				if((flags & WRITEABLE) != 0)
					if(!f.canWrite())
						throw new ValidationException("not writeable");
			}
		}
		this.f = f;
	}

	/**
	 * "Construct" the property from a String argument list.
	 * @param arguments a string of the form "filename,flags"
	 * @exception ValidationException if the arguments cannot be parsed, 
	 * the flags value is not an integer, or the file does not satisfy 
	 * the constraints.
	 */
	public void construct(String arguments) throws ValidationException {
		try {
			int pos = arguments.lastIndexOf(',');

			if(pos < 0)
				throw new ValidationException("not enough arguments");
			flags = Integer.parseInt(arguments.substring(pos + 1));
			if(TProperties.debug > 0)
				fromString(new String(arguments.substring(0, pos)));
			else
				fromString(arguments.substring(0, pos));
		}
		catch(NumberFormatException nf) {
			throw new ValidationException("flag not an integer");
		}
	}

	/**
	 * Change the value of the property from a String representation of the new value.
	 * @param arguments a filename/path string
	 * @exception ValidationException if the filename does not satisfy the constraints
	 */
	public void fromString(String s) throws ValidationException {
		set(new File(s));
	}

	/**
	 * Return a String representation of the value.
	 * @return the filename
	 */
	public String toString() {
		return f.getPath();
	}

	/**
	 * Return a String representation of the arguments necessary to construct the property.
	 * @return a string of the form "filename,flags"
	 */
	public String arguments() {
		return toString() + "," + flags;
	}

	/** Empty constructor used immediately before <B>construct</B> is called */
	TFileProperty() {
	}
}

