package com.trumphurst.utils;

import java.util.*;
import java.io.*;

/**
 * A Dictionary which stores String/Object pairs in a radix trie structure,
 * enabling them to be retrieved in sorted order. For details of the
 * Patricia algorithm, see "Algorithms" by Robert Sedgewick (Addison-Wesley).
 * Comparisons are made at the bit level, and only enough bits to uniquely 
 * identify an individual key need to be compared to find it. This makes it
 * a very efficient way of storing items with large keys. <P>
 * Note that the other Dictionaries in Java can use arbitrary Objects as
 * keys, because they do not sort in order, and thus only need hashCode and
 * isequal to compare keys, both of which exist for all objects. This class
 * needs to access arbitrary bits in the key to do its comparisons. As most
 * keys in real-life are Strings, PatriciaTree merely calls toString on its
 * key to convert it to a String (a no-effect operation if the key is already
 * a String), and uses the String as its real key. Unfortunately, this 
 * precludes using (say) Integers directly as keys (they would be sorted in 
 * alphabetical order, so that 100 is less than 2). It would be necessary to 
 * create a new class whose toString method returned a String which would sort 
 * in the correct order. <P>
 * The alternative would be to create an interface containing the necessary
 * comparison functions, and force the key to implement the interface. However
 * this precludes using <u>any</u> existing class as the key, forcing even the 
 * most common case (Strings as keys) to do extra work.
 */
public class PatriciaTree extends Dictionary {
	static RCSVersion version = new RCSVersion("$Id: PatriciaTree.java 1.2 1997/09/19 17:09:13 nikki Exp nikki $");
	/** The root node of the tree */
	PatriciaNode root;
	/** The number of elements in the tree */
	int count;
	/**
	 * The highest bit which might be set in an extracted char. Note that this
	 * is not 0x8000, as you might expect, because we need an extra value to
	 * use when a particular character position in a String does not exist
	 * (i.e. one String is longer than another), and another value to represent
	 * 0, as every key must have at least one bit set in every position.
	 */
	final static int maxbits = 0x10000;
	/**
	 * A dummy node, to indicate the unused, left-most leaf of the tree. One
	 * dummy is always present in the tree, because the number of leaves in
	 * a binary tree is always one greater than the number of non-leaf nodes.
	 */
	final static PatriciaNode dummy = new PatriciaNode();

	/** 
	 * Internal function to help with printing a tree to System.out.
	 * @param tab the length of the string to return
	 * @param c the character to make up the string
	 * @return a String of length tab consisting of the same repeated character c
	 */
	static String spaces(int tab, char c) {
		StringBuffer s = new StringBuffer(tab);

		while(tab-- > 0)
			s.append(c);
		return s.toString();
	}

	public PatriciaTree() {
	}

	/** 
	 * Internal function to help with printing a tree to System.out. 
	 * @param n root of the sub-tree to print
	 * @param tab prefix string to print in front
	 * @param parent parent of n, or null if n is the root
	 */
	static void printTree(PatriciaNode n, String tab, PatriciaNode parent) {
		String p = (parent == null) ? "" : spaces(parent.toString().length(), ' ');
		String s = spaces(n.toString().length(), '-');
		String c;

		c = (parent == null) ? "" : ((n == parent.right) ? " " : "|");
		if(n.right.isLeaf(n)) {
			System.out.println(tab + c + p + "+" + s + "-" + n.right);
		} else
			printTree(n.right, tab + c + p, n);
		if(parent != null)
			System.out.print(tab + "+" + spaces(parent.toString().length(), '-'));
		System.out.println("+" + n);
		c = (parent == null) ? "" : ((n == parent.left) ? " " : "|");
		if(n.left.isLeaf(n)) {
			System.out.println(tab + c + p + "+" + s + "-" + n.left);
		} else
			printTree(n.left, tab + c + p, n);
	}

	/**
	 * Function to print the tree to System.out. Not used in a production
	 * environment, but helps to understand what is going on during
	 * debugging of PatriciaTree itself.
	 */
	public void print() {
		printTree(root, "", null);
	}

    /**
     * Returns an enumeration of the elements. Use the Enumeration methods 
     * on the returned object to fetch the elements sequentially in key
	 * order.
     * @see PatriciaTree#keys
     * @see Enumeration
     */
	public Enumeration elements() {
		return new PatriciaTreeElementEnumeration(this);
	}

    /**
     * Gets the object associated with the specified key in the PatriciaTree.
     * @param key the key in the hash table. Note that key.toString() is the
	 * used as the actual key (as Strings can be ranked in order).
     * @returns the element for the key, or null if the key
     * 		is not defined in the hash table.
     * @see PatriciaTree#put
	 * @see Object#toString
	 * @see String#toString
     */
    public Object get(Object key) {
		String s = key.toString();
		PatriciaNode t = search(s);

		if(t == null || !s.equals(t.key))
			return null;
		return t.element;
	}

    /**
     * Returns true if the PatriciaTree contains no elements.
     */
    final public boolean isEmpty() {
		return count == 0;
	}

    /**
     * Returns an enumeration of the PatriciaTree's keys. Use the Enumeration 
	 * methods on the returned object to fetch the keys sequentially in order.
     * @see PatriciaTree#elements
     * @see Enumeration
     */
	public Enumeration keys() {
		return new PatriciaTreeKeyEnumeration(this);
	}

    /**
     * Puts the specified element into the PatriciaTree, using the specified
     * key.  The element may be retrieved by doing a get() with the same 
     * key.  The key and the element cannot be null.
     * @param key the specified key. Note that key.toString() is the
	 * used as the actual key (as Strings can be ranked in order).
     * @param value the specified element 
     * @return the old value of the key, or null if it did not have one.
     * @exception NullPointerException If the value of the specified
     * key or element is null.
     * @see PatriciaTree#get
	 * @see Object#toString
	 * @see String#toString
     */
    public Object put(Object key, Object value) {
		if(value == null)
			throw new NullPointerException("Null value in PatriciaTree");
		PatriciaNode t = insert(key.toString());
		Object o = t.element;

		t.element = value;
		return o;
	}

	/**
	 * This function is supposed to remove items from the tree, but it is not
	 * implemented yet, and does nothing but return null.
	 */
	public Object remove(Object key) {
		return null;
	}

    /**
     * Returns the number of elements contained within the PatriciaTree. 
     */
    final public int size() {
		return count;
	}

	/**
	 * Internal function which actually does the work of searching the Tree for
	 * the place where a particular key would be, if it existed.
	 * @param key the key to search for
	 * @return the node containing the key, or the nearest node, or the dummy
	 * node (if the tree is empty)
	 */
	final private PatriciaNode search(String key) {
		if(root == null)
			return dummy;			// Tree is empty

		PatriciaNode p;				// Parent of node x
		PatriciaNode x = root;		// Current node - start at root
		int offset = -1;			// Offset of significant char in key
		int c = 0;					// 'Char' extracted from string

		do {
			p = x;					// Remember parent
			if(offset != x.offset) {// Extract char if necessary
				offset = x.offset;
				c = (offset >= key.length()) ? 1 : (key.charAt(offset) + 2);
				// Equivalent to extractChar(key, offset) - inlined for speed
			}
			if((c & x.bits) == 0)
				x = x.left;			// Relevant bit not set - take left fork
			else
				x = x.right;		// Relevant bit is set - take right fork
		} while(!((x.offset < p.offset) || (x.offset == p.offset && x.bits >= p.bits)));
		// This test is equivalent to while(!x.isLeaf(p)) - inlined for speed
		// i.e. traverse tree till x is a leaf node
		return x;
	}

	/**
	 * Internal function which actually does the work of inserting a new item
	 * in the tree (or returning an existing item if the key already exists).
	 * @param key the key to insert
	 * @return an existing node with the same key, or the new inserted node
	 */
	final private PatriciaNode insert(String key) {
		PatriciaNode t = search(key);	// Nearest key or dummy
		int offset;						// Of significant character in string
		int bit = maxbits;				// Highest significant bit
		int c = 0;						// 'Character' extracted from String

		if(t.key == null) {				// At left-most leaf node, or tree empty
			offset = 0;					// First character is significant one
			if(key.length() != 0) {
				c = key.charAt(0) + 2;	// Extract character (extractChar(key, 0))
				while((c & bit) == 0)	// Find most significant bit
					bit >>>= 1;
			} else
				bit = 1;				// Use least significant bit for empty key
		} else {						// Have found a real key
			if(key.equals(t.key))		// Is it the same as the one we want to insert?
				return t;				// Don't insert, return found key

			int l, r;					// 'Characters' extracted from Strings

			offset = -1;				// Next line will increment to start at 0
			do {
				offset++;				// Process next character
				// Extract key character - equivalent to extractChar(key, offset)
				l = (offset >= key.length()) ? 1 : (key.charAt(offset) + 2);
				// Extract t.key character - equivalent to extractChar(t.key, offset)
				r = (offset >= t.key.length()) ? 1 : (t.key.charAt(offset) + 2);
			} while(l == r);			// Until we find a different character
			while((l & bit) == (r & bit))
				bit >>>= 1;				// Find most significant differing bit
		}

		// Create the new node, which will be a leaf
		t = new PatriciaNode(key, offset, bit);
		if(root == null) {				// Tree empty
			t.bits = maxbits;			// Root must always have bits = maxbits
			t.offset = 0;				// and offset = 0
			root = t;					// Root points to new node
			t.left = dummy;				// Left leaf not used
			t.right = root;				// Right leaf is new key
			count = 1;					// 1 item in tree now
			return t;
		}

		PatriciaNode p;					// Parent of node x
		PatriciaNode x = root;			// Traverse tree starting at root

		offset = -1;					// Offset of significant char in string
		do {
			p = x;						// Remember parent
			if(offset != x.offset) {	// Extract next char if necessary
				offset = x.offset;
				c = (offset >= key.length()) ? 1 : (key.charAt(offset) + 2);
										// extractChar(key, offset) inlined for speed
			}
			if((c & x.bits) == 0)
				x = x.left;				// Relevant bit not set - take left fork
			else
				x = x.right;			// Relevant bit set - take right fork
		} while(!((t.offset < x.offset) || (t.offset == x.offset && t.bits >= x.bits))
			&& !((x.offset < p.offset) || (x.offset == p.offset && x.bits >= p.bits)));
		// while(!t.isLeaf(x) && !x.isLeaf(p)) inlined for efficiency
		// Keep going until the new node belongs above x in the tree, or until we reach a leaf
		if(offset != t.offset) {		// Extract char for t
			offset = t.offset;
			c = (offset >= key.length()) ? 1 : (key.charAt(offset) + 2);
		}
		if((c & t.bits) == 0) {			// If bit is 0
			t.left = t;					// add itself to left fork
			t.right = x;				// point right fork at found node
		} else {						// If bit is 1
			t.right = t;				// add itself to right fork
			t.left = x;					// point left fork at found node
		}
		if(offset != p.offset) {		// Extract char for p
			offset = p.offset;
			c = (offset >= key.length()) ? 1 : (key.charAt(offset) + 2);
		}
		if((c & p.bits) == 0)			// If bit is 0
			p.left = t;					// Add new node to left fork
		else							// If bit is 1
			p.right = t;				// Add new node to right fork
		count++;						// We have added a node
		return t;						// Return new node
	}

	/**
	 * Internal function to find the highest bit which is different between l and r.
	 * Note that all calls to this function have been manually inlined for speed.
	 * @param l, r the items to compare
	 * @return an integer with one bit set, being the most significant bit in which
	 * the two parameters differ.
	 */
	final static int diffBit(int l, int r) {
		int bit = maxbits;

		while((l & bit) == (r & bit))
			bit >>>= 1;
		return bit;
	}
	
	/**
	 * Internal function to return a suitable bit pattern to represent the 
	 * offset'th character of the String key. Note the return value must always
	 * have at least one bit set (i.e. cannot be zero), so we return 1 if the
	 * offset is beyond the end of the string, 2 if the character is '\0', 
	 * 3 if the character is '\1', etc. This gives a range of 1-65537.
	 * Note that all calls to this function have been manually inlined for speed.
	 * @param key the key from which to extract the character
	 * @param offset the character number to extract (0 = first character)
	 * @return an integer in the range 1-65537 which represents the character 
	 * (1 if the string was shorter than offset, otherwise the character plus 2)
	 */
	final static int extractChar(String key, int offset) {
		if(offset >= key.length())
			return 1;
		return key.charAt(offset) + 2;
	}

};

/**
 * A node in a PatriciaTree.
 * <p>
 * The classic Patricia Tree uses fixed-length integers as the keys. In such a tree,
 * it is easy to extract single bits from the integer. We use Strings, so extracting
 * a single bit is a two-stage process - extract the character at a particular offset,
 * then extract the bit from the character.
 * @see PatriciaTree
 */
class PatriciaNode {
	/** Left fork */
	PatriciaNode left;
	/** Right fork */
	PatriciaNode right;
	/** The key */
	String key;
	/** The item stored with this key */
	Object element;
	/** Offset of the character to test */
	int offset;
	/** bit to test in the offset'th character */
	int bits;

	/** Create a "dummy" node, with a null key */
	PatriciaNode() {
		left = right = this;
		offset = 0;
		bits = PatriciaTree.maxbits << 1;
	}

	/** Create a real node */
	PatriciaNode(String key, int offset, int bits) {
		this.key = key;
		this.offset = offset;
		this.bits = bits;
	}

	/**
	 * Used by debug tree print.
	 * @see PatriciaTree#print
	 */
	public String toString() {
		if(key == null)
			return "(empty)";
/*
		return key;
*/
		return key + "[" + Integer.toString(offset)
			+ "] " + Integer.toHexString(bits + 0x100).substring(1) 
			+ " " + Integer.toHexString(PatriciaTree.extractChar(key, offset) + 0x100).substring(1);
	}

	/**
	 * Determines if this node is a leaf node.
	 * 
	 */
	final boolean isLeaf(PatriciaNode parent) {
		return (offset < parent.offset) || (offset == parent.offset && bits >= parent.bits);
	}
}

/**
 * Enumerator over a PatriciaTree which returns the elements in order.
 * Internal class, used by the key and element enumerations.
 * @see PatriciaTreeKeyEnumeration
 * @see PatriciaTreeElementEnumeration
 */
class PatriciaTreeEnumerator {
	/** Stack of parent objects */
	Stack parents = new Stack();
	/** Current node during traversal */
	PatriciaNode node;
	/** Current leaf during traversal */
	PatriciaNode leaf;

	/**
	 * Constructor.
	 * @param tree the tree to traverse
	 */
	PatriciaTreeEnumerator(PatriciaTree tree) {
		node = tree.root;
		if(node != null) {
			followLeftTree(node.left);
			followLeftTree(node.right);
			if(node.left == PatriciaTree.dummy)
				leaf = node.right;
			else
				leaf = node.left;
		}
	}

	/** Have we finished yet */
	public boolean hasMoreElements() {
		return (leaf != null);
	}

	/** Return the next node is sequence */
	PatriciaNode nextNode() {
		if(leaf == null)
			throw new NoSuchElementException("PatriciaTreeEnumerator");
		PatriciaNode r = leaf;		// Value to return

		if(leaf == node.right) {
			node = parents.empty() ? null : (PatriciaNode)parents.pop();
		}
		if(node == null) {
			leaf = null;
		} else {
			followLeftTree(node.right);
			if(leaf == node.left)
				leaf = node.right;
			else
				leaf = node.left;
		}
		return r;
	}

	/** Drop down to a leaf, taking left branch, and stacking nodes */
	void followLeftTree(PatriciaNode n) {
		while(!((n.offset < node.offset) || (n.offset == node.offset && n.bits >= node.bits))) {
//			n.isLeaf(node)) {
			if(n != node.right)
				parents.push(node);
			node = n;
			n = node.left;
		}
	}
}

/**
 * Enumeration to iterate over PatriciaTree keys in order.
 * @see PatriciaTree#keys
 */
class PatriciaTreeKeyEnumeration extends PatriciaTreeEnumerator implements Enumeration {
	/** Constructor.
	 * @param t the tree to traverse
	 */
	PatriciaTreeKeyEnumeration(PatriciaTree t) {
		super(t);
	}

	/** Return the next key in sequence. */
	public Object nextElement() {
		return nextNode().key;
	}
}

/**
 * Enumeration to iterate over PatriciaTree elements in order.
 * @see PatriciaTree#elements
 */
class PatriciaTreeElementEnumeration extends PatriciaTreeEnumerator implements Enumeration {
	/** Constructor.
	 * @param t the tree to traverse
	 */
	PatriciaTreeElementEnumeration(PatriciaTree t) {
		super(t);
	}

	/** Return the next key in sequence. */
	public Object nextElement() {
		return nextNode().element;
	}
}

