package com.trumphurst.controls;

import com.trumphurst.utils.*;
import java.awt.*;

/** ObjectPainter to paint a column of a MultiColumnList with the 
 * representation of the Tree structure. Contracted Trees are shown
 * with a plus (+) in a box, expanded ones with a minus (-). This is
 * followed by the text returned by toString on the object (but this can
 * be individually overridden). The structure of any expanded trees is
 * shown with lines.
 * @see Tree
 * @see TreeListDocument
 * @see MultiColumnList
 */
public class TreePainter extends StringPainter {
	static RCSVersion version = new RCSVersion("$Id: TreePainter.java 1.4 1997/09/19 17:09:41 nikki Exp nikki $");
	public TreePainter(String title) {
		super(title);
	}

	/** Overridable function to paint the identity of the Tree object.
	 * This default version shows the string returned by toString().
	 * @param g Graphics on which to draw.
	 * @param r Rectangle in which to draw. May have a negative width!
	 * @param t The Tree object.
	 * @param clip Whether to clip to the Rectangle.
	 */
	public void paintData(Graphics g, Rectangle r, Tree t, boolean clip) {
		paintString(g, r.x, r.y, t.o.toString());
	}

	/** Business function which actually draws the tree structure etc.
	 * Contracted Trees are shown with a plus (+) in a box, expanded 
	 * ones with a minus (-). This is followed by the identity painted by
	 * paintData. The structure of any expanded trees is shown with lines.
	 */
	public void paintObject(Graphics g, Rectangle r, Object o, boolean clip) {
		Tree t = (Tree)o;
		int lwidth = g.getFontMetrics().stringWidth("0") | 1;

		if(lwidth < 9)
			lwidth = 9;

		int depth = t.depth();
		int x = r.x + lwidth * depth;
		int y = r.y;
		Rectangle tmp = new Rectangle(x, y, r.width - (x - r.x), r.height);

		// Paint optional expansion cross in rectangle
		if(t.first != null) {
			Color old = g.getColor();
			g.setColor(old == Color.white ? Color.black : Color.white);
			g.fillRect(x, y + 2, lwidth, r.height - 4);
			g.setColor(old);
			g.drawRect(x, y + 2, lwidth - 1, r.height - 5);
			g.drawLine(x + 2, y + r.height / 2, x + lwidth - 3, y + r.height / 2);
			if(t.expanded)
				g.drawLine(x + lwidth - lwidth / 2, y + r.height - 3, x + lwidth - lwidth / 2, y + r.height);
			else {
				int off = r.height / 3;
				
				g.drawLine(x + lwidth / 2, y + off, x + lwidth / 2, y + r.height - off - 1);
			}
			tmp.x += lwidth;
			tmp.width -= lwidth;
		}
		paintData(g, tmp, t, clip);
		x -= lwidth / 2;
		// Paint right-most column
		if(depth-- > 0) {
			Tree p = t.parent;
			if(r.contains(x, y)) {
				g.drawLine(x, y + r.height / 2, x + lwidth / 2 - 1, y + r.height / 2);
				if(t != p.last)
					g.drawLine(x, y, x, y + r.height);
				else
					g.drawLine(x, y, x, y + r.height / 2);
			}
			t = p;
			x -= lwidth;
		}
		// Paint remaining columns from right to left
		while(depth-- > 0) {
			Tree p = t.parent;
			boolean line = false;

			if(t != p.last && r.contains(x, y))
				g.drawLine(x, y, x, y + r.height);
			t = p;
			x -= lwidth;
		}
	}
}


