package com.trumphurst.controls;

import com.trumphurst.utils.*;

/** Basic class to hold a tree structure. */
public class Tree {
	static RCSVersion version = new RCSVersion("$Id: Tree.java 1.3 1997/09/19 17:09:40 nikki Exp nikki $");
	/** The parent. */
	Tree parent;
	/** The first child. */
	Tree first;
	/** The last child */
	Tree last;
	/** The next sibling. */
	Tree next;
	/** Indicates whether the children are visible in a View. */
	boolean expanded;
	/** The object held in the Tree. 
	 * Note that this is public, so you can change the Object without
	 * having to change the Tree structure.
	 */
	public Object o;

	/** Construct a Tree, with its attached Object.
	 * @param o the Object to attach.
	 */
	public Tree(Object o) {
		this.o = o;
	}

	/** Are there any children? */
	public boolean hasChildren() {
		return (first != null);
	}

	/** Is this the root? */
	public boolean isRoot() {
		return (parent == null);
	}

	/** Is this the last child of its parent?
	 * This is useful when drawing a representation of the Tree.
	 */
	public boolean isLast() {
		return (parent == null || this == parent.last);
	}

	/** Is this the first child of its parent? */
	public boolean isFirst() {
		return (parent == null || this == parent.first);
	}

	/** Return the parent of this item, or null if root. */
	public Tree getParent() {
		return parent;
	}

	/** Return the first child of this item, or null if there are no children. */
	public Tree getFirst() {
		return first;
	}

	/** Return the last child of this item, or null if there are no children. */
	public Tree getLast() {
		return last;
	}

	/** Return the next sibling of this item, or null if there are no more siblings. */
	public Tree getNext() {
		return next;
	}

	/** Indicates whether children are visible in a View. */
	public boolean isExpanded() {
		return expanded;
	}

	/** Work out the depth of the current item. */
	public int depth() {
		int d = 0;
		Tree p;

		for(Tree a = this; (p = a.parent) != null; a = p)
			d++;
		return d;
	}

	/** Get the root of the current item. */
	public Tree getRoot() {
		Tree a, p;

		for(a = this; (p = a.parent) != null; a = p)
			;
		return a;
	}

	/** Add a child. */
	public void addChild(Tree child) {
		child.parent = this;
		child.next = null;
		if(last != null) {
			last.next = child;
			last = child;
		} else {
			first = last = child;
		}
	}

	/** Delete this item (and all its children) from the Tree.
	 * @return true only if this is the root.
	 */
	public boolean deleteChild() {
		if(parent == null)
			return true;

		Tree prev = null, child = parent.first;

		while(child != null) {
			if(child == this) {
				if(prev == null)
					parent.first = next;
				else
					prev.next = next;
				break;
			}
			prev = child;
			child = child.next;
		}
		if(child != null && this == parent.last)
			parent.last = prev;
		return false;
	}

	/** Run the TreeProcessor on every item in the Tree.
	 * The TreeProcessor continueProcess method is called for each item
	 * in the tree, in a depth-first search. If the continueProcess
	 * method returns false for any item, the search is stopped.
	 * @param p the TreeProcessor to process the items.
	 * @return false if any continueProcess returned false, true otherwise.
	 */
	public boolean processTree(TreeProcessor p) {
		if(!p.continueProcess(this))
			return false;
		if(first != null) {
			if(!first.processTree(p))
				return false;
			for(Tree a = first.next; a != null; a = a.next) {
				if(!a.processTree(p))
					return false;
			}
		}
		return true;
	}

	/** Standard toString method forwards to the attached Object. */
	public String toString() {
		return o.toString();
	}
}


