package com.trumphurst.controls;

import com.trumphurst.utils.*;
import java.util.*;
import java.awt.*;
import java.awt.event.*;

/** A TRowDocument which contains a list of Tree structures.
 * Note that this is a <U>list</U> of tree structures, not a single
 * tree with one root. It was developed to show message threads in
 * a Usenet News reader.<P>
 * When an item is double-clicked, the item will be expanded to show
 * its children (or contracted, if it is already expanded).<P>
 * Perhaps, in a <I>pure</I> Document/View architecture, the expand/
 * contract functionality would be entirely contained in the view.
 * However, this way of doing things works fine, and enables lists of
 * trees to be displayed by the MultiColumnList without a single change
 * to the code.
 * @see Tree
 * @see TMultiColumnList
 */
public class TTreeListDocument extends TRowDocument implements KeyListener {
	static RCSVersion version = new RCSVersion("$Id: TTreeListDocument.java 1.4 1997/09/19 17:09:42 nikki Exp nikki $");

	public TTreeListDocument(String name) {
		super(name);
	}

	/**
	 * Add an object to the list at the specified row.
	 * Overrides the base class, in order to check the item
	 * being added is a Tree.
	 * @exception ClassCastException
	 *	If the item being added is not an instance of Tree.
	 */
	public void addElement(Object item, int index) {
		Tree t = (Tree)item;
		super.addElement(item, index);
	}

	/**
	 * Expand the given row.
	 * @param row The row to expand.
	 * @param children True if the children are also to be expanded recursively.
	 * @return The row number of tha last row added to the list.
	 */
	synchronized public int expand(int row, boolean children) {
		Tree t = (Tree)elementAt(row);

		if(!t.expanded) {
			t.expanded = true;
			for(Tree a = t.first; a != null; a = a.next) {
				addElement(a, ++row);
				if(children)
					row = expand(row, true);
			}
		}
		return row;
	}

	/**
	 * Contract the given row.
	 * @param row The row to contract.
	 */
	synchronized public void contract(int row) {
		Tree t = (Tree)elementAt(row);

		if(t.expanded) {
			t.expanded = false;
			for(Tree a = t.first; a != null; a = a.next) {
				contract(row + 1);
				delElement(row + 1);
			}
		}
	}

	/**
	 * The given row has been actioned (double-clicked).
	 * Expands or contracts the row as necessary.
	 * @param row The row actioned.
	 */
	public void action(int row) {
		Tree t = (Tree)elementAt(row);

		if(t.expanded)
			contract(row);
		else
			expand(row, false);
	}

	/** Handle standard list keyboard stuff. */
	public void keyPressed(KeyEvent e) {
		int sel = getSelectedIndex();
		Tree t = (Tree)getSelectedObject();

		if(t == null)
			return;
		switch(e.getKeyCode()) {
		case KeyEvent.VK_LEFT:
			if(t.expanded)
				contract(sel);
			else if(t.parent != null)
				sel = parentIndex(sel);
			else
				return;
			break;
		case KeyEvent.VK_RIGHT:
			if(!t.expanded)
				expand(sel, false);
			else if(t.first != null)
				sel++;
			else
				return;
			break;
		default:
			switch(e.getKeyChar()) {
			case '+':
				if(!t.expanded)
					expand(sel, false);
				else
					return;
				break;
			case '-':
				if(t.expanded)
					contract(sel);
				else
					return;
				break;
			case '*':
				expand(sel, true);
				break;
			default:
				return;
			}
		}
		select(sel);
		e.consume();
	}
	public void keyReleased(KeyEvent e) {
	}

	public void keyTyped(KeyEvent e) {
	}


	synchronized public int parentIndex(int row) {
		Tree t = ((Tree)elementAt(row)).getParent();
		if(t != null && row > 0)
			while(row-- > 0 && t != elementAt(row))
				;
		return row;
	}

	synchronized public int rootIndex(int row) {
		Tree t = (Tree)elementAt(row);
		if(t != null && row > 0) {
			Tree r = t.getRoot();
			if(r != t)
				while(row-- > 0 && r != elementAt(row))
					;
		}
		return row;
	}
}
