package com.trumphurst.controls;

import com.trumphurst.utils.*;
import java.util.*;
import java.awt.*;
import java.awt.event.*;

/** A document which contains a list of objects, each representing a row,
 * such as would be displayed in a scrolling list. 
 * Allows one object in the list to be "selected".
 * @see TMultiColumnList
 * @see TTreeListDocument
 */
public class TRowDocument extends TDocument implements ItemSelectable {
	static RCSVersion version = new RCSVersion("$Id: TRowDocument.java 1.6 1997/09/19 17:09:38 nikki Exp nikki $");
	/** Event Multicaster */
	protected ItemListener itemListener;
	/** Event Multicaster */
	protected ActionListener actionListener;

	/** The list of objects. */
	protected Vector list = new Vector();
	/** The number of the currently selected row. */
	private int sel = -1;

	public TRowDocument(String name) {
		super(name);
	}

    /**
     * Returns the number of rows in the list.
     */
    public int size() {
		return list.size();
    }

	/**
	 * Add an object to the list at the specified row.
	 * @param item The object to add.
	 * @param index Where to add the object (-1 = at end).
	 */
	synchronized public void addElement(Object item, int index) {
		if (index < -1 || index >= list.size())
			index = -1;
		if (index == -1)
			list.addElement(item);
		else
			list.insertElementAt(item, index);
		updateAllViews();
	}

	/**
	 * Add an object to the list at the end.
	 * @param item The object to add.
	 */
	public void addElement(Object item) {
		addElement(item, -1);
	}

	/**
	 * Returns the object at a given row.
	 * @param index The row to get.
	 * @return The object at that row, or null if index out of range
	 */
	synchronized public Object elementAt(int index) {
		if(index < 0 || index >= list.size())
			return null;

		return list.elementAt(index);
	}

	/**
	 * Returns the row of a given object, or -1 if not found.
	 * @param o The object to find.
	 * @return The row number, or -1 if object not in document
	 */
	synchronized public int indexOf(Object o) {
		return list.indexOf(o);
	}

	/**
	 * Remove one object from the list.
	 * @param index The row number to remove.
	 */
	public void delElement(int index) {
		delElements(index, index);
	}

	/**
	 * Remove multiple objects from the list.
	 * @param start The first row to remove.
	 * @param end Tthe last row to remove.
	 */
	synchronized public void delElements(int start, int end) {
		if(start < 0)
			start = 0;
		if(end >= list.size())
			end = list.size() - 1;
		if(end < start)
			return;
		if(sel >= start) {
			sel -= end - start + 1;
			if(sel < 0)
				sel = -1;
		}
		for (int i = end; i >= start; i--) {
			list.removeElementAt(i);
		}
		updateAllViews();
	}

	/**
	 * Remove all items from the list.
	 */
	public void clear() {
		delElements(0, list.size() - 1);
	}

	/**
	 * Find which row is selected.
	 * @return The selected row, or -1 if none selected.
	 */
	public int getSelectedIndex() {
		return sel;
	}

	/**
	 * Find which object is selected.
	 * @return The object at the selected row, or null if none selected.
	 */
	public Object getSelectedObject() {
		if(sel < 0)
			return null;
		return elementAt(sel);
	}

	/**
	 * Select the given row.
	 * @param row The row to select. Must be in range, or -1 to deselect.
	 */
	synchronized public void select(int row) {
		if(row < -1 || row >= list.size())
			return;
		Object oldsel = null, newsel = null;

		if(sel >= 0 && sel != row) {
			oldsel = getSelectedObject();
			if(itemListener != null)
				itemListener.itemStateChanged(new ItemEvent(this, ItemEvent.DESELECTED, oldsel, ItemEvent.DESELECTED));
		}
		sel = row;
		if(sel >= 0) {
			newsel = getSelectedObject();

			if(itemListener != null)
				itemListener.itemStateChanged(new ItemEvent(this, ItemEvent.SELECTED, newsel, ItemEvent.SELECTED));
		}

		if(oldsel != null)
			updateAllViews(oldsel);
		if(newsel != null)
			updateAllViews(newsel);
	}

	/**
	 * The given row has been actioned (double-clicked or &lt;cr&gt; has been pressed).
	 * Normally overridden in subclasses.
	 * @param row The row actioned.
	 */
	public void action(int row) {
		if(actionListener != null)
			actionListener.actionPerformed(new ActionEvent(this, ActionEvent.ACTION_PERFORMED, getName()));
	}

	/** ItemSelectable - return selected Objects (there is only one). */
	public Object[] getSelectedObjects() {
		Object o = getSelectedObject();

		if(o != null) {
			Object [] r = new Object[1];
			r[0] = o;
			return r;
		}
		return null;
	}

	/** ItemSelectable - add a Listener for when selected item changes. */
	public void addItemListener(ItemListener l) {
		itemListener = AWTEventMulticaster.add(itemListener, l);
	}

	/** ItemSelectable - remove a Listener. */
	public void removeItemListener(ItemListener l) {
		itemListener = AWTEventMulticaster.remove(itemListener, l);
	}

	/** Add a Listener for when selected item is double-clicked. */
	public void addActionListener(ActionListener l) {
		actionListener = AWTEventMulticaster.add(actionListener, l);
	}

	/** Remove a Listener. */
	public void removeActionListener(ActionListener l) {
		actionListener = AWTEventMulticaster.remove(actionListener, l);
	}

}
