package com.trumphurst.controls;

import com.trumphurst.utils.*;
import java.awt.*;

/**
 * Prompt layout is used to layout input components and their prompt labels in a panel.
 * It will Labels in a left column, and other components in a right column, lining
 * the prompts up with their corresponding input components.
 * The constructor has full control over alignment and packing within the columns.
 */
public class PromptLayout extends TFlowLayout {
	static RCSVersion version = new RCSVersion("$Id: PromptLayout.java 1.3 1997/09/19 17:09:45 nikki Exp nikki $");
    /**
     * Constructs a new VFlow Layout with a LEFT, SPREAD alignment and a
     * default 5-unit horizontal and vertical gap.
     */
    public PromptLayout() {
		super(SPREAD, 5, 5);
    }

    /**
     * Constructs a new VFlow Layout with the specified alignment and a
     * default 5-unit horizontal and vertical gap.
     * @param align the alignment value
     */
    public PromptLayout(int align) {
		super(align, 5, 5);
    }

    /**
     * Constructs a new VFlow Layout with the specified alignment and gap
     * values.
     * @param align the alignment value
     * @param hgap the horizontal gap variable
     * @param vgap the vertical gap variable
     */
    public PromptLayout(int align, int hgap, int vgap) {
		super(align, hgap, vgap);
    }

    /**
     * Returns the preferred dimensions for this layout given the components
     * in the specified target container.
     * @param target the component which needs to be laid out
     * @see Container
     * @see #minimumLayoutSize
     */
    public Dimension preferredLayoutSize(Container target) {
		Dimension labels = new Dimension(0, 0);
		Dimension fields = new Dimension(0, 0);
		int nmembers = target.getComponentCount();

		for (int i = 0 ; i < nmembers ; i++) {
			Component m = target.getComponent(i);
			if (m.isVisible()) {
				Dimension d = m.getPreferredSize();

				if(m instanceof Label) {
					// Make sure the next spot for a field is level with top
					fields.height = Math.max(fields.height, labels.height);
					// Add it to the labels
					labels.width = Math.max(labels.width, d.width);
					if (i > 0) {
						labels.height += vgap;
					}
					labels.height += d.height;
				} else {
					// Add it to the fields
					fields.width = Math.max(fields.width, d.width);
					if (i > 0) {
						fields.height += vgap;
					}
					fields.height += d.height;
					// Make sure the next spot for a label is underneath
					labels.height = Math.max(labels.height, fields.height);
				}
			}
		}
		Insets insets = target.getInsets();
		labels.width += fields.width + insets.left + insets.right + hgap * 3;
		labels.height = Math.max(labels.height, fields.height) + insets.top + insets.bottom + vgap * 2;
		return labels;
    }

    /**
     * Returns the minimum dimensions needed to layout the components
     * contained in the specified target container.
     * @param target the component which needs to be laid out 
     * @see #preferredLayoutSize
     */
    public Dimension minimumLayoutSize(Container target) {
		Dimension labels = new Dimension(0, 0);
		int nmembers = target.getComponentCount();
		Dimension fields = new Dimension(0, 0);

		for (int i = 0 ; i < nmembers ; i++) {
			Component m = target.getComponent(i);
			if (m.isVisible()) {
				Dimension d = m.getMinimumSize();
				if(m instanceof Label) {
					// Make sure the next spot for a field is level with top
					fields.height = Math.max(fields.height, labels.height);
					// Add it to the labels
					labels.width = Math.max(labels.width, d.width);
					if (i > 0) {
						labels.height += vgap;
					}
					labels.height += d.height;
				} else {
					// Add it to the fields
					fields.width = Math.max(fields.width, d.width);
					if (i > 0) {
						fields.height += vgap;
					}
					fields.height += d.height;
					// Make sure the next spot for a label is underneath
					labels.height = Math.max(labels.height, fields.height);
				}
			}
		}
		Insets insets = target.getInsets();
		labels.width += fields.width + insets.left + insets.right + hgap * 3;
		labels.height = Math.max(labels.height, fields.height) + insets.top + insets.bottom + vgap * 2;
		return labels;
    }

    /** 
     * Centers the elements in the specified column, if there is any slack.
     * @param target the component which needs to be moved
     * @param x the x coordinate of the labels column
     * @param y the y coordinate
     * @param widthlabels the width of the labels column
     * @param widthfields the width of the fields column
     * @param height the spare height left over
     * @param colStart the beginning of the column
     * @param colEnd the the ending of the column
     */
    private void moveComponents(Container target, int x, int y, int widthlabels, int widthfields, int height, int colStart, int colEnd) {
		int extra = vgap;
		int yfield = y;

		switch (align & SPREAD) {
		case TOP:
			break;
		case VCENTER:
			y += height / 2;
			break;
		case BOTTOM:
			y += height;
			break;
		case SPREAD:
			extra += height / (colEnd - colStart + 1);
			break;
		}
		for (int i = colStart ; i < colEnd ; i++) {
			Component m = target.getComponent(i);
			if (m.isVisible()) {
				Dimension d = m.getSize();
				int nx = x;

				if(m instanceof Label) {
					yfield = Math.max(yfield, y);
					switch(align & EXPAND) {
					case LEFT:
						break;
					case HCENTER:
					case RIGHT:
						nx += (widthlabels - d.width);
						break;
					case EXPAND:
						m.setSize(widthlabels, d.height);
						break;
					}
					m.setLocation(nx, y);
					y += extra + d.height;
				} else {
					switch(align & EXPAND) {
					case LEFT:
					case HCENTER:
						break;
					case RIGHT:
						nx += (widthfields - d.width);
						break;
					case EXPAND:
						m.setSize(widthfields, d.height);
						break;
					}
					m.setLocation(nx + widthlabels + hgap, yfield);
					yfield += extra + d.height;
					y = Math.max(y, yfield);
				}
			}
		}
    }

    /**
     * Lays out the container. This method will actually reshape the
     * components in the target in order to satisfy the constraints of
     * the PromptLayout object. 
     * @param target the specified component being laid out.
     * @see Container
     */
    public void layoutContainer(Container target) {
		Insets insets = target.getInsets();
		Dimension t = target.getSize();
		int maxheight = t.height - (insets.top + insets.bottom + vgap*2);
		int nmembers = target.getComponentCount();
		int x = insets.left + hgap, ylabels = 0, yfields = 0;
		int wlabels = 0, wfields = 0, start = 0;
		int previousLabelWidth = -1;

		for (int i = 0 ; i < nmembers ; i++) {
			Component m = target.getComponent(i);
			if (m.isVisible()) {
				Dimension d = m.getPreferredSize();
				m.setSize(d.width, d.height);

				if(m instanceof Label) {
					previousLabelWidth = wlabels;
					yfields = Math.max(yfields, ylabels);
					if ((ylabels == 0) || ((ylabels + d.height) <= maxheight)) {
						if (ylabels > 0) {
							ylabels += vgap;
						}
						ylabels += d.height;
						wlabels = Math.max(wlabels, d.width);
					} else {
						moveComponents(target, x, insets.top + vgap, wlabels, wfields, maxheight - yfields, start, i);
						ylabels = d.height;
						yfields = 0;
						x += 2 * hgap + wlabels + wfields;
						wlabels = d.width;
						wfields = 0;
						start = i;
					}
				} else {
					if ((yfields == 0) || ((yfields + d.height) <= maxheight)) {
						if (yfields > 0) {
							yfields += vgap;
						}
						yfields += d.height;
						wfields = Math.max(wfields, d.width);
					} else {
						if(previousLabelWidth >= 0) {
							// Last thing was a Label, but we are starting a new column
							// so restore things back to how they were before the label
							// and then process it again in the next column
							wlabels = previousLabelWidth;
							i -= 2;
						}
						moveComponents(target, x, insets.top + vgap, wlabels, wfields, maxheight - yfields, start, i);
						ylabels = yfields = d.height;
						x += 2 * hgap + wlabels + wfields;
						wlabels = 0;
						wfields = d.width;
						start = i;
					}
					ylabels = Math.max(ylabels, yfields);
					previousLabelWidth = -1;
				}
			}
		}
		moveComponents(target, x, insets.top + vgap, wlabels, wfields, maxheight - ylabels, start, nmembers);
    }
}
