package com.trumphurst.controls;

import com.trumphurst.utils.*;
import java.awt.*;

/**
 * A simple progress bar to indicate how long a lengthy task has run.
 */
public class ProgressBar extends Canvas {
	static RCSVersion version = new RCSVersion("$Id: ProgressBar.java 1.5 1997/09/19 17:09:39 nikki Exp nikki $");
	//****************************
	// Constructors
	//****************************
	/**
	 * Constructor.
	 * @param min the minimum value expected.
	 * @param max the maximum value expected.
	 */
	public ProgressBar(int min, int max) {
		this.max = max;
		setMinimum(min);
	}

	/**
	 * Constructor (min = 0).
	 * @param max the maximum value expected.
	 */
	public ProgressBar(int max) {
		this(0, max);
	}

	//****************************
	// JavaBean Property getters/setters
	//****************************

	/**
	 * Move the bar position.
	 * @param current (between min and max).
	 */
	public void setCurrent(int current) {
		if(current > max)
			current = max;
		if(current < min)
			current = min;
		this.current = current;
		repaint();
	}

	/** Return the current bar position. */
	public int getCurrent() {
		return current;
	}

	/**
	 * Reset the minimum value.
	 * @param min the minimum value.
	 */
	public void setMinimum(int min) {
		if(min >= max)
			max = min + 1;
		this.min = min;
		this.current = min;
		repaint();
	}

	/** Return the minimum value. */
	public int getMinimum() {
		return min;
	}

	/**
	 * Reset the maximum values (min = 0).
	 * @param max the maximum value.
	 */
	public void setMaximum(int max) {
		if(min >= max)
			min = max - 1;
		this.max = max;
		this.current = min;
		repaint();
	}

	/** Return the maximum value. */
	public int getMaximum() {
		return max;
	}

	//****************************
	// Overridden public methods
	//****************************

	/**
	 * Standard update routine. 
	 * We optimise to prevent background being cleared.
	 */
	public void update(Graphics g) {
		paint(g);
	}

	/**
	 * Standard paint routine.
	 */
	public void paint(Graphics g) {
		Rectangle r = getBounds();
		int divide = r.width * (current - min) / (max - min);
		g.setColor(Color.blue);
		g.fillRect(0, 0, divide, r.height);
		g.setColor(getBackground());
		g.fillRect(divide, 0, r.width - divide, r.height);
	}

	/** 
	 * Standard preferred size routine.
	 * Actually returns the current size, or 100 x 10 if there isn't one.
	 */
	public Dimension getPreferredSize() {
		Dimension d = getSize();
		if(d.width == 0)
			d.width = 100;
		if(d.height == 0)
			d.height = 12;
		return d;
	}

	//****************************
	// Variables
	//****************************
	/** The minimum expected value. */
	int min;
	/** The maximum expected value. */
	int max;
	/** The current value. */
	int current;
}
