package com.trumphurst.controls;

import trumphurst.utils.*;
import java.awt.*;
import java.awt.event.*;

/**
 * Multi column list or tree control with titles for each column and draggable widths.
 * This is the user interface component (TView) only - the actual data is held in a TRowDocument
 * or TTreeListDocument. The rendering of the data in each column is done by an ObjectPainter
 * supplied to the constructor.
 * @see TRowDocument
 * @see TTreeListDocument
 * @see ObjectPainter
 */
public class MultiColumnList extends Panel implements TView, MouseListener, 
		MouseMotionListener, KeyListener, FocusListener, AdjustmentListener,
		ComponentListener {
	static RCSVersion version = new RCSVersion("$Id: MultiColumnList.java 1.7 1997/09/19 17:09:39 nikki Exp nikki $");
	//****************************
	// Constructors
	//****************************

	/**
	 * Create a new list with the given columns, and attach the given data.
	 * @param t the array of ObjectPainters which will paint the columns.
	 * @param doc the TDocument containing the data.
	 */
	public MultiColumnList(ObjectPainter t[], TRowDocument doc) {
		if(doc != null)
			doc.addView(this);
		painter = new ObjectPainter[t.length];
		System.arraycopy(t, 0, painter, 0, t.length);
		cwidth = new float[t.length];
		// Space everything evenly by default
		for(int i = 0; i < t.length; i++)
			cwidth[i] = 1.0f / t.length;
		cpos = new int[t.length + 1];
		setLayout(null);
		sb = new Scrollbar(Scrollbar.VERTICAL);
		add(sb);
		addMouseListener(this);
		addMouseMotionListener(this);
		addKeyListener(this);
		addFocusListener(this);
		addComponentListener(this);
		sb.addAdjustmentListener(this);
	}

	/**
	 * Create a new list with the given columns.
	 * @param t the array of ObjectPainters which will paint the columns.
	 */
	public MultiColumnList(ObjectPainter t[]) {
		this(t, null);
	}

	//****************************
	// JavaBean Property getters/setters
	//****************************

	/**
	 * Set the proportional widths of each column.
	 * @param widths the widths.
	 * @exception ArrayIndexOutOfBoundsException if not enough widths are supplied.
	 */
	public void setWidths(float widths[]) {
		float max = 0;
		int i;

		// Copy provided array into class variable
		// This will give the documented exception if the parameter is too small
		System.arraycopy(widths, 0, cwidth, 0, painter.length);
		// Calculate total of all passed sizes
		for(i = 0; i < painter.length; i++)
			max += cwidth[i];
		if(max == 0)		// Silly value - space evenly instead
			for(i = 0; i < painter.length; i++)
				cwidth[i] = 1.0f / painter.length;
		else if(max != 1)	// Normalise so they add up to 1
			for(i = 0; i < painter.length; i++)
				cwidth[i] /= max;
		// Resize all the columns accordingly
		respace();
		repaint();
	}

	/**
	 * Get the proportional widths of each column.
	 * @return a float array containing the proportions (adds up to 1).
	 */
	public float[] getWidths() {
		float [] rwidths = new float[cwidth.length];

		System.arraycopy(cwidth, 0, rwidths, 0, cwidth.length);
		return rwidths;
	}

	/** Adjustable - true if use is allowed to drag borders to change column sizes. */
	public boolean isAdjustable() {
		return adjustable;
	}

	/** Adjustable - true if use is allowed to drag borders to change column sizes. */
	public void setAdjustable(boolean on) {
		adjustable = on;
	}

	/** MinimumWidth - minimum width of a column. */
	public int getMinimumWidth() {
		return minwidth;
	}

	/** MinimumWidth - minimum width of a column. Default 6. */
	public void setMinimumWidth(int width) {
		minwidth = width;
	}

	/** DragSensitivity - no of pixels either side of border where cursor 
	 * changes to allow dragging border. If this is greater than MinimumWidth,
	 * it can get difficult to drag very small columns.
	 */
	public int getDragSensitivity() {
		return dragSensitivity;
	}

	/** DragSensitivity - no of pixels either side of border where cursor 
	 * changes to allow dragging border. If this is greater than MinimumWidth,
	 * it can get difficult to drag very small columns. Default 3.
	 */
	public void setDragSensitivity(int s) {
		dragSensitivity = s;
	}

	/** DoubleClickTime - microseconds between 2 mouse clicks for them to 
	 * be counted as a double-click.
	 */
	public int getDoubleClickTime() {
		return doubleClickTime;
	}

	/** DoubleClickTime - microseconds between 2 mouse clicks for them to 
	 * be counted as a double-click. Default 250.
	 */
	public void setDoubleClickTime(int t) {
		doubleClickTime = t;
	}

	/** Colors - the arry of colors (one per column) used to draw each column. 
	 * May be null to default to foregroundColor.
	 */
	public Color[] getColors() {
		if(colors == null)
			return null;
		Color [] c = new Color[cwidth.length];

		System.arraycopy(colors, 0, c, 0, cwidth.length);
		return c;
	}

	/**
	 * Set the array of colors used to draw each column.
	 * @param c	The color array (in column order), or null to
	 * use the default foregroundColor.
	 */
	public void setColors(Color c[]) {
		if(c == null)
			colors = null;
		else {
			if(colors == null)
				colors = new Color[cwidth.length];
			System.arraycopy(c, 0, colors, 0, cwidth.length);
		}
		repaint();
	}

	/** HeadingColor - Color for the heading bar. */
	public Color getHeadingColor() {
		return headingColor;
	}

	/** HeadingColor - Color for the heading bar. Defaults to background. */
	public void setHeadingColor(Color c) {
		headingColor = c;
	}

	/** BarColor - Color for the highlight bar. */
	public Color getBarColors() {
		return barColor;
	}

	/** BarColor - Color for the highlight bar. Default black. */
	public void setBarColors(Color c) {
		barColor = c;
	}

	//****************************
	// Other public methods
	//****************************

    /**
     * Returns the number of columns in the list.
     */
    public int countColumns() {
		return painter.length;
    }

	/** Select a given item in the list.
	 * @param sel the item number to select. Will be adjusted if out of range.
	 */
	public void select(int sel) {
		if(document == null)
			return;
		if(sel < 0)
			sel = 0;
		else if(sel >= document.size())
			sel = document.size() - 1;
		if(sel != document.getSelectedIndex()) {
			document.select(sel);
			compscroll();
			repaint();
		}
	}
	
	/** Make sure a given item in the list is visible on the screen.
	 * @param sel the item number to make visible. Will be adjusted if out of range.
	 */
	public void makeItemVisible(int sel) {
		int oldtop = top;

		if(document == null)
			return;
		if(sel < 0)
			sel = 0;
		else if(sel >= document.size())
			sel = document.size() - 1;
		// Ensure selection visible
		if(sel < top)
			top = sel;
		else if(sel - top >= rows())
			top = sel - rows() + 1;
		if(oldtop != top) {
			compscroll();
			repaint();
		}
	}
	
	/** Called when user double-clicks on item in list, or presses &lt;cr&gt;. */
	public void doubleClicked() {
		if(document != null && bim != null)
			document.action(document.getSelectedIndex());
	}

	/** The Document has changed - update the whole view. */
	public void update() {
		if(bim != null) {
			select(document.getSelectedIndex());
			compscroll();
			repaint();
		}
	}

	/** 
	 * Part of the Document has changed. 
	 * @param o Object representing that part of the document
	 * which has changed.
	 */
	public void update(Object o) {
		if(bim != null)			// Test if o == null too
			update();			// Just update everything for now
	}

	/**
	 * Attach the View to the Document.
	 * @param d the Document.
	 * @exception ClassCastException if the document isn't a TRowDocument.
	 */
	synchronized public void attach(TDocument d) {
		if(document != d) {
			if(document != null)
				document.removeView(this);
			document = (TRowDocument)d;
			if(d != null)
				d.addView(this);
		}
		update();
	}

	/**
	 * Detach the View from the Document.
	 */
	synchronized public void detach(TDocument d) {
		if(document == d) {
			document = null;
			d.removeView(this);
		}
	}

	//****************************
	// Overridden public methods
	//****************************

	/** Enable or disable the control. */
	public void setEnable(boolean enabled) {
		sb.setEnabled(enabled);
		repaint();
	}

	/** Blit the backing image to the front. */
	public void paint(Graphics g) {
		super.paint(g);
		// Draw sunken rectangle round whole control
		g.setColor(getBackground());
		g.draw3DRect(1, 1, getSize().width - 3, getSize().height - 3, hasFocus);

		if (bim == null) {
			// This is the first rendering
			bim = createImage(width, height);
			bg = bim.getGraphics();
			bg.setFont(g.getFont());
			fnm = bg.getFontMetrics();
			fontHeight = fnm.getHeight();
			th = fontHeight + 4;	// Title height
			compscroll();
			if(debugmode)
				bg = g;		// If debugging, draw straight on screen
			render();
		}
		if(!debugmode)
			g.drawImage(bim, in.left + R3Dwidth, in.top + R3Dheight, this);
	}

	/** Called sometime after repaint(). */
	public void update(Graphics g) {
		if (bim != null) {
			if(debugmode)
				bg = g;		// If debugging, draw straight on screen
			render();
		}
		paint(g);
	}

	/** This Panel is a tabstop (can accept focus). */
	public boolean isFocusTraversable() {
		return true;
	}

	/** Standard toString returns "DragGridPanel[x][y]" and Panel info. */
	public String toString() {
		Point p = getLocation();
		return "MultiColumnList(" + p.x + "," + p.y + ")[" + cwidth.length + "]" + super.toString();
	}

	//****************************
	// Event handling
	//****************************

	/** Change the mouse cursor if over a draggable column border. */
	public void mouseMoved(MouseEvent e) {
		debug("mouseMoved", e);
		if(bim != null) {
			if(coldrag > 0					// Already dragging
				|| (isEnabled()					// Or enabled
					&& adjustable				// and allowed to drag
					&& e.getY() - in.top < th	// and in title area
					&& dragColumn(e.getX() - in.top) > 0)) {	// and near dividing line
				setCursor(true);			// Set cursor to dragging arrow
			} else
				setCursor(false);			// Set cursor back to normal
		}
	}

	/** Select a list item or a column to drag. */
	public void mousePressed(MouseEvent e) {
		debug("mousePressed", e);
		if(isEnabled() && bim != null) {
			if(e.getY() - in.top < th) {
			// Press in title bar
				if(adjustable)
					coldrag = dragColumn(e.getX() - in.left);// See if cursor near dividing line
			} else {
			// Item chosen from list
				int row = (e.getY() - in.top - th) / fontHeight + top;

				if (row < document.size()) {
					document.select(row);
					repaint();
				}
			}
		}
	}
	/** Stop dragging a list column. */
	public void mouseReleased(MouseEvent e) {
		debug("mouseReleased", e);
		if(coldrag > 0) {
			coldrag = 0;
			mouseMoved(e);	// This will sort out the cursor shape and everything
		}
	}

	public void mouseClicked(MouseEvent e) {
		debug("mouseClicked", e);
		if(isEnabled() && bim != null && document != null) {
			// Item chosen from list
			int row = (e.getY() - in.top - th) / fontHeight + top;

			if (row < document.size()) {
				document.select(row);
				makeItemVisible(document.getSelectedIndex());
				repaint();
				// Double-click? Surely there should be a better way of detecting one?
				if (row == document.getSelectedIndex() &&
					((e.getClickCount() & 1) != 0 || e.getWhen() - last < doubleClickTime)) {
					last = 0;
					doubleClicked();
				} else
					last = e.getWhen();
			}
		}
	}

	public void mouseEntered(MouseEvent e) {
		debug("mouseEntered", e);
	}

	public void mouseExited(MouseEvent e) {
		debug("mouseExited", e);
	}
	
	/** If a column is selected, change it's width. */
	public void mouseDragged(MouseEvent e) {
		debug("mouseDragged", e);
		if (isEnabled() && coldrag > 0 && bim != null) {
			int x = e.getX() - in.left;		// Compensate for insets
			// Make sure columns either side are at least minwidth wide
			if(x < cpos[coldrag - 1] + minwidth)
				x = cpos[coldrag - 1] + minwidth;
			else if(x > cpos[coldrag + 1] - minwidth)
				x = cpos[coldrag + 1] - minwidth;
			if(cpos[coldrag] != x) {
				// Set border
				cpos[coldrag] = x;
				// Adjust proportional sizes
				cwidth[coldrag - 1] = (cpos[coldrag] - cpos[coldrag - 1]) /
							(float)width;
				cwidth[coldrag] = (cpos[coldrag + 1] - cpos[coldrag]) /
							(float)width;
				repaint();
			}
		}
	}
	/** Change selection bar color if we get the focus. */
	public void focusGained(FocusEvent e) {
		debug("focusGained", e);
		hasFocus = true;
		if(document != null)
			update(document.getSelectedObject());
	}

	/** Change selection bar color if we lose the focus. */
	public void focusLost(FocusEvent e) {
		debug("focusLost", e);
		hasFocus = false;
		if(document != null)
			update(document.getSelectedObject());
	}

	/** Handle standard list keyboard stuff. */
	public void keyPressed(KeyEvent e) {
		debug("keyPressed", e);
		if(!isEnabled() || document == null || document.size() == 0 || bim == null)
			return;

		// Pass key to document first, to see if it wants it.
		if(document instanceof KeyListener) {
			KeyListener kl = (KeyListener)document;
			kl.keyPressed(e);
			if(e.isConsumed())
				return;
		}

		int sel = document.getSelectedIndex();

		switch(e.getKeyCode()) {
		case KeyEvent.VK_DOWN:
			sel++;
			break;
		case KeyEvent.VK_UP:
			sel--;
			break;
		case KeyEvent.VK_HOME:
			sel = 0;
			break;
		case KeyEvent.VK_END:
			sel = document.size() - 1;
			break;
		case KeyEvent.VK_PAGE_DOWN:
			sel += rows();
			break;
		case KeyEvent.VK_PAGE_UP:
			sel -= rows();
			break;
		case KeyEvent.VK_ENTER:
			if(sel >= 0)
				doubleClicked();
		default:
			return;
		}
		select(sel);		// Select will adjust if out of range
		makeItemVisible(document.getSelectedIndex());
		e.consume();
	}
	public void keyReleased(KeyEvent e) {
	}

	public void keyTyped(KeyEvent e) {
	}

	/** Deal with scrollbar movement. */
	public void adjustmentValueChanged(AdjustmentEvent e) {
		debug("adjustmentValueChanged", e);
		if (e.getAdjustable() == sb && bim != null) {
			// scrollbar moved
			// Might be better to just use e.getValue, and not check where it came from?
			top = sb.getValue();
			compscroll();
			repaint();
			}
	}

	/** Called when this container gets resized. */
	public void componentResized(ComponentEvent e) {
		debug("componentResized", e);
		Dimension n = getSize();

		in = getInsets();
		sbwidth = sb.getMinimumSize().width;	// Scroll bar width
		width = n.width - sbwidth - (in.left + in.right) - 2 * R3Dwidth;// List width
		height = n.height - (in.top + in.bottom) - 2 * R3Dheight;		// List height
		// Resize scroll bar
		sb.setBounds(width + in.left + R3Dwidth, in.top + R3Dheight, sbwidth, height);
		// Resize columns in proportion
		respace();
		// Force creation of a new backing image and re-painting
		bim = null;
		repaint();
		compscroll();
	}

	public void componentMoved(ComponentEvent e) {
	}

	public void componentShown(ComponentEvent e) {
	}

	public void componentHidden(ComponentEvent e) {
	}

	//****************************
	// Package and private methods
	//****************************

	/** Compute pixel column widths from proportional widths. */
	void respace() {
		// Start at left column
		cpos[0] = 0;
		for(int i = 0; i < painter.length; i++)
			cpos[i + 1] = cpos[i] + (int)(width * cwidth[i]);
		// Fudge last column in case of rounding errors
		cpos[painter.length] = width;
	}

	/** Re-draw the list into the backing image. */
	synchronized void render() {
		if(bim == null)
			return;

		int fd = fnm.getDescent(),	// useful font metrics
			fa = fnm.getAscent();
		int bot = 0;				// Bottom visible list entry
		int sel = -1;				// Selected list entry

		if(document != null) {
			sel = document.getSelectedIndex();
			bot = Math.min(top + rows() - 1, document.size() - 1);
		}
		// Clear title section and list
		bg.setColor((headingColor == null) ? getBackground() : headingColor);
		bg.fillRect(0, 0, width, th);
		bg.setColor(getBackground());
		bg.fillRect(0, th, width, height - th);

		// Draw each column
		for(int i = 0; i < painter.length; i++) {
			int x = cpos[i],				// Left of column
				w = cpos[i + 1] - x - 1;	// Width of column
			// User-supplied painter for this column
			ObjectPainter cpainter = painter[i];

			// Column title
			bg.setColor((headingColor == null) ? getBackground() : headingColor);
			bg.draw3DRect(x + 1, 1, w - 1, th - R3Dheight, true);
			// Column dividers
			bg.setColor(getBackground());
			bg.draw3DRect(x + 1, th, w - 1, height - th - 1, true);
			bg.setColor(getForeground());
//			if(i > 0)
//				bg.drawLine(x, th, x, height);

			// Create graphic to clip to column size
			Graphics g = bg.create(x + 1, 0, w - 2, height);
			// And a clip rectangle for the column heading
			Rectangle r = new Rectangle(1, 1, w - R3Dwidth, th - R3Dheight);

			// Paint the title by calling user-supplied column painter
			cpainter.paintTitle(g, r, false);

			// Reuse clip rectangle for the list entries
			r.y = th;				// Top row position
			r.height = fontHeight;

			if (!isEnabled())
				g.setColor(Color.lightGray);
			else if (colors != null)
				g.setColor(colors[i]);
			else
				g.setColor(getForeground());
			if(document != null) {
				synchronized(document) {
					bot = Math.min(top + rows() - 1, document.size() - 1);
					// Paint all the visible rows
					for(int j = top; j <= bot; j++, r.y += fontHeight) {
						Object o = document.elementAt(j);

						cpainter.paintObject(g, r, o, false);
					}
				}
			}
			g.dispose();		// Finished with Graphics
		}

		if (isEnabled()) {
			// Mark the selected row
			if (sel >= top && sel <= bot) {
				bg.setColor(getBackground());
				bg.setXORMode(barColor);
				bg.fillRect(0, 0 + th + (sel - top) * fontHeight, width, fontHeight);
				bg.setPaintMode();
			}
		}
	}

	/** Re-compute the size of the scrollbar. */
	private void compscroll() {
		if (bim == null)
			return;		// not visible

		int visible = 1;		// Number of rows visible
		int total = 1;			// Total number of rows

		if(document != null) {
			visible = rows();
			total = document.size() - visible;
		}
		if(total < 0)
			total = 0;
		if(top > total)
			top = total;
		if(top < 0)
			top = 0;
		if(visible <= 0)
			visible = 1;
		// Not too sure if these values are right
		// the scroll bar won't reach the bottom!
		sb.setValues(top, visible, 0, total);
	}

	/** 
	 * Calculate the number of rows visible in the list.
	 * @return the number of rows.
	 */
	private int rows() {
		return Math.min((height - th) / fontHeight, document.size());
	}

	/**
	 * Work out if the mouse cursor is over a draggable border.
	 * You should already have tested adjustable & enabled.
	 * @param x mouse position (you should test in y < th before calling).
	 * @return 1-based column number of border, or 0.
	 */
	private int dragColumn(int x) {
		for(int i = 1; i < painter.length; i++) {
			if(x < cpos[i] - dragSensitivity)
				break;
			if(x <= cpos[i] + dragSensitivity)
				return i;
		}
		return 0;
	}

	/**
	 * Switch cursor to indicate user can drag column widths.
	 * @param drag true to change cursor to drag cursor, false to change it back.
	 */
	private void setCursor(boolean drag) {
		if(drag) {
			if(originalCursor == null) {	// Save original cursor shape
				originalCursor = getCursor();
				setCursor(Cursor.getPredefinedCursor(Cursor.E_RESIZE_CURSOR));
			}
		} else {
			if(originalCursor != null) {
				setCursor(originalCursor);
				originalCursor = null;
			}
		}
	}

	private void debug(String s, Object o) {
		if(trace)
			Debug.print(this, s, o);
	}

	private void debug(String s) {
		if(trace)
			Debug.print(this, s, null);
	}

	//****************************
	// Variables
	//****************************

	/** Width of line round 3DRectangle */
	final static int R3Dwidth = 2;
	/** Height of line round 3DRectangle */
	final static int R3Dheight = 2;
	/** true if user can adjust columns. */
	boolean adjustable = true;
	/** Minimum width of a column. */
	int minwidth = 6;
	/** Drag sensitivity - no of pixels either side of border where cursor 
	 * changes to allow dragging border. If this is greater than minwidth,
	 * it can get difficult to drag very small columns.
	 */
	int dragSensitivity = 3;
	/** Double-click time - microseconds between 2 mouse clicks for them to 
	 * be counted as a double-click.
	 */
	int doubleClickTime = 250;
	/** Color for the heading. */
	Color headingColor;
	/** Color for the highlight bar. */
	Color barColor = Color.black;
	/** optional array of colors, one per column. */
	Color colors[] = null;
	/** Set to true to paint everything to the screen directly, instead
	 * of painting in the background and blitting. */
	private static final boolean debugmode = false;
	/** The document containing the data. */
	TRowDocument document;
	/** ObjectPainters for each column. */
	ObjectPainter painter[];
	/** column x positions. */
	int cpos[];
	/** proportional column widths. */
	float cwidth[];
	/** scrollbar at the right side. */
	Scrollbar sb;
	/** width, minus the scrollbar. */
	int width;
	/** height, minus the title bar. */
	int height;
	/** used space around the border. */
	Insets in;
	/** width of the scrollbar. */
	int sbwidth;
	/** height of title bar. */
	int th;
	/** backing image. */
	Image bim;
	/** backing graphics. */
	Graphics bg;
	/** drawing font metrics. */
	FontMetrics fnm;
	/** drawing font height. */
	int fontHeight = 1;
	/** column being resized. */
	int coldrag = 0;
	/** first row displayed. */
	int top = 0;
	/** last mouse click time. */
	long last;
	/** original cursor type. */
	Cursor originalCursor;
	/** True if this list has the focus */
	boolean hasFocus;

	static final boolean trace = false;

}

