package com.trumphurst.controls;

import com.trumphurst.utils.*;
import java.awt.*;

/**
 * HFlow layout is used to layout components in a panel. It will arrange
 * components left to right until no more buttons fit in the same row.
 * The constructor has full control over alignment and packing within the row.
 */
public class HFlowLayout extends TFlowLayout {
	static RCSVersion version = new RCSVersion("$Id: HFlowLayout.java 1.2 1997/09/19 17:09:46 nikki Exp nikki $");
    /**
     * Constructs a new HFlow Layout with a LEFT, SPREAD alignment and a
     * default 5-unit horizontal and vertical gap.
     */
    public HFlowLayout() {
		super(SPREAD, 5, 5);
    }

    /**
     * Constructs a new HFlow Layout with the specified alignment and a
     * default 5-unit horizontal and vertical gap.
     * @param align the alignment value
     */
    public HFlowLayout(int align) {
		super(align, 5, 5);
    }

    /**
     * Constructs a new HFlow Layout with the specified alignment and gap
     * values.
     * @param align the alignment value
     * @param hgap the horizontal gap variable
     * @param vgap the vertical gap variable
     */
    public HFlowLayout(int align, int hgap, int vgap) {
		super(align, hgap, vgap);
    }

    /**
     * Returns the preferred dimensions for this layout given the components
     * in the specified target container.
     * @param target the component which needs to be laid out
     * @see Container
     * @see #minimumLayoutSize
     */
    public Dimension preferredLayoutSize(Container target) {
		Dimension dim = new Dimension(0, 0);
		int nmembers = target.getComponentCount();

		for (int i = 0 ; i < nmembers ; i++) {
			Component m = target.getComponent(i);
			if (m.isVisible()) {
				Dimension d = m.getPreferredSize();
				dim.height = Math.max(dim.height, d.height);
				if (i > 0) {
					dim.width += hgap;
			}
			dim.width += d.width;
			}
		}
		Insets insets = target.getInsets();
		dim.width += insets.left + insets.right + hgap*2;
		dim.height += insets.top + insets.bottom + vgap*2;
		return dim;
    }

    /**
     * Returns the minimum dimensions needed to layout the components
     * contained in the specified target container.
     * @param target the component which needs to be laid out 
     * @see #preferredLayoutSize
     */
    public Dimension minimumLayoutSize(Container target) {
		Dimension dim = new Dimension(0, 0);
		int nmembers = target.getComponentCount();

		for (int i = 0 ; i < nmembers ; i++) {
			Component m = target.getComponent(i);
			if (m.isVisible()) {
				Dimension d = m.getMinimumSize();
				dim.height = Math.max(dim.height, d.height);
				if (i > 0) {
					dim.width += hgap;
			}
			dim.width += d.width;
			}
		}
		Insets insets = target.getInsets();
		dim.width += insets.left + insets.right + hgap*2;
		dim.height += insets.top + insets.bottom + vgap*2;
		return dim;
    }

    /** 
     * Centers the elements in the specified column, if there is any slack.
     * @param target the component which needs to be moved
     * @param x the x coordinate
     * @param y the y coordinate
     * @param width the spare width left over
     * @param height the height of the row
     * @param rowStart the beginning of the row
     * @param rowEnd the the ending of the row
     */
    private void moveComponents(Container target, int x, int y, int width, int height, int rowStart, int rowEnd) {
		int extra = hgap;

		switch (align & SPREAD) {
		case LEFT:
			break;
		case HCENTER:
			x += width / 2;
			break;
		case BOTTOM:
			x += width;
			break;
		case SPREAD:
			int w = width / (rowEnd - rowStart + 1);
			extra += w;
			x += w;
			break;
		}
		for (int i = rowStart ; i < rowEnd ; i++) {
			Component m = target.getComponent(i);
			if (m.isVisible()) {
				Dimension d = m.getSize();
				int ny = y;
				switch(align & EXPAND) {
				case TOP:
					break;
				case VCENTER:
					ny += (height - d.height) / 2;
					break;
				case BOTTOM:
					ny += (height - d.height);
					break;
				case EXPAND:
					m.setSize(d.width, height);
					break;
				}
				m.setLocation(x, ny);
				x += extra + d.width;
			}
		}
    }

    /**
     * Lays out the container. This method will actually reshape the
     * components in the target in order to satisfy the constraints of
     * the HFlowLayout object. 
     * @param target the specified component being laid out.
     * @see Container
     */
    public void layoutContainer(Container target) {
		Insets insets = target.getInsets();
		Dimension t = target.getSize();
		int maxwidth = t.width - (insets.left + insets.right + hgap*2);
		int nmembers = target.getComponentCount();
		int x = 0, y = insets.top + vgap;
		int colh = 0, start = 0;

		for (int i = 0 ; i < nmembers ; i++) {
			Component m = target.getComponent(i);
			if (m.isVisible()) {
				Dimension d = m.getPreferredSize();
				m.setSize(d.width, d.height);
			
				if ((x == 0) || ((x + d.width) <= maxwidth)) {
					if (x > 0) {
						x += hgap;
					}
					x += d.width;
					colh = Math.max(colh, d.height);
				} else {
					moveComponents(target, insets.left + hgap, y, maxwidth - x, colh, start, i);
					x = d.width;
					y += vgap + colh;
					colh = d.height;
					start = i;
				}
			}
		}
		moveComponents(target, insets.left + hgap, y, maxwidth - x, colh, start, nmembers);
    }
    
}
