
/*
 * DiskSim Storage Subsystem Simulation Environment (Version 2.0)
 * Revision Authors: Greg Ganger
 * Contributors: Ross Cohen, John Griffin, Steve Schlosser
 *
 * Copyright (c) of Carnegie Mellon University, 1999.
 *
 * Permission to reproduce, use, and prepare derivative works of
 * this software for internal use is granted provided the copyright
 * and "No Warranty" statements are included with all reproductions
 * and derivative works. This software may also be redistributed
 * without charge provided that the copyright and "No Warranty"
 * statements are included in all redistributions.
 *
 * NO WARRANTY. THIS SOFTWARE IS FURNISHED ON AN "AS IS" BASIS.
 * CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER
 * EXPRESSED OR IMPLIED AS TO THE MATTER INCLUDING, BUT NOT LIMITED
 * TO: WARRANTY OF FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY
 * OF RESULTS OR RESULTS OBTAINED FROM USE OF THIS SOFTWARE. CARNEGIE
 * MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT
 * TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
 */

/*
 * DiskSim Storage Subsystem Simulation Environment
 * Authors: Greg Ganger, Bruce Worthington, Yale Patt
 *
 * Copyright (C) 1993, 1995, 1997 The Regents of the University of Michigan 
 *
 * This software is being provided by the copyright holders under the
 * following license. By obtaining, using and/or copying this software,
 * you agree that you have read, understood, and will comply with the
 * following terms and conditions:
 *
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose and without fee or royalty is
 * hereby granted, provided that the full text of this NOTICE appears on
 * ALL copies of the software and documentation or portions thereof,
 * including modifications, that you make.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS," AND COPYRIGHT HOLDERS MAKE NO
 * REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED. BY WAY OF EXAMPLE,
 * BUT NOT LIMITATION, COPYRIGHT HOLDERS MAKE NO REPRESENTATIONS OR
 * WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY PARTICULAR PURPOSE OR
 * THAT THE USE OF THE SOFTWARE OR DOCUMENTATION WILL NOT INFRINGE ANY
 * THIRD PARTY PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS. COPYRIGHT
 * HOLDERS WILL BEAR NO LIABILITY FOR ANY USE OF THIS SOFTWARE OR
 * DOCUMENTATION.
 *
 *  This software is provided AS IS, WITHOUT REPRESENTATION FROM THE
 * UNIVERSITY OF MICHIGAN AS TO ITS FITNESS FOR ANY PURPOSE, AND
 * WITHOUT WARRANTY BY THE UNIVERSITY OF MICHIGAN OF ANY KIND, EITHER
 * EXPRESSED OR IMPLIED, INCLUDING WITHOUT LIMITATION THE IMPLIED
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE REGENTS
 * OF THE UNIVERSITY OF MICHIGAN SHALL NOT BE LIABLE FOR ANY DAMAGES,
 * INCLUDING SPECIAL , INDIRECT, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
 * WITH RESPECT TO ANY CLAIM ARISING OUT OF OR IN CONNECTION WITH THE
 * USE OF OR IN CONNECTION WITH THE USE OF THE SOFTWARE, EVEN IF IT HAS
 * BEEN OR IS HEREAFTER ADVISED OF THE POSSIBILITY OF SUCH DAMAGES
 *
 * The names and trademarks of copyright holders or authors may NOT be
 * used in advertising or publicity pertaining to the software without
 * specific, written prior permission. Title to copyright in this software
 * and any associated documentation will at all times remain with copyright
 * holders.
 */

#include "disksim_global.h"
#include "disksim_iosim.h"
#include "disksim_stat.h"
#include "disksim_disk.h"
#include "disksim_ioqueue.h"


/* read-only globals used during readparams phase */
static char *statdesc_seekdiststats	=	"Seek distance";
static char *statdesc_seektimestats	=	"Seek time";
static char *statdesc_rotlatstats	=	"Rotational latency";
static char *statdesc_xfertimestats	=	"Transfer time";
static char *statdesc_postimestats	=	"Positioning time";
static char *statdesc_acctimestats	=	"Access time";


struct disk *getdisk (int diskno)
{
   disk *currdisk;
   ASSERT1((diskno >= 0) && (diskno < MAXDEVICES), "diskno", diskno);
   currdisk = &disksim->diskinfo->disks[diskno];
   if (currdisk->inited == 0) {
      //fprintf (stderr, "getdisk returning a non-initialized currdisk\n");
   }
   return (currdisk);
}


int disk_get_numdisks (void)
{
   return(numdisks);
}


void disk_cleanstats (void)
{
   int i;

   for (i=0; i<MAXDEVICES; i++) {
      disk *currdisk = getdisk (i);
      if (currdisk->inited) {
         ioqueue_cleanstats(currdisk->queue);
      }
   }
}


void disk_param_override (char *paramname, char *paramval, int first, int last)
{
   int i;

   if (first == -1) {
      first = 0;
      last = MAXDEVICES - 1;
   } else if ((first < 0) || (last >= MAXDEVICES) || (last < first)) {
      fprintf(stderr, "Invalid range at disk_param_override: %d - %d\n", first, last);
      exit(0);
   }

   for (i=first; i<=last; i++) {
      disk *currdisk = getdisk (i);
      if (currdisk->inited == 0) {
         continue;
      }

      if ((strcmp(paramname, "ioqueue_schedalg") == 0) ||
	  (strcmp(paramname, "ioqueue_seqscheme") == 0) ||
	  (strcmp(paramname, "ioqueue_cylmaptype") == 0) ||
	  (strcmp(paramname, "ioqueue_to_time") == 0) ||
	  (strcmp(paramname, "ioqueue_timeout_schedalg") == 0) ||
	  (strcmp(paramname, "ioqueue_priority_schedalg") == 0) ||
	  (strcmp(paramname, "ioqueue_priority_mix") == 0)) {
         ioqueue_param_override(currdisk->queue, paramname, paramval);

      } else if (strcmp(paramname, "reading_buffer_whole_servtime") == 0) {
         if (sscanf(paramval, "%lf\n", &reading_buffer_whole_servtime) != 1) {
            fprintf(stderr, "Error reading reading_buffer_whole_servtime in disk_param_override\n");
            exit(0);
         }
      } else if (strcmp(paramname, "buffer_whole_servtime") == 0) {
         if (sscanf(paramval, "%lf\n", &buffer_whole_servtime) != 1) {
            fprintf(stderr, "Error reading buffer_whole_servtime in disk_param_override\n");
            exit(0);
         }
      } else if (strcmp(paramname, "reading_buffer_partial_servtime") == 0) {
         if (sscanf(paramval, "%lf\n", &reading_buffer_partial_servtime) != 1) {
            fprintf(stderr, "Error reading reading_buffer_partial_servtime in disk_param_override\n");
            exit(0);
         }
      } else if (strcmp(paramname, "buffer_partial_servtime") == 0) {
         if (sscanf(paramval, "%lf\n", &buffer_partial_servtime) != 1) {
            fprintf(stderr, "Error reading buffer_partial_servtime in disk_param_override\n");
            exit(0);
         }
      } else if (strcmp(paramname, "readaheadifidle") == 0) {
         if (sscanf(paramval, "%d\n", &currdisk->readaheadifidle) != 1) {
            fprintf(stderr, "Error reading readaheadifidle in disk_param_override\n");
            exit(0);
         }
      } else if (strcmp(paramname, "writecomb") == 0) {
         if (sscanf(paramval, "%d\n", &currdisk->writecomb) != 1) {
            fprintf(stderr, "Error reading writecomb in disk_param_override\n");
            exit(0);
         }
      } else if (strcmp(paramname, "maxqlen") == 0) {
         if (sscanf(paramval, "%d\n", &currdisk->maxqlen) != 1) {
            fprintf(stderr, "Error reading maxqlen in disk_param_override\n");
            exit(0);
         }
      } else if (strcmp(paramname, "hold_bus_for_whole_read_xfer") == 0) {
         if (sscanf(paramval, "%d", &currdisk->hold_bus_for_whole_read_xfer) != 1) {
            fprintf(stderr, "Error reading hold_bus_for_whole_read_xfer in disk_param_override\n");
            exit(0);
         }
      } else if (strcmp(paramname, "hold_bus_for_whole_write_xfer") == 0) {
         if (sscanf(paramval, "%d", &currdisk->hold_bus_for_whole_write_xfer) != 1) {
            fprintf(stderr, "Error reading hold_bus_for_whole_write_xfer in disk_param_override\n");
            exit(0);
         }
      } else if (strcmp(paramname, "almostreadhits") == 0) {
         if (sscanf(paramval, "%d", &currdisk->almostreadhits) != 1) {
            fprintf(stderr, "Error reading almostreadhits in disk_param_override\n");
            exit(0);
         }
      } else if (strcmp(paramname, "sneakyfullreadhits") == 0) {
         if (sscanf(paramval, "%d", &currdisk->sneakyfullreadhits) != 1) {
            fprintf(stderr, "Error reading sneakyfullreadhits in disk_param_override\n");
            exit(0);
         }
      } else if (strcmp(paramname, "sneakypartialreadhits") == 0) {
         if (sscanf(paramval, "%d", &currdisk->sneakypartialreadhits) != 1) {
            fprintf(stderr, "Error reading sneakypartialreadhits in disk_param_override\n");
            exit(0);
         }
      } else if (strcmp(paramname, "sneakyintermediatereadhits") == 0) {
         if (sscanf(paramval, "%d", &currdisk->sneakyintermediatereadhits) != 1) {
            fprintf(stderr, "Error reading sneakyintermediatereadhits in disk_param_override\n");
            exit(0);
         }
      } else if (strcmp(paramname, "readhitsonwritedata") == 0) {
         if (sscanf(paramval, "%d", &currdisk->readhitsonwritedata) != 1) {
            fprintf(stderr, "Error reading readhitsonwritedata in disk_param_override\n");
            exit(0);
         }
      } else if (strcmp(paramname, "writeprebuffering") == 0) {
         if (sscanf(paramval, "%d", &currdisk->writeprebuffering) != 1) {
            fprintf(stderr, "Error reading writeprebuffering in disk_param_override\n");
            exit(0);
         }
      } else if (strcmp(paramname, "preseeking") == 0) {
         if (sscanf(paramval, "%d", &currdisk->preseeking) != 1) {
            fprintf(stderr, "Error reading preseeking in disk_param_override\n");
            exit(0);
         }
      } else if (strcmp(paramname, "neverdisconnect") == 0) {
         if (sscanf(paramval, "%d", &currdisk->neverdisconnect) != 1) {
            fprintf(stderr, "Error reading neverdisconnect in disk_param_override\n");
            exit(0);
         }
      } else if (strcmp(paramname, "numsegs") == 0) {
         if (sscanf(paramval, "%d", &currdisk->numsegs) != 1) {
            fprintf(stderr, "Error reading numsegs in disk_param_override\n");
            exit(0);
         }
      } else if (strcmp(paramname, "segsize") == 0) {
         if (sscanf(paramval, "%d", &currdisk->segsize) != 1) {
            fprintf(stderr, "Error reading segsize in disk_param_override\n");
            exit(0);
         }
      } else if (strcmp(paramname, "numwritesegs") == 0) {
         if (sscanf(paramval, "%d", &currdisk->numwritesegs) != 1) {
            fprintf(stderr, "Error reading numwritesegs in disk_param_override\n");
            exit(0);
         }
      } else if (strcmp(paramname, "dedicatedwriteseg") == 0) {
         int useded;
         if (sscanf(paramval, "%d", &useded) != 1) {
            fprintf(stderr, "Error reading dedicatedwriteseg in disk_param_override\n");
            exit(0);
         }
	 currdisk->dedicatedwriteseg = (segment *)useded;
      } else if (strcmp(paramname, "fastwrites") == 0) {
         if (sscanf(paramval, "%d", &currdisk->fastwrites) != 1) {
            fprintf(stderr, "Error reading fastwrites in disk_param_override\n");
            exit(0);
         }
      } else if (strcmp(paramname, "numsurfaces") == 0) {
	 if (sscanf(paramval, "%d", &currdisk->numsurfaces) != 1) {
	    fprintf(stderr, "Error reading numsurfaces in disk_param_override\n");
	    exit(0);
	 }
      } else if (strcmp(paramname, "numblocks") == 0) {
	 if (sscanf(paramval, "%d", &currdisk->numblocks) != 1) {
	    fprintf(stderr, "Error reading numblocks in disk_param_override\n");
	    exit(0);
	 }
      } else if (strcmp(paramname, "numcyls") == 0) {
	 if (sscanf(paramval, "%d", &currdisk->numcyls) != 1) {
	    fprintf(stderr, "Error reading numcyls in disk_param_override\n");
	    exit(0);
	 }
      } else if (strcmp(paramname, "mapping") == 0) {
	 if (sscanf(paramval, "%d", &currdisk->mapping) != 1) {
	    fprintf(stderr, "Error reading mapping in disk_param_override\n");
	    exit(0);
	 }
      } else {
	 fprintf(stderr, "Unsupported param to override at disk_param_override: %s\n", paramname);
	 exit(0);
      }
   }
}


static void diskstatinit (int diskno, int firsttime)
{
   disk *currdisk = getdisk (diskno);
   diskstat *stat = &currdisk->stat;

   if (firsttime) {
      stat_initialize(statdeffile, statdesc_seekdiststats, &stat->seekdiststats);
      stat_initialize(statdeffile, statdesc_seektimestats, &stat->seektimestats);
      stat_initialize(statdeffile, statdesc_rotlatstats, &stat->rotlatstats);
      stat_initialize(statdeffile, statdesc_xfertimestats, &stat->xfertimestats);
      stat_initialize(statdeffile, statdesc_postimestats, &stat->postimestats);
      stat_initialize(statdeffile, statdesc_acctimestats, &stat->acctimestats);
   } else {
      stat_reset(&stat->seekdiststats);
      stat_reset(&stat->seektimestats);
      stat_reset(&stat->rotlatstats);
      stat_reset(&stat->xfertimestats);
      stat_reset(&stat->postimestats);
      stat_reset(&stat->acctimestats);
   }

   stat->highblkno = 0;
   stat->zeroseeks = 0;
   stat->zerolatency = 0;
   stat->writecombs = 0;
   stat->readmisses = 0;
   stat->writemisses = 0;
   stat->fullreadhits = 0;
   stat->appendhits = 0;
   stat->prependhits = 0;
   stat->readinghits = 0;
   stat->runreadingsize = 0.0;
   stat->remreadingsize = 0.0;
   stat->parthits = 0;
   stat->runpartsize = 0.0;
   stat->rempartsize = 0.0;
   stat->interfere[0] = 0;
   stat->interfere[1] = 0;
   stat->requestedbus = 0.0;
   stat->waitingforbus = 0.0;
   stat->numbuswaits = 0;
}


static void disk_syncset_init (void)
{
   int i, j;
   int synced[128];

   for (i=1; i<=numsyncsets; i++) {
      synced[i] = 0;
   }
   for (i=0; i<MAXDEVICES; i++) {
      disk *currdisk = getdisk (i);
      if (currdisk->inited == 0) {
         continue;
      }
      if (currdisk->syncset == 0) {
         currdisk->rpm -= (currdisk->rpm * currdisk->rpmerr * (double) 0.01 *
			 ((double) 1 - ((double) 2 * DISKSIM_drand48())));
         currdisk->currangle = DISKSIM_drand48();
      } else if ((currdisk->syncset > 0) && (synced[(currdisk->syncset)] == 0)) {
         currdisk->rpm -= (currdisk->rpm * currdisk->rpmerr * (double) 0.01 *
			 ((double) 1 - ((double) 2 * DISKSIM_drand48())));
         currdisk->currangle = DISKSIM_drand48();
         for (j=i; j<MAXDEVICES; j++) {
            disk *currdisk2 = getdisk (j);
            if (currdisk2->inited == 0) {
               continue;
            }
	    if (currdisk2->syncset == currdisk->syncset) {
	       currdisk2->rpm = currdisk->rpm;
	       currdisk2->currangle = currdisk->currangle;
	    }
	 }
	 synced[(currdisk->syncset)] = 1;
      }
   }
}


static void disk_postpass_perdisk (disk *currdisk)
{
   if ((currdisk->contread) && (currdisk->enablecache == FALSE)) {
      fprintf(stderr, "Makes no sense to use read-ahead but not use caching\n");
      exit(0);
   }

   if ((currdisk->contread == BUFFER_NO_READ_AHEAD) &&
       (currdisk->minreadahead > 0)) {
      fprintf(stderr, "'Minimum read-ahead (blks)' forced to zero due to 'Buffer continuous read' value\n");
      currdisk->minreadahead = 0;
   }

   if ((currdisk->maxreadahead < currdisk->minreadahead) ||
       ((currdisk->contread == BUFFER_NO_READ_AHEAD) &&
	(currdisk->maxreadahead > 0))) {
      fprintf(stderr, "'Maximum read-ahead (blks)' forced to zero due to 'Buffer continuous read' value\n");
      currdisk->maxreadahead = currdisk->minreadahead;
   }

   if (currdisk->keeprequestdata >= 0) {
      currdisk->keeprequestdata = abs(currdisk->keeprequestdata - 1);
   }

   if ((currdisk->readanyfreeblocks != FALSE) &&
       (currdisk->enablecache == FALSE)) {
      fprintf(stderr, "Makes no sense to read blocks with caching disabled\n");
      exit(0);
   }

   if ((currdisk->fastwrites) && (currdisk->enablecache == FALSE)) {
      fprintf(stderr, "Can't use fast write if not employing caching\n");
      exit(0);
   }

   /* ripped from the override function */

   if ((currdisk->readaheadifidle != 0) && (currdisk->readaheadifidle != 1)) {
      fprintf(stderr, "Invalid value for readaheadifidle in disk_param_override: %d\n", currdisk->readaheadifidle);
      exit(0);
   }

   if ((currdisk->writecomb != 0) && (currdisk->writecomb != 1)) {
      fprintf(stderr, "Invalid value for writecomb in disk_param_override: %d\n", currdisk->writecomb);
      exit(0);
   }

   if (currdisk->maxqlen < 0) {
      fprintf(stderr, "Invalid value for maxqlen in disk_param_override: %d\n", currdisk->maxqlen);
      exit(0);
   }

   if ((currdisk->hold_bus_for_whole_read_xfer != 0) &&
       (currdisk->hold_bus_for_whole_read_xfer != 1)) {
      fprintf(stderr, "Invalid value for hold_bus_for_whole_read_xfer in disk_param_override: %d\n", currdisk->hold_bus_for_whole_read_xfer);
      exit(0);
   }

   if ((currdisk->hold_bus_for_whole_write_xfer != 0) &&
       (currdisk->hold_bus_for_whole_write_xfer != 1)) {
      fprintf(stderr, "Invalid value for hold_bus_for_whole_write_xfer in disk_param_override: %d\n", currdisk->hold_bus_for_whole_write_xfer);
      exit(0);
   }
   if ((currdisk->hold_bus_for_whole_read_xfer == 1) &&
       ((currdisk->sneakyfullreadhits == 1) ||
	(currdisk->sneakypartialreadhits == 1) ||
	(currdisk->sneakyintermediatereadhits == 1))) {
      fprintf(stderr, "hold_bus_for_whole_read_xfer and one or more sneakyreadhits detected disk_param_override\n");
      exit(0);
   }

   if ((currdisk->almostreadhits != 0) && (currdisk->almostreadhits != 1)) {
      fprintf(stderr, "Invalid value for almostreadhits in disk_param_override: %d\n", currdisk->almostreadhits);
      exit(0);
   }

   if ((currdisk->sneakyfullreadhits != 0) &&
       (currdisk->sneakyfullreadhits != 1)) {
      fprintf(stderr, "Invalid value for sneakyfullreadhits in disk_param_override: %d\n", currdisk->sneakyfullreadhits);
      exit(0);
   }
   if ((currdisk->sneakyfullreadhits == 1) &&
       (currdisk->hold_bus_for_whole_read_xfer == 1)) {
      fprintf(stderr, "hold_bus_for_whole_read_xfer and sneakyfullreadhits detected disk_param_override\n");
      exit(0);
   }

   if ((currdisk->sneakypartialreadhits != 0) &&
       (currdisk->sneakypartialreadhits != 1)) {
      fprintf(stderr, "Invalid value for sneakypartialreadhits in disk_param_override: %d\n", currdisk->sneakypartialreadhits);
      exit(0);
   }
   if ((currdisk->sneakypartialreadhits == 1) &&
       (currdisk->hold_bus_for_whole_read_xfer == 1)) {
      fprintf(stderr, "hold_bus_for_whole_read_xfer and sneakypartialreadhits detected disk_param_override\n");
      exit(0);
   }

   if ((currdisk->sneakyintermediatereadhits != 0) &&
       (currdisk->sneakyintermediatereadhits != 1)) {
      fprintf(stderr, "Invalid value for sneakyintermediatereadhits in disk_param_override: %d\n", currdisk->sneakyintermediatereadhits);
      exit(0);
   }
   if ((currdisk->sneakyintermediatereadhits == 1) &&
       (currdisk->hold_bus_for_whole_read_xfer == 1)) {
      fprintf(stderr, "hold_bus_for_whole_read_xfer and sneakyintermediatereadhits detected disk_param_override\n");
      exit(0);
   }

   if ((currdisk->readhitsonwritedata != 0) &&
       (currdisk->readhitsonwritedata != 1)) {
      fprintf(stderr, "Invalid value for readhitsonwritedata in disk_param_override: %d\n", currdisk->readhitsonwritedata);
      exit(0);
   }

   if ((currdisk->writeprebuffering != 0) &&
       (currdisk->writeprebuffering != 1)) {
      fprintf(stderr, "Invalid value for writeprebuffering in disk_param_override: %d\n", currdisk->writeprebuffering);
      exit(0);
   }

   if ((currdisk->preseeking != 0) && (currdisk->preseeking != 1) &&
       (currdisk->preseeking != 2)) {
      fprintf(stderr, "Invalid value for preseeking in disk_param_override: %d\n", currdisk->preseeking);
      exit(0);
   }

   if ((currdisk->neverdisconnect != 0) && (currdisk->neverdisconnect != 1)) {
      fprintf(stderr, "Invalid value for neverdisconnect in disk_param_override: %d\n", currdisk->neverdisconnect);
      exit(0);
   }

   if (currdisk->numsegs < 1) {
      fprintf(stderr, "Invalid value for numsegs in disk_param_override: %d\n", currdisk->numsegs);
      exit(0);
   }

   if ((currdisk->segsize < 1) ||
       (currdisk->segsize > currdisk->numblocks)) {
      fprintf(stderr, "Invalid value for segsize in disk_param_override: %d\n", currdisk->segsize);
      exit(0);
   }

   if ((currdisk->numwritesegs > currdisk->numsegs) ||
       (currdisk->numwritesegs < 1)) {
      fprintf(stderr, "Invalid value for numwritesegs in disk_param_override: %d\n", currdisk->numwritesegs);
      exit(0);
   }

   if ((currdisk->numsegs <= 1) && (currdisk->dedicatedwriteseg)) {
      fprintf(stderr, "Must have more segments than dedicated write segments\n");
      exit(0);
   }
   if (((int)currdisk->dedicatedwriteseg != 0) &&
       ((int)currdisk->dedicatedwriteseg != 1)) {
      fprintf(stderr, "Invalid value for dedicatedwriteseg in disk_param_override: %d\n", (int)currdisk->dedicatedwriteseg);
      exit(0);
   }

   if ((currdisk->fastwrites != 0) && (currdisk->fastwrites != 1) &&
       (currdisk->fastwrites != 2)) {
      fprintf(stderr, "Invalid value for fastwrites in disk_param_override: %d\n", currdisk->fastwrites);
      exit(0);
   }
   if ((currdisk->fastwrites != 0) && (currdisk->enablecache == FALSE)) {
      fprintf(stderr, "disk_param_override:  Can't use fast write if not employing caching\n");
      exit(0);
   }

   if (currdisk->numsurfaces < 1) {
      fprintf(stderr, "Invalid value for numsurfaces in disk_param_override: %d\n", currdisk->numsurfaces);
      exit(0);
   }

   /* This is probably too difficult to be worthwhile  -rcohen
   if (currdisk->numcyls < 1) {
      fprintf(stderr, "Invalid value for numcyls in disk_param_override: %d\n", currdisk->numcyls);
      exit(0);
   }
   */

   diskmap_initialize (currdisk);
}


static void disk_postpass (void)
{
   int i;

   if (numdisks == 0) {
      return;
   }

   if (reading_buffer_whole_servtime < 0.0) {
      fprintf(stderr, "Invalid value for reading_buffer_whole_servtime in disk_param_override: %f\n", reading_buffer_whole_servtime);
      exit(0);
   }

   if (buffer_whole_servtime < 0.0) {
      fprintf(stderr, "Invalid value for buffer_whole_servtime in disk_param_override: %f\n", buffer_whole_servtime);
      exit(0);
   }

   if (reading_buffer_partial_servtime < 0.0) {
      fprintf(stderr, "Invalid value for reading_buffer_partial_servtime in disk_param_override: %f\n", reading_buffer_partial_servtime);
      exit(0);
   }

   if (buffer_partial_servtime < 0.0) {
      fprintf(stderr, "Invalid value for buffer_partial_servtime in disk_param_override: %f\n", buffer_partial_servtime);
      exit(0);
   }

   for (i = 0; i < MAXDEVICES; i++) {
      disk *currdisk = getdisk (i);
      if (currdisk->inited) {
         disk_postpass_perdisk(currdisk);
      }
   }

   /* syncset stuff, perhaps should be in its own function */
   /* unnecessary for now anyway */

   /*
   for (i = 0; i < MAXDEVICES; i++) {
      disk *currdisk = getdisk (i);
      if (currdisk->inited) {
         currdisk->syncset = 0;
      }
   }
   */

}


void disk_setcallbacks ()
{
   disksim->enablement_disk = disk_enablement_function;
   ioqueue_setcallbacks();
}


static void disk_initialize_diskinfo ()
{
   disksim->diskinfo = DISKSIM_malloc (sizeof(disk_info_t));
   bzero (disksim->diskinfo, sizeof(disk_info_t));

   disksim->diskinfo->disks = (disk*) DISKSIM_malloc(MAXDEVICES * (sizeof(disk)));

   /* important initialization of stuff that gets remapped into diskinfo */
   disk_printhacktime = 0.0;
   global_currtime = 0.0;
   global_currangle = 0.0;
   swap_forward_only = 1;
   addtolatency = 0.0;
   disk_seek_stoptime = 0.0;
   disk_last_seektime = 0.0;
   disk_last_latency = 0.0;
   disk_last_xfertime = 0.0;
   disk_last_acctime = 0.0;
   buffer_partial_servtime = 0.000000001;
   reading_buffer_partial_servtime = 0.000000001;
   buffer_whole_servtime = 0.000000000;
   reading_buffer_whole_servtime = 0.000000000;
}


void disk_initialize (void)
{
   int i, j;
   diskreq *tmpdiskreq;
   segment *tmpseg;
   double tmptime;
   double rotblks;
   double tmpfull;
   double tmpavg;
/*
fprintf (outputfile, "Entered disk_initialize - numdisks %d\n", numdisks);
*/
   StaticAssert (sizeof(segment) <= DISKSIM_EVENT_SIZE);
   StaticAssert (sizeof(diskreq) <= DISKSIM_EVENT_SIZE);

   if (disksim->diskinfo == NULL) {
      disk_initialize_diskinfo();
   }

   disk_setcallbacks();
   disk_postpass();
   disk_syncset_init();
   for (i = 0; i < MAXDEVICES; i++) {
      disk *currdisk = getdisk (i);
      if (currdisk->inited == 0) {
         continue;
      }
      ioqueue_initialize(currdisk->queue, i);
      ioqueue_set_enablement_function (currdisk->queue, &disksim->enablement_disk);
      addlisttoextraq((event **) &currdisk->outwait);
      addlisttoextraq((event **) &currdisk->buswait);
      if (currdisk->currentbus) {
	 if (currdisk->currentbus == currdisk->effectivebus) {
	    currdisk->effectivebus = NULL;
	 }
	 tmpdiskreq = currdisk->currentbus;
	 if (tmpdiskreq->seg) {
	    disk_buffer_remove_from_seg(tmpdiskreq);
	 }
         addlisttoextraq((event **) &tmpdiskreq->ioreqlist);
	 currdisk->currentbus = NULL;
	 addtoextraq((event *) tmpdiskreq);
      }
      if (currdisk->effectivebus) {
	 tmpdiskreq = currdisk->effectivebus;
	 if (tmpdiskreq->seg) {
	    disk_buffer_remove_from_seg(tmpdiskreq);
	 }
         addlisttoextraq((event **) &tmpdiskreq->ioreqlist);
	 currdisk->effectivebus = NULL;
	 addtoextraq((event *) tmpdiskreq);
      }
      if (currdisk->currenthda) {
	 if (currdisk->currenthda == currdisk->effectivehda) {
	    currdisk->effectivehda = NULL;
	 }
	 tmpdiskreq = currdisk->currenthda;
	 if (tmpdiskreq->seg) {
	    disk_buffer_remove_from_seg(tmpdiskreq);
	 }
         addlisttoextraq((event **) &tmpdiskreq->ioreqlist);
	 currdisk->currenthda = NULL;
	 addtoextraq((event *) tmpdiskreq);
      }
      if (currdisk->effectivehda != NULL) {
	 tmpdiskreq = currdisk->effectivehda;
	 if (tmpdiskreq->seg) {
	    disk_buffer_remove_from_seg(tmpdiskreq);
	 }
         addlisttoextraq((event **) &tmpdiskreq->ioreqlist);
	 currdisk->effectivehda = NULL;
	 addtoextraq((event *) tmpdiskreq);
      }
      while (currdisk->pendxfer) {
	 tmpdiskreq = currdisk->pendxfer;
	 if (tmpdiskreq->seg) {
	    disk_buffer_remove_from_seg(tmpdiskreq);
	 }
         addlisttoextraq((event **) &tmpdiskreq->ioreqlist);
	 currdisk->pendxfer = tmpdiskreq->bus_next;
	 addtoextraq((event *) tmpdiskreq);
      }
      currdisk->numinbuses = 0;
      currdisk->outstate = DISK_IDLE;
      currdisk->busy = FALSE;
      currdisk->prev_readahead_min = -1;
      currdisk->extradisc_diskreq = NULL;
      currdisk->currcylno = 0;
      currdisk->currsurface = 0;
      currdisk->currtime = 0.0;
      currdisk->lastflags = READ;
      currdisk->rotatetime = (double) MSECS_PER_MIN / currdisk->rpm;
      currdisk->lastgen = -1;
      currdisk->busowned = -1;
      currdisk->numdirty = 0;
       if (currdisk->seglist == NULL) {
	 currdisk->seglist = (segment *) DISKSIM_malloc(sizeof(segment));
	 currdisk->seglist->next = NULL;
	 currdisk->seglist->prev = NULL;
	 currdisk->seglist->diskreqlist = NULL;
	 currdisk->seglist->recyclereq = NULL;
	 currdisk->seglist->access = NULL;
	 for (j = 1; j < currdisk->numsegs; j++) {
	    tmpseg = (segment *) DISKSIM_malloc(sizeof(segment));
	    tmpseg->next = currdisk->seglist;
	    currdisk->seglist = tmpseg;
            tmpseg->next->prev = tmpseg;
	    tmpseg->prev = NULL;
	    tmpseg->diskreqlist = NULL;
	    tmpseg->recyclereq = NULL;
	    tmpseg->access = NULL;
	 }
	 if (currdisk->dedicatedwriteseg) {
	    currdisk->dedicatedwriteseg = currdisk->seglist;
	 }
      }
      tmpseg = currdisk->seglist;
      for (j = 0; j < currdisk->numsegs; j++) {
	 tmpseg->outstate = BUFFER_IDLE;
	 tmpseg->state = BUFFER_EMPTY;
	 tmpseg->size = currdisk->segsize;
	 while (tmpseg->diskreqlist) {
            addlisttoextraq((event **) &tmpseg->diskreqlist->ioreqlist);
	    tmpdiskreq = tmpseg->diskreqlist;
	    tmpseg->diskreqlist = tmpdiskreq->seg_next;
	    addtoextraq((event *) tmpdiskreq);
	 }
	 /* recyclereq should have already been "recycled" :) by the
	    effectivehda or currenthda recycling above */
	 tmpseg->recyclereq = NULL;
         addlisttoextraq((event **) &tmpseg->access);
	 tmpseg = tmpseg->next;
      }
      for (j=0; j<currdisk->numbands; j++) {
	 tmptime = max(diskseektime(currdisk, 0, 1, 0), diskseektime(currdisk, 0, 1, 1));
	 rotblks = (double) currdisk->bands[j].blkspertrack / currdisk->rotatetime;
         currdisk->bands[j].firstblkno = currdisk->bands[j].firstblkno / rotblks;
	 if (currdisk->bands[j].trackskew == -1.0) {
	    currdisk->bands[j].trackskew = tmptime;
	 } else if (currdisk->bands[j].trackskew == -2.0) {
	    tmptime = (double) ((int) (tmptime * rotblks + 0.999999999));
	    currdisk->bands[j].trackskew = tmptime / rotblks;
	 } else {
	    currdisk->bands[j].trackskew = currdisk->bands[j].trackskew / rotblks;
	 }
	 tmptime = max(diskseektime(currdisk, 1, 0, 0), diskseektime(currdisk, 1, 0, 1));
	 if (currdisk->bands[j].cylskew == -1.0) {
	    currdisk->bands[j].cylskew = tmptime;
	 } else if (currdisk->bands[j].cylskew == -2.0) {
	    tmptime = (double) ((int) (tmptime * rotblks + 0.99999999));
	    currdisk->bands[j].cylskew = tmptime / rotblks;
	 } else {
	    currdisk->bands[j].cylskew = currdisk->bands[j].cylskew / rotblks;
	 }
      }
      if ((currdisk->seektime == THREEPOINT_CURVE) && (currdisk->seekavg > currdisk->seekone)) {
fprintf (outputfile, "seekone %f, seekavg %f, seekfull %f\n", currdisk->seekone, currdisk->seekavg, currdisk->seekfull);
	 tmpfull = currdisk->seekfull;
	 tmpavg = currdisk->seekavg;
	 tmptime = (double) -10.0 * currdisk->seekone;
	 tmptime += (double) 15.0 * currdisk->seekavg;
	 tmptime += (double) -5.0 * currdisk->seekfull;
	 tmptime = tmptime / ((double) 3 * sqrt((double) (currdisk->numcyls)));
	 currdisk->seekavg *= (double) -15.0;
	 currdisk->seekavg += (double) 7.0 * currdisk->seekone;
	 currdisk->seekavg += (double) 8.0 * currdisk->seekfull;
	 currdisk->seekavg = currdisk->seekavg / (double) (3 * currdisk->numcyls);
	 currdisk->seekfull = tmptime;
fprintf (outputfile, "seekone %f, seekavg %f, seekfull %f\n", currdisk->seekone, currdisk->seekavg, currdisk->seekfull);
fprintf (outputfile, "seekone %f, seekavg %f, seekfull %f\n", diskseektime(currdisk, 1, 0, 1), diskseektime(currdisk, (currdisk->numcyls / 3), 0, 1), diskseektime(currdisk, (currdisk->numcyls - 1), 0, 1));
	 if ((currdisk->seekavg < 0.0) || (currdisk->seekfull < 0.0)) {
	    currdisk->seektime = THREEPOINT_CURVE;
	    currdisk->seekfull = tmpfull;
	    currdisk->seekavg = tmpavg;
	 }
      }
      diskstatinit(i, TRUE);
      disk_check_numblocks(currdisk);
   }
}


void disk_resetstats (void)
{
   int i;

   for (i=0; i<MAXDEVICES; i++) {
      disk *currdisk = getdisk (i);
      if (currdisk->inited) {
         ioqueue_resetstats(currdisk->queue);
         diskstatinit(i, 0);
      }
   }
}


static void bandcopy (band *destbands, band *srcbands, int numbands)
{
   int i, j;

   for (i=0; i<numbands; i++) {
      destbands[i].startcyl = srcbands[i].startcyl;
      destbands[i].endcyl = srcbands[i].endcyl;
      destbands[i].firstblkno = srcbands[i].firstblkno;
      destbands[i].blkspertrack = srcbands[i].blkspertrack;
      destbands[i].deadspace = srcbands[i].deadspace;
      destbands[i].trackskew = srcbands[i].trackskew;
      destbands[i].cylskew = srcbands[i].cylskew;
      destbands[i].blksinband = srcbands[i].blksinband;
      destbands[i].sparecnt = srcbands[i].sparecnt;
      destbands[i].numslips = srcbands[i].numslips;
      destbands[i].numdefects = srcbands[i].numdefects;
      for (j=0; j<srcbands[i].numslips; j++) {
         destbands[i].slip[j] = srcbands[i].slip[j];
      }
      for (j=0; j<srcbands[i].numdefects; j++) {
         destbands[i].defect[j] = srcbands[i].defect[j];
         destbands[i].remap[j] = srcbands[i].remap[j];
      }
   }
}


void disk_read_toprints (FILE *parfile)
{
}


static void band_read_specs (disk *currdisk, int numbands, FILE *parfile, int numsurfaces, int sparescheme)
{
   band *bands = currdisk->bands;
   int i;
   int bandno;
   int bandno_expected;
   int startcyl;
   int endcyl;
   int blkspertrack;
   int deadspace;
   int sparecnt;
   int numslips;
   int numdefects;
   int defect;
   int remap;

   for (bandno_expected=1; bandno_expected<=numbands; bandno_expected++) {
      if (fscanf(parfile, "Band #%d\n", &bandno) != 1) {
	 fprintf(stderr, "Error reading band number\n");
	 exit(0);
      }
      if (bandno != bandno_expected) {
	 fprintf(stderr, "Invalid value for band number - %d\n", bandno);
	 exit(0);
      }
      fprintf (outputfile, "Band #%d\n", bandno);
      bandno--;

      getparam_int(parfile, "First cylinder number", &startcyl, 1, 0, 0);
      bands[bandno].startcyl = startcyl;

      getparam_int(parfile, "Last cylinder number", &endcyl, 1, 0, 0);
      bands[bandno].endcyl = endcyl;

      getparam_int(parfile, "Blocks per track", &blkspertrack, 1, 1, 0);
      bands[bandno].blkspertrack = blkspertrack;

      getparam_double(parfile, "Offset of first block", &bands[bandno].firstblkno, 1, (double) 0.0, (double) 0.0);

      getparam_int(parfile, "Empty space at zone front", &deadspace, 1, 0, 0);
      bands[bandno].deadspace = deadspace;

      getparam_double(parfile, "Skew for track switch", &bands[bandno].trackskew, 1, (double) -2.0, (double) 0.0);
      getparam_double(parfile, "Skew for cylinder switch", &bands[bandno].cylskew, 1, (double) -2.0, (double) 0.0);

      getparam_int(parfile, "Number of spares", &sparecnt, 1, 0, 0);
      bands[bandno].sparecnt = sparecnt;

      getparam_int(parfile, "Number of slips", &numslips, 3, 0, MAXSLIPS);
      bands[bandno].numslips = numslips;
      for (i=0; i<numslips; i++) {
         getparam_int(parfile, "Slip", &bands[bandno].slip[i], 1, 0, 0);
      }

      getparam_int(parfile, "Number of defects", &numdefects, 3, 0, MAXDEFECTS);
      bands[bandno].numdefects = numdefects;
      for (i=0; i<numdefects; i++) {
         if (fscanf(parfile, "Defect: %d %d\n", &defect, &remap) != 2) {
	    fprintf(stderr, "Error reading defect #%d\n", i);
	    exit(0);
         }
         if ((defect < 0) || (remap < 0)) {
	    fprintf(stderr, "Invalid value(s) for defect - %d %d\n", defect, remap);
	    exit(0);
         }
         bands[bandno].defect[i] = defect;
         bands[bandno].remap[i] = remap;
         fprintf (outputfile, "Defect: %d %d\n", defect, remap);
      }

      /* fprintf (outputfile, "blksinband %d, sparecnt %d, blkspertrack %d, numtracks %d\n", bands[bandno].blksinband, sparecnt, blkspertrack, ((endcyl-startcyl+1)*numsurfaces)); */
   }
}


static FILE * disk_locate_spec_in_specfile (FILE *parfile, char *brandname, char *line)
{
   FILE *specfile;
   char specname[200];
   char specfilename[200];

   if (!fgets (line, 200, parfile)) {
      fprintf (stderr, "Unable to read disk specification file name from parfile\n");
      exit (0);
   }
   if (sscanf (line, "Disk specification file: %s", specfilename) != 1) {
      fprintf (stderr, "Bad format for `Disk specification file:'");
      exit (0);
   }
   if ((specfile = fopen(specfilename, "r")) == NULL) {
      fprintf(stderr, "Disk specifcation file %s cannot be opened for read access\n", specfilename);
      exit(0);
   }
   sprintf(specname, "Disk brand name: %s\n", brandname);
   while (fgets(line, 200, specfile)) {
      if (strcmp(line, specname) == 0) {
	 return(specfile);
      }
   }
   fprintf(stderr, "Disk specification for %s not found in file %s\n", brandname, specfilename);
   exit(0);
}


void disk_read_specs (FILE *parfile, int diskno, int copies)
{
   char line[201];
   FILE *specfile = parfile;
   char brandname[81];
   int i, j;
   double acctime;
   double seektime;
   char seekinfo[81];
   int extractseekcnt = 0;
   int *extractseekdists = NULL;
   double *extractseektimes = NULL;
   double seekavg;
   int numbands;
   int mapping;
   int enablecache;
   int contread;
   double hp1, hp2, hp3, hp4, hp5, hp6;
   double first10seeks[10];
   disk *currdisk;
   int useded;

   if (disksim->diskinfo == NULL) {
      disk_initialize_diskinfo();
   }

   numdisks += copies;

   /* NOTE: this loop isn't needed anymore */
   //while (diskno < numdisks) {
      currdisk = getdisk (diskno);

      if (fgets(line, 200, parfile) == NULL) {
	 fprintf(stderr, "End of file before disk specifications\n");
	 exit(0);
      }
      specfile = parfile;
      if (sscanf(line, "Disk brand name: %s\n", brandname) == 1) {
	 specfile = disk_locate_spec_in_specfile(parfile, brandname, line);
         if (fgets(line, 200, specfile) == NULL) {
            fprintf(stderr, "End of file before disk specifications\n");
            exit(0);
         }
      }

      if (sscanf(line, "Access time (in msecs): %lf\n", &acctime) != 1) {
         fprintf(stderr, "Error reading access time\n");
         exit(0);
      }
      if ((acctime < 0.0) && (!diskspecialaccesstime(acctime))) {
	 fprintf(stderr, "Invalid value for access time - %f\n", acctime);
	 exit(0);
      }
      fprintf (outputfile, "Access time (in msecs): %f\n", acctime);

      getparam_double(specfile, "Seek time (in msecs)", &seektime, 0, (double) 0.0, (double) 0.0);
      if ((seektime < 0.0) && (!diskspecialseektime(seektime))) {
         fprintf(stderr, "Invalid value for seek time - %f\n", seektime);
         exit(0);
      }

      getparam_double(specfile, "Single cylinder seek time", &currdisk->seekone, 1, (double) 0.0, (double) 0.0);

      if (fscanf(specfile, "Average seek time: %s", seekinfo) != 1) {
	 fprintf(stderr, "Error reading average seek time\n");
	 exit(0);
      }
      fgets(line, 200, specfile);
      if (seektime == EXTRACTION_SEEK) {
	 seekavg = 0.0;
         disk_read_extracted_seek_curve(seekinfo, &extractseekcnt, &extractseekdists, &extractseektimes);
      } else {
	 if (sscanf(seekinfo, "%lf", &seekavg) != 1) {
	    fprintf(stderr, "Invalid value for average seek time: %s\n", seekinfo);
	    exit(0);
	 }
      }
      if (seekavg < 0.0) {
         fprintf(stderr, "Invalid value for average seek time - %f\n", seekavg);
         exit(0);
      }
      fprintf (outputfile, "Average seek time: %s\n", seekinfo);

      getparam_double(specfile, "Full strobe seek time", &currdisk->seekfull, 1, (double) 0.0, (double) 0.0);
      getparam_double(specfile, "Add. write settling delay", &currdisk->seekwritedelta, 0, (double) 0.0, (double) 0.0);

      if (fscanf(specfile, "HPL seek equation values: %lf %lf %lf %lf %lf %lf", &hp1, &hp2, &hp3, &hp4, &hp5, &hp6) != 6) {
	 fprintf(stderr, "Error reading HPL seek equation values\n");
	 exit(0);
      }
      fgets(line, 200, specfile);
      if ((hp1 < 0) | (hp2 < 0) | (hp3 < 0) | (hp4 < 0) | (hp5 < 0) | (hp6 < -1)) {
         fprintf(stderr, "Invalid value for an HPL seek equation values - %f %f %f %f %f %f\n", hp1, hp2, hp3, hp4, hp5, hp6);
         exit(0);
      }
      fprintf (outputfile, "HPL seek equation values: %f %f %f %f %f %f\n", hp1, hp2, hp3, hp4, hp5, hp6);
      if (hp6 != -1) {
	 currdisk->seekone = hp6;
      }

      if ((j = fscanf(specfile, "First 10 seek times: %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf", &first10seeks[0], &first10seeks[1], &first10seeks[2], &first10seeks[3], &first10seeks[4], &first10seeks[5], &first10seeks[6], &first10seeks[7], &first10seeks[8], &first10seeks[9])) == 0) {
	 fprintf(stderr, "Error reading first 10 seek times\n");
	 exit(0);
      }
      fgets(line, 200, specfile);
      if (seektime == FIRST10_PLUS_HPL_SEEK_EQUATION) {
         if ((j < 10) |(first10seeks[0] < 0) | (first10seeks[1] < 0) | (first10seeks[2] < 0) | (first10seeks[3] < 0) | (first10seeks[4] < 0) | (first10seeks[5] < 0) | (first10seeks[6] < 0) | (first10seeks[7] < 0) | (first10seeks[8] < 0) | (first10seeks[9] < 0)) {
            fprintf(stderr, "Invalid value for a First 10 seek times: %d - %f %f %f %f %f %f %f %f %f %f\n", j, first10seeks[0], first10seeks[1], first10seeks[2], first10seeks[3], first10seeks[4], first10seeks[5], first10seeks[6], first10seeks[7], first10seeks[8], first10seeks[9]);
            exit(0);
         }
         fprintf (outputfile, "First 10 seek times:    %f %f %f %f %f %f %f %f %f %f\n", first10seeks[0], first10seeks[1], first10seeks[2], first10seeks[3], first10seeks[4], first10seeks[5], first10seeks[6], first10seeks[7], first10seeks[8], first10seeks[9]);
      } else {
         fprintf (outputfile, "First 10 seek times:    %f\n", first10seeks[0]);
      }

      getparam_double(specfile, "Head switch time", &currdisk->headswitch, 1, (double) 0.0, (double) 0.0);

      getparam_double(specfile, "Rotation speed (in rpms)", &currdisk->rpm, 4, (double) 0.0, (double) 0.0);
      getparam_double(specfile, "Percent error in rpms", &currdisk->rpmerr, 3, (double) 0.0, (double) 100.0);

      getparam_int(specfile, "Number of data surfaces", &currdisk->numsurfaces, 1, 1, 0);
      getparam_int(specfile, "Number of cylinders", &currdisk->numcyls, 1, 1, 0);
      getparam_int(specfile, "Blocks per disk", &currdisk->numblocks, 1, 1, 0);

      getparam_double(specfile, "Per-request overhead time", &currdisk->overhead, 1, (double) 0.0, (double) 0.0);
      getparam_double(specfile, "Time scale for overheads", &currdisk->timescale, 1, (double) 0.0, (double) 0.0);

      getparam_double(specfile, "Bulk sector transfer time", &currdisk->blktranstime, 1, (double) 0.0, (double) 0.0);
      getparam_bool(specfile, "Hold bus entire read xfer:", &currdisk->hold_bus_for_whole_read_xfer);
      getparam_bool(specfile, "Hold bus entire write xfer:", &currdisk->hold_bus_for_whole_write_xfer);

      getparam_bool(specfile, "Allow almost read hits:", &currdisk->almostreadhits);
      getparam_bool(specfile, "Allow sneaky full read hits:", &currdisk->sneakyfullreadhits);
      getparam_bool(specfile, "Allow sneaky partial read hits:", &currdisk->sneakypartialreadhits);
      getparam_bool(specfile, "Allow sneaky intermediate read hits:", &currdisk->sneakyintermediatereadhits);
      getparam_bool(specfile, "Allow read hits on write data:", &currdisk->readhitsonwritedata);
      getparam_bool(specfile, "Allow write prebuffering:", &currdisk->writeprebuffering);
      getparam_int(specfile, "Preseeking level", &currdisk->preseeking, 3, 0, 2);
      getparam_bool(specfile, "Never disconnect:", &currdisk->neverdisconnect);
      getparam_bool(specfile, "Print stats for disk:", &currdisk->printstats);
      getparam_int(specfile, "Avg sectors per cylinder", &currdisk->sectpercyl, 3, 1, currdisk->numblocks);

      getparam_int(specfile, "Max queue length at disk", &currdisk->maxqlen, 1, 0, 0);
      currdisk->queue = ioqueue_readparams(specfile, device_printqueuestats, device_printcritstats, device_printidlestats, device_printintarrstats, device_printsizestats);

      getparam_int(specfile, "Number of buffer segments", &currdisk->numsegs, 1, 1, 0);
      getparam_int(specfile, "Maximum number of write segments", &currdisk->numwritesegs, 3, 1, currdisk->numsegs);
      getparam_int(specfile, "Segment size (in blks)", &currdisk->segsize, 3, 1, currdisk->numblocks);
      getparam_bool(specfile, "Use separate write segment:", &useded);
      currdisk->dedicatedwriteseg = (segment *)useded;

      getparam_double(specfile, "Low (write) water mark", &currdisk->writewater, 3, (double) 0.0, (double) 1.0);
      getparam_double(specfile, "High (read) water mark", &currdisk->readwater, 3, (double) 0.0, (double) 1.0);
      getparam_bool(specfile, "Set watermark by reqsize:", &currdisk->reqwater);

      getparam_bool(specfile, "Calc sector by sector:", &currdisk->sectbysect);

      getparam_int(specfile, "Enable caching in buffer", &enablecache, 3, 0, 2);
      getparam_int(specfile, "Buffer continuous read", &contread, 3, 0, 4);

      getparam_int(specfile, "Minimum read-ahead (blks)", &currdisk->minreadahead, 3, 0, currdisk->segsize);

      getparam_int(specfile, "Maximum read-ahead (blks)", &currdisk->maxreadahead, 3, 0, currdisk->segsize);

      getparam_int(specfile, "Read-ahead over requested", &currdisk->keeprequestdata, 3, -1, 1);

      getparam_bool(specfile, "Read-ahead on idle hit:", &currdisk->readaheadifidle);
      getparam_bool(specfile, "Read any free blocks:", &currdisk->readanyfreeblocks);

      getparam_int(specfile, "Fast write level", &currdisk->fastwrites, 3, 0, 2);

      getparam_int(specfile, "Immediate buffer read", &currdisk->immedread, 3, 0, 2);
      getparam_int(specfile, "Immediate buffer write", &currdisk->immedwrite, 3, 0, 2);
      getparam_bool(specfile, "Combine seq writes:", &currdisk->writecomb);
      getparam_bool(specfile, "Stop prefetch in sector:", &currdisk->stopinsector);
      getparam_bool(specfile, "Disconnect write if seek:", &currdisk->disconnectinseek);
      getparam_bool(specfile, "Write hit stop prefetch:", &currdisk->write_hit_stop_readahead);
      getparam_bool(specfile, "Read directly to buffer:", &currdisk->read_direct_to_buffer);
      getparam_bool(specfile, "Immed transfer partial hit:", &currdisk->immedtrans_any_readhit);

      getparam_double(specfile, "Read hit over. after read", &currdisk->overhead_command_readhit_afterread, 1, (double) 0.0, (double) 0.0);
      getparam_double(specfile, "Read hit over. after write", &currdisk->overhead_command_readhit_afterwrite, 1, (double) 0.0, (double) 0.0);
      getparam_double(specfile, "Read miss over. after read", &currdisk->overhead_command_readmiss_afterread, 1, (double) 0.0, (double) 0.0);
      getparam_double(specfile, "Read miss over. after write", &currdisk->overhead_command_readmiss_afterwrite, 1, (double) 0.0, (double) 0.0);
      getparam_double(specfile, "Write hit over. after read", &currdisk->overhead_command_writehit_afterread, 1, (double) 0.0, (double) 0.0);
      getparam_double(specfile, "Write hit over. after write", &currdisk->overhead_command_writehit_afterwrite, 1, (double) 0.0, (double) 0.0);
      getparam_double(specfile, "Write miss over. after read", &currdisk->overhead_command_writemiss_afterread, 1, (double) 0.0, (double) 0.0);
      getparam_double(specfile, "Write miss over. after write", &currdisk->overhead_command_writemiss_afterwrite, 1, (double) 0.0, (double) 0.0);
      getparam_double(specfile, "Read completion overhead", &currdisk->overhead_complete_read, 1, (double) 0.0, (double) 0.0);
      getparam_double(specfile, "Write completion overhead", &currdisk->overhead_complete_write, 1, (double) 0.0, (double) 0.0);
      getparam_double(specfile, "Data preparation overhead", &currdisk->overhead_data_prep, 1, (double) 0.0, (double) 0.0);
      getparam_double(specfile, "First reselect overhead", &currdisk->overhead_reselect_first, 1, (double) 0.0, (double) 0.0);
      getparam_double(specfile, "Other reselect overhead", &currdisk->overhead_reselect_other, 1, (double) 0.0, (double) 0.0);
      getparam_double(specfile, "Read disconnect afterread", &currdisk->overhead_disconnect_read_afterread, 1, (double) 0.0, (double) 0.0);
      getparam_double(specfile, "Read disconnect afterwrite", &currdisk->overhead_disconnect_read_afterwrite, 1, (double) 0.0, (double) 0.0);
      getparam_double(specfile, "Write disconnect overhead", &currdisk->overhead_disconnect_write, 1, (double) 0.0, (double) 0.0);

      getparam_bool(specfile, "Extra write disconnect:", &currdisk->extra_write_disconnect);
      getparam_double(specfile, "Extradisc command overhead", &currdisk->extradisc_command, 1, (double) 0.0, (double) 0.0);
      getparam_double(specfile, "Extradisc disconnect overhead", &currdisk->extradisc_disconnect1, 1, (double) 0.0, (double) 0.0);
      getparam_double(specfile, "Extradisc inter-disconnect delay", &currdisk->extradisc_inter_disconnect, 1, (double) 0.0, (double) 0.0);
      getparam_double(specfile, "Extradisc 2nd disconnect overhead", &currdisk->extradisc_disconnect2, 1, (double) 0.0, (double) 0.0);
      getparam_double(specfile, "Extradisc seek delta", &currdisk->extradisc_seekdelta, 1, (double) 0.0, (double) 0.0);

      getparam_double(specfile, "Minimum seek delay", &currdisk->minimum_seek_delay, 1, (double) 0.0, (double) 0.0);

      getparam_int(specfile, "LBN-to-PBN mapping scheme", &mapping, 3, 0, LAYOUT_MAX);

      getparam_int(specfile, "Sparing scheme used", &currdisk->sparescheme, 3, NO_SPARING, MAXSPARESCHEME);

      getparam_int(specfile, "Rangesize for sparing", &currdisk->rangesize, 1, 1, 0);

      getparam_int(specfile, "Number of bands", &numbands, 1, 1, 0);

      for (i=diskno; i<(diskno+copies); i++) {
         disk *currdisk2 = getdisk (i);
         currdisk2->inited = 1;
         currdisk2->acctime = acctime;
         currdisk2->seektime = seektime;
         currdisk2->seekone = currdisk->seekone;
         currdisk2->seekavg = seekavg;
         currdisk2->seekfull = currdisk->seekfull;
	 currdisk2->seekwritedelta = currdisk->seekwritedelta;
         currdisk2->hpseek[0] = hp1;
         currdisk2->hpseek[1] = hp2;
         currdisk2->hpseek[2] = hp3;
         currdisk2->hpseek[3] = hp4;
         currdisk2->hpseek[4] = hp5;
         currdisk2->hpseek[5] = hp6;
	 for (j=0; j<10; j++) {
	    currdisk2->first10seeks[j] = first10seeks[j];
	 }
	 currdisk2->extractseekcnt = extractseekcnt;
	 currdisk2->extractseekdists = extractseekdists;
	 currdisk2->extractseektimes = extractseektimes;
         currdisk2->headswitch = currdisk->headswitch;
         currdisk2->rpm = currdisk->rpm;
         currdisk2->rpmerr = currdisk->rpmerr;
         currdisk2->numsurfaces = currdisk->numsurfaces;
         currdisk2->numcyls = currdisk->numcyls;
         currdisk2->numblocks = currdisk->numblocks;
         currdisk2->sparescheme = currdisk->sparescheme;
	 currdisk2->rangesize = currdisk->rangesize;
         currdisk2->printstats = currdisk->printstats;
         currdisk2->sectpercyl = currdisk->sectpercyl;
	 currdisk2->devno = i;
         currdisk2->maxqlen = currdisk->maxqlen;
         currdisk2->write_hit_stop_readahead = currdisk->write_hit_stop_readahead;
         currdisk2->read_direct_to_buffer = currdisk->read_direct_to_buffer;
         currdisk2->immedtrans_any_readhit = currdisk->immedtrans_any_readhit;
         currdisk2->extra_write_disconnect = currdisk->extra_write_disconnect;
         currdisk2->overhead_command_readhit_afterread = currdisk->overhead_command_readhit_afterread;
         currdisk2->overhead_command_readhit_afterwrite = currdisk->overhead_command_readhit_afterwrite;
         currdisk2->overhead_command_readmiss_afterread = currdisk->overhead_command_readmiss_afterread;
         currdisk2->overhead_command_readmiss_afterwrite = currdisk->overhead_command_readmiss_afterwrite;
         currdisk2->overhead_command_writehit_afterread = currdisk->overhead_command_writehit_afterread;
         currdisk2->overhead_command_writehit_afterwrite = currdisk->overhead_command_writehit_afterwrite;
         currdisk2->overhead_command_writemiss_afterread = currdisk->overhead_command_writemiss_afterread;
         currdisk2->overhead_command_writemiss_afterwrite = currdisk->overhead_command_writemiss_afterwrite;
         currdisk2->overhead_complete_read = currdisk->overhead_complete_read;
         currdisk2->overhead_complete_write = currdisk->overhead_complete_write;
         currdisk2->overhead_data_prep = currdisk->overhead_data_prep;
         currdisk2->overhead_reselect_first = currdisk->overhead_reselect_first;
         currdisk2->overhead_reselect_other = currdisk->overhead_reselect_other;
         currdisk2->overhead_disconnect_read_afterread = currdisk->overhead_disconnect_read_afterread;
         currdisk2->overhead_disconnect_read_afterwrite = currdisk->overhead_disconnect_read_afterwrite;
         currdisk2->overhead_disconnect_write = currdisk->overhead_disconnect_write;
         currdisk2->extradisc_command = currdisk->extradisc_command;
         currdisk2->extradisc_disconnect1 = currdisk->extradisc_disconnect1;
         currdisk2->extradisc_inter_disconnect = currdisk->extradisc_inter_disconnect;
         currdisk2->extradisc_disconnect2 = currdisk->extradisc_disconnect2;
         currdisk2->extradisc_seekdelta = currdisk->extradisc_seekdelta;
         currdisk2->minimum_seek_delay = currdisk->minimum_seek_delay;
         currdisk2->readanyfreeblocks = currdisk->readanyfreeblocks;
         currdisk2->overhead = currdisk->overhead;
         currdisk2->timescale = currdisk->timescale;
         currdisk2->blktranstime = currdisk->blktranstime;
         currdisk2->hold_bus_for_whole_read_xfer = currdisk->hold_bus_for_whole_read_xfer;
         currdisk2->hold_bus_for_whole_write_xfer = currdisk->hold_bus_for_whole_write_xfer;
         currdisk2->almostreadhits = currdisk->almostreadhits;
         currdisk2->sneakyfullreadhits = currdisk->sneakyfullreadhits;
         currdisk2->sneakypartialreadhits = currdisk->sneakypartialreadhits;
         currdisk2->sneakyintermediatereadhits = currdisk->sneakyintermediatereadhits;
         currdisk2->readhitsonwritedata = currdisk->readhitsonwritedata;
         currdisk2->writeprebuffering = currdisk->writeprebuffering;
         currdisk2->preseeking = currdisk->preseeking;
         currdisk2->neverdisconnect = currdisk->neverdisconnect;
	 currdisk2->numsegs = currdisk->numsegs;
	 currdisk2->numwritesegs = currdisk->numwritesegs;
	 currdisk2->segsize = currdisk->segsize;
	 currdisk2->dedicatedwriteseg = currdisk->dedicatedwriteseg;
	 currdisk2->writewater = currdisk->writewater;
	 currdisk2->readwater = currdisk->readwater;
	 currdisk2->reqwater = currdisk->reqwater;
	 currdisk2->sectbysect = currdisk->sectbysect;
	 currdisk2->enablecache = enablecache;
	 currdisk2->contread = contread;
	 currdisk2->minreadahead = currdisk->minreadahead;
	 currdisk2->maxreadahead = currdisk->maxreadahead;
	 currdisk2->keeprequestdata = currdisk->keeprequestdata;
	 currdisk2->readaheadifidle = currdisk->readaheadifidle;
	 currdisk2->fastwrites = currdisk->fastwrites;
	 currdisk2->immedread = currdisk->immedread;
	 currdisk2->immedwrite = currdisk->immedwrite;
	 currdisk2->writecomb = currdisk->writecomb;
	 currdisk2->stopinsector = currdisk->stopinsector;
	 currdisk2->disconnectinseek = currdisk->disconnectinseek;
	 currdisk2->seglist = NULL;
         currdisk2->buswait = NULL;
	 currdisk2->currenthda = NULL;
	 currdisk2->effectivehda = NULL;
	 currdisk2->currentbus = NULL;
	 currdisk2->effectivebus = NULL;
	 currdisk2->pendxfer = NULL;
	 currdisk2->outwait = NULL;
         currdisk2->numbands = numbands;
	 currdisk2->mapping = mapping;
	 if (i != diskno) {
	    currdisk2->queue = ioqueue_copy(currdisk->queue);
	 }
         currdisk2->bands = (band *) DISKSIM_malloc(numbands*(sizeof(band)));
	 if (i == diskno) {
            band_read_specs(currdisk, numbands, specfile, currdisk->numsurfaces, currdisk->sparescheme);
         } else {
	    bandcopy(currdisk2->bands, currdisk->bands, numbands);
	 }
      }
      diskno += copies;
   //}
}


void disk_set_syncset (int setstart, int setend)
{
   if ((setstart < 0) || (setend >= MAXDEVICES) || (setend < setstart)) {
      fprintf (stderr, "Bad range passed to disk_set_syncset (%d - %d)\n", setstart, setend);
      exit (0);
   }

   numsyncsets++;
   for (; setstart <= setend; setstart++) {
      disk *currdisk = getdisk (setstart);
      if (currdisk->syncset != 0) {
         fprintf (stderr, "Overlapping synchronized disk sets (%d and %d)\n", numsyncsets, currdisk->syncset);
         exit (0);
      }
      currdisk->syncset = numsyncsets;
   }
}


static void disk_interfere_printstats (int *set, int setsize, char *prefix)
{
   int seq = 0;
   int loc = 0;
   int i;

   if (device_printinterferestats == FALSE)
      return;

   for(i=0; i<setsize; i++) {
      disk *currdisk = getdisk (set[i]);
      seq += currdisk->stat.interfere[0];
      loc += currdisk->stat.interfere[1];
   }
   fprintf (outputfile, "%sSequential interference: %d\n", prefix, seq);
   fprintf (outputfile, "%sLocal interference:      %d\n", prefix, loc);
}


static void disk_buffer_printstats (int *set, int setsize, char *prefix)
{
   int writecombs = 0;
   int readmisses = 0;
   int writemisses = 0;
   int fullreadhits = 0;
   int prependhits = 0;
   int appendhits = 0;
   int readinghits = 0;
   double runreadingsize = 0.0;
   double remreadingsize = 0.0;
   int parthits = 0;
   double runpartsize = 0.0;
   double rempartsize = 0.0;
   double waitingforbus = 0.0;
   int numbuswaits = 0;
   int hits;
   int misses;
   int total;
   int reads;
   int writes;
   int i;

   if (device_printbufferstats == FALSE)
      return;

   for (i=0; i<setsize; i++) {
      disk *currdisk = getdisk (set[i]);
      writecombs += currdisk->stat.writecombs;
      readmisses += currdisk->stat.readmisses;
      writemisses += currdisk->stat.writemisses;
      fullreadhits += currdisk->stat.fullreadhits;
      prependhits += currdisk->stat.prependhits;
      appendhits += currdisk->stat.appendhits;
      readinghits += currdisk->stat.readinghits;
      runreadingsize += currdisk->stat.runreadingsize;
      remreadingsize += currdisk->stat.remreadingsize;
      parthits += currdisk->stat.parthits;
      runpartsize += currdisk->stat.runpartsize;
      rempartsize += currdisk->stat.rempartsize;
      waitingforbus += currdisk->stat.waitingforbus;
      numbuswaits += currdisk->stat.numbuswaits;
   }
   hits = fullreadhits + appendhits + prependhits + parthits + readinghits;
   misses = readmisses + writemisses;
   reads = fullreadhits + readmisses + readinghits + parthits;
   total = hits + misses;
   writes = total - reads;
   if (total == 0) {
      return;
   }
   fprintf(outputfile, "%sNumber of buffer accesses:    %d\n", prefix, total);
   fprintf(outputfile, "%sBuffer hit ratio:        %6d \t%f\n", prefix, hits, ((double) hits / (double) total));
   fprintf(outputfile, "%sBuffer miss ratio:            %6d \t%f\n", prefix, misses, ((double) misses / (double) total));
   fprintf(outputfile, "%sBuffer read hit ratio:        %6d \t%f \t%f\n", prefix, fullreadhits, ((double) fullreadhits / (double) max(1,reads)), ((double) fullreadhits / (double) total));
   fprintf(outputfile, "%sBuffer prepend hit ratio:       %6d \t%f\n", prefix, prependhits, ((double) prependhits / (double) max(1,writes)));
   fprintf(outputfile, "%sBuffer append hit ratio:       %6d \t%f\n", prefix, appendhits, ((double) appendhits / (double) max(1,writes)));
   fprintf(outputfile, "%sWrite combinations:           %6d \t%f\n", prefix, writecombs, ((double) writecombs / (double) max(1,writes)));
   fprintf(outputfile, "%sOngoing read-ahead hit ratio: %6d \t%f \t%f\n", prefix, readinghits, ((double) readinghits / (double) max(1,reads)), ((double) readinghits / (double) total));
   fprintf(outputfile, "%sAverage read-ahead hit size:  %f\n", prefix, (runreadingsize / (double) max(1,readinghits)));
   fprintf(outputfile, "%sAverage remaining read-ahead: %f\n", prefix, (remreadingsize / (double) max(1,readinghits)));
   fprintf(outputfile, "%sPartial read hit ratio: %6d \t%f \t%f\n", prefix, parthits, ((double) parthits / (double) max(1,reads)), ((double) parthits / (double) total));
   fprintf(outputfile, "%sAverage partial hit size:     %f\n", prefix, (runpartsize / (double) max(1,parthits)));
   fprintf(outputfile, "%sAverage remaining partial:    %f\n", prefix, (rempartsize / (double) max(1,parthits)));
   fprintf(outputfile, "%sTotal disk bus wait time: %f\n", prefix, waitingforbus);
   fprintf(outputfile, "%sNumber of disk bus waits: %d\n", prefix, numbuswaits);
}


static void disk_seek_printstats (int *set, int setsize, char *prefix)
{
   int i;
   int zeros = 0;
   statgen * statset[MAXDEVICES];
   double zerofrac;

   if (device_printseekstats == FALSE) {
      return;
   }

   for (i=0; i<setsize; i++) {
      disk *currdisk = getdisk (set[i]);
      zeros += currdisk->stat.zeroseeks;
      statset[i] = &currdisk->stat.seekdiststats;
   }
   zerofrac = (double) zeros / (double) stat_get_count_set(statset, setsize);
   fprintf (outputfile, "%sSeeks of zero distance:\t%d\t%f\n", prefix, zeros, zerofrac);
   stat_print_set(statset, setsize, prefix);

   for (i=0; i<setsize; i++) {
      disk *currdisk = getdisk (set[i]);
      statset[i] = &currdisk->stat.seektimestats;
   }
   stat_print_set(statset, setsize, prefix);
}


static void disk_latency_printstats (int *set, int setsize, char *prefix)
{
   int i;
   int zeros = 0;
   statgen * statset[MAXDEVICES];
   double zerofrac;
   disk *currdisk = NULL;

   if (device_printlatencystats == FALSE) {
      return;
   }

   for (i=0; i<setsize; i++) {
      currdisk = getdisk (set[i]);
      zeros += currdisk->stat.zerolatency;
      statset[i] = &currdisk->stat.rotlatstats;
   }
   if (setsize == 1) {
      fprintf (outputfile, "%sFull rotation time:      %f\n", prefix, currdisk->rotatetime);
   }
   zerofrac = (double) zeros / (double) stat_get_count_set(statset, setsize);
   fprintf (outputfile, "%sZero rotate latency:\t%d\t%f\n", prefix, zeros, zerofrac);
   stat_print_set(statset, setsize, prefix);
}


static void disk_transfer_printstats (int *set, int setsize, char *prefix)
{
   int i;
   statgen * statset[MAXDEVICES];

   if (device_printxferstats) {
      for (i=0; i<setsize; i++) {
         disk *currdisk = getdisk (set[i]);
         statset[i] = &currdisk->stat.xfertimestats;
      }
      stat_print_set(statset, setsize, prefix);
   }
}


static void disk_acctime_printstats (int *set, int setsize, char *prefix)
{
   int i;
   statgen * statset[MAXDEVICES];

   if (device_printacctimestats) {
      for (i=0; i<setsize; i++) {
         disk *currdisk = getdisk (set[i]);
         statset[i] = &currdisk->stat.postimestats;
      }
      stat_print_set(statset, setsize, prefix);
      for (i=0; i<setsize; i++) {
         disk *currdisk = getdisk (set[i]);
         statset[i] = &currdisk->stat.acctimestats;
      }
      stat_print_set(statset, setsize, prefix);
   }
}


void disk_printsetstats (int *set, int setsize, char *sourcestr)
{
   int i;
   struct ioq * queueset[MAXDEVICES];
   int reqcnt = 0;
   char prefix[80];

   sprintf(prefix, "%sdisk ", sourcestr);
   for (i=0; i<setsize; i++) {
      disk *currdisk = getdisk (set[i]);
      queueset[i] = currdisk->queue;
      reqcnt += ioqueue_get_number_of_requests(currdisk->queue);
   }
   if (reqcnt == 0) {
      fprintf (outputfile, "\nNo disk requests for members of this set\n\n");
      return;
   }
   ioqueue_printstats(queueset, setsize, prefix);

   disk_seek_printstats(set, setsize, prefix);
   disk_latency_printstats(set, setsize, prefix);
   disk_transfer_printstats(set, setsize, prefix);
   disk_acctime_printstats(set, setsize, prefix);
   disk_interfere_printstats(set, setsize, prefix);
   disk_buffer_printstats(set, setsize, prefix);
}


void disk_printstats (void)
{
   struct ioq * queueset[MAXDEVICES];
   int set[MAXDEVICES];
   int i;
   int reqcnt = 0;
   char prefix[80];
   int diskcnt;

   fprintf(outputfile, "\nDISK STATISTICS\n");
   fprintf(outputfile, "---------------\n\n");

   sprintf(prefix, "Disk ");

   diskcnt = 0;
   for (i=0; i<MAXDEVICES; i++) {
      disk *currdisk = getdisk (i);
      if (currdisk->inited) {
         queueset[diskcnt] = currdisk->queue;
         reqcnt += ioqueue_get_number_of_requests(currdisk->queue);
         diskcnt++;
      }
   }
   assert (diskcnt == numdisks);

   if (reqcnt == 0) {
      fprintf(outputfile, "No disk requests encountered\n");
      return;
   }
/*
   fprintf(outputfile, "Number of extra write disconnects:   %5d  \t%f\n", extra_write_disconnects, ((double) extra_write_disconnects / (double) reqcnt));
*/
   ioqueue_printstats(queueset, numdisks, prefix);

   diskcnt = 0;
   for (i=0; i<MAXDEVICES; i++) {
      disk *currdisk = getdisk (i);
      if (currdisk->inited) {
         set[diskcnt] = i;
         diskcnt++;
      }
   }
   assert (diskcnt == numdisks);

   disk_seek_printstats(set, numdisks, prefix);
   disk_latency_printstats(set, numdisks, prefix);
   disk_transfer_printstats(set, numdisks, prefix);
   disk_acctime_printstats(set, numdisks, prefix);
   disk_interfere_printstats(set, numdisks, prefix);
   disk_buffer_printstats(set, numdisks, prefix);
   fprintf (outputfile, "\n\n");

   if (numdisks <= 1) {
      return;
   }

   for (i=0; i<numdisks; i++) {
      disk *currdisk = getdisk (set[i]);
      if (currdisk->printstats == FALSE) {
	 continue;
      }
      if (ioqueue_get_number_of_requests(currdisk->queue) == 0) {
	 fprintf(outputfile, "No requests for disk #%d\n\n\n", set[i]);
	 continue;
      }
      fprintf(outputfile, "Disk #%d:\n\n", set[i]);
      fprintf(outputfile, "Disk #%d highest block number requested: %d\n", set[i], currdisk->stat.highblkno);
      sprintf(prefix, "Disk #%d ", set[i]);
      ioqueue_printstats(&currdisk->queue, 1, prefix);
      disk_seek_printstats(&set[i], 1, prefix);
      disk_latency_printstats(&set[i], 1, prefix);
      disk_transfer_printstats(&set[i], 1, prefix);
      disk_acctime_printstats(&set[i], 1, prefix);
      disk_interfere_printstats(&set[i], 1, prefix);
      disk_buffer_printstats(&set[i], 1, prefix);
      fprintf (outputfile, "\n\n");
   }
}

