
/*
 * DiskSim Storage Subsystem Simulation Environment (Version 2.0)
 * Revision Authors: Greg Ganger
 * Contributors: Ross Cohen, John Griffin, Steve Schlosser
 *
 * Copyright (c) of Carnegie Mellon University, 1999.
 *
 * Permission to reproduce, use, and prepare derivative works of
 * this software for internal use is granted provided the copyright
 * and "No Warranty" statements are included with all reproductions
 * and derivative works. This software may also be redistributed
 * without charge provided that the copyright and "No Warranty"
 * statements are included in all redistributions.
 *
 * NO WARRANTY. THIS SOFTWARE IS FURNISHED ON AN "AS IS" BASIS.
 * CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER
 * EXPRESSED OR IMPLIED AS TO THE MATTER INCLUDING, BUT NOT LIMITED
 * TO: WARRANTY OF FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY
 * OF RESULTS OR RESULTS OBTAINED FROM USE OF THIS SOFTWARE. CARNEGIE
 * MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT
 * TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
 */

/*
 * DiskSim Storage Subsystem Simulation Environment
 * Authors: Greg Ganger, Bruce Worthington, Yale Patt
 *
 * Copyright (C) 1993, 1995, 1997 The Regents of the University of Michigan 
 *
 * This software is being provided by the copyright holders under the
 * following license. By obtaining, using and/or copying this software,
 * you agree that you have read, understood, and will comply with the
 * following terms and conditions:
 *
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose and without fee or royalty is
 * hereby granted, provided that the full text of this NOTICE appears on
 * ALL copies of the software and documentation or portions thereof,
 * including modifications, that you make.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS," AND COPYRIGHT HOLDERS MAKE NO
 * REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED. BY WAY OF EXAMPLE,
 * BUT NOT LIMITATION, COPYRIGHT HOLDERS MAKE NO REPRESENTATIONS OR
 * WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY PARTICULAR PURPOSE OR
 * THAT THE USE OF THE SOFTWARE OR DOCUMENTATION WILL NOT INFRINGE ANY
 * THIRD PARTY PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS. COPYRIGHT
 * HOLDERS WILL BEAR NO LIABILITY FOR ANY USE OF THIS SOFTWARE OR
 * DOCUMENTATION.
 *
 *  This software is provided AS IS, WITHOUT REPRESENTATION FROM THE
 * UNIVERSITY OF MICHIGAN AS TO ITS FITNESS FOR ANY PURPOSE, AND
 * WITHOUT WARRANTY BY THE UNIVERSITY OF MICHIGAN OF ANY KIND, EITHER
 * EXPRESSED OR IMPLIED, INCLUDING WITHOUT LIMITATION THE IMPLIED
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE REGENTS
 * OF THE UNIVERSITY OF MICHIGAN SHALL NOT BE LIABLE FOR ANY DAMAGES,
 * INCLUDING SPECIAL , INDIRECT, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
 * WITH RESPECT TO ANY CLAIM ARISING OUT OF OR IN CONNECTION WITH THE
 * USE OF OR IN CONNECTION WITH THE USE OF THE SOFTWARE, EVEN IF IT HAS
 * BEEN OR IS HEREAFTER ADVISED OF THE POSSIBILITY OF SUCH DAMAGES
 *
 * The names and trademarks of copyright holders or authors may NOT be
 * used in advertising or publicity pertaining to the software without
 * specific, written prior permission. Title to copyright in this software
 * and any associated documentation will at all times remain with copyright
 * holders.
 */

#include "disksim_global.h"
#include "disksim_iosim.h"
#include "disksim_stat.h"
#include "disksim_bus.h"
#include "disksim_controller.h"


/* Bus types */

#define BUS_TYPE_MIN	1
#define EXCLUSIVE	1
#define INTERLEAVED	2
#define BUS_TYPE_MAX	2

/* Bus arbitration types */

#define BUS_ARB_TYPE_MIN	1
#define SLOTPRIORITY_ARB	1
#define FIFO_ARB		2
#define BUS_ARB_TYPE_MAX	2

/* Bus states */

#define BUS_FREE	1
#define BUS_OWNED	2

typedef struct {
   int    devtype;
   int    devno;
} slot;

typedef struct bus_ev {
   double time;
   int type;
   struct bus_ev *next;
   struct bus_ev *prev;
   int devno;
   int devtype;
   int busno;
   int slotno;
   ioreq_event *delayed_event;
   double wait_start;
} bus_event;

typedef struct bus {
   int          state;
   int          type;
   int          arbtype;
   bus_event   *owners;
   double       arbtime;
   double       readblktranstime;
   double       writeblktranstime;
   bus_event   *arbwinner;
   int		printstats;
   int          depth;
   int          numslots;
   slot        *slots;
   double	lastowned;
   double	runidletime;
   statgen	arbwaitstats;
   statgen	busidlestats;
} bus;


typedef struct businfo {
   struct bus *buses;
   int numbuses;
   int bus_printidlestats;
   int bus_printarbwaitstats;
} businfo_t;

#define numbuses	(disksim->businfo->numbuses)


static struct bus * getbus (int busno)
{
   ASSERT1((busno >= 0) && (busno < numbuses), "busno",busno);
   return (&disksim->businfo->buses[busno]);
}


void bus_print_phys_config()
{
   int i;
   int j;

   for (i = 0; i < numbuses; i++) {
      struct bus *currbus = getbus(i);
      fprintf (outputfile, "Bus #%d\n", i);
      for (j = 0; j <= currbus->numslots; j++) {
         fprintf (outputfile, "Slot #%d: %d %d\n", j, currbus->slots[j].devtype, currbus->slots[j].devno);
      }
   }
}


void bus_set_to_zero_depth (int busno)
{
   struct bus *currbus = getbus(busno);
   currbus->depth = 0;
}


int bus_get_controller_slot (int busno, int ctlno)
{
   int i;
   int slotno = 0;
   struct bus *currbus = getbus (busno);

   for (i = 1; i <= currbus->numslots; i++) {
      if ((currbus->slots[i].devno == ctlno) && (currbus->slots[i].devtype == CONTROLLER)) {
         slotno = i;
         break;
      }
   }
   if (slotno == 0) {
      fprintf(stderr, "Controller not connected to bus but called bus_get_controller_slot: %d %d\n", ctlno, busno);
      exit(0);
   }
   return(slotno);
}


void bus_set_depths()
{
   int depth;
   int busno;
   int slotno;
   intchar ret;
   int i;
   u_int devno;

   for (depth = 0; depth < MAXDEPTH; depth++) {
/*
fprintf (outputfile, "At depth %d\n", depth);
*/
      for (busno = 0; busno < numbuses; busno++) {
         struct bus *currbus = getbus(busno);
/*
fprintf (outputfile, "At busno %d\n", busno);
*/
         if (currbus->depth == depth) {
/*
fprintf (outputfile, "Busno %d is at this depth - %d\n", busno, depth);
*/
            for (slotno = 1; slotno <= currbus->numslots; slotno++) {
/*
fprintf (outputfile, "Deal with slotno %d of busno %d\n", slotno, busno);
*/
               devno = currbus->slots[slotno].devno;
               switch (currbus->slots[slotno].devtype) {
                  case CONTROLLER:
/*
fprintf (outputfile, "Slotno %d contains controller number %d\n", slotno, devno);
*/
				   ret.value = controller_set_depth(devno, busno, depth, slotno);
				   break;
                  case DISK:
/*
fprintf (outputfile, "Slotno %d contains disk number %d\n", slotno, devno);
*/
				   ret.value = device_set_depth(devno, busno, depth, slotno);
				   break;
                  default:         fprintf(stderr, "Invalid device type in bus slot\n");
                                   exit(0);
               }
/*
fprintf (outputfile, "Back from setting device depth\n");
*/
	       if (ret.value != 0) {
/*
fprintf (outputfile, "Non-zero return - %x\n", ret.value);
*/
		  for (i = 0; i < MAXDEPTH; i++) {
		     if (ret.byte[i] != 0) {
                        struct bus *currbus2 = getbus(ret.byte[i]);
/*
fprintf (outputfile, "Set busno %x to have depth %d\n", ret.byte[i], (depth+1));
*/
		        currbus2->depth = depth + 1;
		     }
		  }
	       }
	    }
         }
      }
   }
}


double bus_get_transfer_time(int busno, int bcount, int read)
{
   double blktranstime;
   struct bus *currbus = getbus(busno);

   blktranstime = (read) ? currbus->readblktranstime : currbus->writeblktranstime;
   return((double) bcount * blktranstime);
}


/*
  **-bus_delay

  Adds a BUS_DELAY_COMPLETE event to the intq.
  */

void bus_delay(int busno, int devtype, int devno, double delay, ioreq_event *curr)
{
   bus_event *tmp = (bus_event *) getfromextraq();
   tmp->type = BUS_DELAY_COMPLETE;
   tmp->time = simtime + delay;
   tmp->devno = devno;
   tmp->busno = busno;
   tmp->devtype = devtype;
   tmp->delayed_event = curr;
   addtointq((event *) tmp);
}


void bus_event_arrive(ioreq_event *ptr)
{
   bus_event *curr = (bus_event *) ptr;

   if (curr->type == BUS_OWNERSHIP_GRANTED) {
      struct bus *currbus = getbus(curr->busno);
      ASSERT(curr == currbus->arbwinner);
      currbus->arbwinner = NULL;
      stat_update (&currbus->arbwaitstats, (simtime - curr->wait_start));
   }

   switch (curr->devtype) {

      case CONTROLLER:
         if (curr->type == BUS_DELAY_COMPLETE) {
	    controller_bus_delay_complete (curr->devno, curr->delayed_event, curr->busno);
         } else {
	    controller_bus_ownership_grant (curr->devno, curr->delayed_event, curr->busno, (simtime - curr->wait_start));
         }
	 break;

      case DISK:
         if (curr->type == BUS_DELAY_COMPLETE) {
	    device_bus_delay_complete (curr->devno, curr->delayed_event, curr->busno);
         } else {
	    device_bus_ownership_grant (curr->devno, curr->delayed_event, curr->busno, (simtime - curr->wait_start));
         }
	 break;

      default:
	 fprintf(stderr, "Unknown device type at bus_event_arrive: %d, type %d\n", curr->devtype, curr->type);
	 exit(0);
   }
   addtoextraq((event *)curr);
}


static bus_event * bus_fifo_arbitration(bus *currbus)
{
   bus_event *ret = currbus->owners;
   ASSERT(currbus->owners != NULL);
   currbus->owners = ret->next;
   return(ret);
}


static bus_event * bus_slotpriority_arbitration(bus *currbus)
{
   bus_event *ret = currbus->owners;

   ASSERT(currbus->owners != NULL);
   if (currbus->owners->next == NULL) {
      currbus->owners = NULL;
      return(ret);
   } else {
      bus_event *trv = ret;
      while (trv->next) {
         if (trv->next->slotno > ret->slotno) {
            ret = trv->next;
         }
         trv = trv->next;
      }
      if (ret == currbus->owners) {
         currbus->owners = ret->next;
         ret->next = NULL;
      } else {
	 trv = currbus->owners;
         while ((trv->next != NULL) && (trv->next != ret)) {
            trv = trv->next;
         }
	 ASSERT(trv->next == ret);
         trv->next = ret->next;
         ret->next = NULL;
      }
   }
   return(ret);
}


static bus_event * bus_arbitrate_for_ownership(bus *currbus)
{
   switch (currbus->arbtype) {

      case SLOTPRIORITY_ARB:
          return(bus_slotpriority_arbitration(currbus));

      case FIFO_ARB:
          return(bus_fifo_arbitration(currbus));

      default:
          fprintf(stderr, "Unrecognized bus arbitration type in bus_arbitrate_for_ownership\n");
          exit(0);
   }
}


/*
  **- bus_ownership_get

  Returns TRUE if the bus is immediately acquired, which only happens for an interleaved bus.
    In this case, this function does nothing else.

  Returns FALSE if there is a delay (always true with SCSI)
    A bus_event of type BUS_OWNERSHIP_GRANTED to this request is created
    If the bus is initially BUS_BUSY, the event is added to the buses.owners queue
	If the bus is initially BUS_FREE, its state is changed to BUS_BUSY and is granted
	  to this request.  (I think:  buses.arbwinner is set to this event).  The event
	  is scheduled on the intq for the current time plus buses.arbtime.

  Note that I don't think this really how SCSI works:  If a higher priority device
  arbitrates for the bus within the arbitration time, it will win.  Maybe this window
  is small enough to be ignored for this level of simulation.

  */

int bus_ownership_get(int busno, int slotno, ioreq_event *curr)
{
   bus_event *tmp;
   struct bus *currbus = getbus (busno);

   if (currbus->type == INTERLEAVED) {
      return(TRUE);
   }
   tmp = (bus_event *) getfromextraq();
   tmp->type = BUS_OWNERSHIP_GRANTED;
   tmp->devno = currbus->slots[slotno].devno;
   tmp->busno = busno;
   tmp->delayed_event = curr;
   tmp->wait_start = simtime;
   if (currbus->state == BUS_FREE) {
      tmp->devtype = currbus->slots[slotno].devtype;
      tmp->time = simtime + currbus->arbtime;
/*
fprintf (outputfile, "Granting ownership immediately - devno %d, devtype %d, busno %d\n", tmp->devno, tmp->devtype, tmp->busno);
*/
      currbus->arbwinner = tmp;
      addtointq((event *) tmp);
      currbus->state = BUS_OWNED;
      currbus->runidletime += simtime - currbus->lastowned;
      stat_update (&currbus->busidlestats, (simtime - currbus->lastowned));
   } else {
      tmp->slotno = slotno;
/*
fprintf (outputfile, "Must wait for bus to become free - devno %d, slotno %d, busno %d\n", tmp->devno, tmp->slotno, tmp->busno);
*/
      tmp->next = NULL;
      if (currbus->owners) {
         bus_event *tail = currbus->owners;
         while (tail->next) {
            tail = tail->next;
         }
         tail->next = tmp;
      } else {
         currbus->owners = tmp;
      }
   }
   return(FALSE);
}


void bus_ownership_release(int busno)
{
   bus_event *tmp;
   struct bus *currbus = getbus(busno);
/*
fprintf (outputfile, "Bus ownership being released - %d\n", busno);
*/
   ASSERT(currbus->arbwinner == NULL);
   if (currbus->owners == NULL) {
/*
fprintf (outputfile, "Bus has become free - %d\n", busno);
*/
      currbus->state = BUS_FREE;
      currbus->lastowned = simtime;
   } else {
      tmp = bus_arbitrate_for_ownership(currbus);
      tmp->devtype = currbus->slots[tmp->slotno].devtype;
      tmp->time = simtime + currbus->arbtime;
/*
fprintf (outputfile, "Bus ownership transfered - devno %d, devtype %d, busno %d\n", tmp->devno, tmp->devtype, tmp->busno);
*/
      currbus->arbwinner = tmp;
      addtointq((event *) tmp);
   }
}


void bus_remove_from_arbitration(int busno, ioreq_event *curr)
{
   bus_event *tmp;
   bus_event *trv;
   struct bus *currbus = getbus(busno);

   if (currbus->arbwinner) {
      if (curr == currbus->arbwinner->delayed_event) {
         if (removefromintq((event *)currbus->arbwinner) != TRUE) {
            fprintf(stderr, "Ongoing arbwinner not in internal queue, at bus_remove_from_arbitration\n");
            exit(0);
         }
         addtoextraq((event *) currbus->arbwinner);
         currbus->arbwinner = NULL;
         bus_ownership_release(busno);
         return;
      }
   }
   if (curr == currbus->owners->delayed_event) {
      tmp = currbus->owners;
      currbus->owners = tmp->next;
   } else {
      trv = currbus->owners;
      while ((trv->next) && (curr != trv->next->delayed_event)) {
         trv = trv->next;
      }
      ASSERT(trv->next != NULL);
      tmp = trv->next;
      trv->next = tmp->next;
   }
   addtoextraq((event *) tmp);
}


void bus_deliver_event(int busno, int slotno, ioreq_event *curr)
{
   int devno;
   struct bus *currbus = getbus(busno);

   ASSERT1((slotno > 0) && (slotno <= currbus->numslots), "slotno", slotno);

/*
fprintf (outputfile, "In middle of bus_deliver_event\n");
*/
   devno = currbus->slots[slotno].devno;
   switch (currbus->slots[slotno].devtype) {

      case CONTROLLER:  controller_event_arrive(devno, curr);
                        break;

      case DISK:        ASSERT(devno == curr->devno);
                        device_event_arrive(curr);
                        break;

      default:          fprintf(stderr, "Invalid device type in bus slot\n");
                        exit(0);
   }
}


int bus_get_data_transfered(ioreq_event *curr, int depth)
{
   intchar slotno;
   intchar busno;
   int checkdevno;
   int ret;
/*
fprintf (outputfile, "Entered bus_get_data_transfered - devno %d, depth %d, busno %x, slotno %x\n", curr->devno, depth, curr->busno, curr->slotno);
*/
   busno.value = curr->busno;
   slotno.value = curr->slotno;
   while (depth) {
      struct bus *currbus = getbus(busno.byte[depth]);
      checkdevno = currbus->slots[(slotno.byte[depth] >> 4)].devno;
      switch (currbus->slots[(slotno.byte[depth] >> 4)].devtype) {
   
         case CONTROLLER:       ret = controller_get_data_transfered(checkdevno, curr->devno);
                                break;
   
         default:       fprintf(stderr, "Invalid device type in bus_get_data_transfered\n");
                        exit(0);
      }
      if (ret != -1) {
         return(ret);
      }
      depth--;
   }
   return(-1);
}


/* temporary global variables */
static int disksim_bus_printidlestats;
static int disksim_bus_printarbwaitstats;


void bus_read_toprints(FILE *parfile)
{
   getparam_bool(parfile, "Print bus idle stats?", &disksim_bus_printidlestats);
   getparam_bool(parfile, "Print bus arbwait stats?", &disksim_bus_printarbwaitstats);
}


void bus_read_specs(FILE *parfile)
{
   struct bus *currbus;
   int specno_expected = 1;
   int specno;
   int busno = 0;
   int copies;
   int i;

   disksim->businfo = DISKSIM_malloc (sizeof(businfo_t));
   bzero ((char *)disksim->businfo, sizeof(businfo_t));

   disksim->businfo->bus_printidlestats = disksim_bus_printidlestats;
   disksim->businfo->bus_printarbwaitstats = disksim_bus_printarbwaitstats;

   getparam_int(parfile, "\nNumber of buses", &numbuses, 1, 0, 0);
   if (numbuses == 0) {
      return;
   }

   disksim->businfo->buses = DISKSIM_malloc (numbuses*sizeof(struct bus));

   while (busno < numbuses) {
      if (fscanf(parfile, "\nBus Spec #%d\n", &specno) != 1) {
         fprintf(stderr, "Error reading spec number\n");
         exit(0);
      }
      if (specno != specno_expected) {
         fprintf(stderr, "Unexpected value for bus spec number: specno %d, expected %d\n", specno, specno_expected);
         exit(0);
      }
      fprintf (outputfile, "\nBus Spec #%d\n", specno);
      specno_expected++;

      getparam_int(parfile, "# buses with Spec", &copies, 1, 0, 0);
      if (copies == 0) {
         copies = numbuses - busno;
      }

      if ((busno + copies) > numbuses) {
         fprintf(stderr, "Too many bus specifications provided\n");
         exit(0);
      }

      currbus = getbus (busno);
      getparam_int(parfile, "Bus Type", &currbus->type, 3, BUS_TYPE_MIN, BUS_TYPE_MAX);
      getparam_int(parfile, "Arbitration type", &currbus->arbtype, 3, BUS_ARB_TYPE_MIN, BUS_ARB_TYPE_MAX);
      getparam_double(parfile, "Arbitration time", &currbus->arbtime, 1, (double) 0.0, (double) 0.0);
      getparam_double(parfile, "Read block transfer time", &currbus->readblktranstime, 1, (double) 0.0, (double) 0.0);
      getparam_double(parfile, "Write block transfer time", &currbus->writeblktranstime, 1, (double) 0.0, (double) 0.0);
      getparam_int(parfile, "Print stats for bus", &currbus->printstats, 3, 0, 1);

      for (i=busno; i<(busno+copies); i++) {
         struct bus *currbus2 = getbus(i);
         currbus2->type = currbus->type;
         currbus2->arbtype = currbus->arbtype;
         currbus2->arbtime = currbus->arbtime;
         currbus2->readblktranstime = currbus->readblktranstime;
         currbus2->writeblktranstime = currbus->writeblktranstime;
         currbus2->printstats = currbus->printstats;
         currbus2->owners = NULL;
         currbus2->depth = -1;
      }
      busno += copies;
   }
}


void bus_read_physorg(FILE *parfile)
{
   int i = 0;
   int j;
   int k;
   int l;
   int cnt;
   int busno;
   char tmp[20];
   char tmpvals[20];
   int numrets;
   int slotcnt;
   char slots[20];
   char devices[40];
   char op;
   int val;
   int endval;
   int devtype;
   int slotadd;
   struct bus *currbus;

   while (i < numbuses) {

      if (fscanf(parfile, "\nBus #%s\n", tmpvals) != 1) {
         fprintf(stderr, "Error reading bus portion of physical organization\n");
         exit(0);
      }
      if ((numrets = sscanf(tmpvals, "%d%s", &busno, tmp)) < 1) {
         fprintf(stderr, "Error parsing bus portion of physical organization\n");
         exit(0);
      } else if (numrets == 1) {
	 /* sprintf(tmp, ""); */
         tmp[0] = (char) 0;
      }
      if (busno != (i + 1)) {
         fprintf(stderr, "Bus numbers in physical organization appear out of order\n");
         exit(0);
      }
      fprintf (outputfile, "\nBus #%d%s\n", busno, tmp);

      j = i + 1;
      if (strcmp(tmp, "") != 0) {
         if (sscanf(tmp, "%c%d", &op, &val) != 2) {
            fprintf(stderr, "Illegal value for 'Bus #' %d in physical organization\n", busno);
            exit(0);
         }
         if ((op != '-') || (val <= busno) || (val > numbuses)) {
            fprintf(stderr, "Illegal value for 'Bus #' %d in physical organization: op %c, val %d, numbuses %d\n", busno, op, val, numbuses);
            exit(0);
         }
         j += val - busno;
      }

      getparam_int(parfile, "# of utilized slots", &slotcnt, 3, 1, MAXSLOTS);

      for (k = i; k < j; k++) {
         struct bus *currbus2 = getbus(k);
         currbus2->numslots = slotcnt;
         if (!(currbus2->slots = (slot *) DISKSIM_malloc((slotcnt+1) * (sizeof(slot))))) {
            fprintf(stderr, "Problem allocating slots for bus #%d\n", busno);
            exit(0);
         }
      }
      currbus = getbus(i);
      if (currbus->numslots == 0)
	 continue;
      cnt = 1;
      while (cnt <= slotcnt) {
         if (fscanf(parfile, "Slots: %s %s\n", devices, slots) != 2) {
            fprintf(stderr, "Error reading 'Slots' for bus %d\n", busno);
            exit(0);
         }
	 if (strcmp(devices, "Controllers") == 0) {
	    devtype = CONTROLLER;
	 } else if (strcmp(devices, "Devices") == 0) {
	    devtype = DEVICE;
	 } else {
            fprintf(stderr, "Invalid value for 'Slots' for bus %d\n", busno);
            exit(0);
	 }
         if ((slots[0] != '#') && ((slots[0] > '9') || (slots[0] < '1'))) {
            fprintf(stderr, "Invalid value for 'Slots' for bus %d\n", busno);
            exit(0);
         }

         if (slots[0] == '#') {
            slotadd = 0;
            if (strcmp(slots, "#") != 0) {
               if (sscanf(slots, "#%d", &val) != 1) {
                  fprintf(stderr, "Invalid value for 'Slots' for bus %d\n", busno);
                  exit(0);
               }
               slotadd = val;
            }
	    for (k=i; k < j; k++) {
               struct bus *currbus2 = getbus(k);
	       currbus2->slots[cnt].devtype = devtype;
               currbus2->slots[cnt].devno = k + slotadd;
	    }
	    cnt++;
         } else {
	    /* sprintf(tmp, ""); */
            tmp[0] = (char) 0;
            sscanf(slots, "%d%s", &val, tmp);
	    if (strcmp(tmp, "") == 0) {
	       for (k = i; k < j; k++) {
                  struct bus *currbus2 = getbus(k);
                  currbus2->slots[cnt].devtype = devtype;
	          currbus2->slots[cnt].devno = val - 1;
               }
	       cnt++;
	    } else if (tmp[0] == '-') {
	       if (sscanf(tmp, "-%d", &endval) != 1) {
		  fprintf(stderr, "A problem with parsing 'Slots' for bus %d\n", busno);
		  exit(0);
	       }
	       if ((endval - val + 1) > slotcnt) {
		  fprintf(stderr, "Too many slot descriptions provided for bus %d\n", busno);
		  exit(0);
	       }
	       for (k = i; k < j; k++) {
		  for (l = cnt; l <= (cnt + endval - val); l++) {
                     struct bus *currbus2 = getbus(k);
                     currbus2->slots[l].devtype = devtype;
	             currbus2->slots[l].devno = val + (l - cnt) - 1;
/*
fprintf (outputfile, "Slot %d contains devtype %d number %d\n", l, devtype, (val + l - cnt - 1));
*/
		  }
               }
	       cnt += endval - val + 1;
	    } else {
               fprintf(stderr, "Invalid value for 'Slots' for bus %d\n", busno);
               exit(0);
	    }
         }
	 fprintf (outputfile, "Slots: %s %s\n", devices, slots);
      }

      i = j;
   }
}


void bus_param_override(char *paramname, char *paramval, int first, int last)
{
   fprintf(stderr, "No param overrides currently supported at bus_param_override\n");
   exit(0);
}


void bus_resetstats()
{
   int i;

   for (i=0; i<numbuses; i++) {
      struct bus *currbus = getbus(i);
      currbus->lastowned = simtime;
      currbus->runidletime = 0.0;
      stat_reset (&currbus->busidlestats);
      stat_reset (&currbus->arbwaitstats);
   }
}


void bus_setcallbacks ()
{
}


void bus_initialize()
{
   int i;

   StaticAssert (sizeof(bus_event) <= DISKSIM_EVENT_SIZE);
   for (i=0; i<numbuses; i++) {
      struct bus *currbus = getbus(i);
      currbus->state = BUS_FREE;
      addlisttoextraq((event **) &currbus->owners);
      stat_initialize (statdeffile, "Arbitration wait time", &currbus->arbwaitstats);
      stat_initialize (statdeffile, "Bus idle period length", &currbus->busidlestats);
   }
   bus_resetstats();
}


void bus_printstats()
{
   int i;
   char prefix[81];

   fprintf (outputfile, "\nBUS STATISTICS\n");
   fprintf (outputfile, "--------------\n\n");

   for (i=0; i<numbuses; i++) {
      struct bus *currbus = getbus(i);
      if (currbus->printstats) {
         sprintf (prefix, "Bus #%d ", i);
         fprintf (outputfile, "Bus #%d\n", i);
         fprintf (outputfile, "Bus #%d Total utilization time: \t%.2f   \t%6.5f\n", i, (simtime - warmuptime - currbus->runidletime), ((simtime - warmuptime - currbus->runidletime) / (simtime - warmuptime)));
         if (disksim->businfo->bus_printidlestats) {
            stat_print (&currbus->busidlestats, prefix);
         }
         if (disksim->businfo->bus_printarbwaitstats) {
            fprintf (outputfile, "Bus #%d Number of arbitrations: \t%d\n", i, stat_get_count (&currbus->arbwaitstats));
            stat_print (&currbus->arbwaitstats, prefix);
         }
         fprintf (outputfile, "\n");
      }
   }
}


void bus_cleanstats()
{
   int i;

   for (i=0; i<numbuses; i++) {
      struct bus *currbus = getbus(i);
      if (currbus->state == BUS_FREE) {
         currbus->runidletime += simtime - currbus->lastowned;
         stat_update (&currbus->busidlestats, (simtime - currbus->lastowned));
      }
   }
}

