
/*
 * DiskSim Storage Subsystem Simulation Environment (Version 2.0)
 * Revision Authors: Greg Ganger
 * Contributors: Ross Cohen, John Griffin, Steve Schlosser
 *
 * Copyright (c) of Carnegie Mellon University, 1999.
 *
 * Permission to reproduce, use, and prepare derivative works of
 * this software for internal use is granted provided the copyright
 * and "No Warranty" statements are included with all reproductions
 * and derivative works. This software may also be redistributed
 * without charge provided that the copyright and "No Warranty"
 * statements are included in all redistributions.
 *
 * NO WARRANTY. THIS SOFTWARE IS FURNISHED ON AN "AS IS" BASIS.
 * CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER
 * EXPRESSED OR IMPLIED AS TO THE MATTER INCLUDING, BUT NOT LIMITED
 * TO: WARRANTY OF FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY
 * OF RESULTS OR RESULTS OBTAINED FROM USE OF THIS SOFTWARE. CARNEGIE
 * MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT
 * TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
 */

/*
 * DiskSim Storage Subsystem Simulation Environment
 * Authors: Greg Ganger, Bruce Worthington, Yale Patt
 *
 * Copyright (C) 1993, 1995, 1997 The Regents of the University of Michigan 
 *
 * This software is being provided by the copyright holders under the
 * following license. By obtaining, using and/or copying this software,
 * you agree that you have read, understood, and will comply with the
 * following terms and conditions:
 *
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose and without fee or royalty is
 * hereby granted, provided that the full text of this NOTICE appears on
 * ALL copies of the software and documentation or portions thereof,
 * including modifications, that you make.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS," AND COPYRIGHT HOLDERS MAKE NO
 * REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED. BY WAY OF EXAMPLE,
 * BUT NOT LIMITATION, COPYRIGHT HOLDERS MAKE NO REPRESENTATIONS OR
 * WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY PARTICULAR PURPOSE OR
 * THAT THE USE OF THE SOFTWARE OR DOCUMENTATION WILL NOT INFRINGE ANY
 * THIRD PARTY PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS. COPYRIGHT
 * HOLDERS WILL BEAR NO LIABILITY FOR ANY USE OF THIS SOFTWARE OR
 * DOCUMENTATION.
 *
 *  This software is provided AS IS, WITHOUT REPRESENTATION FROM THE
 * UNIVERSITY OF MICHIGAN AS TO ITS FITNESS FOR ANY PURPOSE, AND
 * WITHOUT WARRANTY BY THE UNIVERSITY OF MICHIGAN OF ANY KIND, EITHER
 * EXPRESSED OR IMPLIED, INCLUDING WITHOUT LIMITATION THE IMPLIED
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE REGENTS
 * OF THE UNIVERSITY OF MICHIGAN SHALL NOT BE LIABLE FOR ANY DAMAGES,
 * INCLUDING SPECIAL , INDIRECT, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
 * WITH RESPECT TO ANY CLAIM ARISING OUT OF OR IN CONNECTION WITH THE
 * USE OF OR IN CONNECTION WITH THE USE OF THE SOFTWARE, EVEN IF IT HAS
 * BEEN OR IS HEREAFTER ADVISED OF THE POSSIBILITY OF SUCH DAMAGES
 *
 * The names and trademarks of copyright holders or authors may NOT be
 * used in advertising or publicity pertaining to the software without
 * specific, written prior permission. Title to copyright in this software
 * and any associated documentation will at all times remain with copyright
 * holders.
 */

#include "disksim_global.h"
#include "disksim_ioface.h"
#include "disksim_pfface.h"
#include "disksim_iotrace.h"

#ifdef SUPPORT_CHECKPOINTS
#include <unistd.h>
#include <fcntl.h>
#include <sys/mman.h>
#endif


disksim_t *disksim = NULL;

/* legacy hack for HPL traces... */
#define PRINT_TRACEFILE_HEADER	FALSE


/*** Functions to allocate and deallocate empty event structures ***/

/* Creates new, empty events (using malloc) and puts them on the extraq. */
/* Called by getfromextraq when the queue is found to be empty.          */

static void allocateextra ()
{
   int i;
   event *temp = NULL;

   StaticAssert (sizeof(event) == DISKSIM_EVENT_SIZE);
   if ((temp = (event *)DISKSIM_malloc(ALLOCSIZE)) == NULL) {
      fprintf (stderr, "Error allocating space for events\n");
      exit(0);
   }
   for (i=0; i<((ALLOCSIZE/DISKSIM_EVENT_SIZE)-1); i++) {
      temp[i].next = &temp[i+1];
   }
   temp[((ALLOCSIZE/DISKSIM_EVENT_SIZE)-1)].next = disksim->extraq;
   disksim->extraq = temp;
   disksim->extraqlen = ALLOCSIZE / DISKSIM_EVENT_SIZE;
}


/* Deallocates an event structure, adding it to the extraq free pool. */
  
void addtoextraq (event *temp)
{
   if (temp == NULL) {
      return;
   }
   temp->next = disksim->extraq;
   temp->prev = NULL;
   disksim->extraq = temp;
   disksim->extraqlen++;
}


/* Allocates an event structure from the extraq free pool; if empty, */
/* calls allocateextra to create some more.                          */

event * getfromextraq ()
{
   event *temp = NULL;

   temp = disksim->extraq;
   if (disksim->extraqlen == 0) {
      allocateextra();
      temp = disksim->extraq;
      disksim->extraq = disksim->extraq->next;
   } else if (disksim->extraqlen == 1) {
      disksim->extraq = NULL;
   } else {
      disksim->extraq = disksim->extraq->next;
   }
   disksim->extraqlen--;
   temp->next = NULL;
   temp->prev = NULL;
   return(temp);
}


/* Deallocates a list of event structures to the extraq free pool. */

void addlisttoextraq (event **headptr)
{
   event *tmp;

   while ((tmp = *headptr)) {
      *headptr = tmp->next;
      addtoextraq(tmp);
   }
   *headptr = NULL;
}


/* Returns a pointer to a copy of event passed as a parameter.  */

event *event_copy (event *orig)
{
   event *new = getfromextraq();
   memmove((char *)new, (char *)orig, DISKSIM_EVENT_SIZE);
/* bcopy ((char *)orig, (char *)new, DISKSIM_EVENT_SIZE); */
   return((event *) new);
}


/*** Functions to manipulate intq, the queue of scheduled events ***/

/* Prints the intq to the output file, presumably for debug.  */

#if 0
static void printintq ()
{
   event *tmp;
   int i = 0;

   tmp = disksim->intq;
   while (tmp != NULL) {
      i++;
      fprintf (outputfile, "Item #%d: time %f, type %d\n", i, tmp->time, tmp->type);
      tmp = tmp->next;
   }
}
#endif


/* Add an event to the intq.  The "time" field indicates when the event is */
/* scheduled to occur, and the intq is maintained in ascending time order. */

void addtointq (event *temp)
{
   event *run = NULL;

   if ((temp->time + DISKSIM_TIME_THRESHOLD) < simtime) {
      fprintf(stderr, "Attempting to addtointq an event whose time has passed\n");
      fprintf(stderr, "simtime %f, curr->time %f, type = %d\n", simtime, temp->time, temp->type);
      exit(0);
   }

   if (disksim->intq == NULL) {
      disksim->intq = temp;
      temp->next = NULL;
      temp->prev = NULL;
   } else if (temp->time < disksim->intq->time) {
      temp->next = disksim->intq;
      disksim->intq->prev = temp;
      disksim->intq = temp;
      temp->prev = NULL;
   } else {
      run = disksim->intq;
      while (run->next != NULL) {
         if (temp->time < run->next->time) {
            break;
         }
         run = run->next;
      }
      temp->next = run->next;
      run->next = temp;
      temp->prev = run;
      if (temp->next != NULL) {
         temp->next->prev = temp;
      }
   }
}


/* Retrieves the next scheduled event from the head of the intq. */

static event * getfromintq ()
{
   event *temp = NULL;

   if (disksim->intq == NULL) {
      return(NULL);
   }
   temp = disksim->intq;
   disksim->intq = disksim->intq->next;
   if (disksim->intq != NULL) {
      disksim->intq->prev = NULL;
   }
   temp->next = NULL;
   temp->prev = NULL;
   return(temp);
}


/* Removes a given event from the intq, thus descheduling it.  Returns */
/* TRUE if the event was found, FALSE if it was not.                   */

int removefromintq (event *curr)
{
   event *tmp;

   tmp = disksim->intq;
   while (tmp != NULL) {
      if (tmp == curr) {
         break;
      }
      tmp = tmp->next;
   }
   if (tmp == NULL) {
      return(FALSE);
   }
   if (curr->next != NULL) {
      curr->next->prev = curr->prev;
   }
   if (curr->prev == NULL) {
      disksim->intq = curr->next;
   } else {
      curr->prev->next = curr->next;
   }
   curr->next = NULL;
   curr->prev = NULL;
   return(TRUE);
}


/*** Functions to initialize the system and print statistics ***/

void scanparam_int (char *parline, char *parname, int *parptr,
                    int parchecks, int parminval, int parmaxval)
{
   if (sscanf(parline, "%d", parptr) != 1) {
      fprintf(stderr, "Error reading '%s'\n", parname);
      exit(0);
   }
   if (((parchecks & 1) && (*parptr < parminval)) || ((parchecks & 2) && (*parptr > parmaxval))) {
      fprintf(stderr, "Invalid value for '%s': %d\n", parname, *parptr);
      exit(0);
   }
}


void getparam_int (FILE *parfile, char *parname, int *parptr,
                   int parchecks, int parminval, int parmaxval)
{
   char line[201];

   sprintf(line, "%s: %s", parname, "%d");
   if (fscanf(parfile, line, parptr) != 1) {
      fprintf(stderr, "Error reading '%s'\n", parname);
      exit(0);
   }
   fgets(line, 200, parfile); /* this allows comments, kinda ugly hack */
   if (((parchecks & 1) && (*parptr < parminval)) || ((parchecks & 2) && (*parptr > parmaxval))) {
      fprintf(stderr, "Invalid value for '%s': %d\n", parname, *parptr);
      exit(0);
   }
   fprintf (outputfile, "%s: %d\n", parname, *parptr);
}


void getparam_double (FILE *parfile, char *parname, double *parptr,
                      int parchecks, double parminval, double parmaxval)
{
   char line[201];

   sprintf(line, "%s: %s", parname, "%lf");
   if (fscanf(parfile, line, parptr) != 1) {
      fprintf(stderr, "Error reading '%s'\n", parname);
      assert(0);
   }
   fgets(line, 200, parfile);
   if (((parchecks & 1) && (*parptr < parminval)) || ((parchecks & 2) && (*parptr > parmaxval)) || ((parchecks & 4) && (*parptr <= parminval))) {
      fprintf(stderr, "Invalid value for '%s': %f\n", parname, *parptr);
      assert(0);
   }
   fprintf (outputfile, "%s: %f\n", parname, *parptr);
}


void getparam_bool (FILE *parfile, char *parname, int *parptr)
{
   char line[201];

   sprintf(line, "%s %s", parname, "%d");
   if (fscanf(parfile, line, parptr) != 1) {
      fprintf(stderr, "Error reading '%s'\n", parname);
      exit(0);
   }
   fgets(line, 200, parfile);
   if ((*parptr != TRUE) && (*parptr != FALSE)) {
      fprintf(stderr, "Invalid value for '%s': %d\n", parname, *parptr);
      exit(0);
   }
   fprintf (outputfile, "%s %d\n", parname, *parptr);
}


void resetstats ()
{
   if (disksim->external_control | disksim->synthgen | disksim->iotrace) {
      io_resetstats();
   }
   if (disksim->synthgen) {
      pf_resetstats();
   }
}


static void stat_warmup_done (timer_event *timer)
{
   warmuptime = simtime;
   resetstats();
   addtoextraq((event *)timer);
}


static void readparams (FILE *parfile)
{
   char seed[200];
   double warmup;
   int newend = -1;

   if (fscanf(parfile, "\nByte Order (Endian): %s\n", seed) != 1) {
      fprintf(stderr, "Error reading 'Byte Order:'\n");
      exit(0);
   }
   if (strcmp(seed, "Little") == 0) {
      disksim->endian = _LITTLE_ENDIAN;
   } else if (strcmp(seed, "Big") == 0) {
      disksim->endian = _BIG_ENDIAN;
   } else {
      fprintf(stderr, "Invalid byte ordering scheme - %s\n", seed);
      exit(0);
   }

   {
      int tmp = 0x11223344;
      char *tmp2 = (char *) &tmp;
      newend = (tmp2[0] == 0x44) ? _LITTLE_ENDIAN : ((tmp2[0] == 0x11) ? _BIG_ENDIAN : -1);
   }

   if (newend != -1) {
      fprintf (outputfile, "Byte Order (Endian): %s (using detected value: %s)\n", seed, ((newend == _LITTLE_ENDIAN) ? "Little" : "Big"));
      disksim->endian = newend;
   } else {
      fprintf (outputfile, "Byte Order (Endian): %s\n", seed);
   }

   if (fscanf(parfile, "Init Seed: %s\n", seed) != 1) {
      fprintf(stderr, "Error reading 'Seed:'\n");
      exit(0);
   }
   if (strcmp(seed, "TIME") == 0) {
      disksim->seedval = (int) DISKSIM_time();
      DISKSIM_srand48(disksim->seedval);
      fprintf (outputfile, "Init Seed: PID (%d)\n", disksim->seedval);
   } else if (sscanf(seed, "%d", &disksim->seedval) != 1) {
      fprintf(stderr, "Invalid init seed value: %s\n", seed);
      exit(0);
   } else {
      DISKSIM_srand48(disksim->seedval);
      fprintf (outputfile, "Init Seed: %d\n", disksim->seedval);
   }

   if (fscanf(parfile, "Real Seed: %s\n", seed) != 1) {
      fprintf(stderr, "Error reading 'Seed:'\n");
      exit(0);
   }
   if (strcmp(seed, "TIME") == 0) {
      disksim->seedval = (int) DISKSIM_time();
      fprintf (outputfile, "Real Seed: TIME (%d)\n", disksim->seedval);
   } else if (sscanf(seed, "%d", &disksim->seedval) != 1) {
      fprintf(stderr, "Invalid init seed value: %s\n", seed);
      exit(0);
   } else {
      fprintf (outputfile, "Real Seed: %d\n", disksim->seedval);
   }

   if (fscanf(parfile, "Statistic warm-up period: %lf %s\n", &warmup, seed) != 2) {
      fprintf(stderr, "Error reading 'Statistic warm-up period:'\n");
      exit(0);
   }
   fprintf (outputfile, "Statistic warm-up period: %f %s\n", warmup, seed);
   if (warmup > 0.0) {
      if (strcmp(seed, "seconds") == 0) {
         disksim->warmup_event = (timer_event *) getfromextraq();
         disksim->warmup_event->type = TIMER_EXPIRED;
         disksim->warmup_event->time = warmup * (double) 1000.0;
         disksim->warmup_event->func = &disksim->timerfunc_disksim;
      } else if (strcmp(seed, "I/Os") == 0) {
         disksim->warmup_iocnt = ceil(warmup) + 1;
      } else {
         fprintf(stderr, "Invalid type of statistic warm-up period: %s\n", seed);
         exit(0);
      }
   }

   if (fscanf(parfile, "Checkpoint to %s every: %lf %s\n", disksim->checkpointfilename, &warmup, seed) != 3) {
      fprintf(stderr, "Error reading 'Checkpoint to <filename> every:'\n");
      exit(0);
   }
   fprintf (outputfile, "Checkpoint to %s every: %f %s\n", disksim->checkpointfilename, warmup, seed);
   if (warmup > 0.0) {
      if (strcmp(seed, "seconds") == 0) {
         disksim->checkpoint_interval = warmup * (double) 1000.0;
      } else if (strcmp(seed, "I/Os") == 0) {
         disksim->checkpoint_iocnt = ceil(warmup) + 1;
      } else {
         fprintf(stderr, "Invalid type of statistic warm-up period: %s\n", seed);
         exit(0);
      }
   }

   if (fscanf(parfile, "Stat (dist) definition file: %s\n", seed) != 1) {
      fprintf(stderr, "Error reading 'Stat (dist) definition file:'\n");
      exit(0);
   }
   if ((statdeffile = fopen(seed, "r")) == NULL) {
      fprintf(stderr, "Statdeffile %s cannot be opened for read access\n", seed);
      exit(0);
   }
   fprintf (outputfile, "Stat (dist) definition file: %s\n", seed);

   if (fscanf(parfile, "Output file for trace of I/O requests simulated: %s\n", seed) != 1) {
      fprintf(stderr, "Error reading 'Output file for I/O requests simulated:'\n");
      exit(0);
   }
   if ((strcmp(seed, "0") != 0) && (strcmp(seed, "null") != 0)) {
      if ((outios = fopen(seed, "w")) == NULL) {
         fprintf(stderr, "Outios %s cannot be opened for write access\n", seed);
         exit(0);
      }
   }
   fprintf(outputfile, "Output file for I/O requests simulated: %s\n", seed);

   if (strlen(seed) <= 255) {
      strcpy (disksim->outiosfilename, seed);
   } else {
      fprintf (stderr, "Name of outios file is too long (>255 bytes); checkpointing disabled\n");
      disksim->checkpoint_disable = 1;
   }

   if (disksim->external_control | disksim->synthgen | disksim->iotrace) {
      io_readparams(parfile);
   }
   if (disksim->synthgen) {
      pf_readparams(parfile);
   }
}


static void doparamoverrides (char **overrides, int overcnt)
{
   int i;
   int first, last;
   int vals;

   for (i=0; i<overcnt; i+=4) {
      vals = sscanf(overrides[(i+1)], "%d-%d", &first, &last);
      if ((vals != 2) && ((vals != 1) || (first != -1))) {
         fprintf(stderr, "Invalid format for range in doparamoverrides\n");
	 exit(0);
      }
      if (strcmp(overrides[i], "iodriver") == 0) {
	 io_param_override(IODRIVER, overrides[(i+2)], overrides[(i+3)], first, last);
      } else if (strcmp(overrides[i], "iosim") == 0) {
	 io_param_override(IOSIM, overrides[(i+2)], overrides[(i+3)], first, last);
      } else if (strcmp(overrides[i], "sysorg") == 0) {
	 io_param_override(SYSORG, overrides[(i+2)], overrides[(i+3)], first, last);
      } else if (strcmp(overrides[i], "synthio") == 0) {
	 pf_param_override(SYNTHIO_OVERRIDE, overrides[(i+2)], overrides[(i+3)], first, last);
      } else if (strcmp(overrides[i], "disk") == 0) {
	 io_param_override(DISK, overrides[(i+2)], overrides[(i+3)], first, last);
      } else if (strcmp(overrides[i], "controller") == 0) {
	 io_param_override(CONTROLLER, overrides[(i+2)], overrides[(i+3)], first, last);
      } else if (strcmp(overrides[i], "bus") == 0) {
	 io_param_override(BUS, overrides[(i+2)], overrides[(i+3)], first, last);
      } else {
	 fprintf(stderr, "Structure with parameter to override not supported: %s\n", overrides[i]);
	 exit(0);
      }
      fprintf (outputfile, "Override: %s %s %s %s\n", overrides[i], overrides[(i+1)], overrides[(i+2)], overrides[(i+3)]);
   }
}


void disksim_printstats ()
{
   fprintf (outputfile, "\nSIMULATION STATISTICS\n");
   fprintf (outputfile, "---------------------\n\n");
   fprintf (outputfile, "Total time of run:       %f\n\n", simtime);
   fprintf (outputfile, "Warm-up time:            %f\n\n", warmuptime);

   if (disksim->synthgen) {
      pf_printstats();
   }
   if (disksim->external_control | disksim->synthgen | disksim->iotrace) {
      io_printstats();
   }
}


static void setcallbacks ()
{
   disksim->issuefunc_ctlrsmart = NULL;
   disksim->queuefind_ctlrsmart = NULL;
   disksim->wakeupfunc_ctlrsmart = NULL;
   disksim->donefunc_ctlrsmart_read = NULL;
   disksim->donefunc_ctlrsmart_write = NULL;
   disksim->donefunc_cache_empty = NULL;
   disksim->idlework_cache = NULL;
   disksim->concatok_cache = NULL;
   disksim->enablement_disk = NULL;
   disksim->timerfunc_disksim = NULL;
   disksim->timerfunc_ioqueue = NULL;
   disksim->timerfunc_cache = NULL;

   disksim->timerfunc_disksim = stat_warmup_done;
   disksim->external_io_done_notify = NULL;

   io_setcallbacks();
   pf_setcallbacks();
}


static void initialize ()
{
   int val = (disksim->synthgen) ? 0 : 1;

   iotrace_initialize_file (disksim->iotracefile, disksim->traceformat, PRINT_TRACEFILE_HEADER);
   while (disksim->intq) {
      addtoextraq(getfromintq());
   }
   if (disksim->external_control | disksim->synthgen | disksim->iotrace) {
      io_initialize(val);
   }
   if (disksim->synthgen) {
      pf_initialize(disksim->seedval);
   } else {
      DISKSIM_srand48(disksim->seedval);
   }
   simtime = 0.0;
}


void disksim_cleanstats ()
{
   if (disksim->external_control | disksim->synthgen | disksim->iotrace) {
      io_cleanstats();
   }
   if (disksim->synthgen) {
      pf_cleanstats();
   }
}


void disksim_set_external_io_done_notify (void (*func)(ioreq_event *))
{
   disksim->external_io_done_notify = func;
}


event *io_done_notify (ioreq_event *curr)
{
   if (disksim->synthgen) {
      return(pf_io_done_notify(curr));
   }
   if (disksim->external_control) {
      disksim->external_io_done_notify (curr);
   }
   return(NULL);
}


static event * getnextevent ()
{
   event *curr = getfromintq();
   event *temp;

   if (curr) {
      simtime = curr->time;
   }
   if ((curr) && (curr->type == NULL_EVENT)) {
      if ((disksim->iotrace) && io_using_external_event(curr)) {
         if ((temp = io_get_next_external_event(disksim->iotracefile)) == NULL) {
            disksim->stop_sim = TRUE;
         }
      } else {
         fprintf(stderr, "NULL_EVENT not linked to any external source\n");
         exit(0);
      }
      if (temp) {
         temp->type = NULL_EVENT;
         addtointq(temp);
      }
   }
   if (curr == NULL) {
      fprintf (outputfile, "Returning NULL from getnextevent\n");
      fflush (outputfile);
   }
   return(curr);
}


void disksim_checkpoint (char *checkpointfilename)
{
   FILE *checkpointfile;

//printf ("disksim_checkpoint: simtime %f, totalreqs %d\n", simtime, disksim->totalreqs);
   if (disksim->checkpoint_disable) {
      fprintf (outputfile, "Checkpoint at simtime %f skipped because checkpointing is disabled\n", simtime);
      return;
   }

   if ((checkpointfile = fopen(checkpointfilename, "w")) == NULL) {
      fprintf (outputfile, "Checkpoint at simtime %f skipped because checkpointfile cannot be opened for write access\n", simtime);
      return;
   }

   if (disksim->iotracefile) {
      if (strcmp (disksim->iotracefilename, "stdin") == 0) {
         fprintf (outputfile, "Checkpoint at simtime %f skipped because iotrace comes from stdin\n", simtime);
         return;
      }
      fflush (disksim->iotracefile);
      fgetpos (disksim->iotracefile, &disksim->iotracefileposition);
   }

   if (outios) {
      fflush (outios);
      fgetpos (outios, &disksim->outiosfileposition);
   }

   if (outputfile) {
      fflush (outputfile);
      fgetpos (outputfile, &disksim->outputfileposition);
   }

   rewind (checkpointfile);
   fwrite (disksim, disksim->totallength, 1, checkpointfile);
   fflush (checkpointfile);
   fclose (checkpointfile);
}


void disksim_simstop ()
{
   disksim->stop_sim = TRUE;
}


void disksim_register_checkpoint (double attime)
{
   event *checkpoint = getfromextraq ();
   checkpoint->type = CHECKPOINT;
   checkpoint->time = attime;
   addtointq(checkpoint);
}


static void prime_simulation ()
{
   event *curr;

   if (disksim->warmup_event) {
      addtointq((event *)disksim->warmup_event);
      disksim->warmup_event = NULL;
   }
   if (disksim->checkpoint_interval > 0.0) {
      disksim_register_checkpoint (disksim->checkpoint_interval);
   }
   if (disksim->iotrace) {
      if ((curr = io_get_next_external_event(disksim->iotracefile)) == NULL) {
         disksim_cleanstats();
         return;
      }
      if ((disksim->traceformat != VALIDATE) && (disksim->closedios == 0)) {
         curr->type = NULL_EVENT;
      } else if (disksim->closedios) {
         int iocnt;
         io_using_external_event(curr);
         curr->time = simtime + disksim->closedthinktime;
         for (iocnt=1; iocnt < disksim->closedios; iocnt++) {
            curr->next = io_get_next_external_event(disksim->iotracefile);
            if (curr->next) {
               io_using_external_event(curr->next);
               curr->time = simtime + disksim->closedthinktime;
               addtointq(curr->next);
            }
         }
      }
      addtointq(curr);
   }
}


void disksim_simulate_event ()
{
   event *curr;

   if ((curr = getnextevent()) == NULL) {
      disksim_simstop ();

   } else {
      simtime = curr->time;

#ifdef FDEBUG
      fprintf (outputfile, "%f: next event to handle: type %d, temp %x\n", simtime,
	       curr->type, curr->temp);
      fflush (outputfile);
#endif

      if (curr->type == INTR_EVENT) {
	 intr_acknowledge (curr);
      } else if ((curr->type >= IO_MIN_EVENT) && (curr->type <= IO_MAX_EVENT)) {
	 io_internal_event ((ioreq_event *)curr);
      } else if ((curr->type >= PF_MIN_EVENT) && (curr->type <= PF_MAX_EVENT)) {
	 pf_internal_event(curr);
      } else if (curr->type == TIMER_EXPIRED) {
         timer_event *timeout = (timer_event *) curr;
         (*timeout->func) (timeout);
      } else if (curr->type == CHECKPOINT) {
         if (disksim->checkpoint_interval) {
            disksim_register_checkpoint (simtime + disksim->checkpoint_interval);
         }
         disksim_checkpoint (disksim->checkpointfilename);
      } else if (curr->type == STOP_SIM) {
         disksim_simstop ();
      } else if (curr->type == EXIT_DISKSIM) {
         exit (0);
      } else {
         fprintf(stderr, "Unrecognized event in simulate: %d\n", curr->type);
         exit(0);
      }
#ifdef FDEBUG
      fprintf (outputfile, "Event handled, going for next\n");
      fflush (outputfile);
#endif
   }
}


static void disksim_setup_outputfile (char *filename, char *mode)
{
   if (strcmp(filename, "stdout") == 0) {
      outputfile = stdout;
   } else {
      if ((outputfile = fopen(filename,mode)) == NULL) {
         fprintf(stderr, "Outfile %s cannot be opened for write access\n", filename);
         exit(0);
      }
   }

   if (strlen(filename) <= 255) {
      strcpy (disksim->outputfilename, filename);
   } else {
      fprintf (stderr, "Name of output file is too long (>255 bytes); checkpointing disabled\n");
      disksim->checkpoint_disable = 1;
   }
}


static void disksim_setup_iotracefile (char *filename)
{
   if (strcmp(filename, "0") != 0) {
      assert (disksim->external_control == 0);
      disksim->iotrace = 1;
      if (strcmp(filename, "stdin") == 0) {
	 disksim->iotracefile = stdin;
      } else {
	 if ((disksim->iotracefile = fopen(filename,"rb")) == NULL) {
	    fprintf(stderr, "Tracefile %s cannot be opened for read access\n", filename);
	    exit(0);
	 }
      }
   } else {
      disksim->iotracefile = NULL;
   }

   if (strlen(filename) <= 255) {
      strcpy (disksim->iotracefilename, filename);
   } else {
      fprintf (stderr, "Name of iotrace file is too long (>255 bytes); checkpointing disabled\n");
      disksim->checkpoint_disable = 1;
   }
}


void disksim_setup_disksim (int argc, char **argv)
{
   StaticAssert (sizeof(intchar) == 4);
   if (argc < 6) {
      fprintf(stderr,"Usage: %s paramfile outfile format iotrace synthgen?\n", argv[0]);
      exit(0);
   }
#ifdef FDEBUG
   fprintf (stderr, "%s %s %s %s %s %s\n", argv[0], argv[1], argv[2], argv[3],
	    argv[4], argv[5]);
#endif
   if ((argc - 6) % 4) {
      fprintf(stderr, "Parameter file overrides must be four-tuples\n");
      exit(0);
   } 
   if ((disksim->parfile = fopen(argv[1],"r")) == NULL) {
      fprintf(stderr,"%s cannot be opened for read access\n", argv[1]);
      exit(0);
   }

   disksim_setup_outputfile (argv[2], "w");
   fprintf (outputfile, "\nOutput file name: %s\n", argv[2]);
   fflush (outputfile);

   if (strcmp (argv[3], "external") == 0) {
      disksim->external_control = 1;
   } else {
      iotrace_set_format(argv[3]);
   }
   fprintf (outputfile, "Input trace format: %s\n", argv[3]);
   fflush (outputfile);

   disksim_setup_iotracefile (argv[4]);
   fprintf (outputfile, "I/O trace used: %s\n", argv[4]);
   fflush (outputfile);

   if (strcmp(argv[5], "0") != 0) {
      disksim->synthgen = 1;
      disksim->iotrace = 0;
   }
   fprintf (outputfile, "Synthgen to be used?: %s\n\n", argv[5]);
   fflush (outputfile);

   readparams(disksim->parfile);
   if (disksim->parfile != NULL) {
      fclose (disksim->parfile);
      disksim->parfile = NULL;
   }
fprintf(outputfile, "Readparams complete\n");
fflush(outputfile);
   if (argc > 6) {
      doparamoverrides(&argv[6], (argc - 6));
fprintf(outputfile, "Parameter overrides complete\n");
fflush(outputfile);
   }
   initialize();
fprintf(outputfile, "Initialization complete\n");
fflush(outputfile);
   prime_simulation();
}


void disksim_restore_from_checkpoint (char *filename)
{
#ifdef SUPPORT_CHECKPOINTS
   int fd;
   void *startaddr;
   int totallength;
   int ret;

   fd = open (filename, O_RDWR);
   assert (fd >= 0);
   ret = read (fd, &startaddr, sizeof (void *));
   assert (ret == sizeof(void *));
   ret = read (fd, &totallength, sizeof (int));
   assert (ret == sizeof(int));
   disksim = (disksim_t *) mmap (startaddr, totallength, (PROT_READ|PROT_WRITE), (MAP_PRIVATE), fd, 0);
   //printf ("mmap at %p, ret %d, len %d, addr %p\n", disksim, ret, totallength, startaddr);
   //perror("");
   assert (disksim == startaddr);

   disksim_setup_outputfile (disksim->outputfilename, "r+");
   if (outputfile != NULL) {
      ret = fsetpos (outputfile, &disksim->outputfileposition);
      assert (ret >= 0);
   }
   disksim_setup_iotracefile (disksim->iotracefilename);
   if (disksim->iotracefile != NULL) {
      ret = fsetpos (disksim->iotracefile, &disksim->iotracefileposition);
      assert (ret >= 0);
   }
   if ((strcmp(disksim->outiosfilename, "0") != 0) && (strcmp(disksim->outiosfilename, "null") != 0)) {
      if ((outios = fopen(disksim->outiosfilename, "r+")) == NULL) {
         fprintf(stderr, "Outios %s cannot be opened for write access\n", disksim->outiosfilename);
         exit(0);
      }
      ret = fsetpos (outios, &disksim->outiosfileposition);
      assert (ret >= 0);
   }
#else
   assert ("Checkpoint/restore not supported on this platform" == 0);
#endif
   setcallbacks();
}


void disksim_run_simulation ()
{
   while (disksim->stop_sim == FALSE) {
      disksim_simulate_event();
   }
}


void disksim_cleanup_and_printstats ()
{
fprintf(outputfile, "Simulation complete\n");
fflush(outputfile);
   disksim_cleanstats();
   disksim_printstats();

   if (outputfile) {
      fclose (outputfile);
   }
   if (disksim->iotracefile) {
      fclose(disksim->iotracefile);
   }
}


struct disksim * disksim_initialize_disksim_structure (void *addr, int len)
{
   disksim_t *disksim = addr;

   bzero ((char *)disksim, sizeof(disksim_t));
   disksim->startaddr = addr;
   disksim->totallength = len;
   disksim->curroffset = sizeof(disksim_t);
   disksim->closedthinktime = 0.0;
   warmuptime = 0.0;	/* gets remapped to disksim->warmuptime */
   simtime = 0.0;	/* gets remapped to disksim->warmuptime */
   disksim->lastphystime = 0.0;
   disksim->checkpoint_interval = 0.0;
   return (disksim);
}

