
/*
 * DiskSim Storage Subsystem Simulation Environment (Version 2.0)
 * Revision Authors: Greg Ganger
 * Contributors: Ross Cohen, John Griffin, Steve Schlosser
 *
 * Copyright (c) of Carnegie Mellon University, 1999.
 *
 * Permission to reproduce, use, and prepare derivative works of
 * this software for internal use is granted provided the copyright
 * and "No Warranty" statements are included with all reproductions
 * and derivative works. This software may also be redistributed
 * without charge provided that the copyright and "No Warranty"
 * statements are included in all redistributions.
 *
 * NO WARRANTY. THIS SOFTWARE IS FURNISHED ON AN "AS IS" BASIS.
 * CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER
 * EXPRESSED OR IMPLIED AS TO THE MATTER INCLUDING, BUT NOT LIMITED
 * TO: WARRANTY OF FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY
 * OF RESULTS OR RESULTS OBTAINED FROM USE OF THIS SOFTWARE. CARNEGIE
 * MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT
 * TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
 */

/*
 * DiskSim Storage Subsystem Simulation Environment
 * Authors: Greg Ganger, Bruce Worthington, Yale Patt
 *
 * Copyright (C) 1993, 1995, 1997 The Regents of the University of Michigan 
 *
 * This software is being provided by the copyright holders under the
 * following license. By obtaining, using and/or copying this software,
 * you agree that you have read, understood, and will comply with the
 * following terms and conditions:
 *
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose and without fee or royalty is
 * hereby granted, provided that the full text of this NOTICE appears on
 * ALL copies of the software and documentation or portions thereof,
 * including modifications, that you make.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS," AND COPYRIGHT HOLDERS MAKE NO
 * REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED. BY WAY OF EXAMPLE,
 * BUT NOT LIMITATION, COPYRIGHT HOLDERS MAKE NO REPRESENTATIONS OR
 * WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY PARTICULAR PURPOSE OR
 * THAT THE USE OF THE SOFTWARE OR DOCUMENTATION WILL NOT INFRINGE ANY
 * THIRD PARTY PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS. COPYRIGHT
 * HOLDERS WILL BEAR NO LIABILITY FOR ANY USE OF THIS SOFTWARE OR
 * DOCUMENTATION.
 *
 *  This software is provided AS IS, WITHOUT REPRESENTATION FROM THE
 * UNIVERSITY OF MICHIGAN AS TO ITS FITNESS FOR ANY PURPOSE, AND
 * WITHOUT WARRANTY BY THE UNIVERSITY OF MICHIGAN OF ANY KIND, EITHER
 * EXPRESSED OR IMPLIED, INCLUDING WITHOUT LIMITATION THE IMPLIED
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE REGENTS
 * OF THE UNIVERSITY OF MICHIGAN SHALL NOT BE LIABLE FOR ANY DAMAGES,
 * INCLUDING SPECIAL , INDIRECT, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
 * WITH RESPECT TO ANY CLAIM ARISING OUT OF OR IN CONNECTION WITH THE
 * USE OF OR IN CONNECTION WITH THE USE OF THE SOFTWARE, EVEN IF IT HAS
 * BEEN OR IS HEREAFTER ADVISED OF THE POSSIBILITY OF SUCH DAMAGES
 *
 * The names and trademarks of copyright holders or authors may NOT be
 * used in advertising or publicity pertaining to the software without
 * specific, written prior permission. Title to copyright in this software
 * and any associated documentation will at all times remain with copyright
 * holders.
 */

#ifndef DISKSIM_DISK_H
#define DISKSIM_DISK_H

#include "disksim_stat.h"
#include "disksim_ioqueue.h"

#include "disksim_diskmap.h"


/* Seek types */

#define THREEPOINT_LINE         -1.0
#define THREEPOINT_CURVE        -2.0
#define HPL_SEEK_EQUATION	-3.0
#define FIRST10_PLUS_HPL_SEEK_EQUATION	-4.0
#define EXTRACTION_SEEK		-5.0

/* Latency types */

#define AVGROTATE               -1.0
#define SKEWED_FOR_TRACK_SWITCH -2.0
#define NON_SKEWED              -3.0

/* Disk states */

#define DISK_IDLE			1
#define DISK_TRANSFERING		2
#define DISK_WAIT_FOR_CONTROLLER	3

/* Disk buffer states (both internal and external) */

#define BUFFER_EMPTY		1
#define BUFFER_CLEAN		2
#define BUFFER_DIRTY		3
#define BUFFER_READING		4
#define BUFFER_WRITING		5
#define BUFFER_PREEMPT		6
#define BUFFER_IDLE		7
#define BUFFER_CONTACTING	8
#define BUFFER_TRANSFERING	9

/* Disk buffer content match types */

#define BUFFER_COLLISION -1
#define BUFFER_NOMATCH	  0
#define BUFFER_WHOLE	  1     /* read */
#define BUFFER_PARTIAL	  2     /* read */
#define BUFFER_PREPEND    3     /* write */
#define BUFFER_APPEND     4     /* write */

/* Extra disk buffer flag variable */
/* *** MAKE SURE THIS DOESN"T CONFLICT WITH ANY FLAGS IN global.h *** */

#define BUFFER_BACKGROUND	0x10000000

/* Disk buffer continuous read types */

#define BUFFER_NO_READ_AHEAD		0
#define BUFFER_READ_UNTIL_TRACK_END	1
#define BUFFER_READ_UNTIL_CYL_END	2
#define BUFFER_READ_UNTIL_SEG_END	3
#define BUFFER_DEC_PREFETCH_SCHEME      4


/* Disk request flags */
#define SEG_OWNED			0x00000001
#define HDA_OWNED			0x00000002
#define EXTRA_WRITE_DISCONNECT          0x00000004
#define COMPLETION_SENT                 0x00000008
#define COMPLETION_RECEIVED             0x00000010
#define FINAL_WRITE_RECONNECTION_1      0x00000020
#define FINAL_WRITE_RECONNECTION_2      0x00000040

/* Disk preseeking levels */
#define NO_PRESEEK                      0
#define PRESEEK_DURING_COMPLETION	1
#define PRESEEK_BEFORE_COMPLETION       2        /* implies 1 as well */

/* Disk fastwrites levels */
#define NO_FASTWRITE			0
#define LIMITED_FASTWRITE		1
#define FULL_FASTWRITE			2

/* aliases */

#define ioreq_hold_disk tempptr1
#define ioreq_hold_diskreq tempptr2

typedef struct seg {
   double       time;
   int          state;			/* buffer state #define'd above */
   struct seg  *next;
   struct seg  *prev;
   int          startblkno;
   int          endblkno;
   int          outstate;
   int		outbcount;
   int		minreadaheadblkno;      /* min prefetch blkno + 1 */
   int		maxreadaheadblkno;      /* max prefetch blkno + 1 */
   struct diskreq_t *diskreqlist;       /* sorted by ascendingly first blkno */
   /* diskreqlist normally starts as a single request, more can be added by
    * read hits on the segment and write combining */
   int          size;
   ioreq_event *access; /* copy of the active ioreq using this seg  -rcohen */
   int		hold_blkno; 		/* used for prepending */
   int		hold_bcount;		/* sequential writes   */
   struct diskreq_t *recyclereq;        /* diskreq to recycle this seg */
} segment;

typedef struct diskreq_t {
   int			flags;
   ioreq_event 	       *ioreqlist;	/* sorted by ascending blkno */
   struct diskreq_t    *seg_next;	/* list attached to a segment */
   struct diskreq_t    *bus_next;
   int			outblkno;
   int			inblkno;
   segment	       *seg;		/* associated cache segment */
   int          	watermark;
   int			hittype;	/* for cache use */
   double		overhead_done;
   char			space[20];
} diskreq;

typedef struct {
   double  seektime;
   double  latency;
   double  xfertime;
   int     seekdistance;
   int     zeroseeks;
   int     zerolatency;
   int     highblkno;
   statgen seekdiststats;
   statgen seektimestats;
   statgen rotlatstats;
   statgen xfertimestats;
   statgen postimestats;
   statgen acctimestats;
   int     writecombs;
   int     readmisses;
   int     writemisses;
   int     fullreadhits;
   int     appendhits;
   int     prependhits;
   int     readinghits;
   double  runreadingsize;
   double  remreadingsize;
   int     parthits;
   double  runpartsize;
   double  rempartsize;
   int     interfere[2];
   double  requestedbus;
   double  waitingforbus;
   int     numbuswaits;
} diskstat;

typedef struct band {
   int    startcyl;
   int    endcyl;
   int    blkspertrack;
   int    deadspace;
   double firstblkno;
   double cylskew;
   double trackskew;
   int    blksinband;
   int    sparecnt;
   int    numslips;
   int    numdefects;
   int    slip[MAXSLIPS];
   int    defect[MAXDEFECTS];
   int    remap[MAXDEFECTS];
} band; /* zone */

typedef struct disk {
   double       acctime;
   double       seektime;
   double       seekone;
   double       seekavg;
   double       seekfull;
   double	seekwritedelta;
   double	hpseek[6];
   double	first10seeks[10];
   int		extractseekcnt;
   int *	extractseekdists;
   double *	extractseektimes;
   double       headswitch;
   double	rotatetime;
   double	rpmerr;
   double       rpm;
   double	overhead;
   double	timescale;
   int		devno;
   int          inited;
   int          syncset;
   int          numsurfaces;
   int          numblocks;
   int          numcyls;
   int          numbands;	/* numzones */
   int		sparescheme;
   int          rangesize;
   int          mapping;
   band        *bands;		/* zones, means the same thing */
   struct ioq  *queue;		/* list of io's not completed */
   int		sectpercyl;     /* "Avg" value used in suboptimal schedulers */
   int		hold_bus_for_whole_read_xfer;
   int		hold_bus_for_whole_write_xfer;
   int		almostreadhits;
   int		sneakyfullreadhits;
   int		sneakypartialreadhits;
   int		sneakyintermediatereadhits;
   int		readhitsonwritedata;
   int		writeprebuffering;
   int		preseeking;
   int		neverdisconnect;
   int          qlen;
   int          maxqlen;
   int          busy;
   int		prev_readahead_min;
   int		write_hit_stop_readahead;
   int		read_direct_to_buffer;
   int		immedtrans_any_readhit;
   int		readanyfreeblocks;
   int		numsegs;
   int		numwritesegs;
   segment *	dedicatedwriteseg;
   int		segsize;
   double	writewater;
   double	readwater;
   int		reqwater;
   int		sectbysect;
   int		enablecache;
   int		contread;
   int		minreadahead;
   int		maxreadahead;
   int		keeprequestdata;
   int		readaheadifidle;
   int		fastwrites;
   int		numdirty;
   int		immedread;
   int		immedwrite;
   int		immed;
   int		writecomb;
   int		stopinsector;
   int		disconnectinseek;
   int		immedstart;
   int		immedend;
   segment     *seglist;
   double	overhead_command_readhit_afterread;
   double	overhead_command_readhit_afterwrite;
   double	overhead_command_readmiss_afterread;
   double	overhead_command_readmiss_afterwrite;
   double	overhead_command_writehit_afterread;
   double	overhead_command_writehit_afterwrite;
   double	overhead_command_writemiss_afterread;
   double	overhead_command_writemiss_afterwrite;
   double	overhead_complete_read;
   double	overhead_complete_write;
   double	overhead_data_prep;
   double	overhead_reselect_first;
   double	overhead_reselect_other;
   double	overhead_disconnect_read_afterread;
   double	overhead_disconnect_read_afterwrite;
   double	overhead_disconnect_write;
   struct diskreq_t *extradisc_diskreq;
   int		extra_write_disconnect;
   double	extradisc_command;
   double	extradisc_disconnect1;
   double	extradisc_inter_disconnect;
   double	extradisc_disconnect2;
   double	extradisc_seekdelta;
   double	minimum_seek_delay;
   int		firstblkontrack;
   int		endoftrack;
   int          currcylno;
   int          currsurface;
   double       currangle;
   double       currtime;
   int          lastgen;
   int		lastflags;
   int          printstats;
   double	starttrans;
   double	blktranstime;
   int		outstate;
   diskreq     *pendxfer;
   diskreq     *currenthda;
   diskreq     *effectivehda;
   diskreq     *currentbus;
   diskreq     *effectivebus;
   int		blksdone;
   int		busowned;
   ioreq_event *buswait;
   ioreq_event *outwait;
   int          numinbuses;
   int          inbuses[MAXINBUSES];
   int          depth[MAXINBUSES];
   int          slotno[MAXINBUSES];
   diskstat     stat;
} disk;


typedef struct disk_info {
   disk *disks;
   int numdisks;
   int disk_printhack;   /* disk_printhack = 0, 1, or 2 */
   double disk_printhacktime;
   int numsyncsets;
   int extra_write_disconnects;
/* From disksim_diskmap.c -- gross mechanism for returning a value */
   int remapsector;
/* From disksim_diskctlr.c */
   int bandstart;
   int swap_forward_only;
/* From disksim_diskmech.c */
   int trackstart;   /* really yucky usage */
   double addtolatency;
   int global_seekdistance;  /* really yucky usage */
   double disk_seek_stoptime;
   int disk_last_distance;
   double disk_last_seektime;
   double disk_last_latency;
   double disk_last_xfertime;
   double disk_last_acctime;
   int disk_last_cylno;
   int disk_last_surface;
/* From disksim_diskcache.c */
   int LRU_at_seg_list_head;
/* "globals" are used instead of locals. values may be set and passed */
/* do NOT assume these have useful values at any point in time        */
/* unless they have just been set                                     */
   int global_currcylno;
   int global_currsurface;
   double global_currtime;
   double global_currangle;
/* *ESTIMATED* command processing overheads for buffer cache hits.  These */
/* values are not actually used for determining request service times.    */
   double buffer_partial_servtime;
   double reading_buffer_partial_servtime;
   double buffer_whole_servtime;
   double reading_buffer_whole_servtime;
} disk_info_t;

/* one remapping #define for each variable in disk_info_t */
//#define disks                     (disksim->diskinfo->disks)
#define numdisks                  (disksim->diskinfo->numdisks)
#define disk_printhack            (disksim->diskinfo->disk_printhack)
#define disk_printhacktime        (disksim->diskinfo->disk_printhacktime)
#define numsyncsets               (disksim->diskinfo->numsyncsets)
#define extra_write_disconnects   (disksim->diskinfo->extra_write_disconnects)
#define remapsector               (disksim->diskinfo->remapsector)
#define bandstart                 (disksim->diskinfo->bandstart)
#define swap_forward_only         (disksim->diskinfo->swap_forward_only)
#define trackstart                (disksim->diskinfo->trackstart)
#define addtolatency              (disksim->diskinfo->addtolatency)
#define global_seekdistance       (disksim->diskinfo->global_seekdistance)
#define disk_seek_stoptime        (disksim->diskinfo->disk_seek_stoptime)
#define disk_last_distance        (disksim->diskinfo->disk_last_distance)
#define disk_last_seektime        (disksim->diskinfo->disk_last_seektime)
#define disk_last_latency         (disksim->diskinfo->disk_last_latency)
#define disk_last_xfertime        (disksim->diskinfo->disk_last_xfertime)
#define disk_last_acctime         (disksim->diskinfo->disk_last_acctime)
#define disk_last_cylno           (disksim->diskinfo->disk_last_cylno)
#define disk_last_surface         (disksim->diskinfo->disk_last_surface)
#define LRU_at_seg_list_head      (disksim->diskinfo->LRU_at_seg_list_head)
#define global_currcylno          (disksim->diskinfo->global_currcylno)
#define global_currsurface        (disksim->diskinfo->global_currsurface)
#define global_currtime           (disksim->diskinfo->global_currtime)
#define global_currangle          (disksim->diskinfo->global_currangle)
#define buffer_partial_servtime   (disksim->diskinfo->buffer_partial_servtime)
#define reading_buffer_partial_servtime (disksim->diskinfo->reading_buffer_partial_servtime)
#define buffer_whole_servtime     (disksim->diskinfo->buffer_whole_servtime)
#define reading_buffer_whole_servtime (disksim->diskinfo->reading_buffer_whole_servtime)

/* disksim_diskmech.c functions */

int diskspecialseektime (double seektime);
int diskspecialaccesstime (double acctime);
void disk_read_extracted_seek_curve (char *filename, int *cntptr, int **distsptr, double **timesptr);
double diskseektime (disk *currdisk, int distance, int headswitch, int read);
double disklatency (disk *currdisk, band *currband, double rotstarttime, int blkno, int bcount, int immedaccess);
double diskxfertime (disk *currdisk, band *currband, int blkno, int reqsize);
double diskacctime (disk *currdisk, band *currband, int type, int rw, double reqtime, int cylno, int surfaceno, int blkno, int bcount, int immedaccess);
void disk_acctimestats (disk *currdisk, int distance, double seektime, double latency, double xfertime, double acctime);

/* disksim_diskctlr.c functions */

int  disk_buffer_stopable_access (disk *currdisk, diskreq *currdiskreq);
int  disk_enablement_function (ioreq_event *);

/* disksim_diskcache.c functions */

segment * disk_buffer_select_segment (disk *currdisk, diskreq *currdiskreq, int set_extradisc);
segment * disk_buffer_recyclable_segment (disk *currdisk, int isread);
diskreq * disk_buffer_seg_owner (segment *seg, int effective);
int  disk_buffer_attempt_seg_ownership (disk *currdisk, diskreq *currdiskreq);
int  disk_buffer_get_max_readahead (disk *currdisk, segment *seg, ioreq_event *curr);
int  disk_buffer_block_available (disk *currdisk, segment *seg, int blkno);
int  disk_buffer_reusable_segment_check (disk *currdisk, segment *currseg);
int  disk_buffer_overlap (segment *seg, ioreq_event *curr);
int  disk_buffer_check_segments (disk *currdisk, ioreq_event *currioreq, int* buffer_reading);
void disk_buffer_set_segment (disk *currdisk, diskreq *currdiskreq);
void disk_buffer_segment_wrap (segment *seg, int endblkno);
void disk_buffer_remove_from_seg (diskreq *currdiskreq);
void disk_interferestats (disk *currdisk, ioreq_event *curr);


/* externalized disksim_disk*.c functions (should be here?) */

struct disk *getdisk (int diskno);
void    disk_read_toprints (FILE *parfile);
void    disk_read_specs (FILE *parfile, int diskno, int copies);
void    disk_set_syncset (int setstart, int setend);
void    disk_param_override (char *paramname, char *paramval, int first, int last);
void    disk_setcallbacks (void);
void    disk_initialize (void);
void    disk_resetstats (void);
void    disk_printstats (void);
void    disk_printsetstats (int *set, int setsize, char *sourcestr);
void    disk_cleanstats (void);
int     disk_set_depth (int diskno, int inbusno, int depth, int slotno);
int     disk_get_depth (int diskno);
int     disk_get_inbus (int diskno);
int     disk_get_busno (ioreq_event *curr);
int     disk_get_slotno (int diskno);
int     disk_get_number_of_blocks (int diskno);
int     disk_get_maxoutstanding (int diskno);
int     disk_get_numdisks (void);
int     disk_get_numcyls (int diskno);
double  disk_get_blktranstime (ioreq_event *curr);
int     disk_get_avg_sectpercyl (int devno);
void    disk_get_mapping (int maptype, int diskno, int blkno, int *cylptr, int *surfaceptr, int *blkptr);
void    disk_event_arrive (ioreq_event *curr);
int     disk_get_distance (int diskno, ioreq_event *req, int exact, int direction);
double  disk_get_servtime (int diskno, ioreq_event *req, int checkcache, double maxtime);
double  disk_get_acctime (int diskno, ioreq_event *req, double maxtime);
void    disk_bus_delay_complete (int devno, ioreq_event *curr, int sentbusno);
void    disk_bus_ownership_grant (int devno, ioreq_event *curr, int busno, double arbdelay);


#endif   /* DISKSIM_DISK_H */

