
/*
 * DiskSim Storage Subsystem Simulation Environment (Version 2.0)
 * Revision Authors: Greg Ganger
 * Contributors: Ross Cohen, John Griffin, Steve Schlosser
 *
 * Copyright (c) of Carnegie Mellon University, 1999.
 *
 * Permission to reproduce, use, and prepare derivative works of
 * this software for internal use is granted provided the copyright
 * and "No Warranty" statements are included with all reproductions
 * and derivative works. This software may also be redistributed
 * without charge provided that the copyright and "No Warranty"
 * statements are included in all redistributions.
 *
 * NO WARRANTY. THIS SOFTWARE IS FURNISHED ON AN "AS IS" BASIS.
 * CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER
 * EXPRESSED OR IMPLIED AS TO THE MATTER INCLUDING, BUT NOT LIMITED
 * TO: WARRANTY OF FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY
 * OF RESULTS OR RESULTS OBTAINED FROM USE OF THIS SOFTWARE. CARNEGIE
 * MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT
 * TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
 */


#include "disksim_global.h"
#include "disksim_iosim.h"
#include "disksim_device.h"
#include "disksim_disk.h"
#include "disksim_simpledisk.h"


/* This remaps device numbers amongst multiple types of devices (each */
/* of which keeps internal arrays based on the remapped numbers). So, */
/* the device implementations need to be careful to call back to get  */
/* the remapping info...                                              */

/* To avoid the corresponding confusion, we will simply remap devnos  */
/* to themselves for now.  So, each device implementation should      */
/* allocate MAXDEVICES structures and track which of them are used.   */


/* private remapping #defines for variables from device_info_t */
#define numdevices                  (disksim->deviceinfo->numdevices)
	/* per-devno device type */
#define devicetypes                 (disksim->deviceinfo->devicetypes)
	/* per-devno device number (for when remapping them) */
#define devicenos                   (disksim->deviceinfo->devicenos)
	/* number of devices of each type */
#define maxdeviceno                 (disksim->deviceinfo->maxdeviceno)



static void device_initialize_deviceinfo ()
{
   if (disksim->deviceinfo == NULL) {
      disksim->deviceinfo = DISKSIM_malloc (sizeof(device_info_t));
      bzero (disksim->deviceinfo, sizeof(device_info_t));
   }
}


void device_read_toprints (FILE *parfile)
{
   /* read toprints here and let all devices see them (via disksim struct) */
   device_initialize_deviceinfo();

   getparam_bool(parfile, "Print device queue stats?", &device_printqueuestats);
   getparam_bool(parfile, "Print device crit stats?", &device_printcritstats);
   getparam_bool(parfile, "Print device idle stats?", &device_printidlestats);
   getparam_bool(parfile, "Print device intarr stats?", &device_printintarrstats);
   getparam_bool(parfile, "Print device size stats?", &device_printsizestats);
   getparam_bool(parfile, "Print device seek stats?", &device_printseekstats);
   getparam_bool(parfile, "Print device latency stats?", &device_printlatencystats);
   getparam_bool(parfile, "Print device xfer stats?", &device_printxferstats);
   getparam_bool(parfile, "Print device acctime stats?", &device_printacctimestats);
   getparam_bool(parfile, "Print device interfere stats?", &device_printinterferestats);
   getparam_bool(parfile, "Print device buffer stats?", &device_printbufferstats);

   /* then call for each type of device (in case there's something special) */
   disk_read_toprints (parfile);
   simpledisk_read_toprints (parfile);
}


void device_read_specs (FILE *parfile)
{
   int specno;
   int specno_expected = 1;
   int devno = 0;
   int copies;
   int devicetype;
   char spectype[81];
   int i;

   getparam_int(parfile, "\nNumber of storage devices", &numdevices, 3, 0, MAXDEVICES);
   if (numdevices == 0) {
      return;
   }

   while (devno < numdevices) {
      if (fscanf(parfile, "\nDevice Spec #%d\n", &specno) != 1) {
         fprintf(stderr, "Error reading device spec number\n");
         exit(0);
      }
      if (specno != specno_expected) {
         fprintf(stderr, "Unexpected value for device spec number: specno %d, expected %d\n", specno, specno_expected);
         exit(0);
      }
      fprintf (outputfile, "\nDevice Spec #%d\n", specno);
      specno_expected++;

      getparam_int(parfile, "# devices with Spec", &copies, 1, 0, 0);
      if (copies == 0) {
         copies = numdevices - devno;
      }

      if ((devno+copies) > numdevices) {
         fprintf(stderr, "Too many device specifications provided\n");
         exit(0);
      }

      /* read/interpret param to identify device type */
      if (fscanf(parfile, "Device type for Spec: %s\n", spectype) != 1) {
         fprintf(stderr, "Error reading Device type for Spec\n");
         exit(0);
      }
      if (strcmp(spectype, "disk") == 0) {
         devicetype = DEVICETYPE_DISK;
      } else if (strcmp(spectype, "simpledisk") == 0) {
         devicetype = DEVICETYPE_SIMPLEDISK;
      } else {
         fprintf(stderr, "Unexpected value for device spec number: specno %d, expected %d\n", specno, specno_expected);
         exit(0);
      }
      fprintf (outputfile, "\nDevice type for Spec: %s\n", spectype);

      /* set the remapping variables properly */
      for (i=0; i<copies; i++) {
         devicetypes[(devno+i)] = devicetype;
         //devicenos[(devno+i)] = maxdeviceno[devicetype] + i;
         devicenos[(devno+i)] = devno + i;
      }

      /* call appropriate read_specs() function with *remapped* devno */
      if (devicetype == DEVICETYPE_DISK) {
         disk_read_specs (parfile, devno, copies);
      } else if (devicetype == DEVICETYPE_SIMPLEDISK) {
         simpledisk_read_specs (parfile, devno, copies);
      } else {
         fprintf(stderr, "Unknown devicetype in device_read_specs: %d\n", devicetype);
         exit(0);
      }

      maxdeviceno[devicetype] += copies;
      devno += copies;
   }
}


void device_read_syncsets (FILE *parfile)
{
   int i,j;
   int setno;
   int size;
   int setstart;
   int setend;
   int devicetype;
   int syncsetcnt;

   if (fscanf(parfile,"\nNumber of synchronized sets: %d\n\n", &syncsetcnt) != 1) {
      fprintf(stderr, "Error reading number of synchronized sets\n");
      exit(0);
   }
   if (syncsetcnt < 0) {
      fprintf(stderr, "Invalid value for number of synchronized sets - %d\n", syncsetcnt);
      exit(0);
   }
   fprintf (outputfile, "\nNumber of synchronized sets: %d\n\n", syncsetcnt);

   for (i=1; i<=syncsetcnt; i++) {
      if (fscanf(parfile, "Number of devices in set #%d: %d\n", &setno, &size) != 2) {
         fprintf(stderr, "Error reading number of devices in synchronized set (#%d)\n", i);
         exit(0);
      }
      if (setno != i) {
         fprintf(stderr, "Synchronized sets not appearing in correct order\n");
         exit(0);
      }
      if ((size < 1) || (size > 31)) {
         fprintf(stderr, "Invalid value for size of synchronized set #%d\n", setno);
         exit(0);
      }
      fprintf (outputfile, "Number of devices in set #%d: %d\n", setno, size);

      if (fscanf(parfile, "Synchronized devices: %d-%d\n", &setstart, &setend) != 2) {
         fprintf(stderr, "Error reading 'Synchronized devices: ' for set #%d\n", setno);
         exit(0);
      }
      if ((setstart >= setend) || (setstart < 1) || (setend > numdevices) || (setstart + size - setend - 1)) {
         fprintf(stderr, "Invalid set of synchronized devices in set #%d\n", setno);
         exit(0);
      }
      fprintf (outputfile, "Synchronized devices: %d-%d\n", setstart, setend);

      /* verify that all set members are of same device type */
      devicetype = devicetypes[setstart];
      for (j=(setstart-1); j<setend; j++) {
         if (devicetypes[j] != devicetype) {
            fprintf(stderr, "Can't have mismatching devicetypes in a syncset (%d != %d)\n", devicetypes[j], devicetype);
            /* Might want to just return (and put this message in the output */
            /* file) rather than exiting...                                  */
            exit(0);
         }
      }

      /* call the appropriate one */
      if (devicetype == DEVICETYPE_DISK) {
         disk_set_syncset ((setstart-1), (setend-1));
      } else if (devicetype == DEVICETYPE_SIMPLEDISK) {
         simpledisk_set_syncset ((setstart-1), (setend-1));
      } else {
         fprintf(stderr, "Unknown value for device type: devicetype %d\n", devicetype);
         exit(0);
      }
   }
}


void device_param_override (char *paramname, char *paramval, int first, int last)
{
   int i;

   if (first == -1) {
      first = 0;
      last = numdevices - 1;
   } else if ((first < 0) || (last >= numdevices) || (last < first)) {
      fprintf(stderr, "Invalid range at device_param_override: %d - %d\n", first, last);
      exit(0);
   }

   /* go thru the devices and call individually as appropriately... */
   for (i=first; i<=last; i++) {
      if (devicetypes[i] == DEVICETYPE_DISK) {
         disk_param_override (paramname, paramval, i, i);
      } else if (devicetypes[i] == DEVICETYPE_SIMPLEDISK) {
         simpledisk_param_override (paramname, paramval, i, i);
      } else {
         fprintf(stderr, "Unknown value for device type: devicetype %d, devno %d\n", devicetypes[i], i);
         exit(0);
      }
   }
}


void device_setcallbacks (void)
{
   /* call for each type of device */
   disk_setcallbacks ();
   simpledisk_setcallbacks ();
}


void device_initialize (void)
{
   /* call for each type of device */
   disk_initialize ();
   simpledisk_initialize ();
}


void device_resetstats (void)
{
   /* call for each type of device */
   disk_resetstats ();
   simpledisk_resetstats ();
}


void device_printstats (void)
{
   /* call for each type of device */
   disk_printstats ();
   simpledisk_printstats ();
}


void device_printsetstats (int *set, int setsize, char *sourcestr)
{
   int i;
   int devicetype = devicetypes[set[0]];

   /* verify that all set members are of same device type */
   for (i=0; i<setsize; i++) {
      if (devicetypes[set[i]] != devicetype) {
         //fprintf(stderr, "Can't have mismatching devicetypes in device_printsetstats (%d != %d)\n", devicetypes[set[i]], devicetype);
         /* Might want to just return (and put this message in the output */
         /* file) rather than exiting...                                  */
         //exit(0);
         return;
      }
   }

   /* call the appropriate one */
   if (devicetype == DEVICETYPE_DISK) {
      disk_printsetstats (set, setsize, sourcestr);
   } else if (devicetype == DEVICETYPE_SIMPLEDISK) {
      simpledisk_printsetstats (set, setsize, sourcestr);
   } else {
      fprintf(stderr, "Unknown value for device type: devicetype %d\n", devicetype);
      exit(0);
   }
}


void device_cleanstats (void)
{
   /* call for each type of device */
   disk_cleanstats ();
   simpledisk_cleanstats ();
}


int device_set_depth (int devno, int inbusno, int depth, int slotno)
{
   ASSERT1 ((devno >= 0) && (devno < numdevices), "devno", devno);

   /* call the appropriate one, given overall devno */
   if (devicetypes[devno] == DEVICETYPE_DISK) {
      return ( disk_set_depth (devno, inbusno, depth, slotno) );
   } else if (devicetypes[devno] == DEVICETYPE_SIMPLEDISK) {
      return ( simpledisk_set_depth (devno, inbusno, depth, slotno) );
   } else {
      fprintf(stderr, "Unknown value for device type: devicetype %d, devno %d\n", devicetypes[devno], devno);
      exit(0);
   }
}


int device_get_depth (int devno)
{
   ASSERT1 ((devno >= 0) && (devno < numdevices), "devno", devno);

   /* call the appropriate one, given overall devno */
   if (devicetypes[devno] == DEVICETYPE_DISK) {
      return ( disk_get_depth (devno) );
   } else if (devicetypes[devno] == DEVICETYPE_SIMPLEDISK) {
      return ( simpledisk_get_depth (devno) );
   } else {
      fprintf(stderr, "Unknown value for device type: devicetype %d, devno %d\n", devicetypes[devno], devno);
      exit(0);
   }
}


int device_get_inbus (int devno)
{
   ASSERT1 ((devno >= 0) && (devno < numdevices), "devno", devno);

   /* call the appropriate one, given overall devno */
   if (devicetypes[devno] == DEVICETYPE_DISK) {
      return ( disk_get_inbus (devno) );
   } else if (devicetypes[devno] == DEVICETYPE_SIMPLEDISK) {
      return ( simpledisk_get_inbus (devno) );
   } else {
      fprintf(stderr, "Unknown value for device type: devicetype %d, devno %d\n", devicetypes[devno], devno);
      exit(0);
   }
}


int device_get_busno (ioreq_event *curr)
{
   ASSERT1 ((curr->devno >= 0) && (curr->devno < numdevices), "curr->devno", curr->devno);

   /* call the appropriate one, given overall curr->devno */
   if (devicetypes[curr->devno] == DEVICETYPE_DISK) {
      return ( disk_get_busno (curr) );
   } else if (devicetypes[curr->devno] == DEVICETYPE_SIMPLEDISK) {
      return ( simpledisk_get_busno (curr) );
   } else {
      fprintf(stderr, "Unknown value for device type: devicetype %d, devno %d\n", devicetypes[curr->devno], curr->devno);
      exit(0);
   }
}


int device_get_slotno (int devno)
{
   ASSERT1 ((devno >= 0) && (devno < numdevices), "devno", devno);

   /* call the appropriate one, given overall devno */
   if (devicetypes[devno] == DEVICETYPE_DISK) {
      return ( disk_get_slotno (devno) );
   } else if (devicetypes[devno] == DEVICETYPE_SIMPLEDISK) {
      return ( simpledisk_get_slotno (devno) );
   } else {
      fprintf(stderr, "Unknown value for device type: devicetype %d, devno %d\n", devicetypes[devno], devno);
      exit(0);
   }
}


int device_get_number_of_blocks (int devno)
{
   ASSERT1 ((devno >= 0) && (devno < numdevices), "devno", devno);

   //   fprintf(stderr, "device_get_number_of_blocks::  entry\n");

   /* call the appropriate one, given overall devno */
   if (devicetypes[devno] == DEVICETYPE_DISK) {
      return ( disk_get_number_of_blocks (devno) );
   } else if (devicetypes[devno] == DEVICETYPE_SIMPLEDISK) {
      return ( simpledisk_get_number_of_blocks (devno) );
   } else {
      fprintf(stderr, "Unknown value for device type: devicetype %d, devno %d\n", devicetypes[devno], devno);
      exit(0);
   }
}


int device_get_maxoutstanding (int devno)
{
   ASSERT1 ((devno >= 0) && (devno < numdevices), "devno", devno);

   /* call the appropriate one, given overall devno */
   if (devicetypes[devno] == DEVICETYPE_DISK) {
      return ( disk_get_maxoutstanding (devno) );
   } else if (devicetypes[devno] == DEVICETYPE_SIMPLEDISK) {
      return ( simpledisk_get_maxoutstanding (devno) );
   } else {
      fprintf(stderr, "Unknown value for device type: devicetype %d, devno %d\n", devicetypes[devno], devno);
      exit(0);
   }
}


int device_get_numdevices (void)
{
   return ( numdevices );
}


int device_get_numcyls (int devno)
{
   ASSERT1 ((devno >= 0) && (devno < numdevices), "devno", devno);

   /* call the appropriate one, given overall devno */
   if (devicetypes[devno] == DEVICETYPE_DISK) {
      return ( disk_get_numcyls (devno) );
   } else if (devicetypes[devno] == DEVICETYPE_SIMPLEDISK) {
      return ( simpledisk_get_numcyls (devno) );
   } else {
      fprintf(stderr, "Unknown value for device type: devicetype %d, devno %d\n", devicetypes[devno], devno);
      exit(0);
   }
}


double device_get_blktranstime (ioreq_event *curr)
{
   ASSERT1 ((curr->devno >= 0) && (curr->devno < numdevices), "curr->devno", curr->devno);

   /* call the appropriate one, given overall curr->devno */
   if (devicetypes[curr->devno] == DEVICETYPE_DISK) {
      return ( disk_get_blktranstime (curr) );
   } else if (devicetypes[curr->devno] == DEVICETYPE_SIMPLEDISK) {
      return ( simpledisk_get_blktranstime (curr) );
   } else {
      fprintf(stderr, "Unknown value for device type: devicetype %d, devno %d\n", devicetypes[curr->devno], curr->devno);
      exit(0);
   }
}


int device_get_avg_sectpercyl (int devno)
{
   ASSERT1 ((devno >= 0) && (devno < numdevices), "devno", devno);

   /* call the appropriate one, given overall devno */
   if (devicetypes[devno] == DEVICETYPE_DISK) {
      return ( disk_get_avg_sectpercyl (devno) );
   } else if (devicetypes[devno] == DEVICETYPE_SIMPLEDISK) {
      return ( simpledisk_get_avg_sectpercyl (devno) );
   } else {
      fprintf(stderr, "Unknown value for device type: devicetype %d, devno %d\n", devicetypes[devno], devno);
      exit(0);
   }
}


void device_get_mapping (int maptype, int devno, int blkno, int *cylptr, int *surfaceptr, int *blkptr)
{
   ASSERT1 ((devno >= 0) && (devno < numdevices), "devno", devno);

   /* call the appropriate one, given overall devno */
   if (devicetypes[devno] == DEVICETYPE_DISK) {
      disk_get_mapping (maptype, devno, blkno, cylptr, surfaceptr, blkptr);
   } else if (devicetypes[devno] == DEVICETYPE_SIMPLEDISK) {
      simpledisk_get_mapping (maptype, devno, blkno, cylptr, surfaceptr, blkptr);
   } else {
      fprintf(stderr, "Unknown value for device type: devicetype %d, devno %d\n", devicetypes[devno], devno);
      exit(0);
   }
}


void device_event_arrive (ioreq_event *curr)
{
   ASSERT1 ((curr->devno >= 0) && (curr->devno < numdevices), "curr->devno", curr->devno);

   /* call the appropriate one, given overall curr->devno */
   if (devicetypes[curr->devno] == DEVICETYPE_DISK) {
      disk_event_arrive (curr);
   } else if (devicetypes[curr->devno] == DEVICETYPE_SIMPLEDISK) {
      simpledisk_event_arrive (curr);
   } else {
      fprintf(stderr, "Unknown value for device type: devicetype %d, devno %d\n", devicetypes[curr->devno], curr->devno);
      exit(0);
   }
}


int device_get_distance (int devno, ioreq_event *req, int exact, int direction)
{
   ASSERT1 ((devno >= 0) && (devno < numdevices), "devno", devno);

   /* call the appropriate one, given overall devno */
   if (devicetypes[devno] == DEVICETYPE_DISK) {
      return ( disk_get_distance (devno, req, exact, direction) );
   } else if (devicetypes[devno] == DEVICETYPE_SIMPLEDISK) {
      return ( simpledisk_get_distance (devno, req, exact, direction) );
   } else {
      fprintf(stderr, "Unknown value for device type: devicetype %d, devno %d\n", devicetypes[devno], devno);
      exit(0);
   }
}


double device_get_servtime (int devno, ioreq_event *req, int checkcache, double maxtime)
{
   ASSERT1 ((devno >= 0) && (devno < numdevices), "devno", devno);

   /* call the appropriate one, given overall devno */
   if (devicetypes[devno] == DEVICETYPE_DISK) {
      return ( disk_get_servtime (devno, req, checkcache, maxtime) );
   } else if (devicetypes[devno] == DEVICETYPE_SIMPLEDISK) {
      return ( simpledisk_get_servtime (devno, req, checkcache, maxtime) );
   } else {
      fprintf(stderr, "Unknown value for device type: devicetype %d, devno %d\n", devicetypes[devno], devno);
      exit(0);
   }
}


double device_get_acctime (int devno, ioreq_event *req, double maxtime)
{
   ASSERT1 ((devno >= 0) && (devno < numdevices), "devno", devno);

   /* call the appropriate one, given overall devno */
   if (devicetypes[devno] == DEVICETYPE_DISK) {
      return ( disk_get_acctime (devno, req, maxtime) );
   } else if (devicetypes[devno] == DEVICETYPE_SIMPLEDISK) {
      return ( simpledisk_get_acctime (devno, req, maxtime) );
   } else {
      fprintf(stderr, "Unknown value for device type: devicetype %d, devno %d\n", devicetypes[devno], devno);
      exit(0);
   }
}


void device_bus_delay_complete (int devno, ioreq_event *curr, int sentbusno)
{
   ASSERT1 ((devno >= 0) && (devno < numdevices), "devno", devno);

   /* call the appropriate one, given overall devno */
   if (devicetypes[devno] == DEVICETYPE_DISK) {
      disk_bus_delay_complete (devno, curr, sentbusno);
   } else if (devicetypes[devno] == DEVICETYPE_SIMPLEDISK) {
      simpledisk_bus_delay_complete (devno, curr, sentbusno);
   } else {
      fprintf(stderr, "Unknown value for device type: devicetype %d, devno %d\n", devicetypes[devno], devno);
      exit(0);
   }
}


void device_bus_ownership_grant (int devno, ioreq_event *curr, int busno, double arbdelay)
{
   ASSERT1 ((devno >= 0) && (devno < numdevices), "devno", devno);

   /* call the appropriate one, given overall devno */
   if (devicetypes[devno] == DEVICETYPE_DISK) {
      disk_bus_ownership_grant (devno, curr, busno, arbdelay);
   } else if (devicetypes[devno] == DEVICETYPE_SIMPLEDISK) {
      simpledisk_bus_ownership_grant (devno, curr, busno, arbdelay);
   } else {
      fprintf(stderr, "Unknown value for device type: devicetype %d, devno %d\n", devicetypes[devno], devno);
      exit(0);
   }
}

