#ifndef __GRAPHCUT_VK3_H__
#define __GRAPHCUT_VK3_H__
/*********************************************************************
 * Software License Agreement (Modified BSD License)
 *
 * Copyright (c) 2010, Daniel Munoz
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *   * Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *   * Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *   * Neither the name of the copyright holders' organizations nor the
 *     names of its contributors may be used to endorse or promote products
 *     derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *********************************************************************/

#include <maxflow-v3.01/graph.h>

#include <submodular_graphcut/submodular_graphcut.h>

namespace submodular_graphcut
{
// --------------------------------------------------------------
/*!
 * \brief Wraps around Vladimir Kolmogorov's maxflow-v3.0 library,
 *        which is under a "research-only" license.
 */
// --------------------------------------------------------------
class GraphcutVK3: public SubmodularGraphcut
{
  public:
    // --------------------------------------------------------------
    /*!
     * \brief Instantiates empty submodular function
     *
     * \param node_num_max An estimate of the number of nodes in the graph
     * \param edge_num_max An estimate of the number of edges in the graph
     */
    // --------------------------------------------------------------
    GraphcutVK3(unsigned int node_num_max,
                unsigned int edge_num_max)
    {
      m_graph = new Graph<double, double, double> (node_num_max, edge_num_max);
    }

    ~GraphcutVK3()
    {
      delete m_graph;
    }

    // --------------------------------------------------------------
    /*!
     * \see SubmodularGraphcut::addVariable
     */
    // --------------------------------------------------------------
    inline virtual EnergyVar addVariable()
    {
      return m_graph->add_node(1);
    }

    // --------------------------------------------------------------
    /*!
     * \see SubmodularGraphcut::minimize
     */
    // --------------------------------------------------------------
    inline virtual double minimize()
    {
      return m_const_offset + m_graph->maxflow();
    }

    // --------------------------------------------------------------
    /*!
     * \see SubmodularGraphcut::getValue
     */
    // --------------------------------------------------------------
    inline virtual unsigned int getValue(const EnergyVar& x)
    {
      return m_graph->what_segment(x) == Graph<double, double, double>::SINK;
    }

  protected:
    // --------------------------------------------------------------
    /*!
     * \see SubmodularGraphcut::add_tweights
     */
    // --------------------------------------------------------------
    inline virtual void add_tweights(const EnergyVar& x,
                                     double A,
                                     double B)
    {
      m_graph->add_tweights(x, A, B);
    }

    // --------------------------------------------------------------
    /*!
     * \see SubmodularGraphcut::add_edge
     */
    // --------------------------------------------------------------
    inline virtual void add_edge(const EnergyVar& x,
                                 const EnergyVar& y,
                                 double A,
                                 double B)
    {
      m_graph->add_edge(x, y, A, B);
    }

  private:
    Graph<double, double, double>* m_graph;
};
}
#endif
