#ifndef __GRAPHCUT_BGL_H__
#define __GRAPHCUT_BGL_H__
/*********************************************************************
 * Software License Agreement (Modified BSD License)
 *
 * Copyright (c) 2010, Daniel Munoz
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *   * Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *   * Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *   * Neither the name of the copyright holders' organizations nor the
 *     names of its contributors may be used to endorse or promote products
 *     derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *********************************************************************/

#include <boost/config.hpp>
#include <boost/graph/kolmogorov_max_flow.hpp>
#include <boost/graph/adjacency_list.hpp>
#include <boost/graph/graph_utility.hpp>
#include <boost/graph/graph_traits.hpp>

#include <submodular_graphcut/submodular_graphcut.h>

namespace submodular_graphcut
{
class GraphcutBGL: public SubmodularGraphcut
{
  public:
    // Change properties of the graph as needed
    typedef boost::adjacency_list_traits<boost::vecS, boost::vecS, boost::directedS> Traits;

    typedef boost::adjacency_list<boost::vecS, boost::vecS, boost::directedS, boost::property<
        boost::vertex_name_t, std::string, boost::property<boost::vertex_index_t, unsigned int,
            boost::property<boost::vertex_color_t, boost::default_color_type, boost::property<
                boost::vertex_distance_t, double, boost::property<boost::vertex_predecessor_t,
                    Traits::edge_descriptor> > > > >, boost::property<boost::edge_capacity_t,
        double, boost::property<boost::edge_residual_capacity_t, double, boost::property<
            boost::edge_reverse_t, Traits::edge_descriptor> > > > Graph;

    // --------------------------------------------------------------
    /*!
     * \brief Instantiates empty energy function
     */
    // --------------------------------------------------------------
    GraphcutBGL();

    ~GraphcutBGL()
    {
      delete m_graph;
    }

    // --------------------------------------------------------------
    /*!
     * \see SubmodularGraphcut::addVariable
     */
    // --------------------------------------------------------------
    inline virtual EnergyVar addVariable()
    {
      return boost::add_vertex(*m_graph);
    }

    // --------------------------------------------------------------
    /*!
     * \brief Minimizes the current energy function
     *
     * \return The minimum energy value
     */
    // --------------------------------------------------------------
    inline virtual double minimize()
    {
      return m_const_offset + boost::kolmogorov_max_flow(*m_graph, m_source, m_sink);
    }

    // --------------------------------------------------------------
    /*!
     * \brief Returns the binary value for the variable
     *
     * This should be called after calling minimize()
     *
     * \return 0 or 1 for the value of x in the minimizing solution
     */
    // --------------------------------------------------------------
    virtual unsigned int getValue(const EnergyVar& x);

  protected:
    // --------------------------------------------------------------
    /*!
     * \see SubmodularGraphcut::add_tweights
     */
    // --------------------------------------------------------------
    virtual void add_tweights(const EnergyVar& x,
                              double A,
                              double B);

    // --------------------------------------------------------------
    /*!
     * \see SubmodularGraphcut::add_edge
     */
    // --------------------------------------------------------------
    virtual void add_edge(const EnergyVar& x,
                          const EnergyVar& y,
                          double A,
                          double B);

  private:
    EnergyVar m_source;
    EnergyVar m_sink;
    Graph* m_graph;
};
}
#endif
