function [x_new,y_new,s_new,scale_new] = refine_int_pts(x, y, s, lap_pyramid, scales_list)
%
% [xnew,ynew,snew, final_offsets] = refine_int_pts(x, y, s, lap_pyramid, scales_list)
%
%  
% ------------
% Andrew Stein
%

MAX_REFINEMENTS = 10;

[nrows, ncols, num_levels] = size(lap_pyramid);

% Note that the passed-in laplacian pyramid has an extra level at top and
% bottom (for local extrema detection and so we can do this voxel
% refinement, which requires extraction of a small cube of data around the
% interest point).  This means that the values in s, which represent which
% level of the pyramid the interest point is on, should never be less than 2.

% Get the scale of each interest point according to the list of scales for
% this pyramid and the interest points' scale indices:
scale = scales_list(s);

to_remove = [];

for(i=1:length(x))
    
    % Note that the spacing along the scale dimension is not uniform
    % or unit since the scales increase geometrically as we go through the
    % pyramid.
    local_cube = lap_pyramid(y(i)+[-1:1], x(i)+[-1:1], s(i)+[-1:1]);
    ds = scales_list(s(i)+[0 1]) - scales_list(s(i)+[-1 0]);
    [Dp,Dpp] = ComputeDerivatives_3D(local_cube, [1 1], [1 1], ds);
    offsets = -inv(Dpp)*Dp;
    
    % If the offset indicates we are more than 1 voxel away from the
    % true max position, we should move over a _single_ sample and do the
    % interpolation around that point.  We keep refining until we converge
    % or we do too many refinements.
    refinements = 0;
    while( (any(abs(offsets(1:2))>0.5) | offsets(3)>.5*ds(2) | offsets(3)<-.5*ds(1)) & refinements < MAX_REFINEMENTS)
        refinements = refinements + 1;
        x(i) = round(x(i) + max(-1, min(1, offsets(1))));
        y(i) = round(y(i) + max(-1, min(1, offsets(2))));
        scale(i) = scale(i) + max(-ds(1), min(ds(2), offsets(3)));
        scale(i) = max(scales_list(2), min(scales_list(end-1), scale(i)));
        [closest, s(i)] = min(abs(scale(i)-scales_list));
        
        % If we move out of the image, break out of this refinement,
        % and this point will be removed when it gets checked again
        % below
        if(x(i)<=1 | x(i)>=ncols | y(i)<=1 | y(i)>=nrows)
            break;
        end
        
        local_cube = lap_pyramid(y(i)+[-1:1], x(i)+[-1:1], s(i)+[-1:1]);
        ds = scales_list(s(i)+[0 1]) - scales_list(s(i)+[-1 0]);
        [Dp,Dpp] = ComputeDerivatives_3D(local_cube, [1 1], [1 1], ds);
        offsets = -inv(Dpp)*Dp;
    end
    
    % Go ahead and move the interest point up to one sample's distance
    % in the direction computed above (without rounding), but if we've
    % move outside the image, or even onto the image border, this point 
    % should be removed.
    x_new(i) = x(i) + max(-1, min(1, offsets(1)));
    y_new(i) = y(i) + max(-1, min(1, offsets(2)));
    if(x_new(i)<=1 | x_new(i)>=ncols | y_new(i)<=1 | y_new(i)>=nrows)
        to_remove(end+1) = i;
    end
    
    % Make sure we get a scale that's within the pyramid:
    scale(i) = scale(i) + max(-ds(1), min(ds(2), offsets(3)));
    scale_new(i) = max(scales_list(1), min(scales_list(end), scale(i)));
    [closest, s_new(i)] = min(abs(scale_new(i)-scales_list));
       
end

% % Also remove any points not above the threshold, now that we've let them
% % come to rest at the local extremum
% lap_val = myInterp3(lap_pyramid, x_new, y_new, s_new, NaN);
% to_remove = [to_remove find(abs(lap_val) < 0.01)];

% Get rid of any points marked for deletion:
x_new(to_remove) = [];
y_new(to_remove) = [];
s_new(to_remove) = [];
scale_new(to_remove) = [];

% disp(['   Removing ' num2str(length(to_remove)) ' while doing subvoxel refinement.'])

