function [img_pyramid, lap_pyramid, varargout] = ScaleSpaceRepresentation(img, cracks, sigma_0, samples_per_octave)
%
% [img_pyramid, lap_pyramid, <crack_pyramid>, <dx_pyramid, dy_pyramid>, <scales_list>] = 
%         ScaleSpaceRepresentation(img, <cracks>, <sigma_0>, <samples_per_octave>)
%
% == Overview: ==
%
%  Given an image, and optionally crack data as well, this function builds a 
%  Guassian pyramid of the image.  It then approximates a Laplacian pyramid by
%  building a Difference of Gaussian pyramid. 
%
%
% == More details: ==
%
%  If crack data is specified, iterative diffusion is used to do the
%  Gaussian filtering.  Otherwise (if unspecified or []), normal convolution 
%  with a Gaussian kernel is performed. 
%
%  Each octave of the pyramid represents a halving (or doubling) of the
%  original image.  To provide for finding local extrema in the Laplacian
%  pyramid, each octave will have samples_per_octave+2 levels, where
%  samples_per_octave defaults to 3.  Additional octaves are created until
%  the minimum spatial dimension of the octave is less than 4 pixels.  Note
%  that the image pyramid will have an extra level at the top of each octave
%  since differencing of adjacent levels is used to build the Difference of
%  Gaussian pyarmid.
%
%  If the initial smoothing, sigma_0 is unspecified, it defaults to 1.
%
%  If cracks are provided, a crack_pyramid _must_ be requested as output IF
%  any further outputs are desired (i.e. derivatives or a scales_list).  If
%  cracks are NOT provided, the crack_pyramid is not available as an output.
%
%  BOTH dx_pyramid and dy_pyramid must be requested to get either.
%
%  The list of scales corresponding to the scale of the Laplacian at each
%  level in an octave of the Laplacian pyramid (modulo the factor of two
%  corresponding to which octave) can always be requested as the last output
%  argument.
%
% ----------------
% Andrew Stein, 2005
%

USE_CONSTANT_ITERATIONS = false;

% When to stop halving the image and creating a new octave
min_dimension = 8;

if(USE_CONSTANT_ITERATIONS)
    constant_iterations = 25;
end

% For diffusion, if used.  The smaller the better (and slower).  This must
% be less than 0.25 for numerical stability.
timestep = 0.05;

if(size(img,3)==3)
    disp('Converting image to grayscale')
    img = rgb2gray(img);
end

if(~isa(img, 'double'))
    disp('Converting image to DOUBLE');
    img = double(img);
end

use_diffusion = true;
if(nargin<4)
    disp('Using default parameter samples_per_octave = 3')
    samples_per_octave = 3;
    if(nargin<3)
        disp('Using default parameter sigma_0 = 1')
        sigma_0 = 1;
        if(nargin<2)
            %         % If no crack info is given, still use diffusion for smoothing
            %         use_diffusion = true;
            %         cracks = uint8(zeros(size(img)));
            
            % If no crack info is provided use gaussian smoothing
            use_diffusion = false;
        end
        
    elseif(isempty(cracks))
        use_diffusion = false;
    end
end

% Only bother computing derivatives if they are requested as outputs
do_derivatives = true;
if((~use_diffusion & (nargout < 4)) | (use_diffusion & (nargout < 5)))
    do_derivatives = false;
end

% Compute the list of scales we want for our Laplacian pyramid
samples_per_octave = 3; 
k = 2^(1/samples_per_octave);
scales_list = sigma_0*k.^[-1:samples_per_octave]; % need one extra level on either end to look for extrema

% Actually need one extra scale for smoothing so we can do Gaussian differencing
gauss_scales = sigma_0*k.^[-1:(samples_per_octave+1)];

first_level_img = img;
if(use_diffusion)
    sampled_cracks = cracks;
    
    if(USE_CONSTANT_ITERATIONS)
        constant_iterations = 50;
        timesteps_list = gauss_scales.^2 / (2*constant_iterations);
        first_level_img = mexDiffuse_cracks(first_level_img, sampled_cracks, constant_iterations, timesteps_list(1));
    else
        iterations_list = round( (gauss_scales.^2) / (2*timestep) );  
        first_level_img = mexDiffuse_cracks(first_level_img, sampled_cracks, iterations_list(1), timestep);
    end
    
else
    g = myGaussian(gauss_scales(1));
    first_level_img = imfilter( imfilter(first_level_img, g, 'conv'), g', 'conv');
end
        
i_octave = 1;
while(min(size(first_level_img)) > min_dimension)
    
    scale_factor = 2^(i_octave-1);
    
    if(use_diffusion)
        % If crack information is provided, use local diffusion to do the
        % smoothing and respect boundary information.  Note that we
        % subtract off the iterations necessary for the initial smoothing
        % because the first level has already been smoothed by that much
        % (whether done above in the case of the first octave or because of
        % downsampling for higher octaves)
        
        crack_pyramid{i_octave} = sampled_cracks;
        if(do_derivatives)
            if(USE_CONSTANT_ITERATIONS)
                [img_pyramid{i_octave}, dx{i_octave}, dy{i_octave}] = mexDiffuse_cracks(first_level_img, ...
                    sampled_cracks, constant_iterations, timesteps_list-timesteps_list(1));
            else
                [img_pyramid{i_octave}, dx{i_octave}, dy{i_octave}] = mexDiffuse_cracks(first_level_img, ...
                    sampled_cracks, iterations_list-iterations_list(1), timestep);
            end                
            
            % We don't actually need the last level of derivatives, so toss
            % them:
            dx{i_octave}(:,:,end) = [];
            dy{i_octave}(:,:,end) = [];
        else
            if(USE_CONSTANT_ITERATIONS)
                img_pyramid{i_octave} = mexDiffuse_cracks(first_level_img, ...
                    sampled_cracks, constant_iterations, timesteps_list-timesteps_list(1));
            else
                img_pyramid{i_octave} = mexDiffuse_cracks(first_level_img, ...
                    sampled_cracks, iterations_list-iterations_list(1), timestep);
            end
                
        end
        
    else 
        % If there's no crack information, just use simple Gaussian smoothing
        
        img_pyramid{i_octave} = zeros([size(first_level_img) length(gauss_scales)]);
        img_pyramid{i_octave}(:,:,1) = first_level_img;
        for(i=2:length(gauss_scales))
            % Get a Gaussian filter to smooth the previous level up to the
            % amount of smoothing needed for this level (instead of
            % resmoothing from the first level each time)
            addl_sigma = sqrt(k^2 - 1)*gauss_scales(i-1);
            g = myGaussian(addl_sigma);
            
            img_pyramid{i_octave}(:,:,i) = imfilter( imfilter(img_pyramid{i_octave}(:,:,i-1), ...
                g, 'replicate', 'conv'), g', 'replicate', 'conv');
                       
        end
        
        if(do_derivatives)
            dx{i_octave} = (img_pyramid{i_octave}(:,[2:end end],1:end-1) - img_pyramid{i_octave}(:,[1 1:end-1],1:end-1))/2;
            dy{i_octave} = (img_pyramid{i_octave}([2:end end],:,1:end-1) - img_pyramid{i_octave}([1 1:end-1],:,1:end-1))/2;
        end
        
    end

    % Now approximate the Laplacian pyramid using differences of
    % appropriately-spaced Gaussians:
    %     norm_factor = (k-1); % not really necessary for extrema detection
    lap_pyramid{i_octave} = (img_pyramid{i_octave}(:,:,2:end) - img_pyramid{i_octave}(:,:,1:end-1));%/norm_factor;
    
    % Downsample the image (and crack data) to get ready for the next
    % scale octave
    if(use_diffusion)
        [first_level_img, sampled_cracks] = downsample_cracks(img_pyramid{i_octave}(:,:,end-1), sampled_cracks);
    else
        %first_level_img = img_pyramid{i_octave}(1:2:end,1:2:end,end-1);
        temp = img_pyramid{i_octave}(:,:,end-1);
        if(mod(size(temp, 2), 2) == 1)
            temp = temp(:,[1:end end]);
        end
        if(mod(size(temp, 1), 2) == 1)
            temp = temp([1:end end],:);
        end
        first_level_img = (temp(1:2:end,1:2:end) + temp(1:2:end,2:2:end) + temp(2:2:end,1:2:end) + temp(2:2:end,2:2:end))/4;
    end
       
    i_octave = i_octave + 1;
    
end

% Assign outputs
if(use_diffusion)
    if(nargout > 2)
        varargout{1} = crack_pyramid;
        if(do_derivatives)
            varargout{2} = dx;
            varargout{3} = dy;
            if(nargout==6)
                varargout{4} = scales_list;
            end
        elseif(nargout==4)
            varargout{2} = scales_list;
        end
    end
else
    if(do_derivatives)
        varargout{1} = dx;
        varargout{2} = dy;
        if(nargout==5)
            varargout{3} = scales_list;
        end
    elseif(nargout == 3)
        varargout{1} = scales_list;
    end
end
        
            
    
    