;;; markup - hypertext in scheme, producing latex and html
;;; Copyright (C) 1995  Scott Draves <spot@cs.cmu.edu>
;;;
;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2 of the License, or
;;; (at your option) any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.


; JAR's markup language. jar@ai.mit.edu
; Some comments added by Olin Shivers@ai.mit.edu
; features added by Scott `it works for me' Draves

; Source language:
;   file ::= token*
;   token ::= constituent* | other-char | paragraph-break | { token* }
; Semantics:
;   Most things are passed through unchanged.

(define *current-output-file*)

(define (markup backend filename)
  (define (fin fn ext)
    (call-with-input-file (string-append filename ".mar")
      (lambda (in)
	(printf "[reading ~A]~%" filename)
	(let ((f (string-append filename "." ext)))
	  (set! *current-output-file* f)
	  (call-with-output-file f
	    (lambda (out)
	      (do-it fn in out)))))))
  (case backend
    ((html) (fin html-file "html"))
    ((latex) (fin latex-file "tex"))
    ((parse) (fin parse-file "parse"))
    ((pass) (fin pass-items "pass"))
    (else (error "unrecognized backend" backend))))

; Table of contents

(define *toc* #f)
(define *chapters* '())
(define *chapter* #f)
(define *sections* '())
(define *section* #f)
(define *subsections* '())
(define toc-file-name "toc.scm")

(define (do-it back-end in out)
  (set! *toc* #f)
  (start-chapter #f)
  (back-end in out)
  (if *toc*
      (call-with-output-file toc-file-name
	(lambda (toc-out)
	  (write *toc* toc-out)))))

(define (toc)
  (if (not *toc*)
      (set! *toc*
	    (call-with-input-file toc-file-name read)))
  *toc*)

; chapter

(define (chapter name items)
  (end-chapter)
  (start-chapter (cons name items)))

(define (start-chapter ch)
  (set! *chapter* ch)
  (set! *sections* '())
  (start-section #f))

(define (end-chapter)
  (end-section)
  (if (or *chapter*
	  (not (null? *sections*)))
      (set! *chapters*
	    (cons (cons *chapter* (reverse *sections*))
		  *chapters*))))


; section

(define (section name items)
  (end-section)
  (if (not *chapter*)
      '?)
  (start-section (cons name items)))

(define (start-section sec)
  (set! *section* sec)
  (set! *subsections* '()))

(define (end-section)
  (end-subsection)
  (if (or *section*
	  (not (null? *subsections*)))
      (set! *sections*
	    (cons (cons *section* (reverse *subsections*))
		  *sections*))))

; subsection

(define (subsection name items)
  (end-subsection)
  (start-subsection (cons name items)))

(define (start-subsection stuff)
  (set! *subsections* (cons stuff *subsections*)))

(define (end-subsection)
  '?)


; {pass ...}

; ITEMS is a list of items.
; - Chars and strings are simply copied to port OUT.
; - Lists are recursively copied, surrounded by {...} braces.

(define pass-items
  (lambda (items out)
    (let recur ((items items))
      (for-each (lambda (item)
		  (cond ((or (char? item) (string? item))
			 (display item out))
			((null? item)
			 (display "{}" out))
			((pair? item)
			 (if (equal? (car item) "paragraph")
			     (begin (newline out) (newline out))
			     (begin (write-char #\{ out)
				    (recur item)
				    (write-char #\} out))))
			(else (warn "weird item" item))))
		items))))

(define (to-string x)
  (cond ((string? x) x)
	((char? x) (string x))
	;; gag
	((null? x) "{}")
	((pair? x)  (string-append "{" (list-to-string x) "}"))
	(else (call-error "whazza?" to-string x))))

(define (list-to-string x)
  (apply string-append (map to-string x)))

(define (parse-file in out)
  (set! *level* 0)
  (read-items (lambda (item) (format out "~S~%" item))
	      in #t)
  (go-to-level 0 out))

; ----------------------------------------
; Input stream processing

; Read a sequence of items from port IN, and apply ACTION to each one.
; If TOP? is true,  the sequence is supposed to be terminated by EOF.
; If TOP? is false, the sequence is supposed to be terminated by a #\}
; item.

(define (read-items action in top?)
  (let loop ()
    (let ((c (read-char in)))
      (cond ((eof-object? c)
	     (if (not top?)
		 (warn "missing }")))
	    ((char=? c #\})
	     (if top?
		 (begin (warn "extraneous }")
			(action c)
			(loop))))
	    (else
	     ((char-action c) action in)
	     (loop))))))

(define (fast-char-function fun)
  (let ((v (make-vector ascii-limit #f)))
    (do ((i 0 (+ i 1)))
	((= i ascii-limit))
      (vector-set! v i (fun (ascii->char i))))
    (lambda (c) (vector-ref v (char->ascii c)))))

; why isn't this just !white?
; need real distinct lexical environments
(define constituent?
  (fast-char-function
   (let ((winners (string->list ".+-/:?!|")))
     (lambda (c)
       (or (char-alphabetic? c)
	   (char-numeric? c)
	   (memq c winners))))))

; Read a compound item from port IN, 
; and apply ACTION to the list of items read.

(define (read-compound-item action in)	; {command ... }
  (let ((l '()))
    (read-items (lambda (item)
		  (set! l (cons item l)))
		in
		#f)
    (action (reverse l))))

; Suppose we just read a character C.
; (CHAR-ACTION C) returns a function (lambda (action in) ...)
; that does different things depending on the value of C:
; - newline
;   double-newline => ("paragraph")
;   If the following char in the input stream is also newline, 
;   gobble it, and do (ACTION '("paragraph")).
;   Otherwise, just do (ACTION #\newline).
; - {
;   Read a compound item. Apply ACTION to the list read. 
; - constituent char
;   Read a maximal string composed of constituent chars,
;   and apply ACTION to it.
; - otherwise
;   (ACTION C)

(define paragraph-mark '("paragraph"))

(define char-action
  (fast-char-function
   (lambda (c)
     (cond ((char=? c #\newline)
	    (lambda (action in)
	      (action (if (eq? (peek-char in) #\newline)
			  (begin (read-char in)
				 paragraph-mark)
			  c))))
	   ((char=? c #\{)
	    read-compound-item)
	   ((constituent? c)
	    (lambda (action in)
	      (let loop ((l (list c)) (n 1))
		(let ((c (peek-char in)))
		  (if (or (eof-object? c)
			  (not (constituent? c)))
		      (action (reverse-list->string l n))
		      (loop (cons (read-char in) l) (+ n 1)))))))
	   (else
	    (lambda (action in)
	      (action c)))))))

(define (whitespace-delimited-word items)
  (if (null? items)
      ""
      (let ((hd (car items))
	    (tl (cdr items)))
	(if (char? hd)
	    (if (char-whitespace? hd)
		""
		(string-append (string hd)
			       (whitespace-delimited-word tl)))
	    (string-append hd
			   (whitespace-delimited-word tl))))))

(define (item-is-white? item)
  (or (equal? item paragraph-mark)
      (and (char? item)
	   (char-whitespace? item))))

(define (drop-until-whitespace items)
  (if (null? items)
      items
      (if (item-is-white? (car items))
	  items
	  (drop-until-whitespace (cdr items)))))

(define (drop-whitespace items)
  (if (null? items)
      items
      (if (item-is-white? (car items))
	  (drop-whitespace (cdr items))
	  items)))

(define (inserting items back-item out)
  (let ((fn (car items)))
    (call-with-input-file fn
      (lambda (in)
	(display "  [Inserting ") (display fn)
	(read-items (lambda (item)
		      (back-item item out))
		    in #t)
	(display "]")
	(newline)))))


; doesn't handle ()s yet
(define (items->list items)
  (let ((items (drop-whitespace items)))
    (if (null? items)
	'()
	(let ((a (car items))
	      (rest (items->list (cdr items))))
	  (cons (if (pair? a)
		    a
		    (string->sexp a))
		rest)))))

;;; take item-handler instead of list-handler?
(define (if-handler list-handler)
  (lambda (items out)
    (let* ((args (items->list items))
	   (var (car args))
	   (args (cdr args))
	   (nargs (length args)))
      (cond ((= nargs 2)
	     (list-handler (list-ref args (if (markup-eval var) 0 1))
			   out))
	    (else (warn "bad if" items))))))

(define (def-handler table lister)
  (lambda (items out)
    (let ((name (car items))
	  (body (car (drop-whitespace (cdr items)))))
      (table-set! table name
		  (lambda (items out)
		    (lister body out))))))

(define (sort-entry-list items)
  (define (entry->string e)
    (car (drop-whitespace (cdr e))))
  (define (entry-< a b)
    (string-ci<? (entry->string a)
		 (entry->string b)))
  (define (is-entry? e)
     (and (pair? e)
	  (equal? (car e) "entry")))
  (sort-list (filter is-entry? items) entry-<))

(define (field-defined? name object)
  (define (loop flds)
    (cond ((null? flds) #f)
	  ((string=? (car flds) name) #t)
	  (else (loop (cddr flds)))))
  (loop (cdr object)))

; object = (name . alist)
(define (get-field name object . default)
  (define (loop flds)
    (cond ((null? flds) (if (null? default)
			    (error "not found: ~S" name)
			    (car default)))
	  ((string=? (car flds) name)
	   (if (pair? (cdr flds))
	       (car (cdr flds))
	       (error "malformed object: ~S" object)))
	  (else (loop (cddr flds)))))
  (loop (cdr object)))



(define (load-handler items)
  (load (list-to-string (drop-whitespace items))
	(force user-env)))

; (could run dot in background)
(define (dot-handler items out do-item)
  (let* ((label (whitespace-car items))
	 (fname (string-append label ".ps"))
	 (text (whitespace-cdr items)))
    (run (| (begin (pass-items text (current-output-port)))
	    (dot -Tps))
	 (> ,fname))
    (do-item (list "ps" fname) out)))

(define (plot-handler items out do-item)
  (let* ((label (whitespace-car items))
	 (fname (string-append label ".ps"))
	 (text (whitespace-cdr items)))
    (run (| (begin (display "load \"init.plt\"")
		   (newline)
		   (pass-items text (current-output-port)))
	    (gnuplot))
	 (> ,fname))
    (do-item (list "ps" fname) out)))

(define (alias-handler table)
  (lambda (items out)
    (table-set! table
		(car items)
		(table-ref table
			   (car (drop-whitespace (cdr items)))))))

; (from big-util.scm, but works on strings)
(define (remove-dupes list)
  (do ((list list (cdr list))
       (res  '()  (if (member (car list) res)
                      res
                      (cons (car list) res))))
      ((null-list? list)
       res)))

(define (last-word items)
  (cond ((whitespace-null? items) "")
	((whitespace-null? (whitespace-cdr items))
	 (whitespace-car items))
	(else (last-word (whitespace-cdr items)))))

(define *citations* '())


; (make a bib db)
(define (bib-db-handler items do-entry)
  (let* ((numbered *numbered-abc-bib*)
	 (bib0 (strip-whitespace items))
	 (bib (map strip-whitespace bib0)) ; surely wrong XXX
	 (cits (remove-dupes *citations*))
	 (cit->entry (lambda (c) (or (assoc c bib)
				     `(,c "type" "undefined"))))
	 (by-tag (lambda (a b)
		   (string-ci<? (car a) (car b))))
	 (key (lambda (entry)
		(or (let ((a (get-field "author" entry #f)))
		      (and a (if (atom? (whitespace-car a))
				 (last-word a)
				 (last-word (whitespace-car a)))))
		    (let ((a (get-field "title" entry #f)))
		      (and a (apply string-append (strip-whitespace a))))
		    (car entry))))
	 (by-first-author (lambda (a b)
			    (string-ci<? (key a) (key b))))
	 (sorted-cits (map (lambda (e)
			     (do-entry (car e) (format-bib-entry e))
			     (car e))
			   (sort-list (map cit->entry cits)
				      (if numbered
					  by-first-author
					  by-tag)))))
    (if numbered
	(make-forward-file sorted-cits))))

(define *bib-show-urls* #f)

(define (format-bib-entry entry)
  (let ((type (get-field "type" entry))
	(comma-list one-or-comma-list-macro)
	(dot #\.) (comma #\,) (space #\space) (colon #\:)
	(def? (lambda (s) (field-defined? s entry)))
	(field (lambda (s) (get-field s entry '("?"))))
	(name (whitespace-car entry)))
    (append
     (cond ((string=? "TR" type)
	    `(,@(comma-list (field "author"))
	      ,dot ,space
	      ,@(field "title") ,dot ,space
	      ,@(field "tr-id") ,dot))
	   ((string=? "in-proceedings" type)
	    `(,@(comma-list (field "author"))
	      ,dot ,space
	      ,@(field "title") ,dot ,space
	      ("em" ,@(field "book-title"))
	      ,comma ,space ,@(field "year") ,dot))
	   ((string=? "article" type)
	    `(,@(comma-list (field "author"))
	      ,dot ,space
	      ,@(field "title") ,dot ,space
	      ("em" ,@(field "journal")) ,comma ,space
	      ,@(field "volume") ,colon ,@(field "pages") ,dot))
	   ((string=? "phd-thesis" type)
	    `(,@(comma-list (field "author"))
	      ,dot ,space
	      ("em" ,@(field "title")) ,dot ,space
	      ,@(field "school") ,space ,@(field "year") ,dot))
	   ((string=? "book" type)
	    `(,@(comma-list (field "author"))
	      ,dot ,space
	      ("em" ,@(field "title")) ,dot ,space
	      ,@(field "publisher") ,space ,@(field "year") ,dot))
	   ((string=? "software" type)
	    `(,@(field "title")
	      ,dot ,space ,@(field "publisher") ,space
	      ,@(field "year") ,dot))
	   ((string=? "undefined" type)
	    `(,(car entry) ,space "undefined" ,dot))
	   (else
	    `(,name "has bad type type: " ,type)))
     (if (and *bib-show-urls* (def? "url"))
	 `(,space ("link" ,space ,@(field "url") ,space "original") ,dot)
	 '())
     (if (and *bib-show-urls* (def? "local-url"))
	 `(,space ("link" ,space ,@(field "local-url") ,space "local") ,dot)
	 '()))))
  
(define (whitespace-null? items)
  (null? (drop-whitespace items)))

(define (whitespace-car items)
  (let ((d (drop-whitespace items)))
    (car d)))

(define (whitespace-cdr items)
  (let ((d (drop-whitespace items)))
    (drop-until-whitespace (cdr d))))

; seems familiar...
(define (strip-whitespace items)
  (if (whitespace-null? items)
      '()
      (cons (whitespace-car items)
	     (strip-whitespace (whitespace-cdr items)))))

(define (one-or-comma-list-macro items)
  (cond ((atom? (whitespace-car items)) items)
	(else (append (whitespace-car items)
		      (let ((n (whitespace-cdr items)))
			(if (whitespace-null? n)
			    '()
			    (append '(#\, #\space)
				    (one-or-comma-list-macro n))))))))

(define (integer x)
  (let ((tr (truncate x)))
    (if (exact? tr) tr
	(inexact->exact tr))))

(define (distance->pixels s)
  (let* ((l (string-length s))
	 (units (substring s (- l 2) l))
	 (conv (cadr (or (assoc units '(("in" 72.0) ("mm" 2.83) ("cm" 28.3)))
			 '("none" 1))))
	 (n (string->number (substring s 0 (- l 2)))))
    (integer (* n conv))))
	 
    
	 


(define winner "----------------------------------------")    ;40

