;;; markup - hypertext in scheme, producing latex and html
;;; Copyright (C) 1995  Scott Draves <spot@cs.cmu.edu>
;;;
;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2 of the License, or
;;; (at your option) any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.


;;; table-driver graphical data-structure layout.
;;; install handlers for your own types

(define-structure layout
  (export layout->ps
	  layout->ps*
	  layout->dot
	  layout->dot-begin
	  layout-add-type
	  reset-layout-fns
	  display-layout)
  (open dot quick-stdio spot-util match fluids tables features scsh scheme)
  (begin

    ;; damn native hash tables don't handle pairs, so we have to build
    ;; our own.

    (define (hash-sexpr val)
      (define (loop val n)
	(if (< n 5)
	    (match val
		   ((a . d) (let ((n (+ 1 n)))
			      (+ (* 2 (loop a n)) (loop d n))))
		   (_ (hash-atom val)))
	    1))
      (loop val 0))

    (define (hash-atom obj)
      (cond ((symbol? obj) (string-hash (symbol->string obj)))
	    ((integer? obj)
	     (if (< obj 0) (- -1 obj) obj))
	    ((char? obj) (+ 333 (char->integer obj)))
	    ((eq? obj #f) 3001)
	    ((eq? obj #t) 3003)
	    ((null? obj) 3005)
	    (else 0)))

    (define make-sexpr-table
      (make-table-maker eq? hash-sexpr))

    (define (display-layout . args)
      (let ((tmp (create-temp-file "gv")))
	(with-out-file tmp
		       (apply layout->ps args)
		       (run (gv ,tmp)))
	(delete-file tmp)))

    (define (pair->dot val name)
      (match val
	     ((a . d)
	      ;; order important
	      (let* ((a-lab (layout->dot a (string-append name ":a")))
		     (d-lab (layout->dot d (string-append name ":d"))))
		(cons-box name a-lab d-lab))
	      #t)
	     (_ #f)))
    

    ;; list of functions, one for each type.  they are called in
    ;; sequence on a value, until one handles it and returns a true
    ;; value. when a object system is settled on, then use methods.
    (define type-layout-fns (list pair->dot))

    (define (layout-add-type fn)
      (set! type-layout-fns
	    (cons fn type-layout-fns)))

    (define (reset-layout-fns)
      (set! type-layout-fns (list pair->dot)))

    (define seen-fluid-table (make-fluid #f))

    ;; laysout VAL, including the line from AT.  returns a label
    ;; string, writes dot code to stdout.  if the value is drawn
    ;; boxed, the dot output will include the arrow *to* VAL from AT.
    ;; AT is the context, a node name[:label], just a string.  this is
    ;; a terrible dependence on dot.  otps is just the link style
    (define (layout->dot val at . opts)
      (define (ln to)
	;; should be in dot package
	((match opts
	   (('dotted) link-dotted)
	   ((or ('solid) ()) link-solid))
	 at to)
	"")
      (cond ((and (pair? val)
		  (table-ref (fluid seen-fluid-table) val))
	     => ln)
	    (else (let ((name (new-node-name)))
		    (if (pair? val)
			(table-set! (fluid seen-fluid-table)
				    val name))
		    (let loop ((fns type-layout-fns))
		      (match fns
			     (() (dot-quote-string (format #f "~S" val)))
			     ((fn . rest)
			      (if (fn val name)
				  (ln name)
				  (loop rest)))))))))

    (define (layout->dot-begin val name)
      (let-fluid seen-fluid-table
		 (make-sexpr-table)
		 (lambda ()
		   (layout->dot val name))))
			  

    ;; (layout->ps val ['lr] [h [w]])
    ;; writes ps to stdout
    (define (layout->ps val . opts)
      (let parse-args ((lr #f) (opts opts))
	(define (doit w h)
	  (let ((head (string-append "digraph g {~%"
				     "center=1;~%"
				     "ordering=out;~%"
				     (if lr "rankdir=LR;~%" "")
				     "size=\"~S,~S\";~%"
				     "page=\"8.5,11\";~%"))
		(tail "}\n"))
	    (run (| (begin (printf head w h)
			   (let-fluid seen-fluid-table
				      (make-sexpr-table)
				      make-entry-point)
			   (printf tail))
		    (dot -Tps))
		 stdports)))
	(define (make-entry-point)
	  (let* ((at (new-node-name))
		 (lab (layout->dot val at)))
	    (bullet at lab)))
	(match opts
	       (('lr . opts) (parse-args #t opts))
	       ('() (doit 7.5 10))
	       ((h) (doit 5 h)) ; fit latex column
	       ((w h) (doit w h)))))

    (define (layout->ps* vals . opts)
      (let parse-args ((lr #f) (opts opts))
	(define (doit w h)
	  (let ((head (string-append "digraph g {~%"
				     "center=1;~%"
				     "ordering=out;~%"
				     (if lr "rankdir=LR;~%" "")
				     "size=\"~S,~S\";~%"
				     "page=\"8.5,11\";~%"))
		(tail "}\n"))
	    (run (| (begin (printf head w h)
			   (for-each
			    (lambda (v)
			      (define (make-entry-point)
				(let* ((at (new-node-name))
				       (lab (layout->dot v at)))
				  (bullet at lab)))
			      (let-fluid seen-fluid-table
					 (make-sexpr-table)
					 make-entry-point))
			    vals)
			   (printf tail))
		    (dot -Tps))
		 stdports)))
	
	(match opts
	       (('lr . opts) (parse-args #t opts))
	       ('() (doit 7.5 10))
	       ((h) (doit 5 h)) ; fit latex column
	       ((w h) (doit w h)))))
      

    ))
