;;; markup - hypertext in scheme, producing latex and html
;;; Copyright (C) 1995  Scott Draves <spot@cs.cmu.edu>
;;;
;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2 of the License, or
;;; (at your option) any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

; Latex generation

(define *in-quotation?* #f)

(define (latex-file in out)
  (set! *in-quotation?* #f)
  (read-items (lambda (item) (latex item out))
	      in #t))

; anything that you {def } goes into here, and stays, even across
; multiple invocations?
(define latex-emitters (make-string-table))

(define (define-latex-emitter name proc)
  ;; assumes symbol names are lower case
  (table-set! latex-emitters (symbol->string name) proc))



(define (latex item out)
  (cond ((pair? item)
	 (if (string? (car item))
	     (let ((probe (table-ref latex-emitters (car item)))
		   ;; Skip whitespace following command name
		   (items (drop-whitespace (cdr item))))
	       (if probe
		   (probe items out)
		   (begin (warn "unrecognized command name" (car item))
			  ((latex-command-emitter (car item))
			   items
			   out))))
	     (begin (warn "peculiar item" (car item))
		    (latex-list item out))))
	((char? item)
	 (latex-write-char item out))
	((string? item)
	 (if (string=? item winner)     ;Kludge for code --> transparencies
	     (display "\\newpage" out)
	     (display item out)))
	(else (warn "unrecognized item" item))))

(define (latex-write-char item out)
  (case item
    ;; ((#\\) (display "\\backslash{}" out))
    ((#\<) (display "$<$" out)) ; XXX don't need some of these, right?
    ((#\>) (display "$>$" out)) ; XXX either remove them or restore $ in code env
    ((#\&) (display "\\&" out))
    ((#\#) (display "\\#" out))
    ((#\$) (display "\\$" out))
    ; ((#\") (display (if *in-quotation?* "''" "``") out)
	;   (set! *in-quotation?* (not *in-quotation?*)))
    ((#\%) (display "\\%" out))
    ((#\_) (display "\\_" out))
    ;; ^ ~
    (else (write-char item out))))

(define (latex-list items out)
  (for-each (lambda (item) (latex item out))
	    items))

(define (latex-command-emitter name)
  (lambda (items out)
    (emit-latex-command name items out)))

(define (emit-latex-command name items out)
  (write-char #\\ out) (display name out) (write-char #\{ out)
  (latex-list items out)
  (write-char #\} out))

(define (latex-section-emitter kind)    ;\section{bar}\label{foo}
  (lambda (items out)
    (let ((label (car items))
	  (items (drop-whitespace (cdr items))))
      (write-char #\\ out) (display kind out) (write-char #\{ out)
      (latex-list items out)
      (display "}\\label{" out)
      (display label out)
      (write-char #\} out))))

(define (latex-group-emitter header)
  (lambda (items out)
    (write-char #\{ out) (display header out)
    (latex-list items out)
    (write-char #\} out)))

(define (latex-environment-emitter kind)
  (lambda (items out)
    (display "\\begin{" out) (display kind out) (write-char #\} out)
    (latex-list items out)
    (display "\\end{" out) (display kind out) (write-char #\} out)))

(define (latex-itemized-emitter kind)
  (lambda (items out)
    (display "\\begin{" out) (display kind out) (write-char #\} out)
    (let ((item-marker (car items)))
      (for-each (lambda (item)
		  (if (equal? item item-marker)
		      (display "\\item{}" out)
		      (latex item out)))
		items))
    (display "\\end{" out) (display kind out) (write-char #\} out)))

(define (latex-accent-emitter kind)
  (lambda (items out)
    (let ((char (car items)))
      (if (not (= 1 (string-length char)))
	  (warn kind " accent of more than one character" char))
      (format out "\\~A~A" kind char))))

(define-latex-emitter 'alias
  (alias-handler latex-emitters))

(define-latex-emitter 'begin
  latex-list)

(define-latex-emitter 'brace		;{...}
  (lambda (items out)
    (display "\\{" out)
    (latex-list items out)
    (display "\\}" out)))


(define-latex-emitter 'angle
  (lambda (items out)
    (display "$\\langle$" out)
    (latex-list items out)
    (display "$\\rangle$" out)))

(define-latex-emitter 'lbrace		;left curly brace
  (lambda (items out)
    (display "\\{" out)))
(define-latex-emitter 'rbrace		;right curly brace
  (lambda (items out)
    (display "\\}" out)))

(define-latex-emitter 'c		;Code font
  (latex-group-emitter "\\tt{}"))

(define-latex-emitter 'chapter
  (latex-section-emitter "chapter"))

(define *numbered-abc-bib* #f)
(define *forward-file* "forward-decls.tex")

(define (make-forward-file cits)
  (with-out-file
   *forward-file*
   (let loop ((cits cits) (n 1))
     (cond ((null? cits) 'ok)
	   (else (format #t "\\newcommand{\\~A}{~S}~%"
			 (latex-encode-tag (car cits)) n)
		 (loop  (cdr cits) (+ n 1)))))))

; gag XXX
(define (latex-encode-tag s)
  (string-append "tag"
		 (list->string
		  (map (lambda (c)
			 (if (char-alphabetic? c)
			     c
			     (ascii->char
			      (+ (char->ascii #\a)
				 (remainder (char->ascii c)
					    (- (char->ascii #\z)
					       (char->ascii
						#\a)))))))
		       (string->list s)))))

(define-latex-emitter 'cite
  (lambda (items out)
    (let ((tag (whitespace-car items)))
      (set! *citations* (cons tag *citations*))
      (write-char #\[ out)
      (if *numbered-abc-bib*
	  (format out "\\~A" (latex-encode-tag tag))
	  (latex tag out))
      (write-char #\] out))))

(define-latex-emitter 'code		;Needs \input{code.tex}
  (lambda (items out)
    (let ((kind "code"))
      (display "\\begin{" out) (display kind out) (write-char #\} out)
      (for-each (lambda (item)
		  (latex-code-item item out))
		items)
      (display "\\end{" out) (display kind out) (write-char #\} out))))

(define-latex-emitter 'small-code
  (lambda (items out)
    (let ((kind "code"))
      (display "\\begin{" out) (display kind out) (write-char #\} out)
      (display "\\small{}" out)
      (for-each (lambda (item)
		  (latex-code-item item out))
		items)
      (display "\\normalsize{}" out)
      (display "\\end{" out) (display kind out) (write-char #\} out))))

(define-latex-emitter 'code2		;Needs \input{code.tex}
  (lambda (items out)
    (let ((kind "code*"))
      (display "\\begin{" out) (display kind out) (write-char #\} out)
      (for-each (lambda (item)
		  (latex-code-item item out))
		items)
      (display "\\end{" out) (display kind out) (write-char #\} out))))

(define (latex-code-item item out)
  (cond ((not (pair? item))
	 (cond ((string? item)
		(latex-code-string-item item out))
	       ((memq item '(#\? #\! #\.))
	        ;; Try to prevent extra space following.
		(display "\\hbox{" out)
		(write-char item out)
		(write-char #\} out))
	       ((memq item '(#\" #\< #\> #\_))
		(write-char item out))
	       ((eq? item #\\)
		(display "\\char`\\\\" out))
	       (else (latex item out))))
	((equal? (car item) "paragraph")
	 (codeskip out))
	((equal? (car item) "insert")
	 (inserting (drop-whitespace (cdr item))
		    latex-code-item
		    out))
	(else
	 (latex item out))))

(define (latex-code-string-item item out)
  (let ((probe (table-ref latex-code-string-emitters item)))
    (if probe
	(display probe out)
	(latex item out))))

(define latex-code-string-emitters (make-string-table))

(define (define-latex-code-string-emitter name string)
  ;; assumes symbol names are lower case
  (table-set! latex-code-string-emitters (symbol->string name) string))

; (define-latex-code-string-emitter 'lambda "\hbox{$\lambda$}")



(define (codeskip out)
  (newline out)
  (display "\\codeskip" out)
  (newline out))

(define-latex-emitter 'codeskip
  (lambda (items out)
    (codeskip out)))

(define-latex-emitter 'comment		;No action
  (lambda (items out)
    (display "{}" out)))  ;prevent paragraph breaks

(define-latex-emitter 'hrule		;Horizontal rule
  (lambda (items out)
    (newline out) (newline out) ;vertical
    (display "\\bigskip\\hrule\\bigskip" out)
    (newline out) (newline out)))

(define-latex-emitter 'newline
  (lambda (items out)
    (display "\\\\" out)))

(define-latex-emitter 'skip
  (lambda (items out)
    (display "\\medskip" out)))

(define-latex-emitter 'ednote		;Editorial note
  (lambda (items out)
    (emit-latex-command (if (< (length items) 25)
			    "ednote"
			    "longednote")
			items out)))

(define emit-latex-emphasized
  (lambda (items out)
    (display "{\\em{}" out)
    (latex-list items out)
    (display "\\/}" out)))    ;italic correction

(define-latex-emitter 'em emit-latex-emphasized) 

(define-latex-emitter 'center
  (latex-environment-emitter "center"))

(define-latex-emitter 'enumerate	;Numbered list
  (latex-itemized-emitter "enumerate"))

(define-latex-emitter 'definitions
  (lambda (items out)
    (write-string "\\begin{description}" out)
    (let ((item-marker (car items)))
      (let loop ((items items))
	(cond ((null? items)
	       (write-string "\\end{description}" out))
	      ((equal? (car items) item-marker)
	       (display "\\item[" out)
	       (latex-list (car (drop-whitespace (cdr items)))
			  out)
	       (display "]" out)
	       (loop (cdr (drop-whitespace (cdr items)))))
	      (else (latex (car items) out)
		    (loop (cdr items))))))))

; XXX i don't have the latex defn of this
; (define-latex-emitter 'evalsto    	; -->
  ; (latex-command-emitter "evalsto"))

(define-latex-emitter 'evalsto
  (lambda (items out) (write-string "$\\rightarrow$" out)))

(define-latex-emitter 'syntax-production
  (lambda (items out) (write-string "$\\longrightarrow$" out)))

(define-latex-emitter 'non-terminal
  (lambda (items out)
    (display "{\\rm{}\\em{}" out)
    (latex-list items out)
    (display "\\/}" out))) 

(define-latex-emitter 'lattice-lt-eq
  (lambda (items out) (write-string "$\\sqsubseteq$" out)))
(define-latex-emitter 'lattice-lt
  (lambda (items out) (write-string "$\\sqsubset$" out)))
(define-latex-emitter 'iff
  (lambda (items out) (write-string "$\\Leftrightarrow$" out)))
(define-latex-emitter 'abstract
  (latex-environment-emitter "abstract"))

(define-latex-emitter 'grave (latex-accent-emitter "`"))
(define-latex-emitter 'acute (latex-accent-emitter "'"))
(define-latex-emitter 'tilde (latex-accent-emitter "~"))
(define-latex-emitter 'cedilla (latex-accent-emitter "c"))
(define-latex-emitter 'circumflex (latex-accent-emitter "^"))
(define-latex-emitter 'umlaut (latex-accent-emitter "\""))
(define-latex-emitter 'slash
  (lambda (items out)
    (let ((char (car items)))
      (format out "{\\~A}" char))))
(define-latex-emitter 'ring
  (lambda (items out)
    (let ((char (car items)))
      (format out "{\\~A~A}" char char))))


(define-latex-emitter 'footnote		;Footnote
  (latex-command-emitter "footnote"))

(define-latex-emitter 'insert
  (lambda (items out)
    (inserting items latex out)))

(define-latex-emitter 'itemize		;Unnumbered list
  (latex-itemized-emitter "itemize"))

(define-latex-emitter 'label		;Anchor
  (latex-command-emitter "label"))

(define-latex-emitter 'link		;Hypertext link {link URL text}
  (lambda (items out)
    (let ((url (whitespace-delimited-word items))
	  (text (whitespace-cdr items)))
      (if (whitespace-null? text)
	  (format out "{\\tt ~A}" url)
	  (latex-list text out)))))

(define-latex-emitter 'ref
  (lambda (items out)
    (let ((tag (car items)))
      (write-string tag out))))

(define (latex-sorted-entry-emitter kind)
  (lambda (items out)
    (format out "\\begin{~A}" kind)
    (latex-list (sort-entry-list items) out)
    (format out "\\end{~A}" kind)))

(define-latex-emitter 'bib (latex-sorted-entry-emitter "description"))
(define-latex-emitter 'notes (latex-sorted-entry-emitter "description"))

(define-latex-emitter 'entry
  (lambda (items out)
    (let ((tag (car items)))
      (format out "\\item[~A]" tag)
      (latex-list (cdr items) out))))

(define-latex-emitter 'note
  (lambda (items out)
    (let ((tag (car items)))
      (format out "[note ~A]" tag))))

(define (latex-do-bib-entry out)
  (lambda (tag entry)
    (format out "\\bibitem[~A]{~A}" tag tag)
    (latex-list entry out)
    (newline out)))

(define-latex-emitter 'bib-db
  (lambda (items out)
    (format out "\\begin{thebibliography}{99}~%")
    (bib-db-handler items (latex-do-bib-entry out))
    (format out "\\end{thebibliography}")))

(define latex-code-macros-location "code")
(define latex-use-times-package #t)

(define-latex-emitter 'document
  (lambda (items out)
    (let* ((doc (strip-whitespace items))
	   (class (get-field "style" doc))
	   (options (cond ((string=? class "article") "[twocolumn]")
			  ((string=? class "book") "[12pt]")
			  (else ""))))
      (set! *citations* '())
      (format out "\\documentclass~A{~A}~%" options class)
      (format out "\\usepackage{~Aepsfig,amssymb}~%~A~%"
	      (if latex-use-times-package "times," "")
	      tex-init-string)
      (if (string=? class "book")
	  (format out
		  (string-append
		   "\\newenvironment{abstract}{\\titlepage \\null\\vfil"
		   "\\begin{center}\\bfseries Abstract \\end{center}}~%~%")))
      (format out "\\title{")
      (latex-list (get-field "title" doc) out)
      (format out "}~%\\author{")
      (latex-list (get-field "author" doc) out)
      (cond ((field-defined? "thanks" doc)
	     (format out "\\thanks{")
	     (latex-list (get-field "thanks" doc) out)
	     (format out "}~%")))
      (cond ((and (string=? "article" class)
		  (field-defined? "institute" doc))
	     (format out "\\\\")
	     (latex-list (get-field "institute" doc) out)))
      (format out "}~%")
      (cond ((string=? "llncs" class)
	     (format out "\\institute{")
	     (latex-list (get-field "institute" doc) out)
	     (format out "}~%")))
      (format out "\\begin{document}\\maketitle~%")
      (if *numbered-abc-bib* (format out "\\input{~A}~%" *forward-file*))
      (if (field-defined? "body" doc)
	  (latex-list (get-field "body" doc) out))
      (if (field-defined? "files" doc)
	  (for-each (lambda (f)
		      (format out "\\input{~A.tex}~%" f)
		      (markup 'latex f))
		    (strip-whitespace (get-field "files" doc))))
      (format out "\\end{document}~%"))))

(define-latex-emitter 'minipage
  (lambda (items out)
    (let ((w (whitespace-car items)))
      (format out "\\begin{minipage}{~A}" w)
      (latex-list (whitespace-cdr items) out)
      (format out "\\end{minipage}"))))

(define-latex-emitter 'hspace
  (lambda (items out)
    (format out "\\hspace{~A}" (whitespace-car items))))

(define-latex-emitter 'vspace
  (lambda (items out)
    (format out "\\vspace{~A}" (whitespace-car items))))

(define-latex-emitter 'see
  (lambda (items out)
    (let ((tag (car items)))
      (format out "(see ~A)" tag))))

;;; (replace with <--> icon)
(define-latex-emitter 'expand
  (lambda (items out)
    (let ((tag (car items)))
      (format out "(see ~A for more)" tag))))


(define-latex-emitter 'load
  (lambda (items out)
    (load-handler items)))

(define-latex-emitter 'include
  (lambda (items out)
    (let ((f (whitespace-car items)))
      (format out "\\input{~A.tex}" f)
      (markup 'latex f))))

(define-latex-emitter 'dot
  (lambda (items out)
    (dot-handler items out latex)))

(define-latex-emitter 'plot
  (lambda (items out)
    (plot-handler items out latex)))

(define user-env
  (delay (let* ((env (interaction-environment))
		(xfer (lambda (name val)
			(eval `(define ,name #f) env)
			((eval `(lambda (_x) (set! ,name _x)) env) val))))
	   (xfer 'define-latex-code-string-emitter
		 define-latex-code-string-emitter)
	   env)))

(define-latex-emitter 'm		;Math mode
  (lambda (items out)
    (write-char #\$ out)
    (pass-items items out)    ; for _ < > etc.
    (write-char #\$ out)))

(define-latex-emitter 'mm		;Math display
  (lambda (items out)
    (display "$$" out)
    (pass-items items out)
    (display "$$" out)))

(define-latex-emitter 'as-image
  (lambda (items out)
    (latex-list items out)))

(define-latex-emitter 'paragraph
  (lambda (items out)
    (newline out) (newline out)))

(define-latex-emitter 'part
  (lambda (items out)
    (display "\\input{" out)
    (display (car (drop-whitespace items)) out)
    (display ".tex}" out)))

(define-latex-emitter 'pass pass-items)

(define-latex-emitter 'quote
  (latex-environment-emitter "quote"))

(define-latex-emitter 'refer		;Cross-ref number
  (latex-command-emitter "ref"))

(define-latex-emitter 'section
  (latex-section-emitter "section"))

(define-latex-emitter 'subsection
  (latex-section-emitter "subsection"))

(define-latex-emitter 'subsubsection
  (latex-section-emitter "subsubsection"))

(define-latex-emitter 'strong
  (latex-group-emitter "\\bf{}"))

(define-latex-emitter 'title		;HTML only
  (lambda (items out) #f))

(define-latex-emitter 'var		;Variable name
  emit-latex-emphasized)

(define-latex-emitter 'when-latex
  (lambda (items out)
    (latex-list items out)))

(define-latex-emitter 'when-html
  (lambda (items out) #f))

(define-latex-emitter 'write		;for debugging
  (lambda (items out)
    (write items out)
    (newline out)))

(define-latex-emitter 'ps
  (lambda (items out)
    (let ((psfile (car items)))
      ;; scaling?
      (format out "\\psfig{figure=~A}" psfile))))

(define-latex-emitter 'table
  (lambda (items out)
    (define (last? x)
      (whitespace-null? (whitespace-cdr x)))
    (let ((column-spec (whitespace-car (whitespace-car items))))
      (format out "\\begin{tabular}{~A}~%" column-spec)
      (let next-row
	  ((rows (whitespace-cdr items)))
	(let next-entry
	    ((entries (whitespace-car rows)))
	  (if (equal? "span" (whitespace-car entries))
	      (let* ((d (whitespace-cdr entries))
		     (n (string->number
			 (whitespace-car d)))
		     (dd (whitespace-cdr d)))
		(format out "\\multicolumn{~A}{c}{" n)
		(latex-list (whitespace-car dd) out)
		(format out "}")
		(cond ((last? dd) 'ok)
		      (else (format out "&")
			    (next-entry (whitespace-cdr dd)))))
	      (begin
		(latex-list (whitespace-car entries) out)
		(cond ((last? entries) 'ok)
		      (else (format out "&")
			    (next-entry (whitespace-cdr entries)))))))
	(cond ((last? rows)
	       (format out "~%\\end{tabular}"))
	      (else (format out "\\\\~%")
		    (next-row (whitespace-cdr rows))))))))

; first arg should be label
(define-latex-emitter 'figure
  (lambda (items out)
    (let ((label (car items))
	  (items (cdr items)))
      (format out "\\begin{figure}")
      (latex (whitespace-car items) out)
      (format out "\\caption{")
      (format out "\\label{~A}" label)
      (latex-list (whitespace-cdr items) out)
      (format out "}\\end{figure}"))))

(define-latex-emitter 'figure-two-column
  (lambda (items out)
    (let ((label (car items))
	  (items (cdr items)))
      (format out "\\begin{figure*}")
      (latex (whitespace-car items) out)
      (format out "\\caption{")
      (format out "\\label{~A}" label)
      (latex-list (whitespace-cdr items) out)
      (format out "}\\end{figure*}"))))

(define-latex-emitter 'frame-box
  (lambda (items out)
    (format out "\\fbox{")
    (latex-list items out)
    (format out "}")))

; negspaces: three in times, two in cmr
(define-latex-emitter 'syntax-bracket
  (lambda (items out)
    (format out "[\\negthinspace{}\\negthinspace{}[")
    (latex-list items out)
    (format out "]\\negthinspace{}\\negthinspace{}]")))

(define-latex-emitter 'if (if-handler latex-list))
(define-latex-emitter 'def (def-handler latex-emitters latex-list))

