;;; markup - hypertext in scheme, producing latex and html
;;; Copyright (C) 1995  Scott Draves <spot@cs.cmu.edu>
;;;
;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2 of the License, or
;;; (at your option) any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

; HTML generation

(define *level* 0)
(define *html-tex-image-counter* 0)
(define *html-tex-filename*)
(define *html-tex-memo-table*)

(define (html-file in out)
  (set! *level* 0)
  (read-items (lambda (item) (html item out))
	      in #t)
  (go-to-level 0 out))

(define html-emitters (make-string-table))

; maps labels to filenames
(define html-refer-table (make-string-table))

(define (remember-label lab)
  (table-set! html-refer-table lab
	      *current-output-file*))

(define (define-html-emitter name proc)
  ;; assumes symbol names are lower case
  (table-set! html-emitters (symbol->string name) proc))

(define (html item out)
  (cond ((pair? item)
	 (if (string? (car item))
	     (let ((probe (table-ref html-emitters (car item)))
		   ;; Skip whitespace following command name
		   (items (drop-whitespace (cdr item))))
	       (if probe
		   (probe items out)
		   (begin (warn "unrecognized command name" (car item))
			  (html-list item out))))
	     (begin (warn "peculiar item" (car item))
		    (html-list item out))))
	((char? item)
	 (html-write-char item out))
	((string? item)
	 (if (string=? item winner)     ;Kludge for code --> transparencies
	     (display "<hr>" out)
	     (display item out)))
	(else (warn "unrecognized item" item))))

(define (html-write-char item out)
  (case item
    ((#\<) (display "&lt;" out))
    ((#\>) (display "&gt;" out))
    ((#\&) (display "&amp;" out))
    (else (write-char item out))))

(define (html-list items out)
  (for-each (lambda (item) (html item out))
	    items))

(define (html-environment-emitter kind)
  (lambda (items out)
    (write-char #\< out) (display kind out) (write-char #\> out)
    (html-list items out)
    (display "</"   out) (display kind out) (write-char #\> out)))

(define (html-section-emitter kind)
  (lambda (items out)
    (let ((label (car items))
	  (items (cdr items)))
      (remember-label label)
      (display "<a name=\"" out)
      (display label out)
      (display "\"><" out) (display kind out) (write-char #\> out)
      (html-list items out)
      (display "</"   out) (display kind out)
      (display "></a>" out))))

(define (html-itemized-emitter kind)
  (lambda (items out)
    (write-char #\< out) (display kind out) (write-char #\> out)
    (let ((item-marker (car items)))
      (for-each (lambda (item)
		  (if (equal? item item-marker)
		      (display "<li>" out)
		      (html item out)))
		items))
    (display "</"   out) (display kind out) (write-char #\> out)))

(define (html-accent-emitter kind)
  (lambda (items out)
    (let ((char (car items)))
      (if (not (= 1 (string-length char)))
	  (warn kind " accent of more than one character" char))
      (format out "&~A~A;" char kind))))

(define-html-emitter 'alias
  (alias-handler html-emitters))

(define-html-emitter 'begin
  html-list)

(define-html-emitter 'brace		;{...}
  (lambda (items out)
    (write-char #\{ out)
    (html-list items out)
    (write-char #\} out)))

(define-html-emitter 'lbrace		;left curly brace
  (lambda (items out)
    (write-char #\{ out)))

(define-html-emitter 'rbrace		;right curly brace
  (lambda (items out)
    (write-char #\} out)))

(define-html-emitter 'c (html-environment-emitter "code"))
(define-html-emitter 'chapter (html-section-emitter "h1"))

; just assuming bib.html XXX
(define-html-emitter 'cite
  (lambda (items out)
    (let ((tag (whitespace-car items)))
      (set! *citations* (cons tag *citations*))
      (format out "<a href=\"bib.html#~A\">[~A]</a>"
	      tag tag))))

(define (html-sorted-entry-emitter kind)
  (lambda (items out)
    (format out "<~A>" kind)
    (html-list (sort-entry-list items) out)
    (format out "</~A>" kind)))

(define-html-emitter 'notes (html-sorted-entry-emitter "dl"))

(define-html-emitter 'code		;Code display
  (lambda (items out)
    (display "<pre>" out)
    (html-list items out)
    (display "</pre>" out)))

(define-html-emitter 'code2		;Code display
  (lambda (items out)
    (display "<pre>" out)
    (html-list items out)
    (display "</pre>" out)))

(define-html-emitter 'angle
  (lambda (items out)
    (display "&lt;" out)
    (html-list items out)
    (display "&gt;" out)))

(define-html-emitter 'comment		;No action
  (lambda (items out) #f))

(define-html-emitter 'ednote		;Editorial note
  (lambda (items out)
    (write-char #\[ out)
    (html-list items out)
    (write-char #\] out)))

(define-html-emitter 'em		;Emphasized
  (html-environment-emitter "em"))

(define-html-emitter 'center
  (html-environment-emitter "center"))

(define-html-emitter 'enumerate		;Numbered list
  (html-itemized-emitter "ol"))

;;; fails if blank lines follow immediately XXX
;;; count {paragraph} as whitespace?
(define-html-emitter 'definitions
  (lambda (items out)
    (write-string "<dl>" out)
    (let ((item-marker (car items)))
      (let loop ((items items))
	(cond ((null? items)
	       (write-string "</dl>" out))
	      ((equal? (car items) item-marker)
	       (display "<dt>" out)
	       (html-list (car (drop-whitespace (cdr items)))
			  out)
	       (display "<dd>" out)
	       (loop (cdr (drop-whitespace (cdr items)))))
	      (else (html (car items) out)
		    (loop (cdr items))))))))

; i can't find any way in HTML to force hspace
(define-html-emitter 'hspace
  (lambda (items out)
    (display " " out)))

(define-html-emitter 'evalsto
  (lambda (items out)
    (display "-->" out)))

(define-html-emitter 'syntax-production
  (lambda (items out)
    (display "::=" out)))

(define-html-emitter 'syntax-bracket
  (lambda (items out)
    (format out "[[")
    (html-list items out)
    (format out "]]")))

(define-html-emitter 'frame-box
  (if #f
      (lambda (items out)
	(format out "<table border><td>")
	(html-list items out)
	(format out "</td></table>"))
      (lambda (items out)
	(let ((s (call-with-string-output-port
		  (lambda (p) (latex-list items p)))))
	  (tex->html (string-append "\\fbox{" s "}") out)))))

(define-html-emitter 'non-terminal
  (html-environment-emitter "var"))

(define-html-emitter 'grave (html-accent-emitter "grave"))
(define-html-emitter 'acute (html-accent-emitter "acute"))
(define-html-emitter 'tilde (html-accent-emitter "tilde"))
(define-html-emitter 'cedilla (html-accent-emitter "cedil"))
(define-html-emitter 'circumflex (html-accent-emitter "circ"))
(define-html-emitter 'umlaut (html-accent-emitter "uml"))
(define-html-emitter 'slash (html-accent-emitter "slash"))
(define-html-emitter 'ring (html-accent-emitter "ring"))

(define-html-emitter 'iff
  (lambda (items out)
    (display "<==>" out)))

(define-html-emitter 'lattice-lt-eq
  (lambda (items out)
    (display "<=" out)))

(define-html-emitter 'lattice-lt
  (lambda (items out)
    (display "<" out)))

(define-html-emitter 'hrule		;Horizontal rule
  (lambda (items out)
    (display "<hr>" out)))

(define-html-emitter 'newline
  (lambda (items out)
    (display "<br>" out)))

(define-html-emitter 'skip
  (lambda (items out)
    (format out "<pre>~%~%</pre>~%")))

(define-html-emitter 'insert            ;Insert a file
  (lambda (items out)
    (inserting items html out)))

(define-html-emitter 'itemize		;Unnumbered list
  (html-itemized-emitter "ul"))

(define-html-emitter 'label		;Hypertext label
  (lambda (items out)
    (display "<a name=\"" out)
    (display (car items) out)
    (display "\">" out)
    (html-list (cdr items) out)
    (display "</a>" out)))

; {link http://mumble.../foo#label stuff ...}

;;; should skip whitespace after the url
(define-html-emitter 'link		;Hypertext link {link URL text}
  (lambda (items out)
    (let ((url (whitespace-delimited-word items))
	  (text (drop-whitespace (drop-until-whitespace items))))
      (format out "<a href=\"~A\">" url)
      (if (null? text)
	  (format out "<code>~A</code>" url)
	  (html-list text out))
      (write-string "</a>" out))))

(define-html-emitter 'refer
  (lambda (items out)
    (let ((tag (car items)))
      (cond ((table-ref html-refer-table tag)
	     => (lambda (file)
		  (format out "<a href=\"~A#~A\">~A</a>" file tag tag)))
	    (else (printf "undefined reference: ~A~%" tag)
		  (format out "<a href=nowhere>~A</a>" tag))))))

(define-html-emitter 'note
  (lambda (items out)
    (let ((tag (car items)))
      (format out "<a href=\"notes.html#~A\">[note ~A]</a>" tag tag))))

; should be an option to put them all on their own page, like the notes
(define-html-emitter 'footnote
  (lambda (items out)
    (format out "[footnote: ")
    (html-list items out)
    (format out "]")))

(define-html-emitter 'see
  (lambda (items out)
    (let ((tag (car items)))
      (format out "(see <a href=\"~A.html\">~A</a>)" tag tag))))

;;; replace with <--> icon
(define-html-emitter 'expand
  (lambda (items out)
    (let ((tag (car items)))
      (format out "(see <a href=\"~A.html\">~A</a> for more)" tag tag))))

(define (html-until-whitespace items out)
  (let loop ((items items))		;Copy out the URL
    (if (null? items)
	items
	(if (and (char? (car items))
		 (char-whitespace? (car items)))
	    items
	    (begin (html (car items) out)
		   (loop (cdr items)))))))

(define-html-emitter 'load
  (lambda (items out)
    (load-handler items)))

(define-html-emitter 'dot
  (lambda (items out)
    (dot-handler items out html)))

(define-html-emitter 'plot
  (lambda (items out)
    (plot-handler items out html)))

(define html-figure-handler
  (lambda (items out)
    (let ((label (car items))
	  (items (cdr items)))
      (remember-label label)
      (format out "<hr><a name=~A>" label)
      (html (whitespace-car items) out)
      (format out "<br>Figure ~A:" label)
      (html-list (whitespace-cdr items) out)
      (format out "</a><hr>"))))

(define-html-emitter 'figure html-figure-handler)
(define-html-emitter 'figure-two-column html-figure-handler)

(define-html-emitter 'minipage
  (lambda (items out)
    (let ((w (distance->pixels (whitespace-car items))))
      (format out "<table width=~A><tr><td>" w)
      (html-list (whitespace-cdr items) out)
      (format out "</td></tr></table>"))))

(define *html-tex-completions*)

(define (tex->html s out)
  (define (finish ps-file)
    (format out "<img align=absmiddle src=~S alt=~S>"
	    (replace-extension ps-file ".gif") s))
  (cond ((table-ref *html-tex-memo-table* s)
	 => finish)
	(else
	 (let ((ps-file (if #f (format #f "~A.texhtml.~S.ps"
				       *html-tex-filename*
				       *html-tex-image-counter*)
			    (format #f "~S.ps"
				    *html-tex-image-counter*))))
	   (textops-queue s ps-file)
	   (table-set! *html-tex-memo-table* s ps-file)
	   (set! *html-tex-image-counter*
		 (+ 1 *html-tex-image-counter*))
	   (finish ps-file)
	   (set! *html-tex-completions*
		 (cons (lambda ()
			 (pstogif-file ps-file)
			 (delete-file ps-file))
		       *html-tex-completions*))))))

(define-html-emitter 'm
  (lambda (items out)
    ;; argh, clearly we want a read-macro here to change the lexical
    ;; environment
    (tex->html (string-append "$" (list-to-string items) "$") out)))

(define-html-emitter 'mm
  (lambda (items out)
    (display "<p><i>" out)
    (pass-items items out)
    (display "</i><p>" out)))

(define-html-emitter 'as-image
  (lambda (items out)
    (let ((s (call-with-string-output-port
	      (lambda (p) (latex-list items p)))))
      (tex->html s out))))

(define-html-emitter 'paragraph		;Paragraph separator
  (lambda (items out)
    (newline out)
    (display "<p>" out)
    (newline out)))

(define-html-emitter 'part
  (lambda (items out)
    (scan-file-for-toc (car (drop-whitespace items))
		       out)))

(define (scan-file-for-toc file out)

  (define (toc-item items level file)
    (go-to-level level out)
    (display "<li><a href=\"" out)
    (display file out)
    (display ".html#" out)
    (display (car items) out)
    (display "\">" out)
    (html-list (cdr items) out)
    (display "</a>" out)
    (newline out))

  (define (scan-for-toc file)
    (let ((mfile (string-append file ".mar")))
      (call-with-input-file mfile
	(lambda (in)
	  (display "  [Scanning ") (display mfile) (display "]") (newline)
	  (read-items (lambda (item)
			(if (pair? item)
			    (let ((items (drop-whitespace (cdr item))))
			      (cond ((equal? (car item) "chapter")
				     (toc-item items 1 file))
				    ((equal? (car item) "section")
				     (toc-item items 2 file))
				    ((equal? (car item) "subsection")
				     (toc-item items 3 file))
				    ((member (car item) '("part" "insert"))
				     (scan-for-toc (car items)))))))
		      in #t)))))

  (scan-for-toc file))


(define (go-to-level level out)
  (let loop ()
    (if (< *level* level)
	(begin (display "<ul> " out)
	       (set! *level* (+ *level* 1))
	       (loop))))
  (let loop ()
    (if (> *level* level)
	(begin (display "</ul>" out)
	       (newline out)
	       (set! *level* (- *level* 1))
	       (loop)))))

(define-html-emitter 'pass pass-items)

(define-html-emitter 'quote		;Indented
  (html-environment-emitter "ul"))

(define-html-emitter 'section
  (html-section-emitter "h2"))

(define-html-emitter 'strong		;Boldface
  (html-environment-emitter "strong"))

(define-html-emitter 'subsection
  (html-section-emitter "h3"))

(define-html-emitter 'subsubsection
  (html-section-emitter "h4"))

(define-html-emitter 'title		;Title (HTML only)
  (html-environment-emitter "title"))

(define-html-emitter 'var		;Variable name
  (html-environment-emitter "var"))

(define-html-emitter 'when-html
  html-list)

(define-html-emitter 'when-latex
  (lambda (items out) #f))

(define-html-emitter 'write		;for debugging
  (lambda (items out)
    (write items out)
    (newline out)))

(define-html-emitter 'ps
  (lambda (items out)
    (let* ((psfile (car items))
	   (gifile (pstogif-file psfile)))
      (format out "<img src=~S><a href=~S>ps</a>" gifile psfile))))

; can't we bring in the right margin?
(define-html-emitter 'abstract
  (lambda (items out)
    (format out "<a name=\"abstract\">")
    (format out "<h2>Abstract</h2>")
    (format out "<ul>")
    (html-list items out)
    (format out "</ul></a>")))

(define (tail l pred)
  (if (null? l)
      #f
      (if (pred (car l))
	  l
	  (tail (cdr l) pred))))

(define-html-emitter 'if (if-handler html-list))
(define-html-emitter 'def (def-handler html-emitters html-list))

(define (string->sexp s)
  (let ((port (make-string-input-port s)))
    (read port)))

(define (markup-eval exp) (eval exp (interaction-environment)))


(define-html-emitter 'document
  (lambda (items out)
    (let ((doc (strip-whitespace items)))
      (set! *citations* '())
      (set! *html-tex-completions* '())
      (set! *html-tex-image-counter* 0)
      (set! *html-tex-filename* (whitespace-car doc))
      (set! *html-tex-memo-table* (make-string-table))
      (format out "<center><h1>")
      (html-list (get-field "title" doc) out)
      (format out "</h1><h2>")
      (latex-list (get-field "author" doc) out)
      (format out "</h2><h2>")
      (latex-list (get-field "institute" doc) out)
      (format out "</h2></center>")
      (if (field-defined? "body" doc)
	  (html-list (get-field "body" doc) out))
      (if (field-defined? "files" doc)
	  (begin (format out "<ul>")
		 (for-each (lambda (f)
			     (format out "<li><a href=\"~A.html\">~A</a>~%" f f)
			     (markup 'html f))
			   (strip-whitespace (get-field "files" doc)))
		 (format out "</ul>")))
      (textops-flush)
      (for-each (lambda (f) (f)) *html-tex-completions*))))

(define (html-do-bib-entry out)
  (lambda (tag entry)
    (format out "<a name=\"~A\"><dt>~A<dd>" tag tag)
    (html-list entry out)
    (format out "</a>~%")))

(define-html-emitter 'bib-db
  (lambda (items out)
    (format out "<h2>References</h2><dl>~%")
    (bib-db-handler items (html-do-bib-entry out))
    (format out "</dl>~%")))

(define-html-emitter 'table
  (lambda (items out)
    (let ((items (whitespace-cdr items)))
      (define (do-cols items)
	(cond ((whitespace-null? items) 'done)
	      (else (if (equal? (whitespace-car items) "span")
			(let* ((d (whitespace-cdr items))
			       (n (string->number
				   (whitespace-car d)))
			       (dd (whitespace-cdr d)))
			  (format out "<td colspan=~A>" n)
			  (html-list (whitespace-car dd) out)
			  (format out "</td>")
			  (do-cols (whitespace-cdr dd)))
			(begin (format out "<td>")
			       (html-list (whitespace-car items) out)
			       (format out "</td>")
			       (do-cols (whitespace-cdr items)))))))
      (define (do-rows items)
	(cond ((whitespace-null? items) 'done)
	      (else (format out "<tr>")
		    (do-cols (whitespace-car items))
		    (format out "</tr>~%")
		    (do-rows (whitespace-cdr items)))))
      (format out "<table>")
      (do-rows items)
      (format out "</table>"))))

