(* Tactics for type-checking *)

Require Import Syntax.

Let emp := (nil:linctx).

Lemma tp_Pair_emp : forall P1 P2 p1 p2,
  pat_tp emp p1 P1 -> pat_tp emp p2 P2 ->
  pat_tp emp (Pair p1 p2) (Times P1 P2).
intros; rewrite (app_nil_end emp); eauto with focus.
Qed.

Lemma tp_Nat_emp : forall p Delta, pat_tp Delta p Nat -> Delta = nil.
intros; induction p; inversion H; auto.
Qed.

Hint Resolve tp_Pair_emp : focus.
Hint Resolve tp_Nat_emp : focus.

Ltac patinv H :=
  match (type of H) with
  | pat_tp _ _ (Arrow _ _) => inversion_clear H
  | pat_tp _ _ One => inversion_clear H
  | pat_tp _ _ (Times _ _) =>
    let Delta1 := fresh "Delta" in let Delta2 := fresh "Delta" in
    let P1 := fresh "P" in let P2 := fresh "P" in
    let p1 := fresh "p" in let p2 := fresh "p" in
    let H1 := fresh "H" in let H2 := fresh "H" in
      (inversion_clear H as [ | | Delta1 Delta2 P1 P2 p1 p2 H1 H2 | | | | | | | ];
        patinv H1; patinv H2)
  | pat_tp _ _ Zero => inversion_clear H
  | pat_tp _ _ (Plus _ _) =>
    let Delta' := fresh "Delta" in
    let P1 := fresh "P" in let P2 := fresh "P" in
    let p' := fresh "p" in
    let H' := fresh "H" in
      (inversion_clear H as [ | | | Delta' P1 P2 p' H' | Delta' P1 P2 p' H' | | | | | ];
        patinv H')
  | pat_tp _ _ Bool => inversion_clear H
  | pat_tp _ _ Nat =>
    let p' := fresh "p" in
    let H' := fresh "H" in
      (inversion_clear H as [ | | | | | | | | p' H' | ])
  | pat_tp _ _ D =>
    let Delta' := fresh "Delta" in
    let p' := fresh "p" in
    let H' := fresh "H" in
      (inversion_clear H as [ | | | | | | | | | Delta' p' H' ];
        patinv H')
  | _ => trivial
  end.

Ltac checkpat :=
  (match goal with
  | [ |- pat_tp _ _ (Arrow _ _) ] => eapply tp_Fvar
  | [ |- pat_tp _ Unit One ] => apply tp_Unit
  | [ H1 : pat_tp _ _ ?P, H2 : pat_tp _ _ ?Q |- pat_tp _ _ (Times ?P ?Q) ] => apply (tp_Pair H1 H2)
  | [ H : pat_tp _ _ ?P |- pat_tp _ _ (Plus ?P _) ] => apply (tp_In1 H)
  | [ H : pat_tp _ _ ?Q |- pat_tp _ _ (Plus _ ?Q) ] => apply (tp_In2 H)
  | [ |- pat_tp _ Tr Bool ] => apply tp_Tr
  | [ |- pat_tp _ Fa Bool ] => apply tp_Fa
  | [ |- pat_tp _ Ze Nat] => apply tp_Ze
  | [ H : pat_tp _ _ Nat |- pat_tp _ _ Nat ] => apply (tp_Su H)
  | [ H : pat_tp _ _ (Arrow D D) |- pat_tp _ _ D ] => apply (tp_In1 H)
  end) ||
  (match goal with
  | [ |- pat_tp _ _ (Times ?P ?Q) ] => eapply tp_Pair
  | [ |- pat_tp _ _ (Plus ?P _) ] => eapply tp_In1
  | [ |- pat_tp _ _ (Plus _ ?Q) ] => eapply tp_In2
  | [ |- pat_tp _ _ Nat ] => (eapply tp_Ze || eapply tp_Su)
  | [ |- pat_tp _ _ D ] => eapply tp_Fn
   end) ||
  trivial.

Ltac checkexp :=
  match goal with
  | [ |- exp_tp _ (Return _) _ ] => eapply ReturnTp
  | [ |- exp_tp _ (EComp _ _ _) _ ] => eapply ECompTp; compute; trivial
  | [ |- exp_tp _ (EAppV _ _) _ ] => eapply EAppVTp
  | [ |- exp_tp _ (EAppE _ _) _ ] => eapply EAppETp
  | _ => trivial
  end.

Ltac checkval :=
  match goal with
  | [ |- val_tp _ _ _ ] => eapply ValTp
  | _ => trivial
  end.
Ltac checkfnc :=
  match goal with
  | [ |- fnc_tp _ (Lam _) _ _ ] => 
    let p := fresh "p" in
    let Delta := fresh "Delta" in
    let H := fresh "H" in
    (apply LamTp; intros p Delta H; patinv H; simpl; eauto)
  | [ |- fnc_tp _ Id ?P ?P ] => apply IdTp
  | [ |- fnc_tp _ (IdVar _) _ _ ] => eapply IdVarTp
  | [ |- fnc_tp _ (Fix _) _ _ ] => eapply FixTp
  | _ => trivial
  end.
Ltac checksub :=
  match goal with
  | [ |- sub_tp _ (Sub nil) nil ] => eapply SubNilTp
  | [ |- sub_tp _ (Sub _) _ ] => eapply SubConsTp
  | _ => trivial
  end.

Hint Extern 2 (pat_tp _ _ _) => checkpat : focus.
Hint Extern 2 (val_tp _ _ _) => checkval : focus.
Hint Extern 2 (fnc_tp _ _ _ _) => checkfnc : focus.
Hint Extern 2 (sub_tp _ _ _) => checksub : focus.
Hint Extern 2 (exp_tp _ _ _) => checkexp : focus.

Ltac checklam := 
  let p := fresh "p" in
  let Delta := fresh "Delta" in
  let H := fresh "H" in
  (compute; apply LamTp; intros p Delta H; patinv H; simpl; eauto with focus).
Hint Extern 4 (fnc_tp _ _ _ _) => checklam : focus.

Ltac checktm :=
  repeat
  (compute ;
   match goal with
   | [ |- pat_tp _ _ _ ] => checkpat
   | [ |- val_tp _ _ _ ] => checkval
   | [ |- fnc_tp _ _ _ _ ] => checkfnc
   | [ |- sub_tp _ _ _ ] => checksub
   | [ |- exp_tp _ _ _ ] => checkexp
   | _ => compute
   end;
   eauto with focus).
