Require Export Debruijn2.

Set Implicit Arguments.

(* types *)
Inductive tp : Set := 
  | One : tp
  | Times : tp -> tp -> tp
  | Zero : tp
  | Plus : tp -> tp -> tp
  | Arrow : tp -> tp -> tp
  | Bool : tp
  | Nat : tp
  | D : tp.

(* patterns and pattern-typing *)
Inductive pat : Set :=
  | Fvar : pat
  | Unit : pat
  | Pair : pat -> pat -> pat
  | In1 : pat -> pat | In2 : pat -> pat
  | Tr : pat | Fa : pat
  | Ze : pat | Su : pat -> pat
  | Fn : pat -> pat.

Inductive hyp : Set := ArrowH : tp -> tp -> hyp.
Definition linctx := frame hyp.

Inductive pat_tp : linctx -> pat -> tp -> Prop :=
  | tp_Fvar : forall P Q, pat_tp (ArrowH P Q::nil) Fvar (Arrow P Q)
  | tp_Unit : pat_tp nil Unit One
  | tp_Pair : forall Delta1 Delta2 P1 P2 p1 p2,
      pat_tp Delta1 p1 P1 -> pat_tp Delta2 p2 P2 ->
      pat_tp (Delta1 ++ Delta2) (Pair p1 p2) (Times P1 P2)
  | tp_In1 : forall Delta P1 P2 p1,
      pat_tp Delta p1 P1 ->
      pat_tp Delta (In1 p1) (Plus P1 P2)
  | tp_In2 : forall Delta P1 P2 p2,
      pat_tp Delta p2 P2 ->
      pat_tp Delta (In2 p2) (Plus P1 P2)
  | tp_Tr : pat_tp nil Tr Bool
  | tp_Fa : pat_tp nil Fa Bool
  | tp_Ze : pat_tp nil Ze Nat
  | tp_Su : forall p, pat_tp nil p Nat -> pat_tp nil (Su p) Nat
  | tp_Fn : forall Delta p,
      pat_tp Delta p (Arrow D D) ->
      pat_tp Delta (Fn p) D.

(* values, functions, expressions, and substitutions *)
Inductive val : Set :=
  | Val : pat -> sub -> val
with fnc : Set :=
  | Lam : (pat -> exp) -> fnc
  | Id : fnc
  | IdVar : index -> fnc
  | Fix : fnc -> fnc
with exp : Set :=
  | Return  : val -> exp
  | EComp : fnc -> index -> val -> exp
  | EAppV : fnc -> val -> exp
  | EAppE : fnc -> exp -> exp
  | EFail : exp
with sub : Set :=
  | Sub : frame fnc -> sub.

Hint Constructors pat val fnc exp sub : focus.

(* Term typing *)
Definition intctx := stack hyp.

Inductive val_tp : intctx -> val -> tp -> Prop :=
  | ValTp :
    forall Gamma p sigma P Delta,
      pat_tp Delta p P -> sub_tp Gamma sigma Delta ->
      val_tp Gamma (Val p sigma) P

with fnc_tp : intctx -> fnc -> tp -> tp -> Prop :=
  | LamTp :
    forall Gamma phi P Q,
      (forall p Delta, pat_tp Delta p P -> exp_tp (Delta :: Gamma) (phi p) Q) ->
      fnc_tp Gamma (Lam phi) P Q
  | IdTp : forall Gamma P, fnc_tp Gamma Id P P
  | IdVarTp :
    forall Gamma f P Q,
      lookup_stack f Gamma = Some (ArrowH P Q) -> fnc_tp Gamma (IdVar f) P Q
  | FixTp :
    forall Gamma F P Q,
      fnc_tp (((ArrowH P Q) :: nil) :: Gamma) F P Q ->
      fnc_tp Gamma (Fix F) P Q

with exp_tp : intctx -> exp -> tp -> Prop :=
  | ReturnTp :
    forall Gamma V P, val_tp Gamma V P -> exp_tp Gamma (Return V) P
  | ECompTp :
    forall Gamma F g V R P Q,
      lookup_stack g Gamma = Some (ArrowH P Q) ->
      val_tp Gamma V P -> fnc_tp Gamma F Q R ->
      exp_tp Gamma (EComp F g V) R
  | EAppVTp :
    forall Gamma F V P Q,
      val_tp Gamma V P -> fnc_tp Gamma F P Q ->
      exp_tp Gamma (EAppV F V) Q
  | EAppETp :
    forall Gamma F E P Q,
      exp_tp Gamma E P -> fnc_tp Gamma F P Q ->
      exp_tp Gamma (EAppE F E) Q

with sub_tp : intctx -> sub -> linctx -> Prop :=
  | SubNilTp :
    forall Gamma, sub_tp Gamma (Sub nil) nil
  | SubConsTp : 
    forall Gamma F subs P Q Delta,
      fnc_tp Gamma F P Q -> sub_tp Gamma (Sub subs) Delta -> 
      sub_tp Gamma (Sub (F :: subs)) (ArrowH P Q::Delta).

Hint Constructors pat_tp val_tp fnc_tp exp_tp sub_tp : focus.

(* Code to generate good mutual induction principles -- thanks Xavier Leroy! *)

Scheme valtp_ind_2 := Induction for val_tp Sort Prop
  with fnctp_ind_2 := Induction for fnc_tp Sort Prop
  with exptp_ind_2 := Induction for exp_tp Sort Prop
  with subtp_ind_2 := Induction for sub_tp Sort Prop.

Definition conj4 (A B C D: Prop) (X:A) (Y:B) (Z:C) (W:D) :=
  conj X (conj Y (conj Z W)).

Definition typing_mutual_ind
  (P : forall (i : intctx) (v : val) (t : tp), val_tp i v t -> Prop)
  (P0 : forall (i : intctx) (f : fnc) (t t0 : tp),
    fnc_tp i f t t0 -> Prop)
  (P1 : forall (i : intctx) (e : exp) (t : tp), exp_tp i e t -> Prop)
  (P2 : forall (i : intctx) (s : sub) (l : linctx),
    sub_tp i s l -> Prop) :=
  fun h1 h2 h3 h4 h5 h6 h7 h8 h9 h10 h11 =>
    conj4 (valtp_ind_2 P P0 P1 P2 h1 h2 h3 h4 h5 h6 h7 h8 h9 h10 h11)
          (fnctp_ind_2 P P0 P1 P2 h1 h2 h3 h4 h5 h6 h7 h8 h9 h10 h11)
          (exptp_ind_2 P P0 P1 P2 h1 h2 h3 h4 h5 h6 h7 h8 h9 h10 h11)
          (subtp_ind_2 P P0 P1 P2 h1 h2 h3 h4 h5 h6 h7 h8 h9 h10 h11).
