Require Import Syntax.
Require Import ArithLem.

Set Implicit Arguments.

(* Operational semantics *)

Fixpoint shift_val (i : nat) (V : val) {struct V} : val :=
  match V with
    Val p sigma => Val p (shift_sub i sigma)
  end

with shift_fnc (i : nat) (F : fnc) {struct F} : fnc :=
  match F with
    Lam f => Lam (fun p => shift_exp (S i) (f p))
  | Id => Id
  | IdVar (i',k) => IdVar (if le_gt_dec i i' then S i' else i', k)
  | Fix F => Fix (shift_fnc (S i) F)
  end

with shift_exp (i : nat) (E : exp) {struct E} : exp :=
  match E with
  | Return V => Return (shift_val i V)
  | EComp F (i',k) V =>
    EComp (shift_fnc i F) (if le_gt_dec i i' then S i' else i', k)
          (shift_val i V)
  | EAppV F V => EAppV (shift_fnc i F) (shift_val i V)
  | EAppE F E => EAppE (shift_fnc i F) (shift_exp i E)
  | EFail => EFail
  end

with shift_sub (i : nat) (sigma : sub) {struct sigma} : sub :=
  match sigma with
  | Sub subs => Sub (List.map (fun F => shift_fnc i F) subs)
  end.

Definition lookup_sub (k : nat) (sigma : sub) : fnc :=
  match sigma with
    | Sub subs =>
      match lookup_frame k subs with
        | Some F => F
        | None => Lam (fun _ => EFail)
      end
  end.


Fixpoint sub_val (i : nat) (sigma : sub) (V : val) {struct V} : val :=
  match V with
    Val p sigma' => Val p (sub_sub i sigma sigma')
  end

with sub_fnc (i : nat) (sigma : sub) (F : fnc) {struct F} : fnc :=
  match F with
    Lam f => Lam (fun p => sub_exp (S i) (shift_sub 0 sigma) (f p))
  | Id => Id
  | IdVar (i',k) => if eq_nat_decide i i' then lookup_sub k sigma
                    else let i'' := if le_lt_dec i i' then i'-1 else i' in
                      IdVar (i'',k)
  | Fix F => Fix (sub_fnc (S i) (shift_sub 0 sigma) F)
  end

with sub_exp (i : nat) (sigma : sub) (E : exp) {struct E} : exp :=
  match E with
  | Return V => Return (sub_val i sigma V)
  | EComp F (i',k) V =>
    if eq_nat_decide i i' then
      let G := lookup_sub k sigma in
        EAppE (sub_fnc i sigma F) (EAppV G (sub_val i sigma V))
      else let i'' := if le_lt_dec i i' then i'-1 else i' in
        EComp (sub_fnc i sigma F) (i'',k) (sub_val i sigma V)
  | EAppV F V => EAppV (sub_fnc i sigma F) (sub_val i sigma V)
  | EAppE F E => EAppE (sub_fnc i sigma F) (sub_exp i sigma E)
  | EFail => EFail
  end

with sub_sub (i : nat) (sigma : sub) (sigma' : sub) {struct sigma'} : sub :=
  match sigma' with
  | Sub subs => Sub (List.map (fun F => sub_fnc i sigma F) subs)
  end.

(* transition relation *)
Inductive step : exp -> exp -> Prop :=
 | reduce : forall p sigma phi,
            step (EAppV (Lam phi) (Val p sigma)) (sub_exp 0 sigma (phi p))
 | app_id : forall V, step (EAppV Id V) (Return V)
 | eval_arg : forall F E E', step E E' -> step (EAppE F E) (EAppE F E')
 | eval_val : forall F V, step (EAppE F (Return V)) (EAppV F V)
 | unroll : forall F V,
            step (EAppV (Fix F) V) (EAppV (sub_fnc 0 (Sub ((Fix F)::nil)) F) V).
Hint Constructors step : focus_sem.

(* functional version of transition relation *)
Fixpoint stepf (E : exp) {struct E} : exp :=
 match E with
   | EAppV (Lam phi) (Val p sigma) => sub_exp 0 sigma (phi p)
   | EAppV Id V => Return V
   | EAppV (Fix F) V => EAppV (sub_fnc 0 (Sub ((Fix F)::nil)) F) V
   | EAppE F (Return V) => EAppV F V
   | EAppE F E => EAppE F (stepf E)
   | _ => E
 end.

(* iterate n times *)
Fixpoint stepn (E : exp) (n:nat) {struct n} : exp :=
  match n with O => E | S n => stepn (stepf E) n end.
