#ifndef SSAPRE_H
#define SSAPRE_H

#include <cfa/cfa.h>
#include <suifvm/suifvm.h>
#include <bvd/bvd.h>
#include <ssa/ssa.h>
#include <machine/machine.h>
#include <cfg/cfg.h>
#include <stack>
#include <stdlib.h>
#include <algorithm>
#include <iostream>
#define DEBUG_SSAPRE

//#ifndef SUPPRESS_PRAGMA_INTERFACE
//#pragma interface "pre/ssapre.h"
//#endif

using namespace ssa;

class SSApre;

class DominanceChild
{
  public:
    DominanceChild(int node_number) { block_number = node_number; }
    ~DominanceChild() { if (next) delete next; }

    unsigned block_number;
    DominanceChild *next;
};

class Occ {
 public:
  bool reload, save, live;

  enum kind {PHI,PHIOP,REAL};
  virtual kind is_a() = 0;

  static const int TOP = -1;
  static const int END = -2;
  int pos;
 
  int block;

  static const int BOTTOM = 0;
  int version;

  virtual void print(FILE *) = 0;

 protected:
  Occ();
};

// NOTE - when we construct a RealOcc, (in the expression gathering
// phase) we need to tell it its position in the BB.  We should know this
// anyway, since we've gotten to it by some sort of iterator.
class RealOcc : public Occ {
 public:
  RealOcc(Instr *mi, int pos, SSApre *ssa);
  virtual ~RealOcc();

  Opnd old_arg_name(int pos);
  Opnd new_arg_name(int pos);
  int num_args();

  Opnd new_dst_name();
  Opnd old_dst_name();

  void print(FILE *);
  kind is_a() { return REAL; }
 private:
  Instr *mi;
  Opnd *old_args;
  Opnd *new_args;
  Opnd new_dst;
  Opnd old_dst;
  int nargs;
};

class PhiOp;

class PHINode : public Occ {
 public:
  PHINode(SSApre *ssa, int block);
  virtual ~PHINode();

  bool can_be_avail, down_safe, later;
  kind is_a() { return PHI; }
  void print(FILE *);

  int num_opnds();
  bool will_be_avail() { return can_be_avail && !later; }
  Occ *get_opnd(int i);
  void set_opnd(int i, Occ *op) { opnds[i] = op; }

 private:
  int nopnds;
  Occ **opnds;
};

class PhiOp : public Occ {
 public:
  PhiOp(SSApre *ssa, PHINode *phi);
  bool has_real_use; 
  bool insert(Vector<Occ *> &def_point);

  kind is_a() { return PHIOP; }
  PHINode *get_phi();
  void print(FILE *);
 private:
  PHINode *phi;
};

class ExprKey {
 public:
  ExprKey( const ExprKey &a );
  ExprKey( RegSymCatalog *c, int op, RealOcc *occ );
  ~ExprKey();
  Vector<int> *makeVec();
  bool operator==( const ExprKey &a) const;
  bool operator <(const ExprKey &a) const;

 private:
  int *opnd_list;
  int ct;
  int opcode;
};


class SSApre : public SsaForm {
 public:
  
    void initialize();
    void do_opt_unit(OptUnit*);
    void finalize();
    
    // set pass options
    void set_debug_arg(int da)          { debuglvl = da; }
 protected:

    friend class ExprPRE;
    friend class Occ;
    friend class RealOcc;
    friend class PHINode;
    friend class PhiOp;

 private:
    // Determine whether or not we should try to do pre on the expression
    // in this instruction.
    bool is_interesting_expr(Instr *); 
};

inline void debug(char *fmt, ...) {
#ifdef DEBUG_SSAPRE
  va_list vargs;
  va_start(vargs,fmt);
  vprintf(fmt, vargs);
  va_end(vargs);
  fflush(stdout);
#endif
}

class ExprPRE {
 public:
  ExprPRE(Vector<int> &arg_ids, Vector<Occ *> &occ, SSApre *ssa);

  void doPRE();
  
 private:
  Vector<Occ *>  all;
  SSApre *ssa;
  Vector<PHINode*> PHI;
  // t is the "old name" of the temporary for Expr
  Opnd t;
  // the opcode of our expr.
  int opcode;

  // Things required for rename:
  int arg_ct;                     // Num of args in this expression
  Vector<int> arg_ids;            // List of ids in old_opnd_table for expr
  int NextVersion;                // Next version number to assign to h
  stack<Occ*> OccStack;           // stack of occurences of h
  List<Occ*> rename_worklist;     // worklist for do_Rename_2()

  map<int, PHINode *> PHI_for_block;

  // For each BB, the set of occurrences in that BB.
  // We may want to consider a multimap instead 
  map<int, deque<Occ *> *> occ_for_block;

  // the unique OCCURRENCE that defines each version of h
  Vector<Occ *> def_point;

  // For each def_point F, the list of PhiOps defined by F. 
  multimap<Occ *, PhiOp *> uses;

  // avail_def has one entry for each version of h 
  Vector<Occ *> avail_def; 

  void Set_var_phis(PhiNode *phi);
  void do_PHI_insertion();
  void add_PHI(int node);

  void do_Rename_1();
  void do_Rename_2();
  void preorder_rename( DominanceChild *);
  bool phiop_is_bottom( PhiOp *, Opnd );
  bool real_dom_phi(RealOcc *, PHINode *);
  
  void do_DeadStoreElim();

  void reset_downsafe(PhiOp *);
  void do_DownSafety();

  void compute_can_be_avail();
  void reset_can_be_avail(PHINode *);
  void compute_later();
  void reset_later(PHINode *);
  void do_WillBeAvail();

  bool dominates(Occ* x1, Occ *x2);
  Opnd update_opnd(int, int, Opnd);
  void finalize_visit(int, Opnd *);
  void do_Finalize();

  void do_Extraneous_PHI_removal();

  void do_CodeMotion();
};

#endif




