/* file "ssa/ssa_form_skel.h" -- transformers to and from SSA form */

/*
    Copyright (c) 2000-2001 The President and Fellows of Harvard College

    All rights reserved.

    This software is provided under the terms described in
    the "machine/copyright.h" include file.
*/

#ifndef SSA_SSA_FORM_SKEL_H
#define SSA_SSA_FORM_SKEL_H

#include <machine/copyright.h>

#ifdef USE_PRAGMA_INTERFACE
#pragma interface "ssa/ssa_form.h"
#endif

#include <machine/machine.h>


/*
 * Literature:   This implementation adheres quite closely to that developed by
 * the Massively Scalar Compiler Project at Rice University.  (See
 * <URL:http://www.cs.rice.edu/MSCP/>.)  A description of the Rice innovations
 * in SSA transformation is at <URL:http://www.cs.rice.edu/~harv/ssa.ps>.
 *
 *
 * Terminology:
 *
 * location:  A conventional variable or register, suitable for SSA
 * 	transformation.
 *
 * name:   (An Opnd.)  Either an original location operand suitable for SSA
 * 	replacement (an "old" name, or location name), or the SSA replacement
 * 	itself (a "new" name, or value name).
 *
 * id:	An integer that identifies a name.  There are location ids for
 * 	old names and value ids for SSA names.
 */


/*
 * A phi-node contains inputs, called sources, which are the names of the
 * parameters corresponding to incoming control paths.  (We use the
 * abbreviation "src" where BHS uses "parm".)
 *
 * It also holds both the original name ("old_name") of the location
 * represented (variable or register) and the SSA name ("new_name") for the
 * value defined by the node.
 *
 * A phi-node can be marked useless if all its sources are equivalent.
 * The library optionally removes useless phi-nodes.
 */

class PhiNode
{
  public:
    PhiNode(int nsrcs = 0);
    ~PhiNode();

    int srcs_size() const;

    Opnd get_src(int pos) const;
    void set_src(int pos, Opnd);

    Opnd get_old_name() const;
    void set_old_name(Opnd);
    Opnd get_new_name() const;
    void set_new_name(Opnd);

    bool get_useless() const;
    void set_useless(bool);

  protected:
    Opnd old_name;
    Opnd new_name;
    bool useless;
    int nsrcs;
    Opnd *srcs;
};


/*
 * map_opnds
 *
 * Scan the sources of a phi-node, applying a filter to each in turn.
 *
 * Unlike map_opnds(Instr*, ...), this function does NOT change the source
 * operands of the phi-node.
 */

void map_opnds(PhiNode*, OpndFilter&);

/*
 * An ``operation'' is either a phi-node or an instruction that defines one
 * or more values.
 */

class Operation
{
  public:
    Operation() : phi_node(&null_indicator) { }
    Operation(InstrHandle handle) { set_instr_handle(handle); }
    Operation(PhiNode *phi_node)  { set_phi_node(phi_node); }

    PhiNode* get_phi_node() const { return phi_node; }
    InstrHandle get_instr_handle() const { return handle; }

    void set_instr_handle(InstrHandle h) { handle = h; phi_node = NULL; }
    void set_phi_node(PhiNode *p) { phi_node = p; }

    bool is_instr_handle() const { return phi_node == NULL; }
    bool is_phi_node() const { return phi_node != NULL; }
    bool is_null() const { return phi_node == &null_indicator; }

  protected:
    InstrHandle handle;
    PhiNode *phi_node;

    static PhiNode null_indicator;
};

/*
 * An ``occurrence'' designates a use of an SSA name.  It is represented by
 * the containing block and either (the handle of) an instruction in that
 * block, or a phi-node and a source position within it.
 *
 * For an instruction occurrence, it is necessary in general to map a
 * filter over the instruction's operands to find/replace the actual SSA
 * name(s) of interest.
 */
class Occurrence : public Operation
{
  public:
    Occurrence(InstrHandle h, CfgNode *b)
	: Operation(h), block(b), phi_src_pos(-1)  { }
    Occurrence(PhiNode *p, CfgNode *b, int pos)
	: Operation(p), block(b), phi_src_pos(pos) { }

    CfgNode *get_block() const { return block; }
    int get_phi_src_pos() const { return phi_src_pos; }
    void set_phi_src_pos(int pos) { phi_src_pos = pos; }

  protected:
    CfgNode *block;
    int phi_src_pos;
};

/*
 * Classes defined in ssa_form.cpp
 */
class LivenessInfo;
class PhiSrcMapEntry;
class NameStackEntry;
class CopyData;
class DominanceChild;
class AssignmentListEntry;

/*
 * An instance of class SsaForm give access to all information about the
 * SSA form of a CFG.
 *
 * Method `build' transforms a CFG into SSA form.  It takes a simple bitset
 * of flags (see below) that control options.
 *
 * Method `restore' transforms the CFG to conventional form.  Depending on
 * optimizations performed at build time or by the client, it may or may
 * not be valid to restore the original names.  If argument `not_old_names'
 * is true, restoration of original names is suppressed.
 *
 * Method `loc_count' returns the number of original names replaced by SSA
 * names.  Method `value_count' returns the number of SSA names.
 *
 * Method `value_catalog' returns a map from SSA names to their integer
 * ids.  (OpndCatalog is defined in the BVD library.)  The table produced
 * by `value_table' gives the opposite map from ids to names.
 *
 * The `unique_def' and `use_chain' methods give use-def and def-use
 * "chains" for the SSA form, assuming that the corresponding option flags
 * are passed to `build'.  In single-assignment form, the def for each use
 * is unique, and so there's not an actual use-def chain as such.
 *
 * Method `phi_nodes' maps from a basic block to the set of phi-nodes for
 * the block.  Method `phi_node_block' gives the block for any phi-node.
 *
 * Method `record_use_swap' takes note of a replacement (`in' replacing
 * `out') among the use occurrences of SSA names in an instruction or
 * phi-node.  It doesn't alter the instruction or phi-node; it just updates
 * internal records (e.g., def-use chains) to reflect the swap.  In the
 * phi-node case, it is given the position of the use among the source
 * operands of the phi-node.
 *
 *   NB: If the name being swapped in is a newly-created SSA name, its
 *   definition must already have been processed by record_def before
 *   record_use_swap is called.
 *
 * Method `record_all_uses' takes note of all the use occurrences within a
 * new instruction or phi-node.  It modifies internal records (e.g.,
 * def-use chains) to reflect insertion of the instruction or phi-node into
 * the code.
 *
 * Method `record_def' takes note of a new definition, in the form either
 * of an instruction or a phi-node.  It does not change the instruction or
 * phi-node, but looks there to find the SSA name defined.
 *
 *   In the instruction case, it is given the original (non-SSA) name for
 *   the definee, as well as the position of the relevant destination in
 *   the instruction.
 *
 *   In the phi-node case, it is given the block to which the phi-node has
 *   already been attached.  The phi-node's old_name (non-SSA) and new_name
 *   fields must already have been filled in.
 *
 *
 * There are also print methods for debugging.
 */
class SsaForm
{
  public:
    SsaForm() { }

    void build(Cfg*, unsigned flags = 0);	// transform to SSA form
    void restore(bool not_old_names = false);	// restore conventional form

    int loc_count() const;			// number of original locations
    int value_count() const;			// number of SSA defs (plus one)
						
    OpndCatalog* value_catalog() const;		// numbering map for SSA defs
    const Vector<Opnd>& value_table() const;	// map from value ids to "names"
						
    Operation unique_def(Opnd) const;		// defining operation for value
    const List<Occurrence>&			// all uses of value		
	use_chain(Opnd) const;
						
    List<PhiNode*>& phi_nodes(CfgNode*);	// all phi-nodes of basic block
    CfgNode *phi_node_block(PhiNode*) const;	// block containing phi-node

    void record_use_swap(Opnd out, Opnd in, InstrHandle);
						// note use swap (in for out)
    void record_use_swap(Opnd out, Opnd in, PhiNode*, int src_pos);
						//	  (ditto)
    void record_all_uses(InstrHandle);		// note all uses in new instr
    void record_all_uses(PhiNode*);		// note all uses in new phi-node
    void record_def(Opnd old_name, InstrHandle, int dst_pos = 0);
						// note new SSA def for old var
    void record_def(PhiNode*, CfgNode *block);	// note new phi-node in block

    void print_phi_nodes(FILE*, CfgNode*);
    void print(FILE*);

    //  protected:
    OptUnit *unit;
    Cfg *unit_cfg;

    unsigned _loc_count;		// virtual and symbolic registers
    unsigned def_count;			// number of original defs plus one
    unsigned block_count;		// basic blocks

    RegSymCatalog *old_opnd_catalog;	// original opnd to its numeric id
    RegSymCatalog *new_opnd_catalog;	// ssa opnd to numeric id

    Vector<Opnd> old_opnd_table;	// numeric id to original opnd
    Vector<Opnd> new_opnd_table;	// numeric id to ssa opnd

    Vector<List<PhiNode*> > block_phi_nodes;
					// basic block id to its phi nodes
    CfgNode ***edge_maps;		// edge_maps[b][j] == src j's pred of b
    Vector<Opnd> name_map;		// name_map [value_id] == original name
    Vector<CfgNode*> block_map;		// block_map[value_id] == defining block
    Vector<Instr*> instr_map;		// instr_map[value_id] == defining instr

    Vector<int> formal_value_ids; 	// identifiers of SSA names for formals

    LivenessInfo *live_in;
    LivenessInfo *live_out;
    Vector<Operation> use_def_chains;
    Vector<List<Occurrence> > def_use_chains;

    bool cfg_renumbered;		// true while CFG is in SSA form
    PhiSrcMapEntry **phi_src_map;

    void print_def_use_chains(FILE*);
    void print_use_def_chains(FILE*);

    int enroll_old_opnd(Opnd opnd);
    int lookup_old_opnd(Opnd opnd) const;
    int enroll_new_opnd(Opnd opnd);
    int lookup_new_opnd(Opnd opnd) const;
    int (SsaForm::* enroll_opnd)(Opnd);	// One of enroll_{old,new}_opnd

    void record_use_swap_either(Opnd out, Opnd in, Occurrence);
    int record_def_either(Opnd old_name, Opnd new_name, Operation, CfgNode*);

    void record_liveness_info(NatSet **sets, bool live_out_info);
    void push_name(NameStackEntry **stack,
		   NatSet *pushed, int *node_list_pointer,
		   unsigned old_name_number, Opnd new_name);
    void rename_locs(CfgNode *block, NameStackEntry **name_stacks);
    void rewrite_liveness_info(LivenessInfo *set, NameStackEntry **name_stacks);
    int find_root(unsigned value_id, unsigned *loc_names);
    int phi_src_compare(PhiNode *phi_node, unsigned *loc_names);
    bool cfg_edges_split();
    void replace_phi_nodes();
    void insert_copies(CfgNode *block, CopyData *copy_stats);
    void set_comment(Instr*, IdString comment);

    DominanceInfo *dominator_analysis;
    DominanceChild **dominator_children;
    bool build_minimal_form, build_semi_pruned_form, build_pruned_form,
	build_use_def_chains, build_def_use_chains,
	fold_copies, drop_useless_phi_nodes,
	report_undefined_locs, keep_live_in_info, keep_live_out_info;
    NatSet **private_live_in;
    NatSet **private_live_out;
    AssignmentListEntry **assignments;
    NatSet *global_locs;
    NatSet **phi_node_uses;
    NatSet *pushed;
    unsigned new_name_counter;
    unsigned *phi_srcs_indices;
    bool replaced_phi_nodes;
    NatSet *dst_list;
    NatSet *worklist;

    // Classes for mapping the operands of instructions and phi-nodes

    friend class CountOpnds;
    friend class AddToLiveIn;
    friend class AddToCurKilled;
    friend class AddUsesToGlobal;
    friend class AddDefsToKilled;
    friend class FormAssignmentLists;
    friend class ReplaceUsedLocs;
    friend class RenameDefinedLocs;
    friend class ChangeNames;
    friend class RewriteTheUse;
    friend class ReplaceSourceNames;
    friend class RestoreRegisters;
    friend class RecordUses;
};


/*
 * Flags for SsaForm::build options.
 *
 * MINIMAL, SEMI_PRUNED, and PRUNED are mutually exclusive.  Minimal form
 * ignores liveness in placing phi-nodes.  Pruned form only inserts
 * phi-nodes at merge points where the defined value is known to be live.
 * Semi-pruned form is a compromise that doesn't require full-blown
 * liveness analysis.  It places phi-nodes only for names known to be live
 * on entry to _some_ basic block.  ("Minimal" form is a terrible name for
 * the form that inserts zillions of unneeded phi-nodes.  Avoid it.)
 *
 * BUILD_DEF_USE_CHAINS and BUILD_USE_DEF_CHAINS direct the library to
 * record def-use and use-def mappings, which are available through methods
 * of class SsaForm.  (The use-def info isn't really a "chain", of
 * course, since SSA form has only one def for each use.)
 *
 * FOLD_COPIES causes the transformation into SSA form to eliminate "move"
 * instructions involving SSA names by propagating the source of a move to
 * the occurrences of its destination name.
 *
 * DROP_USELESS_PHI_NODES causes a phi-node to be omitted if every path to
 * its basic block has the same SSA value for the original name.
 *
 * PRINT_WARNINGS turns some warnings that are suppressed by default.
 */
namespace ssa {

    enum {
        MINIMAL                = 1<<0,
        SEMI_PRUNED            = 1<<1,
        PRUNED                 = 1<<2,
        BUILD_DEF_USE_CHAINS   = 1<<3,
        BUILD_USE_DEF_CHAINS   = 1<<4,
        FOLD_COPIES            = 1<<5,
        DROP_USELESS_PHI_NODES = 1<<6,
        PRINT_WARNINGS         = 1<<7
    };

} // namespace ssa

extern "C" void init_ssa(SuifEnv*);

#endif /* SSA_SSA_FORM_SKEL_H */
