/* file "ssa/ssa_form.cpp" */

/*
    Copyright (c) 2000-2001 The President and Fellows of Harvard College

    All rights reserved.

    This software is provided under the terms described in
    the "machine/copyright.h" include file.
*/

#include <machine/copyright.h>

#ifdef USE_PRAGMA_INTERFACE
#pragma implementation "ssa/ssa_form.h"
#endif

#include <machine/machine.h>
#include <suifvm/suifvm.h>
#include <cfg/cfg.h>
#include <cfa/cfa.h>
#include <bvd/bvd.h>

#include <ssa/ssa_form.h>

#ifdef USE_DMALLOC
#include <dmalloc.h>
#define new D_NEW
#endif

using namespace ssa;

typedef List<PhiNode*>::iterator PhiNodeHandle;
typedef List<Occurrence>::iterator UseHandle;

// Operation::null_indicator's address is used to distinguish null values
// of class Operation.

PhiNode Operation::null_indicator;


static void
map_formal_defs(OptUnit *unit, OpndFilter &filter)
{
    for (int i = get_formal_param_count(unit) - 1; i >= 0; --i)
	filter(opnd_var(get_formal_param(unit, i)), OpndFilter::OUT);
}

/*
 * Return true iff `opnd' is a candidate for replacement by an SSA
 * "variable".  It must be a virtual register or a local variable whose
 * address is not taken.
 *
 * For now, we don't put code involving machine registers into SSA form.
 * We are implementing an approach described by Leung and George (PLDI '99)
 * for relaxing this restriction.  */

static bool
is_ssa_candidate(Opnd opnd)
{
    if (is_var(opnd))
	return is_auto(get_var(opnd)) && !is_addr_taken(get_var(opnd));
    return is_virtual_reg(opnd);
}

class LivenessInfo
{
  public:
    unsigned size;
    unsigned *names;
};

class DominanceChild
{
  public:
    DominanceChild(int node_number) { block_number = node_number; }
    ~DominanceChild() { if (next) delete next; }

    unsigned block_number;
    DominanceChild *next;
};

/*
 * A `phi source' is the parameter of a phi-node.  BHS calls this a `parm'.
 *
 * The `phi_src_map' entry for a block (`parm_map' in BHS) pairs each
 * successor of block with the corresponding index into the phi-node source
 * vectors of that successor.  Suppose S is a successor of block B, and
 * that B is designated the 2nd predecessor of S.  Then the phi_src_map
 * entry for B contains the pair <2, S>.
 */
struct PhiSrcMapEntry
{
    unsigned index;
    CfgNode *succ_block;
    PhiSrcMapEntry *next;
};

/*
 * An AssignmentListEntry links a definition to the CFG block that
 * contains it.  Assignment lists are anchored in the `assignments' array,
 * which is indexed by original location id number.
 */
struct AssignmentListEntry
{
    AssignmentListEntry(CfgNode *block_ptr, AssignmentListEntry *next_ptr)
      { block = block_ptr; next = next_ptr;}
    ~AssignmentListEntry() { delete next; }
    CfgNode *block;
    AssignmentListEntry *next;
};

/*
 * A "name", in Rice's terminology, is the representative of a location.
 * In our implementation, it has type Opnd.  An "old name" is an operand of
 * the untransformed program; a "new name" is one generated by conversion
 * to SSA form.
 *
 * The "name stack" for a given location is used in generating its new
 * names for SSA form.  The `next' field connects current stack members.
 * The `next_pushed' field connects entries for distinct locations that
 * need to popped at the same time.  It's an index into an array of name
 * stacks.
 */

struct NameStackEntry
{
    Opnd new_name;
    NameStackEntry *next;
    int next_pushed;
};

/*
 * CopyData keeps book on how one location is used in inserted copies during
 * rewrite_phi_nodes.  An instance of CopyData is always an entry in the
 * copy_stats table, which is indexed by the id of an SSA name.  That name is
 * thus an implicit part of the copy described by a table entry.
 *
 * src			Original source of the copy for which this entry is dst.
 * src_name_to_use	Source name to substitute for this in subsequent copies.
 * used_by_count	Number of times this is used as the src in another copy.
 * is_live_out		True iff this entry is live leaving current block.
 * temp_name		Temp name for this item, if there is one, else null.
 * new_name		Name stack used to substitute temp names for phi names.
 *
 * src_name_to_use is used for the `swap problem'.  temp_name and new_name
 * are used for the `lost-copy problem'.
 */

struct CopyData
{
    CopyData() { used_by_count = 0; is_live_out = false; new_name = NULL; }
    Opnd src;
    unsigned used_by_count;
    bool is_live_out;
    Opnd src_name_to_use;
    Opnd temp_name;
    NameStackEntry *new_name;
};


// --------------------  Filters for scanning instructions  --------------------

class CountOpnds : public OpndFilter
{
  public:
    CountOpnds(SsaForm *ssa_form) : ssa_form(ssa_form) { }
    Opnd operator()(Opnd, InOrOut);

  protected:
    SsaForm *ssa_form;
};

Opnd
CountOpnds::operator()(Opnd opnd, InOrOut usage)
{
    if (ssa_form->old_opnd_catalog->enroll(opnd))
	++ssa_form->_loc_count;
    return opnd;
}

class AddToLiveIn : public OpndFilter
{
  public:
    AddToLiveIn(SsaForm *ssa_form, NatSet *cur_killed, NatSet *cur_live_in)
	: ssa_form(ssa_form), cur_killed(cur_killed), cur_live_in(cur_live_in)
    { }
    Opnd operator()(Opnd, InOrOut);

  private:
    SsaForm *ssa_form;
    NatSet *cur_killed;
    NatSet *cur_live_in;
};

Opnd
AddToLiveIn::operator()(Opnd opnd, InOrOut in_or_out)
{
    if (in_or_out == IN)
    {
	int old_opnd_id = (ssa_form->*(ssa_form->enroll_opnd))(opnd);
	if (old_opnd_id >= 0 && !cur_killed->contains(old_opnd_id))
	    cur_live_in->insert(old_opnd_id);
    }
    return opnd;
}

class AddToCurKilled : public OpndFilter
{
  public:
    AddToCurKilled(SsaForm *ssa_form, NatSet *cur_killed)
	: ssa_form(ssa_form), cur_killed(cur_killed)
    { }
    Opnd operator()(Opnd, InOrOut);

  private:
    SsaForm *ssa_form;
    NatSet *cur_killed;
};

Opnd
AddToCurKilled::operator()(Opnd opnd, InOrOut in_or_out)
{
    if (in_or_out == OUT)
    {
	int old_opnd_id = (ssa_form->*(ssa_form->enroll_opnd))(opnd);
	if (old_opnd_id >= 0)
	    cur_killed->insert(old_opnd_id);
    }
    return opnd;
}

class AddUsesToGlobal : public OpndFilter
{
  public:
    AddUsesToGlobal(SsaForm *ssa_form, NatSet *killed_in_block)
	: ssa_form(ssa_form), killed_in_block(killed_in_block)
    { }
    Opnd operator()(Opnd, InOrOut);

  private:
    SsaForm *ssa_form;
    NatSet *killed_in_block;
};

Opnd
AddUsesToGlobal::operator()(Opnd opnd, InOrOut in_or_out)
{
    if (in_or_out == IN)
    {
	int old_opnd_id = ssa_form->enroll_old_opnd(opnd);
	if (old_opnd_id >= 0 && !killed_in_block->contains(old_opnd_id))
	    ssa_form->global_locs->insert(old_opnd_id);
    }
    return opnd;
}

class AddDefsToKilled : public OpndFilter
{
  public:
    AddDefsToKilled(SsaForm *ssa_form, NatSet *killed_in_block)
	: ssa_form(ssa_form), killed_in_block(killed_in_block)
    { }
    Opnd operator()(Opnd, InOrOut);

  private:
    SsaForm *ssa_form;
    NatSet *killed_in_block;
};

Opnd
AddDefsToKilled::operator()(Opnd opnd, InOrOut in_or_out)
{
    if (in_or_out == OUT)
    {
	int old_opnd_id = ssa_form->enroll_old_opnd(opnd);
	if (old_opnd_id >= 0)
	    killed_in_block->insert(old_opnd_id);
    }
    return opnd;
}

class FormAssignmentLists : public OpndFilter
{
  public:
    FormAssignmentLists(SsaForm *ssa_form, CfgNode *block)
	: ssa_form(ssa_form), block(block)
    { }
    Opnd operator()(Opnd, InOrOut);

  private:
    SsaForm *ssa_form;
    CfgNode *block;
};

Opnd
FormAssignmentLists::operator()(Opnd opnd, InOrOut in_or_out)
{
    if (in_or_out == OUT)
    {
	int old_opnd_id = ssa_form->enroll_old_opnd(opnd);
	if (old_opnd_id >= 0)
	{
	    ssa_form->def_count++;
	    if (ssa_form->build_minimal_form ||
		ssa_form->global_locs->contains(old_opnd_id))
	    {
		AssignmentListEntry *first = ssa_form->assignments[old_opnd_id];

		if (!first || first->block != block)
		{
		    AssignmentListEntry *new_entry =
			new AssignmentListEntry(block, first);
		    ssa_form->assignments[old_opnd_id] = new_entry;
		}
	    }
	}
    }
    return opnd;
}

class ReplaceUsedLocs : public OpndFilter
{
  public:
    ReplaceUsedLocs(SsaForm *ssa_form, NameStackEntry **name_stacks,
		    InstrHandle handle, CfgNode *block)
	: ssa_form(ssa_form), name_stacks(name_stacks),
	  handle(handle), block(block)
    { }
    Opnd operator()(Opnd, InOrOut);

  protected:
    SsaForm *ssa_form;
    NameStackEntry **name_stacks;
    InstrHandle handle;
    CfgNode *block;
};

Opnd
ReplaceUsedLocs::operator()(Opnd opnd, InOrOut in_or_out)
{
    if (in_or_out == OUT)
	return opnd;

    int old_opnd_id = ssa_form->lookup_old_opnd(opnd);
    if (old_opnd_id < 0)
	return opnd;

    if (!name_stacks[old_opnd_id])
    {
	// Rice tries to eliminate any reference to an uninitialized location,
	// substituting an unrelated value if necessary.  We leave the code as
	// is.
	if (ssa_form->report_undefined_locs)
	    fprintf(stderr, "Found an uninitialized location: "
		    "opcode = %d, location #%d.\n",
		    get_opcode(*handle), old_opnd_id);
    }
    else
    {
	opnd = name_stacks[old_opnd_id]->new_name;
	if (ssa_form->build_def_use_chains)
	{
	    unsigned value_id = ssa_form->lookup_new_opnd(opnd);
	    claim(value_id < ssa_form->def_count);
	    Occurrence occurrence(handle, block);
	    ssa_form->def_use_chains[value_id].push_back(occurrence);
	}
    }
    return opnd;
}

class RenameDefinedLocs : public OpndFilter
{
  public:
    RenameDefinedLocs(SsaForm *ssa_form, InstrHandle handle,
			   CfgNode *block, NameStackEntry **name_stacks,
			   int *pushed_list)
	: ssa_form(ssa_form), handle(handle), block(block),
	  name_stacks(name_stacks), pushed_list(pushed_list)
    { }
    Opnd operator()(Opnd, InOrOut);

  private:
    SsaForm *ssa_form;
    InstrHandle handle;
    CfgNode *block;
    NameStackEntry **name_stacks;
    int *pushed_list;
};

Opnd
RenameDefinedLocs::operator()(Opnd opnd, InOrOut in_or_out)
{
    if (in_or_out == OUT)
    {
	int old_opnd_id = ssa_form->lookup_old_opnd(opnd);
	if (old_opnd_id >= 0)
	{
	    Opnd new_opnd = opnd_reg(get_type(opnd));
	    int new_opnd_id = ssa_form->enroll_new_opnd(new_opnd);
	    claim(new_opnd_id >= 0);

	    ssa_form->push_name(&name_stacks[old_opnd_id], ssa_form->pushed,
				pushed_list, old_opnd_id, new_opnd);
	    if (ssa_form->build_use_def_chains)
		ssa_form->use_def_chains[new_opnd_id] =
		    Operation(handle);
	    ssa_form->name_map [new_opnd_id] = opnd;
	    ssa_form->block_map[new_opnd_id] = block;
	    ssa_form->instr_map[new_opnd_id] = *handle;

	    opnd = new_opnd;
	}
    }
    return opnd;
}

class ChangeNames : public OpndFilter
{
  public:
    ChangeNames(SsaForm *ssa_form, unsigned *loc_names)
	: ssa_form(ssa_form), loc_names(loc_names)
    { }
    Opnd operator()(Opnd, InOrOut);

  private:
    SsaForm *ssa_form;
    unsigned *loc_names;
};

Opnd
ChangeNames::operator()(Opnd opnd, InOrOut)
{
    int new_opnd_id = ssa_form->lookup_new_opnd(opnd);
    if (new_opnd_id >= 0)
	return ssa_form->new_opnd_table[loc_names[new_opnd_id]];
    return opnd;
}

class RewriteTheUse : public OpndFilter
{
  public:
    RewriteTheUse(SsaForm *ssa_form, Opnd opnd_to_replace,
		  unsigned replacement_id)
	: ssa_form(ssa_form), opnd_to_replace(opnd_to_replace),
	  replacement_id(replacement_id)
    { }
    Opnd operator()(Opnd, InOrOut);

  private:
    SsaForm *ssa_form;
    Opnd opnd_to_replace;
    unsigned replacement_id;
};

Opnd
RewriteTheUse::operator()(Opnd opnd, InOrOut in_or_out)
{
    if ((in_or_out == IN) && (opnd == opnd_to_replace))
	opnd = ssa_form->new_opnd_table[replacement_id];
    return opnd;
}

class ReplaceSourceNames : public OpndFilter
{
  public:
    ReplaceSourceNames(SsaForm *ssa_form, CopyData *copy_stats)
	: ssa_form(ssa_form), copy_stats(copy_stats)
    { }
    Opnd operator()(Opnd, InOrOut);

  private:
    SsaForm *ssa_form;
    CopyData *copy_stats;
};

Opnd
ReplaceSourceNames::operator()(Opnd opnd, InOrOut in_or_out)
{
    if (in_or_out == IN)
    {
	int new_opnd_id = ssa_form->lookup_new_opnd(opnd);
	if ((new_opnd_id >= 0) && copy_stats[new_opnd_id].new_name)
	    opnd = copy_stats[new_opnd_id].new_name->new_name;
    }
    return opnd;
}

class RestoreRegisters : public OpndFilter
{
  public:
    RestoreRegisters(SsaForm *ssa_form) : ssa_form(ssa_form) { }
    Opnd operator()(Opnd, InOrOut);

  private:
    SsaForm *ssa_form;
};

Opnd
RestoreRegisters::operator()(Opnd opnd, InOrOut)
{
    int value_number = ssa_form->lookup_new_opnd(opnd);
    if (value_number >= 0)
    {
	if ((unsigned)value_number < ssa_form->def_count)
	    opnd = ssa_form->name_map[value_number];
	else
	{
	    opnd = opnd_reg(get_type(opnd));
	    warn("Creating $vr%d from thin air for value %d",
		  get_reg(opnd), value_number);
	}
    }
    return opnd;
}

/*
 * Update the def-use chain for an SSA name found as an input operand of a
 * newly planted instruction or phi-node.
 */

class RecordUses : public OpndFilter
{
  public:
    RecordUses(SsaForm *ssa_form, InstrHandle ih)
	: ssa_form(ssa_form),
	  occurrence(ih, get_parent_node(*ih))
    { }
    RecordUses(SsaForm *ssa_form, PhiNode *phi)
	: ssa_form(ssa_form),
	  occurrence(phi, ssa_form->phi_node_block(phi), 0)
    { }
    Opnd operator()(Opnd name, InOrOut);

  protected:
    SsaForm *ssa_form;
    Occurrence occurrence;
};

Opnd
RecordUses::operator()(Opnd name, InOrOut in_or_out)
{
    if (in_or_out == IN)
    {
	int name_id = ssa_form->lookup_new_opnd(name);
	if (name_id >= 0)
	    ssa_form->def_use_chains[name_id].push_back(occurrence);
	if (occurrence.is_phi_node())
	    occurrence.set_phi_src_pos(occurrence.get_phi_src_pos() + 1);
    }
    return name;
}


// ----------------------------	 class SsaForm  --------------------------------

void
SsaForm::build(Cfg *cfg, unsigned flags)
{
    unit_cfg = cfg;
    unit = to<OptUnit>(cfg->get_parent());
    assignments = NULL;

    old_opnd_catalog = new RegSymCatalog(false, is_ssa_candidate);
    new_opnd_catalog = new RegSymCatalog(false, is_ssa_candidate);

    // Declare the size of the old and new operand tables after we figure
    // out how many operands there are going to be.

    def_count = 0;
    live_in = NULL;
    live_out = NULL;
    cfg_renumbered = false;
    new_name_counter = 1;
    replaced_phi_nodes = false;
    phi_src_map = NULL;

    enroll_opnd = &SsaForm::enroll_old_opnd;

    build_minimal_form	   = !!(flags & MINIMAL);
    build_semi_pruned_form = !!(flags & SEMI_PRUNED);
    build_pruned_form	   = !!(flags & PRUNED);
    build_def_use_chains   = !!(flags & BUILD_DEF_USE_CHAINS);
    build_use_def_chains   = !!(flags & BUILD_USE_DEF_CHAINS);
    fold_copies		   = !!(flags & FOLD_COPIES);
    drop_useless_phi_nodes = !!(flags & DROP_USELESS_PHI_NODES);
    keep_live_in_info	   = false; 		// for now, we don't save ..
    keep_live_out_info	   = false;		// .. liveness info from SSA
    report_undefined_locs  = debuglvl > 0;

    unsigned flavor = flags & (MINIMAL | SEMI_PRUNED | PRUNED);

    claim(   flavor == MINIMAL
	  || flavor == SEMI_PRUNED
	  || flavor == PRUNED,
	"Ambiguous SSA kind.  Use exactly one of:\n"
	"\tssa::MINIMAL, ssa::SEMI_PRUNED, or ssa::PRUNED.\n");

    // Iterate through every block, counting each one, and count every new
    // operand in each block.

    _loc_count = 0;
    block_count = 0;

    CountOpnds count_opnds(this);		// bumps _loc_count
    map_formal_defs(unit, count_opnds);		// count formals first

    for (int i = 0; i < nodes_size(unit_cfg); i++)
    {
	CfgNode *block = get_node(unit_cfg, i);
	block_count++;

	InstrHandle h = instrs_start(block);
	for (int j = instrs_size(block); j > 0; --j, ++h)
	    map_opnds(*h, count_opnds);
    }

    old_opnd_table.resize(_loc_count);

    // Compute dominance frontiers and build dominator tree

    dominator_analysis = new DominanceInfo(unit_cfg);
    dominator_analysis->find_dominators();
    dominator_analysis->find_dom_frontier();

    dominator_children = new (DominanceChild*)[block_count+1];

    for (unsigned i = 0; i < block_count + 1; ++i)
	dominator_children[i] = NULL;

    for (int i = 0; i < size(unit_cfg); i++)
    {
        CfgNode *block = get_node(unit_cfg, i);
        int block_number = get_number(block);
        CfgNode *immediate_dominator =
	    dominator_analysis->immed_dom(block_number);

        if (immediate_dominator != NULL)
	{
            // Node has a parent
            int dominator_number = get_number(immediate_dominator);

            DominanceChild *temp_ptr = new DominanceChild(block_number);
            if (dominator_children[dominator_number] == NULL)
	    {
                dominator_children[dominator_number] = temp_ptr;
                temp_ptr->next = NULL;
	    }
            else
	    {
                temp_ptr->next = dominator_children[dominator_number];
                dominator_children[dominator_number] = temp_ptr;
	    }
	}
    }

    // Now that block count and location count are known, initialize tables
    // having those sizes.

    if (build_pruned_form || keep_live_in_info)
        private_live_in = new (NatSet*)[block_count+1];
    if  (keep_live_out_info)
        private_live_out = new (NatSet*)[block_count+1];
    assignments = new (AssignmentListEntry*)[_loc_count];
    for (unsigned i = 0; i < _loc_count; ++i)
         assignments[i] = NULL;
    if (!build_minimal_form)
        global_locs = new NatSetSparse();

    block_phi_nodes.resize(block_count + 1);

    edge_maps = new (CfgNode**)[block_count+1];

    CfgNodeListRpo rpo(unit_cfg);
    for (CfgNodeHandle h = rpo.start(); h != rpo.end(); ++h)
    {
	CfgNode *block = *h;
	if (block != get_entry_node(unit_cfg))
	{
	    unsigned pred_count = preds_size(block);
	    edge_maps[get_number(block)] = new (CfgNode*)[pred_count];
	}
    }

    phi_src_map = new (PhiSrcMapEntry*)[block_count + 1];
    for (unsigned l = 0; l < block_count + 1; l++)
      phi_src_map[l] = NULL;

    if (build_pruned_form || keep_live_in_info || keep_live_out_info)
    {
        // Perform liveness analysis.

	NatSet **killed = new (NatSet*)[block_count + 1];

	for (CfgNodeHandle h = rpo.start(); h != rpo.end(); ++h)
	{
	    CfgNode *block = *h;
	    Instr *inst;
	    unsigned block_num = get_number(block);

	    NatSet *cur_live_in;
	    NatSet *cur_live_out;

	    private_live_in[block_num] = new NatSetSparse();
	    cur_live_in = private_live_in[block_num];

	    if (cfg_renumbered || keep_live_out_info)
	    {
		private_live_out[block_num] = new NatSetSparse();
		cur_live_out = private_live_out[block_num];
	    }

	    if (block != get_exit_node(unit_cfg))
	    {
		killed[block_num] = new NatSetSparse();
		NatSet *cur_killed = killed[block_num];

		if (cfg_renumbered)
		{
		    List<PhiNode*> block_phi_nodes = phi_nodes(block);
		    for (PhiNodeHandle h = block_phi_nodes.begin();
			 h != block_phi_nodes.end(); ++h)
		    {
			PhiNode *phi_node = *h;
			int value_id =
			    lookup_new_opnd(phi_node->get_new_name());
			claim(value_id >= 0);
			cur_killed->insert(value_id);
		    }
		}


		InstrHandle list_iterator = instrs_start(block);
		for (int j = 0; list_iterator != instrs_end(block);
		     j++, list_iterator++)
		{
		    inst = *list_iterator;
		    AddToLiveIn add_to_live_in(this, cur_killed, cur_live_in);
		    map_opnds(inst, add_to_live_in);

		    AddToCurKilled add_to_cur_killed(this, cur_killed);
		    map_dst_opnds(inst, add_to_cur_killed);
		}
	    }
	}
	if (cfg_renumbered)
	{
	    phi_node_uses = new (NatSet*)[block_count + 1];

	    CfgNodeListRpo rpo(unit_cfg);
	    for (CfgNodeHandle h = rpo.start(); h != rpo.end(); h++)
	    {
		CfgNode *block = *h;

		unsigned block_num = get_number(block);
		phi_node_uses[block_num] = new NatSetSparse;

		for (PhiSrcMapEntry *pair = phi_src_map[block_num];
		     pair; pair = pair->next)
		{
		    unsigned phi_src_index = pair->index;
		    CfgNode *succ = pair->succ_block;

		    List<PhiNode*> succ_phi_nodes = phi_nodes(succ);
		    for (PhiNodeHandle h = succ_phi_nodes.begin();
			 h != succ_phi_nodes.end(); ++h)
		    {
			int phi_node_number =
			    lookup_new_opnd((*h)->get_src(phi_src_index));
			claim(phi_node_number >= 0, "Null src in phi-node");
			phi_node_uses[block_num]->insert(phi_node_number);
		    }
		}
	    }
	}

	bool changed;
	NatSet *temp = new NatSetSparse;

	do {
	    CfgNode *block;

	    changed = false;
	    CfgNodeListRpo rpo(unit_cfg);
	    for (CfgNodeHandle h = rpo.start(); h != rpo.end(); h++)
	    {
		block = *h;

		if (block != get_exit_node(unit_cfg))
		{
		    unsigned block_num = get_number(block);
		    NatSet *current_live_in = private_live_in[block_num];
		    NatSet *current_live_out =
			(cfg_renumbered || keep_live_out_info)
			    ? private_live_out[block_num]
			    : NULL;

		    CfgNode *successor;
		    bool first_successor = true;

		    for (CfgNodeHandle sh = succs_start(block);
			 sh != succs_end(block); ++sh)
		    {
			successor = *sh;
			if (first_successor)
			{
			    // Originally copied pointers - trying this
			    *temp = *private_live_in[get_number(successor)];
			    first_successor = false;
			}
			else
			    *temp += *(private_live_in[get_number(successor)]);
		    }

		    if (cfg_renumbered)
			*temp += *(phi_node_uses[get_number(block)]);

		    if ((cfg_renumbered || keep_live_out_info) &&
			(current_live_out != temp))
			current_live_out = temp;

		    NatSet *killed_complement =
			new NatSetSparse(*(killed[block_num]));
		    killed_complement->complement();
		    *temp *= *killed_complement;
		    *temp += *current_live_in;

		    if (*current_live_in != *temp)
		    {
			changed = true;
			*current_live_in = *temp;
		    }
		}
	    }
	} while (changed);

	if (keep_live_in_info)
	    record_liveness_info(private_live_in, false);
	if (keep_live_out_info)
	    record_liveness_info(private_live_out, true);

	if (!keep_live_out_info)
	{
	    // May want to add some other deletes here now that arena delete is
	    // gone
	}
    }

    // <<Find assignments>>
    if (!build_minimal_form)
    {
	// <<Find global locations>>, i.e. those whose live ranges cross
	// basic-block boundaries.

	NatSet *killed_in_block = new NatSetSparse;

	CfgNodeListRpo rpo(unit_cfg);
	for (CfgNodeHandle h = rpo.start(); h != rpo.end(); ++h)
	{
	    CfgNode *block = *h;
	    killed_in_block->remove_all();

	    InstrHandle list_iterator = instrs_start(block);
	    for (int j = 0; j < instrs_size(block);
		 j++, list_iterator++)
	    {
		Instr *instr = *list_iterator;

		AddUsesToGlobal add_uses_to_global(this, killed_in_block);
		map_opnds(instr, add_uses_to_global);
		AddDefsToKilled add_defs_to_killed(this, killed_in_block);
		map_dst_opnds(instr, add_defs_to_killed);
	    }
	}
    }

    // <<Accumulate assignment blocks for global locations>>

    for (CfgNodeHandle bh = rpo.start(); bh != rpo.end(); ++bh)
    {
	CfgNode *block = *bh;
	FormAssignmentLists form_assignment_lists(this, block);

	InstrHandle ih = instrs_start(block);
	for (int j = instrs_size(block); j > 0; --j, ++ih)
	{
	    Instr *instr = *ih;

	    // <<Examine all the definitions in [[instr]]>>
	    if (has_note(instr, k_proc_entry))
		map_formal_defs(unit, form_assignment_lists);
	    else
		map_dst_opnds(instr, form_assignment_lists);
	}
    }

    // <<Place phi-nodes>>

    int loc;
    NatSet *in_list = new NatSetSparse;
    NatSet *has_node = new NatSetSparse;

    if (build_minimal_form)
    {
	int max_slot = old_opnd_catalog->num_slots();
	for (loc = 0; loc < max_slot; ++loc)
	{
	    // <<Place the phiNodes for [[loc]]>>
	    AssignmentListEntry *assignment_list = assignments[loc]; // worklist

	    has_node->remove_all();
	    has_node->insert(get_number(get_exit_node(unit_cfg)));

	    // <<Initialize the set [[in_list]]>>

	    in_list->remove_all();
	    for (AssignmentListEntry *assignment = assignment_list;
		 assignment != NULL;
		 assignment = assignment->next)
		in_list->insert(get_number(assignment->block));

	    while (assignment_list)
	    {
		CfgNode *block = assignment_list->block;
		assignment_list = assignment_list->next;

		// <<Place phi-nodes for [[loc]] in [[block]]'s iterated dominance
		//   frontier and update [[assignment_list]]>>

		const NatSet *frontier =
		    dominator_analysis->dom_frontier(get_number(block));

		NatSetIter frontier_iterator = frontier->iter();
		for (/* */; frontier_iterator.is_valid(); frontier_iterator.next())
		{
		    unsigned frontier_index = frontier_iterator.current();
		    CfgNode *frontier_block =
			get_node(unit_cfg, frontier_index);

		    if (!has_node->contains(frontier_index) &&
			(!build_pruned_form ||
			 (private_live_in[frontier_index]->contains(loc))))
		    {
			has_node->insert(frontier_index);

			// <<Place a phi-node for [[loc]] in
			//   [[frontier_block]]>>

			unsigned pred_count = preds_size(frontier_block);

			PhiNode *new_node = new PhiNode(pred_count);
			new_node->set_old_name(old_opnd_table[loc]);
			def_count++;		// for the phi-node just created
			block_phi_nodes[frontier_index].push_back(new_node);

			// <<Put [[frontier_block]] into [[assignment_list]]

			if (!in_list->contains(frontier_index))
			{
			    AssignmentListEntry *new_node =
				new AssignmentListEntry(frontier_block,
							assignment_list);
			    assignment_list = new_node;
			    in_list->insert(frontier_index);
			}
		    }
		}
	    }
	}
    }
    else			// (building semi-pruned or pruned)
    {
	NatSetIter ss_iterator = global_locs->iter();
	for (/* */; ss_iterator.is_valid(); ss_iterator.next())
	{
	    loc = ss_iterator.current();

	    // <<Place the phiNodes for [[loc]]>>

	    AssignmentListEntry *assignment_list = assignments[loc]; // worklist

	    has_node->remove_all();
	    has_node->insert(get_number(get_exit_node(unit_cfg)));

	    // <<Initialize the set [[in_list]]>>

	    in_list->remove_all();
	    for (AssignmentListEntry *an = assignment_list; an; an = an->next)
		in_list->insert(get_number(an->block));

	    while (assignment_list)
	    {
		CfgNode *block = assignment_list->block;

		assignment_list = assignment_list->next;
		const NatSet *frontier =
		    dominator_analysis->dom_frontier(get_number(block));

		NatSetIter frontier_iter = frontier->iter();
		for (/* */; frontier_iter.is_valid(); frontier_iter.next())
		{
		    unsigned frontier_index = frontier_iter.current();
		    CfgNode *frontier_block =
			get_node(unit_cfg, frontier_index);

		    if (!has_node->contains(frontier_index) &&
			(!build_pruned_form ||
			 (private_live_in[frontier_index]->contains(loc))))
		    {
			has_node->insert(frontier_index);

			unsigned pred_count = preds_size(frontier_block);

			PhiNode *new_node = new PhiNode(pred_count);
			Opnd old_opnd = old_opnd_table[loc];
			new_node->set_old_name(old_opnd);

			def_count++;	// count new phi-node as a def
			block_phi_nodes[frontier_index].push_back(new_node);

			if (!in_list->contains(frontier_index))
			{
			    AssignmentListEntry *new_node =
				new AssignmentListEntry(frontier_block,
							assignment_list);
			    assignment_list = new_node;
			    in_list->insert(frontier_index);
			}
		    }
		}
	    }
	}
    }

    // We are done with data-flow information.  On to renaming locations.
    // <<Rename resources>>

    NameStackEntry **name_stacks = new (NameStackEntry*)[_loc_count];
    for (unsigned l = 0; l < _loc_count; ++l)
	name_stacks[l] = NULL;

    // <<Initialize variables during renaming>>
    pushed = new NatSetSparse;

    phi_srcs_indices = new (unsigned)[block_count + 1];
    for (unsigned l = 0; l < block_count + 1; ++l)
	phi_srcs_indices[l] = 0;

    // <<Initialize variables whose size depends on [[def_count]]>>

    block_map.resize(def_count, NULL);
    instr_map.resize(def_count, NULL);
    name_map .resize(def_count, NULL);

    new_opnd_table.resize(def_count);

    if (build_use_def_chains)
	use_def_chains.resize(def_count, Operation());

    if (build_def_use_chains)
	def_use_chains.resize(def_count, List<Occurrence>());

    rename_locs(get_entry_node(unit_cfg), name_stacks);

    if (debuglvl >= 5)
    {
	fputs("Original operands:\n", stderr);
	for (unsigned i = 0; i < _loc_count; ++i)
	{
	    fprintf(stderr, "%3d: ", i);
	    fprint (stderr, old_opnd_table[i]);
	    putc('\n', stderr);
	}
	fputs("SSA operands:\n", stderr);
	for (unsigned i = 0; i < def_count; ++i)
	{
	    fprintf(stderr, "%3d: ", i);
	    fprint (stderr, new_opnd_table[i]);
	    fprintf(stderr, "\twas: ");
	    fprint (stderr, name_map[i]);
	    putc('\n', stderr);
	}
    }

    delete [] phi_srcs_indices;
    delete [] name_stacks;

    // <<Fold useless phi-nodes, if necessary>>

    if (drop_useless_phi_nodes)
    {
        if (build_def_use_chains && !(keep_live_in_info || keep_live_out_info))
	{
	    // <<Remove useless phi-nodes using the worklist method>>

	    NatSet *worklist = new NatSetSparse;
	    PhiNode **phi_node_map = new (PhiNode*)[def_count];

	    unsigned number_removed = 0;
	    unsigned *loc_names;

	    // <<Initialize [[loc_names]]>>

	    loc_names = new (unsigned)[def_count];
	    for (unsigned i = 0; i < def_count; ++i)
		loc_names[i] = i;

	    // <<Initialize [[phi_node_map]] and the worklist for removing
	    //   useless phi-nodes>>

	    CfgNodeListRpo rpo(unit_cfg);
	    for (CfgNodeHandle h = rpo.start(); h != rpo.end(); ++h)
	    {
		CfgNode *block = *h;

		List<PhiNode*> block_phi_nodes = phi_nodes(block);
		for (PhiNodeHandle h = block_phi_nodes.begin();
		     h != block_phi_nodes.end(); ++h)
		{
		    PhiNode *phi_node = *h;
		    Opnd phi_node_name = phi_node->get_new_name();
		    unsigned phi_node_number =
			lookup_new_opnd(phi_node_name);
		    claim(phi_node_number < def_count);
		    phi_node_map[phi_node_number] = phi_node;
		    worklist->insert(phi_node_number);
		}
	    }

	    while (worklist->size() > 0)
	    {
		// <<Examine the uses of the next phi-node on the worklist>>

		unsigned next_phi_node = worklist->iter().current();
		PhiNode *phi_node = phi_node_map[next_phi_node];
		int replacement_id = phi_src_compare(phi_node, loc_names);

		worklist->remove(next_phi_node);
		if (replacement_id >= 0)
		{
		    if (loc_names[next_phi_node] == next_phi_node)
			++number_removed;

		    // <<Rewrite uses and add any new phi-nodes to the
		    //   worklist>>

		    List<Occurrence> chain = def_use_chains[next_phi_node];
		    for (UseHandle h = chain.begin(); h != chain.end(); ++h)
		    {
			Occurrence use = *h;
			if (use.is_phi_node())
			{
			    // <<Rewrite the parameter and add it to the
			    //   worklist>>
			    // (This scrap descriptor is misleading: we actually
			    // add to the worklist the phi-node whose parameter
			    // we are rewriting.)

			    PhiNode *target = use.get_phi_node();
			    Opnd new_src = new_opnd_table[replacement_id];
			    target->set_src(use.get_phi_src_pos(), new_src);
			    Opnd new_name = target->get_new_name();
			    worklist->insert(lookup_new_opnd(new_name));
			}
			else
			{
			    // <<Rewrite the use>>
			    Opnd to_be_replaced = new_opnd_table[next_phi_node];

			    // Search through uses until operand to be replaced
			    // is found and replace it.
			    RewriteTheUse filter(this, to_be_replaced,
						 replacement_id);
			    map_opnds(*use.get_instr_handle(), filter);
			}
		    }

		    // <<Move the def-use list>>
		    end_splice(def_use_chains[replacement_id],
			       def_use_chains[next_phi_node]);
		    phi_node->set_useless(true);
		}
	    }
	    debug(5, "Number of dead phi-nodes removed: %d", number_removed);
	}
        else
	{
	    // <<Remove useless phi-nodes using the iterative method>>

	    unsigned *loc_names, number_removed = 0;
	    bool changed = true;

	    // <<Initialize [[loc_names]]>>

	    loc_names = new (unsigned)[def_count];
	    for (unsigned i = 0; i < def_count; ++i)
		loc_names[i] = i;

	    while (changed)
	    {
		changed = false;

		// <<Search the graph for useless phi-nodes and set [[changed]]
		//   if one is found>>

		CfgNodeListRpo rpo(unit_cfg);
		for (CfgNodeHandle h = rpo.start(); h != rpo.end(); ++h)
		{
		    CfgNode *block = *h;

		    List<PhiNode*> block_phi_nodes = phi_nodes(block);
		    for (PhiNodeHandle h = block_phi_nodes.begin();
			 h != block_phi_nodes.end(); ++h)
		    {
			PhiNode *phi_node = *h;
			Opnd phi_name = phi_node->get_new_name();
			unsigned phi_id = lookup_new_opnd(phi_name);
			int replacement_id =
			    phi_src_compare(phi_node, loc_names);

			if (replacement_id >= 0 &&
			    replacement_id != find_root(phi_id, loc_names))
			{
			    if (loc_names[phi_id] == phi_id)
				number_removed++;
			    loc_names[phi_id] = replacement_id;
			    changed = true;
			    phi_node->set_useless(true);
			}
		    }
		}
		if (!changed)
		{
		    // <<Compress the trees>>
		    for (unsigned i = 0; i < def_count; ++i)
			loc_names[i] = find_root(i, loc_names);
		}
	    }

	    // <<Rewrite the graph using the new names in [[loc_names]]>>

	    CfgNodeListRpo rpo(unit_cfg);
	    for (CfgNodeHandle h = rpo.start(); h != rpo.end(); ++h)
	    {
		CfgNode *block = *h;

		List<PhiNode*> block_phi_nodes = phi_nodes(block);
		for (PhiNodeHandle h = block_phi_nodes.begin();
		     h != block_phi_nodes.end(); ++h)
		{
		    PhiNode *phi_node = *h;

		    for (int i = 0; i < phi_node->srcs_size(); ++i)
		    {
			Opnd src = phi_node->get_src(i);
			int src_id = lookup_new_opnd(src);
			Opnd new_opnd = new_opnd_table[loc_names[src_id]];
			phi_node->set_src(i, new_opnd);
		    }
		}
		InstrHandle list_iterator = instrs_start(block);
		for (int j = 0; j < instrs_size(block); ++j, ++list_iterator)
		{
		    Instr *instr = *list_iterator;
		    ChangeNames opnd_function(this, loc_names);
		    map_opnds(instr, opnd_function);
		}
		if (keep_live_in_info)
		{
		    // <<Rewrite liveness information to account for removal of
		    //   phi-nodes>>
		    LivenessInfo *current = &live_in[get_number(block)];
		    unsigned *names = current->names;

		    for (unsigned i = 0; i < current->size; ++i)
			names[i] = loc_names[names[i]];
		}
		if (keep_live_out_info)
		{
		    // <<Rewrite liveness information to account for removal of
		    //   phi-nodes>>
		    LivenessInfo *current = &live_out[get_number(block)];
		    unsigned *names = current->names;

		    for (unsigned i = 0; i < current->size; ++i)
			names[i] = loc_names[names[i]];
		}
	    }
	    debug(5, "Number of dead phi-nodes removed: %d", number_removed);
	}

	// <<Remove the useless phi-nodes' data structures from the graph>>

        CfgNodeListRpo rpo(unit_cfg);
        for (CfgNodeHandle h = rpo.start(); h != rpo.end(); ++h)
        {
	    List<PhiNode*> &phi_list = phi_nodes(*h);

	    for (PhiNodeHandle h = phi_list.begin(); h != phi_list.end(); /* */)
	    {
		PhiNodeHandle this_h = h++;
		PhiNode *phi_node = *this_h;
		if (phi_node->get_useless())
                {
		    phi_list.erase(this_h);
		    delete phi_node;
                }
            }
        }
    }

    if (keep_live_out_info)
    {
        unsigned i;
        NatSet *temp = new NatSetSparse;
        CfgNode *block;
        CfgNode *successor;

        CfgNodeListRpo rpo(unit_cfg);
        for (CfgNodeHandle h = rpo.start(); h != rpo.end(); ++h)
        {
	    block = *h;

	    unsigned block_num = get_number(block);
	    LivenessInfo *cur_live_out = &live_out[block_num];

	    if (block == get_entry_node(unit_cfg) ||
		block == get_exit_node(unit_cfg))
            {
		live_out[block_num].size = 0;
		continue;
            }
	    temp->remove_all();
	    CfgNodeHandle succ_iterator = succs_start(block);
	    for (/* */; succ_iterator != succs_end(block); ++succ_iterator)
            {
		successor = *succ_iterator;
		LivenessInfo *cur_live_in = &live_in[get_number(successor)];

		for (i = 0; i < cur_live_in->size; ++i)
		    temp->insert(cur_live_in->names[i]);
            }
	    cur_live_out->size = 0;
	    cur_live_out->names = new (unsigned)[temp->size()];
	    NatSetIter ss_iterator = temp->iter();
	    for (/* */; ss_iterator.is_valid(); ss_iterator.next())
            {
                i = ss_iterator.current();
		cur_live_out->names[cur_live_out->size++] = i;
            }
        }
        keep_live_out_info = false; /* we only do this once */
    }
    keep_live_in_info = false; /* we only do this once -- this information
				  was built inside of rename_locs */
    cfg_renumbered = true;

    if (debuglvl >= 4)
    {
	fprintf(stderr, "\nJust after building SSA form:\n");
	this->print(stderr);
    }
} // build

void
SsaForm::replace_phi_nodes()
{
    CopyData *copy_stats = new CopyData[def_count];

    if (fold_copies && !cfg_edges_split())
    {
        NatSet **private_live_in;

	enroll_opnd = &SsaForm::enroll_new_opnd;

        private_live_in = new (NatSet*)[block_count + 1];
        private_live_out = new (NatSet*)[block_count + 1];

	NatSet **killed = new (NatSet*)[block_count + 1];

	CfgNode *block;
	CfgNodeListRpo rpo(unit_cfg);
	for (CfgNodeHandle h =  rpo.start(); h != rpo.end(); ++h)
	{
	    block = *h;
	    unsigned block_num = get_number(block);

	    NatSet *cur_live_in;
	    NatSet *cur_live_out;

	    private_live_in[block_num] = new NatSetSparse;
	    cur_live_in = private_live_in[block_num];

	    if (cfg_renumbered || keep_live_out_info)
	    {
		private_live_out[block_num] = new NatSetSparse;
		cur_live_out = private_live_out[block_num];
	    }

	    if (block != get_exit_node(unit_cfg))
	    {
		killed[block_num] = new NatSetSparse;
		NatSet *cur_killed = killed[block_num];

		if (cfg_renumbered)
		{
		    List<PhiNode*> block_phi_nodes = phi_nodes(block);
		    for (PhiNodeHandle h = block_phi_nodes.begin();
			 h != block_phi_nodes.end(); ++h)
		    {
			PhiNode *phi_node = *h;
			Opnd opnd = phi_node->get_new_name();
			int value_id = lookup_new_opnd(opnd);
			cur_killed->insert(value_id);
		    }
		}

		InstrHandle list_iterator = instrs_start(block);
		for (int j = 0; j < instrs_size(block); ++j, ++list_iterator)
		{
		    Instr *instr = *list_iterator;

		    AddToLiveIn add_to_live_in(this, cur_killed, cur_live_in);
		    map_src_opnds(instr, add_to_live_in);

		    AddToCurKilled add_to_cur_killed(this, cur_killed);
		    map_dst_opnds(instr, add_to_cur_killed);
		}
	    }
	}
	if (cfg_renumbered)
	{
	    phi_node_uses = new (NatSet*)[block_count + 1];

	    CfgNode *block;
	    CfgNodeListRpo rpo(unit_cfg);
	    for (CfgNodeHandle h = rpo.start(); h != rpo.end(); ++h)
	    {
                block = *h;

                unsigned block_num = get_number(block);
                phi_node_uses[block_num] = new NatSetSparse;

		for (PhiSrcMapEntry *pair = phi_src_map[block_num];
		     pair; pair = pair->next)
		{
                    unsigned phi_src_index = pair->index;
                    CfgNode *succ_block = pair->succ_block;

		    List<PhiNode*> succ_block_phi_nodes = phi_nodes(succ_block);
		    for (PhiNodeHandle h = succ_block_phi_nodes.begin();
			 h != succ_block_phi_nodes.end(); ++h)
		    {
			Opnd src = (*h)->get_src(phi_src_index);
			int src_id = lookup_new_opnd(src);
			if (src_id >= 0)
			    phi_node_uses[block_num]->insert(src_id);
		    }
		}
	    }
	}

	bool changed;
	NatSet *temp = new NatSetSparse;

	do {
	    CfgNode *block;

	    changed = false;
	    CfgNodeListRpo rpo(unit_cfg);
	    for (CfgNodeHandle h = rpo.start(); h != rpo.end(); ++h)
	    {
		block = *h;

		if (block != get_exit_node(unit_cfg))
		{
		    unsigned block_num = get_number(block);
		    NatSet *current_live_in = private_live_in[block_num];
		    NatSet *current_live_out =
			(cfg_renumbered || keep_live_out_info)
			    ? private_live_out[block_num]
			    : NULL;

		    CfgNode *successor;
		    bool first_successor = true;

		    for (CfgNodeHandle sh = succs_start(block);
			 sh != succs_end(block); ++sh)
		    {
			successor = *sh;
			if (first_successor)
			{
			    // Originally copied pointers - trying this
			    *temp = *private_live_in[get_number(successor)];
			    first_successor = false;
			}
			else
			    *temp += *(private_live_in[get_number(successor)]);
		    }

		    if (cfg_renumbered)
			*temp += *(phi_node_uses[get_number(block)]);

		    if ((cfg_renumbered || keep_live_out_info) &&
			(current_live_out != temp))
			current_live_out = temp;

		    NatSetSparse *killed_complement =
			new NatSetSparse(*(killed[block_num]));
		    killed_complement->complement();
		    *temp *= *killed_complement;
		    *temp += *current_live_in;

		    if (*current_live_in != *temp)
		    {
			changed = true;
			*current_live_in = *temp;
		    }
		}
	    }
	} while (changed);

	if (keep_live_in_info)
	    record_liveness_info(private_live_in, false);
	if (keep_live_out_info)
	    record_liveness_info(private_live_out, true);

	if (!keep_live_out_info)
	{
	    // May want to add some other deletes here now that arena
	    // delete is gone
	}
    }
    else
        private_live_out = NULL; /* we rely on this value to be NULL in the
                                    case that we don't have to build the
                                    liveness analysis */
    pushed = new NatSetSparse;
    dst_list = new NatSetSparse;
    worklist = new NatSetSparse;

    insert_copies(get_entry_node(unit_cfg), copy_stats);

    replaced_phi_nodes = true;
} // replace_phi_nodes


/*
 * Restore CFG to conventional (non-SSA) form.
 *
 * As explained in BHS, original names cannot be restored in general if
 * copy-folding is turned on, or if the client has performed some other
 * optimization that can cause the live ranges of different SSA names for
 * the same original name to overlap.  The caller indicates that such an
 * optimization has been performed by binding argument `not_old_names' to
 * true.
 */
void
SsaForm::restore(bool not_old_names)
{
    if (!cfg_renumbered)
	return;

    not_old_names |= fold_copies;

    if (not_old_names && !replaced_phi_nodes)
	replace_phi_nodes();

    CfgNodeListRpo rpo(unit_cfg);
    for (CfgNodeHandle nh = rpo.start(); nh != rpo.end(); ++nh)
    {
	CfgNode *block = *nh;
	
	for (InstrHandle ih = instrs_start(block);
	     ih != instrs_end(block); ++ih)
        {
	    Instr *instr = *ih;
	    if (not_old_names)
            {
		// Original names aren't being restored, so we need to be
		// sure that the value of each formal parameter that has a
		// SSA name is transferred to that SSA variable at proc
		// entry.  In code that has been emitted by the gen pass,
		// this is not an issue, since formals are always moved
		// explicitly from their argument-passing locations to
		// local "symbolic registers".  But for SUIFvm code, we
		// must insert a copy for each each formal.  A cheesy way
		// to recognize the SUIFvm target is to check for an empty
		// register file.

		if (has_note(instr, k_proc_entry) && reg_allocatables()->is_empty())
		    for (int i = get_formal_param_count(unit) - 1; i >= 0; --i)
		    {
			Opnd formal = opnd_var(get_formal_param(unit, i));
			int id = formal_value_ids[i];
			if (id < 0)
			    continue;
			Opnd value  = new_opnd_table[id];
			int opcode = opcode_move(get_type(value));
			Instr *copy = new_instr_alm(value, opcode, formal);

			if (debuglvl > 0)
			    set_comment(copy, "Copy formal to SSA variable");

			insert_after(block, ih, copy);
		    }
	    }
	    else
	    {
		RestoreRegisters opnd_function(this);
		map_opnds(instr, opnd_function);
	    }
        }
    }
    cfg_renumbered = false;

    if (debuglvl >= 4)
    {
	fprintf(stderr, "\nJust after restoring from SSA form:\n");
	this->print(stderr);
	fprintf(stderr, "\nEnd of just-after-restore report\n" );
    }
} // restore

void
SsaForm::print(FILE *fp)
{
    for (CfgNodeHandle h = start(unit_cfg); h != end(unit_cfg); ++h) {
	putc('\n', fp);
	print_phi_nodes(fp, *h);
	fprint(fp, *h, true);	// show code
    }


    void print_def_use_chains(FILE*);
    void print_use_def_chains(FILE*);

}

void
SsaForm::print_phi_nodes(FILE *fp, CfgNode *block)
{
    const char *header = "phi nodes:\n";
    List<PhiNode*> block_phi_nodes = phi_nodes(block);
    for (PhiNodeHandle h = block_phi_nodes.begin();
	 h != block_phi_nodes.end(); ++h)
    {
	PhiNode *phi_node = *h;
	fprintf(fp, header);
	header = "";
	fprintf(fp, "[%lx]", (unsigned long)phi_node);
        fprintf(fp, "\tOrig: ");
	fprint (fp, phi_node->get_old_name());
        fprintf(fp, "\tSSA: ");
	fprint (fp, phi_node->get_new_name());

        fprintf(fp, "\tSources: [ ");
	const char *sep = "";
	for (int i = 0; i < phi_node->srcs_size(); ++i) {
	    fprintf(fp, sep);
	    sep = "\t";
	    fprint (fp, phi_node->get_src(i));
	}
        fprintf(fp, " ]\n");
    }
}

int
SsaForm::loc_count() const
{
    return _loc_count;
}

int
SsaForm::value_count() const
{
    return def_count;
}

OpndCatalog*
SsaForm::value_catalog() const
{
    return new_opnd_catalog;
}

const Vector<Opnd>&
SsaForm::value_table() const
{
    return new_opnd_table;
}

Operation
SsaForm::unique_def(Opnd opnd) const
{
    if (build_use_def_chains)
    {
	int new_opnd_id = lookup_new_opnd(opnd);
	if (new_opnd_id >= 0)
	    return use_def_chains[new_opnd_id];
    }
    return Operation();
}

const List<Occurrence>&
SsaForm::use_chain(Opnd opnd) const
{
    static List<Occurrence> none;

    if (build_def_use_chains)
    {
	int new_opnd_id = lookup_new_opnd(opnd);
	if (new_opnd_id >= 0)
	    return def_use_chains[new_opnd_id];
    }
    return none;
}

CfgNode*
SsaForm::phi_node_block(PhiNode *phi_node) const
{
    Opnd phi_node_name = phi_node->get_new_name();
    unsigned phi_node_id = lookup_new_opnd(phi_node_name);

    claim(phi_node_id >= 0, "Unregistered phi-node definee");

    return block_map[phi_node_id];
}

List<PhiNode*>&
SsaForm::phi_nodes(CfgNode *block)
{
    return block_phi_nodes[get_number(block)];
}

/*
 * record_use_swap
 *
 * Reflect the replacement of a use occurrence (out) by another one (in).
 *
 * The actual change is performed by the caller.  The record_use_swap
 * methods just update the def-use chains, if those are being maintained.
 *
 * NB: If the name being swapped in is a newly-created SSA name, its
 * definition must already have been processed by record_def before
 * record_use_swap is called.
 */
void
SsaForm::record_use_swap(Opnd out, Opnd in, InstrHandle ih)
{
    if (build_def_use_chains)
	record_use_swap_either(out, in, Occurrence(ih, get_parent_node(*ih)));
}

void
SsaForm::record_use_swap(Opnd out, Opnd in, PhiNode *phi, int src_pos)
{
    if (build_def_use_chains)
	record_use_swap_either(out, in,
			       Occurrence(phi, phi_node_block(phi), src_pos));
}

void
SsaForm::record_use_swap_either(Opnd out, Opnd in, Occurrence occurrence)
{
    int out_id = lookup_new_opnd(out);
    if (out_id >= 0)
    {
	List<Occurrence> &chain = def_use_chains[out_id];
	for (UseHandle h = chain.begin(); h != chain.end(); /* */)
	{
	    UseHandle this_h = h++;
	    if ((*this_h).get_instr_handle() == occurrence.get_instr_handle() ||
		(*this_h).get_phi_node()     == occurrence.get_phi_node())
		chain.erase(this_h);
	}
    }
    int in_id = lookup_new_opnd(in);
    if (in_id >= 0)
	def_use_chains[in_id].push_back(occurrence);
}

/*
 * record_all_uses
 *
 * Take note of all the use occurrences within a newly planted instruction
 * or phi-node.
 *
 * Assume that any SSA names among these operands have already been
 * enrolled, e.g., by calling record_def on their defining instructions.
 * Update the def-use chains of any SSA names found among the input
 * operands.
 */

void
SsaForm::record_all_uses(InstrHandle ih)
{
    if (build_def_use_chains)
    {
	RecordUses record_uses(this, ih);
	map_src_opnds(*ih, record_uses);
    }
}

void
SsaForm::record_all_uses(PhiNode *phi)
{
    if (build_def_use_chains)
    {
	RecordUses record_uses(this, phi);
	map_opnds(phi, record_uses);
    }
}

/*
 * record_def
 *
 * Reflect the introduction of a new definition, either in the form of an
 * instruction or a phi-node.  In either case:
 *
 *   The new "name" is already in place in the defining construct.  Both
 *   the old name and new name may need to be recorded in internal counters
 *   and tables.
 *
 * When the defining construct is an instruction:
 *
 *   The old name is passed as an argument, as is the position of the new
 *   definee in the destination vector of the instruction.  The instruction
 *   is entered in the instr_map slot for the new name.
 *
 * When the defining construct is a phi-node:
 *
 *   The old name and new name are both found in the phi-node, which must
 *   be entered by the caller in phi_nodes(block), where block is the CFG
 *   node to which the phi-node is being attached.  That block is also
 *   passed as an argument.
 *
 */
void
SsaForm::record_def(Opnd old_name, InstrHandle ih, int dst_pos)
{
    Opnd new_name = get_dst(*ih, dst_pos);
    CfgNode *block = get_parent_node(*ih);
    int new_id = record_def_either(old_name, new_name, Operation(ih), block);
    instr_map[new_id] = *ih;    
}

void
SsaForm::record_def(PhiNode *phi, CfgNode *block)
{
    Opnd old_name = phi->get_old_name();
    Opnd new_name = phi->get_new_name();
    record_def_either(old_name, new_name, Operation(phi), block);
}

int
SsaForm::record_def_either(Opnd old_name, Opnd new_name, Operation operation,
			   CfgNode *block)
{
    int old_id = enroll_old_opnd(old_name);
    if (old_id >= (signed)_loc_count)
    {
	_loc_count = old_id + 1;
	old_opnd_table.resize(_loc_count, NULL);
    }
    old_opnd_table[old_id] = old_name;

    int new_id = enroll_new_opnd(new_name);
    claim(new_id >= 0, "Definee should be an acceptable SSA name");

    if (new_id >= (signed)def_count)
    {
	def_count = new_id + 1;
	new_opnd_table.resize(def_count, NULL);
	name_map .resize(def_count, NULL);
	block_map.resize(def_count, NULL);
	instr_map.resize(def_count, NULL);
	if (build_use_def_chains)
	    use_def_chains.resize(def_count, Operation());
	if (build_def_use_chains)
	    def_use_chains.resize(def_count, List<Occurrence>());
    }
    new_opnd_table[new_id] = new_name;
    name_map [new_id] = old_name;
    block_map[new_id] = block;

    if (build_use_def_chains)
	use_def_chains[new_id] = operation;

    return new_id;
}

// FIXME: Why does the following print all of the sources of phi-node, when each
// represents just one of those?

void
SsaForm::print_def_use_chains(FILE *fp)
{
    unsigned i;

    fprintf(fp, "\n\n PRINTING DEF-USE CHAINS\n\n");
    for (i = 0; i < def_count; ++i)
    {
        List<Occurrence> chain = def_use_chains[i];

        fprintf(fp, "*/*/*/*/*/*/*/*/*/*/*/*/*/*/*/*/*/*/*/*/*/*/*/*/*/*/*\n");
        fprintf(fp, "LOC NAME: %d\n", i);

	for (UseHandle h = chain.begin(); h != chain.end(); ++h)
        {
	    Occurrence use = *h;

            fprintf(fp, "----------\n");
            fprintf(fp, "\tIs_Phi_Node: ");
            if (use.is_phi_node())
            {
                int j;
                PhiNode *phi_node = use.get_phi_node();

                fprintf(fp, "true\n");
                fprintf(fp, "\tPhi Node Sources(old_name):\n\t\t");
                for (j = 0; j < phi_node->srcs_size(); ++j)
                {
                    unsigned src = lookup_new_opnd(phi_node->get_src(j));

                    fprintf(fp, "%u(%lx)   ", src,
			    (bool)(src)
				? (unsigned long) &name_map[src]
				: (unsigned long)0);
                }
                fprintf(fp, "\n");
		fprintf(fp, "\tPosition: %d\n", use.get_phi_src_pos());
            }
            else
            {
                fprintf(fp, "false\n");
                fprintf(fp, "\tOpcode = %d\n",
			get_opcode(*use.get_instr_handle()));
            }
        }
        fprintf(fp, "*/*/*/*/*/*/*/*/*/*/*/*/*/*/*/*/*/*/*/*/*/*/*/*/*/*/*\n");
    }
    fprintf(fp, "\n\n***DONE*** PRINTING DEF-USE CHAINS\n\n");
} // print_def_use_chains

void
SsaForm::print_use_def_chains(FILE *fp)
{
    unsigned i;

    fprintf(fp, "\n\nPRINTING USE-DEF CHAINS\n\n");
    for (i = 1; i < def_count; i++)
    {
        Operation def = use_def_chains[i];

        fprintf(fp, "--------------------------------------------\n");
        fprintf(fp, "Loc Name: %d", i);
        fprintf(fp, "\tOriginal Name: %d\n", (int) &name_map[i]);
        fprintf(fp, "\tIs_Phi_Node: ");

        if (def.is_phi_node())
        {
            int j;
            PhiNode *phi_node = def.get_phi_node();

            fprintf(fp, "true\n");
            fprintf(fp, "\tPhi Node Sources(old_name):\n\t\t");
            for (j = 0; j < phi_node->srcs_size(); ++j)
            {
                unsigned src = lookup_new_opnd(phi_node->get_src(j));

                fprintf(fp, "%u(%lx)   ", src,
			(bool)(src) ? (unsigned long) &name_map[src]
				    : (unsigned long)0);
            }
            fprintf(fp, "\n");
        }
        else
        {
            fprintf(fp, "false\n");
            fprintf(fp, "\tOpcode = %d\n", get_opcode(*def.get_instr_handle()));
        }

        fprintf(fp, "--------------------------------------------\n");
    }
    fprintf(fp, "\n\n***DONE*** PRINTING USE-DEF CHAINS\n\n");
} // print_use_def_chains

int
SsaForm::enroll_old_opnd(Opnd opnd)
{
    int opnd_id = -1;
    old_opnd_catalog->enroll(opnd, &opnd_id);
    if (opnd_id >= 0)
    {
	claim((unsigned)opnd_id < old_opnd_table.size());
	old_opnd_table[opnd_id] = opnd;
    }
    return opnd_id;
}

int
SsaForm::lookup_old_opnd(Opnd opnd) const
{
    int opnd_id = -1;
    old_opnd_catalog->lookup(opnd, &opnd_id);
    return opnd_id;
}

int
SsaForm::enroll_new_opnd(Opnd opnd)
{
    int opnd_id = -1;
    new_opnd_catalog->enroll(opnd, &opnd_id);
    if (opnd_id >= 0)
    {
      //claim((unsigned)opnd_id < new_opnd_table.size());
	new_opnd_table[opnd_id] = opnd;
    }
    return opnd_id;
}

int
SsaForm::lookup_new_opnd(Opnd opnd) const
{
    int opnd_id = -1;
    new_opnd_catalog->lookup(opnd, &opnd_id);
    return opnd_id;
}

void
SsaForm::record_liveness_info(NatSet **sets, bool live_out_info)
{
    unsigned i, j;
    LivenessInfo *temp = new LivenessInfo[block_count + 1];
    for (int k = 0; (unsigned)k < block_count + 1; ++k)
    {
        temp[k].size = 0;
        temp[k].names = NULL;
    }

    // Does this need to change to a block iterator?
    for (i = 1; i <= block_count; ++i)
    {
        if (sets[i])
        {
            unsigned size = sets[i]->size();

            temp[i].names = new (unsigned)[size];
            NatSetIter set_iterator = sets[i]->iter();
            for (/* */; set_iterator.is_valid(); set_iterator.next())
	    {
                j = set_iterator.current();
                temp[i].names[temp[i].size++] = j;
	    }
        }
    }

    if (live_out_info)
        live_out = temp;
    else
        live_in = temp;

} // record_liveness_info

/*
 * Generate new names for phi nodes.
 */
void
SsaForm::rename_locs(CfgNode *block, NameStackEntry **name_stacks)
{
    int pushed_list = -1;

    pushed->remove_all();

    List<PhiNode*> block_phi_nodes = phi_nodes(block);
    for (PhiNodeHandle h = block_phi_nodes.begin();
	 h != block_phi_nodes.end(); ++h)
    {
	PhiNode *phi_node = *h;
	Opnd old_name = phi_node->get_old_name();
	Opnd new_name = opnd_reg(get_type(old_name));
	phi_node->set_new_name(new_name);

	unsigned old_name_id = lookup_old_opnd(old_name);
	unsigned new_name_id = enroll_new_opnd(new_name);
	claim(old_name_id < _loc_count &&
	      new_name_id < def_count);

	name_map[new_name_id] = old_name;
	block_map[new_name_id] = block;
	push_name(&name_stacks[old_name_id], pushed, &pushed_list,
		  old_name_id, new_name);
	pushed->insert(old_name_id);

	if (build_use_def_chains)
	    use_def_chains[new_name_id] = Operation(phi_node);
    }

    if (keep_live_in_info & (block != unit_cfg->get_entry_node() &&
			     block != unit_cfg->get_exit_node()))
        rewrite_liveness_info(&live_in[block->get_number()], name_stacks);

    InstrHandle h = instrs_start(block);
    for (int j = size(block); j > 0; --j)
    {
	InstrHandle handle = h++;
	Instr *instr = *handle;
	ReplaceUsedLocs opnd_function(this, name_stacks, handle, block);
	map_opnds(instr, opnd_function);

	if (fold_copies && is_move(instr))
	{
	    Opnd src = get_src(instr, 0);
	    Opnd dst = get_dst(instr, 0);
	    int dst_id = lookup_old_opnd(dst);

	    // Can only fold a copy if both opnds are candidates for
	    // renaming as SSA values.  Dst opnd is one of these if it has
	    // an id.

	    if (dst_id >= 0 && is_ssa_candidate(src))
	    {
		push_name(&name_stacks[dst_id], pushed, &pushed_list,
			  dst_id, src);
		delete remove(block, handle);
	    }
	    else
		goto not_foldable;
	}
	else
      not_foldable:
	{
	    RenameDefinedLocs
		rename_defined_locs(this, handle, block,
				    name_stacks, &pushed_list);
	    if (!has_note(instr, k_proc_entry))
	    {
		map_dst_opnds(instr, rename_defined_locs);
	    }
	    else
	    {
		map_formal_defs(unit, rename_defined_locs);
		int count = get_formal_param_count(unit);
		formal_value_ids.resize(count);
		for (int i = 0; i < count; ++i)
		{
		    Opnd formal = opnd_var(get_formal_param(unit, i));
		    int id = lookup_old_opnd(formal);
		    if (id >= 0)
			id = lookup_new_opnd(name_stacks[id]->new_name);
		    formal_value_ids[i] = id;
		}
	    }
	}
    }

    // Fill in the phi nodes of block's successors.  A block can appear
    // more than once in block's successor list, but must only be processed
    // once.

    static BitVector done;
    done.set_to_zero();

    CfgNodeHandle sh = succs_start(block);    
    for (int k = succs_size(block); k > 0; --k, ++sh)
    {
	CfgNode *succ_block = *sh;
	unsigned succ_block_num = get_number(succ_block);
	if (done.get_bit(succ_block_num))
	    continue;
	done.set_bit(succ_block_num, true);

	unsigned phi_srcs_index = phi_srcs_indices[succ_block_num]++;
	List<PhiNode*> succ_phi_nodes = phi_nodes(succ_block);

	for (PhiNodeHandle h = succ_phi_nodes.begin();
	     h != succ_phi_nodes.end(); ++h)
	{
	    PhiNode *phi_node = *h;

	    int old_name_id = lookup_old_opnd(phi_node->get_old_name());
	    claim(old_name_id >= 0);

	    NameStackEntry *top_of_stack = name_stacks[old_name_id];

	    if (top_of_stack)
	    {
		Opnd new_name = top_of_stack->new_name;	// avoids derefs

		phi_node->set_src(phi_srcs_index, new_name);
		if (build_def_use_chains)
		{
		    unsigned new_name_id = lookup_new_opnd(new_name);
		    claim(new_name_id >= 0);

		    Occurrence occurrence(phi_node, succ_block, phi_srcs_index);
		    def_use_chains[new_name_id].push_back(occurrence);
		}
	    }
	}

	if (!phi_nodes(succ_block).empty())
	{
	    PhiSrcMapEntry *new_node = new PhiSrcMapEntry;

	    new_node->index = phi_srcs_index;
	    new_node->succ_block = succ_block;
	    new_node->next = phi_src_map[get_number(block)];
	    phi_src_map[get_number(block)] = new_node;
	}
	edge_maps[succ_block_num][phi_srcs_index] = block;
    }

    for (DominanceChild *child = dominator_children[get_number(block)];
	 child != NULL;
	 child = child->next)
	rename_locs(get_node(unit_cfg, child->block_number), name_stacks);

    while (pushed_list >= 0)
    {
	int next_node = name_stacks[pushed_list]->next_pushed;

	NameStackEntry *temp_ptr = name_stacks[pushed_list];
	name_stacks[pushed_list] = name_stacks[pushed_list]->next;
	temp_ptr->next = NULL;
	delete temp_ptr;
	pushed_list = next_node;
    }
} // rename_locs

void
SsaForm::rewrite_liveness_info(LivenessInfo *set, NameStackEntry **name_stacks)
{
    unsigned i, size = set->size;

    set->size = 0;
    for (i = 0; i < size; i++)
    {
        claim(name_stacks[set->names[i]]);
	set->names[set->size++] =
	    lookup_new_opnd(name_stacks[set->names[i]]->new_name);
    }
} // rewrite_liveness_info

void
SsaForm::push_name(NameStackEntry **stack,
		   NatSet *pushed, int *node_list_pointer,
		   unsigned old_name, Opnd new_name)
{
    if (pushed->contains(old_name))
        (*stack)->new_name = new_name;
    else
    {
	NameStackEntry *new_node = new NameStackEntry;

        new_node->new_name = new_name;
        new_node->next_pushed = *node_list_pointer;
        *node_list_pointer = old_name;
        new_node->next = *stack;
        *stack = new_node;
        pushed->insert(old_name);
    }
} // push_name

int
SsaForm::find_root(unsigned value_id, unsigned *loc_names)
{
    if (loc_names[value_id] != value_id)
        loc_names[value_id] =
	    find_root(loc_names[value_id], loc_names);
    return loc_names[value_id];
} // find_root

int
SsaForm::phi_src_compare(PhiNode *phi_node, unsigned *loc_names)
{
    int replacement_id = -1;
    int phi_node_id = lookup_new_opnd(phi_node->get_new_name());

    for (int i = 0; i < phi_node->srcs_size(); ++i)
    {
        Opnd phi_src = phi_node->get_src(i);
        int phi_src_id = lookup_new_opnd(phi_src);

	if (phi_src_id >= 0)
	{
	    phi_src_id = find_root(phi_src_id, loc_names);
	    if (phi_src_id != phi_node_id)
	    {
		if (replacement_id == -1)
		    replacement_id = phi_src_id;
		else if (replacement_id != phi_src_id)
		    return -1;
	    }
	}
    }
    claim(replacement_id >= 0);
    return replacement_id;
} // phi_src_compare

/*
 * Return true unless the CFG has a critical back edge.  A back edge is one
 * whose head dominates its tail.  A critical edge is one whose tail has
 * multiple successors and whose head has multiple predecessors.  We know
 * that the head block of a back edge has other predecessors, so we only
 * check the tail block's successor count once we find a back edge.  Note
 * that we must not count twice if a tail block has two edges to the same
 * successor.
 */
bool
SsaForm::cfg_edges_split()
{
    CfgNodeListRpo rpo(unit_cfg);
    for (CfgNodeHandle h = rpo.start(); h != rpo.end(); ++h)
    {
	CfgNode *block = *h;
	bool back_edge_found = false;
	int succ_count = 0;
	static BitVector done;
	done.set_to_zero();

        CfgNodeHandle sh = succs_start(block);
        for (int k = succs_size(block); k > 0; --k, ++sh)
        {
	    CfgNode *succ = *sh;
	    int succ_number = get_number(succ);
	    if (done.get_bit(succ_number))
		continue;
	    done.set_bit(succ_number, true);
	    succ_count++;

	    // Rice says: "This only detects back edges for natural loops.
	    // We should switch to Tarjan's method."
	    if (dominator_analysis->dominates(succ, block))
		back_edge_found = true;

	    if (back_edge_found && succ_count > 1)
		return false;
        }
    }
    return true;
} // cfg_edges_split

/*
 * Recursive helper of replace_phi_nodes.  Described in BHS section
 * <<Inserting the copies>>.
 */
void
SsaForm::insert_copies(CfgNode *block, CopyData *copy_stats)
{
    int pushed_list = -1;

    pushed->remove_all();
    dst_list->remove_all();

    // <<Walk through the instructions in the block, replacing uses with
    // new names>>.
    // Start with the uses occurring in the phi-nodes.

    List<PhiNode*> block_phi_nodes = phi_nodes(block);
    for (PhiNodeHandle h = block_phi_nodes.begin();
	 h != block_phi_nodes.end(); ++h)
    {
	PhiNode *phi_node = *h;
	claim(phi_node->srcs_size() == preds_size(block));

	for (int i = 0; i < phi_node->srcs_size(); ++i)
	{
	    Opnd phi_src = phi_node->get_src(i);
	    int src_id = lookup_new_opnd(phi_src);
	    if (src_id >= 0 && copy_stats[src_id].new_name)
		phi_node->set_src(i, copy_stats[src_id].new_name->new_name);
	}
    }
    // Now visit each phi node again.  If its SSA name is live out of this
    // block, BHS pushes that name onto its own name stack, ostensibly to
    // avoid damaging a copy-to-temporary that has already been inserted
    // during the processing of another block.

    // I claim that no such copy-to-temp can exist for a phi-node name that
    // is live out of the current block.  Such a phi-node has to be at the
    // head of a back edge from any block where its name is live out, and
    // hence it dominates such a block, and will have been processed before
    // it.
    //
    //  I therefore simply check the name stack of the live-out SSA name to
    //  be sure that it's empty.

    block_phi_nodes = phi_nodes(block);
    for (PhiNodeHandle h = block_phi_nodes.begin();
	 h != block_phi_nodes.end(); ++h)
    {
	Opnd name = (*h)->get_new_name();
	int name_id = lookup_new_opnd(name);

	if (private_live_out &&
	    private_live_out[get_number(block)]->contains(name_id))
#ifdef BHS_PUSH_LIVE_NAME
	    push_name(&(copy_stats[name_id].new_name),
		      pushed, &pushed_list, name_id, name);
#else
	    claim(copy_stats[name_id].new_name == NULL,
		  "Expected empty SSA name stack");
#endif
    }
    // Now replace uses with new names in the block's instructions.

    InstrHandle handle = instrs_start(block);
    for (int j = instrs_size(block); j > 0; --j, ++handle)
    {
	ReplaceSourceNames opnd_function(this, copy_stats);
	map_opnds(*handle, opnd_function);
    }

    // <<Insert copies into [[block]] from all of its successors>>
    //   <<Phi-node replacement pass 1: initialize the data structures>>

    PhiSrcMapEntry *next = phi_src_map[get_number(block)];
    while (next)
    {
	unsigned index = next->index;
	CfgNode *succ_block = next->succ_block;

	List<PhiNode*> succ_block_phi_nodes = phi_nodes(succ_block);
	for (PhiNodeHandle h = succ_block_phi_nodes.begin();
	     h != succ_block_phi_nodes.end(); ++h)
	{
	    PhiNode *phi_node = *h;
	    Opnd src_name = phi_node->get_src(index);
	    Opnd dst_name = phi_node->get_new_name();
	    int src_id = lookup_new_opnd(src_name);
	    int dst_id = lookup_new_opnd(dst_name);

	    if (src_id >= 0 && copy_stats[src_id].new_name) {
		src_name = copy_stats[src_id].new_name->new_name;
		src_id = lookup_new_opnd(src_name);
	    }
	    if (src_id >= 0 && dst_id != src_id)
	    {
		claim(dst_id >= 0);

		// <<Add the statistics for this copy into the [[copy_stats]]
		// array>>
		copy_stats[dst_id].src = src_name;
		copy_stats[dst_id].is_live_out =
		    (private_live_out)
			? private_live_out[get_number(block)]->
		    	    contains(dst_id)
			: false;
		copy_stats[dst_id].src_name_to_use = dst_name;
		copy_stats[src_id].src_name_to_use = src_name;
		copy_stats[src_id].used_by_count++;
		dst_list->insert(dst_id);
	    }
	}
	next = next->next;
    }

    // <<Phi-node replacement pass 2: set up the worklist for inserting
    // copies for phi nodes>>
    
    for (NatSetIter dst_iter = dst_list->iter();
	 dst_iter.is_valid(); dst_iter.next())
    {
	unsigned dst = dst_iter.current();
	if (copy_stats[dst].used_by_count == 0)
	    worklist->insert(dst);
    }

    // <<Phi-node replacement pass 3: iterate over the worklist,
    // inserting copies>>

    unsigned dst_list_size;
    do {
	while (worklist->size() > 0)
	{
	    // <<Choose a copy off of the worklist, insert it, and remove
	    // it from [[destination_list]]>>

	    unsigned dst_id = worklist->iter().current();
	    Opnd dst_name = new_opnd_table[dst_id];
	    Opnd src_name = copy_stats[dst_id].src;
	    int src_id = lookup_new_opnd(src_name);

	    worklist->remove(dst_id);
	    if (copy_stats[dst_id].is_live_out
		&& is_null(copy_stats[dst_id].temp_name))
	    {
		copy_stats[dst_id].is_live_out = false;

		// <<Insert a copy to a temporary to store the old value>>
		// Caution: we obscure the enclosing bindings of variables
		// {src,dst}_{name,id}.  The ones here are for the copy-to-
		// temporary instruction.

		Opnd src_name = dst_name;
		unsigned src_id = dst_id;
		Opnd dst_name = opnd_reg(get_type(src_name));
		int dst_id = enroll_new_opnd(dst_name);
		
		copy_stats[src_id].used_by_count = 0;

		// <<Insert a copy for the temp>>
		// Insert this copy "at the phi node", i.e., at the start
		// of the block.

		int opcode = opcode_move(get_type(dst_name));
		Opnd src_name_to_use = copy_stats[src_id].src_name_to_use;
		Instr *new_instr =
		    new_instr_alm(dst_name, opcode, src_name_to_use);

		if (debuglvl > 0)
		    set_comment(new_instr,
				"Copy to temp at phi node to save old value");

		// <<Insert [[new_instruction]] into [[block]]>>
		InstrHandle new_handle = prepend(block_map[src_id], new_instr);

		name_map [dst_id] = name_map[src_id];
		block_map[dst_id] = block;
		instr_map[dst_id] = new_instr;

		// <<Add the use-def chain, if necessary>>
		if (build_use_def_chains)
		    use_def_chains[dst_id] = Operation(new_handle);

		copy_stats[src_id].temp_name = dst_name;
		copy_stats[src_id].src_name_to_use = dst_name;
		push_name(&(copy_stats[src_id].new_name),
			  pushed, &pushed_list, src_id, dst_name);
	    }
	    Instr *new_instr =
		new_instr_alm(dst_name,
			      opcode_move(get_type(dst_name)),
			      copy_stats[src_id].src_name_to_use);

	    if (debuglvl > 0)
	    {
		Opnd old_name = name_map[dst_id];
		char buf[BUFSIZ];
		if (is_reg(old_name))
		    sprintf(buf, "$vr%d", get_reg(old_name));
		else
		    sprintf(buf, "`%s'", get_name(get_var(old_name)).chars());
		String comment = String("Copy to phi-defined name for ") + buf;
		set_comment(new_instr, comment);
	    }

	    // <<Insert [[new_instruction]] into [[block]]>>
	    // Put it at the end of the block, but before a CTI if there is one.

	    insert_before(block, get_cti_handle(block), new_instr);

	    copy_stats[src_id].used_by_count = 0;
	    if (dst_list->contains(src_id) &&
		copy_stats[src_id].is_live_out)
		push_name(&(copy_stats[src_id].new_name),
			  pushed, &pushed_list, src_id, dst_name);
	    copy_stats[src_id].is_live_out = false;
	    copy_stats[src_id].src_name_to_use = dst_name;

	    if (dst_list->contains(src_id))
		worklist->insert(src_id);
	    dst_list->remove(dst_id);
	}

	dst_list_size = dst_list->size();
	if (dst_list_size)
	{
	    // <<Insert a copy to a temporary to break the cycle>>
	    unsigned src_id = dst_list->iter().current();
	    Opnd src_name = new_opnd_table[src_id];
	    Opnd dst_name;
	    unsigned dst_id;

	    if (is_null(copy_stats[src_id].temp_name))
	    {
		dst_name = opnd_reg(get_type(src_name));
		dst_id = enroll_new_opnd(dst_name);
		copy_stats[src_id].temp_name = dst_name;

		// <<Insert a copy for the temp>>
		Instr *new_instr =
		    new_instr_alm(dst_name,
				  opcode_move(get_type(dst_name)),
				  copy_stats[src_id].src_name_to_use);

    ListNote<IdString> note;						     //D
    String comment = "Copy to temp at block end to break cycle";	     //D
    note.append(comment);						     //D
    set_note(new_instr, k_comment, note);				     //D

		// <<Insert the copy into [[block]]>>.
		// A cycle-breaking copy should go at the end of [[block]].

		CfgNode *block = block_map[src_id];
		InstrHandle end = get_cti_handle(block);
		InstrHandle new_handle = insert_before(block, end, new_instr);

		name_map[dst_id] = name_map[src_id];
		block_map[dst_id] = block;
		instr_map[dst_id] = new_instr;
		if (build_use_def_chains)
		    use_def_chains[dst_id] = Operation(new_handle);
	    }
	    else
	    {
		dst_name = copy_stats[src_id].temp_name;
		dst_id = lookup_new_opnd(dst_name);
	    }

	    // Source's value will be obtained from the temp location from
	    // now on, so mark the source dead and zero its count of uses.

	    copy_stats[src_id].used_by_count = 0;
	    copy_stats[src_id].is_live_out = false;
	    copy_stats[src_id].src_name_to_use = dst_name;
	    push_name(&(copy_stats[src_id].new_name),
		      pushed, &pushed_list, src_id, dst_name);
	    worklist->insert(src_id);
	}
    } while (dst_list_size > 0);

    // <<Recur over all of the block's children in the dominator tree>>

    for (DominanceChild *child = dominator_children[get_number(block)];
	 child != NULL;
	 child = child->next)
	insert_copies(get_node(unit_cfg, child->block_number), copy_stats);

    // <<Pop any names added during this run>>

    while (pushed_list >= 0)
    {
        int next_node = copy_stats[pushed_list].new_name->next_pushed;

        copy_stats[pushed_list].new_name =
	    copy_stats[pushed_list].new_name->next;
        pushed_list = next_node;
    }
} // insert_copies

void
SsaForm::set_comment(Instr *instr, IdString comment)
{
    ListNote<IdString> note;
    note.append(comment);
    set_note(instr, k_comment, note);
}



// ----------------------------	 class PhiNode  --------------------------------

PhiNode::PhiNode(int srcs_size)
{
    useless = false;
    nsrcs = srcs_size;
    srcs = new Opnd[srcs_size];

    for (int i = 0; i < srcs_size; i++)
	srcs[i] = opnd_null();
}

PhiNode::~PhiNode()
{
  delete [] srcs;
}

int
PhiNode::srcs_size() const
{
    return nsrcs;
}

Opnd
PhiNode::get_src(int index) const
{
    claim(index < nsrcs);
    return srcs[index];
}

void
PhiNode::set_src(int index, Opnd src)
{
    claim(index < nsrcs);
    srcs[index] = src;
}

Opnd
PhiNode::get_old_name() const
{
    return old_name;
}

void
PhiNode::set_old_name(Opnd name)
{
    old_name = name;
}

Opnd
PhiNode::get_new_name() const
{
    return new_name;
}

void
PhiNode::set_new_name(Opnd name)
{
    new_name = name;
}

bool
PhiNode::get_useless() const
{
    return useless;
}

void
PhiNode::set_useless(bool useless)
{
    this->useless = useless;
}

void
map_opnds(PhiNode *phi_node, OpndFilter &filter)
{
    int size = phi_node->srcs_size();
    for (int i = 0; i < size; ++i)
	filter(phi_node->get_src(i), OpndFilter::IN);
}

// -----------------------------------------------------------------------------


extern "C" void
init_ssa(SuifEnv *suif_env)
{
    static bool init_done = false;

    if (init_done)
        return;
    init_done = true;

    init_machine(suif_env);
    init_cfg(suif_env);
    init_cfa(suif_env);
    init_bvd(suif_env);
}
