(* Top Level *)

(*
 * Use in a SML read/eval/print loop
 * and for building stand-along executables
 *)

signature TOP =
sig
    val load : Ast.env -> string list -> Ast.env (* Top.load env [<file>,...] = env' *)

    val lambda : string -> OS.Process.status         (* Top.lambda "<command line arguments>" = status *)
    val main : string * string list -> OS.Process.status (* for stand-alone executable *)
end (* signature TOP *)

structure Top :> TOP =
struct

structure G = GetOpt  (* from $/smlnj-lib/Util/getopt-sig.sml *)
structure A = Ast
structure PP = Ast.Print

(************************)
(* Command Line Options *)
(************************)

datatype option =
         Lang of string
       | Abort of string
       | Subtyping of string
       | Warn of string
       | Verbose of int
       | Help of bool

(* printing error/success messages to stdErr *)
fun say s = TextIO.output (TextIO.stdErr, s ^ "\n")

val usage =
    if "sml" = #file (OS.Path.splitDirFile (CommandLine.name ()))
    then "Top.lambda \"<option>* <file>*\";"
    else "lambda <option>* <file>*"
val header = Flags.version ^ "\n" ^ "Usage: " ^ usage ^ "\nwhere <option> is"
val options : option G.opt_descr list =
    [
     {short = "l", long = ["lang"],
      desc = G.ReqArg ((fn l => Lang(l)), "<language>"),
      help = "Language, one of 'lam', 'poly', 'cbv', or 'prf'; default to file name extension"},
     {short = "a", long = ["abort"],
      desc = G.ReqArg ((fn a => Abort(a)), "<reason>"),
      help = "Abort, one of 'error' (default) or 'warning'"},
     {short = "s", long = ["subtyping"],
      desc = G.ReqArg ((fn b => Subtyping(b)), "<true/false>"),
      help = "Use subtyping, one of 'false' (default) or 'true'"},
     {short = "w", long = ["warn"],
      desc = G.ReqArg ((fn b => Warn(b)), "<true/false>"),
      help = "Print warnings, one of 'true' (default) or 'false'"},
     {short = "v", long = ["verbose"],
      desc = G.NoArg (fn () => Verbose(2)),
      help = "Give verbose status messages"},
     {short = "q", long = ["quiet"],
      desc = G.NoArg (fn () => Verbose(0)),
      help = "Run quietly"},
     {short = "d", long = ["debug"],
      desc = G.NoArg (fn () => Verbose(3)),
      help = "Print some debugging information"},
     {short = "h", long = ["help"],
      desc = G.NoArg (fn () => Help(true)),
      help = "Give short usage message and exit"}
    ]

val usage_info = G.usageInfo {header = header, options = options}

exception OS_FAILURE
exception OS_SUCCESS

fun exit_failure msg = ( say msg ; raise OS_FAILURE )
fun exit_success msg = ( say msg ; raise OS_SUCCESS )

fun get_options (args) =
    G.getOpt {argOrder = G.RequireOrder,
              options = options,
              errFn = exit_failure}
             args

fun process_option (Lang(s)) =
    (case (Flags.parseLang s, !Flags.lang)
      of (NONE, _) => exit_failure ("language " ^ s ^ " not recognized")
       | (SOME(l), NONE) => Flags.lang := SOME(l)
       | (SOME(l), SOME(l')) => if l = l' then ()
                                else exit_failure ("cannot override language spec"))
  | process_option (Abort(s)) =
    (case Flags.parseAbort s
      of NONE => exit_failure ("abort reason " ^ s ^ " not recognized")
       | SOME(a) => Flags.abort := a)
  | process_option (Subtyping(s)) =
    (case Flags.parseSubtyping s
      of NONE => exit_failure ("subtyping flag " ^ s ^ " not recognized")
       | SOME(b) => Flags.subtyping := b)
  | process_option (Warn(s)) =
    (case Flags.parseSubtyping s
      of NONE => exit_failure ("warning flag " ^ s ^ " not recognized")
       | SOME(b) => Flags.warn := b)
  | process_option (Verbose(level)) = Flags.verbosity := level
  | process_option (Help(true)) = exit_success usage_info
  | process_option (Help(false)) = ()

fun check_compat_options () =
    (case (!Flags.lang, !Flags.subtyping)
      of (SOME(Flags.Prf), true) => exit_failure ("subtyping is incompatible with proof checking")
       | _ => ())

(*********************************)
(* Loading and Elaborating Files *)
(*********************************)

fun readable file = OS.FileSys.access (file, [OS.FileSys.A_READ])

fun reset () = (* do not reset flags, only internal state *)
    ( ParseState.reset ()
    ; ErrorMsg.reset ()
    )

fun select_language filename =
    case (!Flags.lang, Option.map Flags.parseLang (OS.Path.ext filename))
     of (SOME(l), _) => l         (* declared language takes precedence *)
      | (_, SOME(SOME(l))) => l   (* file name extension, .lam, .poly, .cbv, .prf *)
      | (NONE, _) => Flags.CBV    (* no or unrecognized extension *)

(* command line options applied before execution *)
fun apply_options line =
    let val args = String.tokens Char.isSpace line
        val (options, filenames) = get_options args (* may exit_failure(msgs) *)
        val () = List.app process_option options
        val () = case filenames
                  of nil => ()
                   | (_::_) => exit_failure ("spurious options: "
                                             ^ List.foldr (fn (arg,msg) => arg ^ " " ^ msg) "" filenames)
    in () end

(* pragmas indicated at the top of file applied before execution *)
fun apply_pragmas (A.Pragma("#options",line,_)::decs) =
    let val () = if !Flags.verbosity >= 1
                 then TextIO.print ("#options" ^ line ^ "\n")
                 else ()
        val () = apply_options line
    in apply_pragmas decs end
  | apply_pragmas (A.Pragma("#test",line,ext)::decs) =
    (* ignore #test pragma *)
    apply_pragmas decs
  | apply_pragmas (A.Pragma(pragma,line,ext)::decs) =
    ( ErrorMsg.error ext ("unrecognized pragma: " ^ pragma)
    ; raise ErrorMsg.Error )
  | apply_pragmas decs = decs

(* load a set of files *)
fun load env (file::filenames) =
    let val () = reset ()     (* internal lexer and parser state *)
        val () = Flags.lang := SOME(select_language file)
        val decs = Parse.parse file (* may raise ErrorMsg.Error *)
        val decs' = apply_pragmas decs (* remove pragmas; may raise ErrorMsg.Error *)
        val () = check_compat_options ()
    in (* do not allow for mutually recursive definitions *)
        case Elab.elab_decs env decs'
         of SOME(env') => load env' filenames
          | NONE => raise ErrorMsg.Error (* error during elaboration *)
    end
  | load env nil = env

fun exit_on_empty_files nil = exit_success Flags.version
  | exit_on_empty_files (_::_) = ()

(* main function to run file *)
fun test raise_exn args =
    (* reset flags *)
    let val () = Flags.reset()
        (* get and apply options *)
        val (options, filenames) = get_options args
        val () = List.app process_option options
        val () = exit_on_empty_files filenames
        (* parse and load file, i.e., generate an environment *)
        val env = load nil filenames
            handle ErrorMsg.Error => exit_failure "% parsing or type-checking failed"
                 | e => if raise_exn then raise e   (* for debugging purposes *)
                        else exit_failure "% internal error (uncaught exception)"
    in
        exit_success "% success"
    end handle OS_SUCCESS => OS.Process.success
             | OS_FAILURE => OS.Process.failure

fun lambda argstring = test true (String.tokens Char.isSpace argstring)
fun main (name, args) = test false args

end (* structure Top *)
