(* Flags *)
(* Author: Frank Pfenning <fp@cs.cmu.edu> *)

(* 
 * User option given on the command line or in the
 * #options pragma in the source file, plus the
 * version number and date
 *)

signature FLAGS =
sig
    (* language *)
    datatype lang = Lam | Poly | CBV | Prf
    val parseLang : string -> lang option
    val pp_lang : lang -> string

    val parseSubtyping : string -> bool option
    val pp_subtyping : bool -> string

    datatype abort = OnError | OnWarning
    val parseAbort : string -> abort option
    val pp_abort : abort -> string

    val abort : abort ref
    val subtyping : bool ref
    val warn : bool ref
    val lang : lang option ref
    val verbosity : int ref
    val help : bool ref

    (* reset all flags to their default value *)
    val reset : unit -> unit

    val version : string
end  (* signature FLAGS *)

structure Flags :> FLAGS =
struct

(* language *)
datatype lang = Lam | Poly | CBV | Prf

fun parseLang "lam" = SOME(Lam)
  | parseLang "poly" = SOME(Poly)
  | parseLang "cbv" = SOME(CBV)
  | parseLang "prf" = SOME(Prf)
  | parseLang _ = NONE

fun pp_lang Lam = "lam"
  | pp_lang Poly = "poly"
  | pp_lang CBV = "cbv"
  | pp_lang Prf = "prf"

fun parseSubtyping "true" = SOME(true)
  | parseSubtyping "false" = SOME(false)
  | parseSubtyping _ = NONE

fun pp_subtyping true = "true"
  | pp_subtyping false = "false"

datatype abort = OnError | OnWarning
fun parseAbort "error" = SOME(OnError)
  | parseAbort "warning" = SOME(OnWarning)
  | parseAbort _ = NONE
fun pp_abort (OnError) = "error"
  | pp_abort (OnWarning) = "warning"

val abort = ref (OnError)
val lang = ref (NONE : lang option) (* Lam = untyped, Poly = polymorphic, CBV = call-by-value *)
val subtyping = ref false       (* true = use subtyping, false = use type equivalence *)
val warn = ref true             (* true = produce warnings *)
val verbosity = ref 1           (* ~1 = print nothing, 0 = quiet, 1 = normal, 2 = verbose, 3 = debug *)
val help = ref false

fun reset () =
    ( lang := NONE              (* default: based on file extension *)
    ; abort := OnError          (* default: abort only on error, not warning *)
    ; subtyping := false        (* default: don't use subtyping *)
    ; warn := true
    ; verbosity := 1
    ; help := false )

val version = "Lambda v1.1 (Fri Nov 05 2021)"

end (* structure Flags *)
