(* Evaluation *)
(* Frank Pfenning *)

signature EVAL =
sig

    exception BoundExceeded
    exception NoMatch

    val eval : Ast.env -> int -> Ast.exp -> Ast.exp * int

    exception NotComparable of string
    val eq_val : Ast.env -> Ast.exp -> Ast.exp -> bool

end (* signature EVAL *)

structure Eval :> EVAL =
struct

structure A = Ast
structure PP = Ast.Print
structure TC = TypeCheck

exception BoundExceeded
exception NoMatch

local
    val gas : int ref = ref 0
in
fun set_gas n = ( gas := n )
fun get_gas () = !gas
fun tick () = if !gas = 0 then raise BoundExceeded
              else gas := !gas - 1
end

(* we ignore type-level expressions during evaluation *)

fun match v (A.VarPat(x)) theta = SOME((v,x)::theta)
  | match (A.Pair(v1,v2)) (A.PairPat(p1,p2)) theta =
    (case match v1 p1 theta
      of NONE => NONE
       | SOME(theta1) => match v2 p2 theta1)
  | match (A.Unit) (A.UnitPat) theta = SOME(theta)
  | match (A.Inject(i,v)) (A.InjectPat(j,p)) theta =
    if i = j then match v p theta else NONE
  | match (A.Fold(v)) (A.FoldPat(p)) theta = match v p theta
  | match (A.Pack(tau,v)) (A.PackPat(a,p)) theta = match v p theta
  | match v p theta = raise Match

fun project ((j,e)::texps) (i:A.tag) =
    if i = j then e else project texps i
  | project nil i = raise Match

fun eval env (A.Var _) = raise Match
  | eval env (e as A.Lam _) = e
  | eval env (A.App(e1,e2)) =
    (case eval env e1
      of A.Lam(x,_,e1') => ( tick() ; eval env (A.subst_exp_exp (eval env e2) x e1') )
       | _ => raise Match)
  | eval env (e as A.TpLam _) = e
  | eval env (A.TpApp(e,tau)) =
    (case eval env e
      of A.TpLam(a,e') => ( tick() ; eval env e' ) (* (A.subst_tp_exp tau a e') *)
       | _ => raise Match)
  | eval env (A.Pair(e1,e2)) = A.Pair(eval env e1, eval env e2)
  | eval env (A.Unit) = A.Unit
  | eval env (A.Inject(i,e)) = A.Inject(i,eval env e)
  | eval env (e as A.Record _) = e (* lazy records *)
  | eval env (A.Project(e,i)) =
    (case eval env e
      of A.Record(texps) => eval env (project texps i)
       | _ => raise Match)
  | eval env (A.Case(e,branches)) = match_branches env (eval env e) branches
  | eval env (A.Fold(e)) = A.Fold(eval env e)
  | eval env (A.Unfold(e)) =
    (case eval env e
      of A.Fold(v) => (tick () ; v)
       | _ => raise Match)
  | eval env (A.Pack(tau,e)) = A.Pack(tau, eval env e)
  | eval env (e as A.Fix(g,_,e')) = ( tick () ; eval env (A.subst_exp_exp e g e') )
  | eval env (A.Def(x)) = eval env (A.expand_exp env x)
  | eval env (A.Marked(marked_e)) = eval env (Mark.data marked_e)
and match_branches env v ((p,e)::branches) =
    (case match v p nil
      of NONE => match_branches env v branches
       | SOME(eta) => ( tick () ; eval env (A.subst_exp eta e)) )
  | match_branches env v nil = raise NoMatch

val eval = fn env => fn n => fn e =>
           ( set_gas n ; (eval env e , n - get_gas() ) )
           handle Overflow => raise BoundExceeded

exception NotComparable of string
                           
fun eq_val env (A.Lam _) (A.Lam _) = raise NotComparable("functions")
  | eq_val env (A.TpLam _) (A.TpLam _) = raise NotComparable("type functions")
  | eq_val env (A.Pair(v1,v2)) (A.Pair(w1,w2)) = eq_val env v1 w1 andalso eq_val env v2 w2
  | eq_val env (A.Unit) (A.Unit) = true
  | eq_val env (A.Inject(i,v)) (A.Inject(j,w)) = (i = j) andalso eq_val env v w
  | eq_val env (A.Record _) (A.Record _) = raise NotComparable("lazy records (pairs)")
  | eq_val env (A.Fold(v)) (A.Fold(w)) = eq_val env v w
  | eq_val env (A.Pack(tau,v)) (A.Pack(sigma,w)) = raise NotComparable("values of existential type") (* conservative *)
  (* Var, App, TpApp, Project, Case, Unfold, Fix, Defn, Marked *)
  | eq_val env _ _ = raise Match (* not a value or not of equal type *)

end (* structure Eval *)
