###############################################################################
# Driverlib.pm - A package of helper functions for Perl Autolab drivers
# 
# Copyright (c) 2005 David R. O'Hallaron, All rights reserved.
# May not be used, modified, or copied without permission.
#
# $Id: Driverlib.pm,v 1.12 2006/09/08 20:45:40 autolab Exp $
###############################################################################

package Driverlib;

use File::Basename;
use Socket;
use File::Copy;
use File::Basename;
use FileHandle;
use Socket;
use POSIX qw(ctime);
use POSIX qw(mktime);
use POSIX qw(strftime);
use POSIX qw(ceil);
use POSIX ":sys_wait_h";
use POSIX qw(setpgid);
use POSIX ":signal_h";

# Autogenerated header file with lab-specific constants
use lib ".";
use Driverhdrs;

require Exporter;
@ISA = qw(Exporter);
@EXPORT = qw(&driver_post &system_with_timeout &printd &read_file &read_testdirective $DEBUG);

use strict;

$Driverlib::DEBUG = 0;

#####
# Public functions
#

#
# driver_post - This is the routine that a driver calls when 
#    it needs to transmit an autoresult string to Autolab.
#
sub driver_post ($$) {
    my $userid = shift;       # User id for this submission
    my $result = shift;       # Autoresult string
    my $autograded = shift;   # Set if called by an autograder

    # Echo the autoresult string to stdout if the driver was called
    # by an autograder
    if ($autograded) {
	print "\n";
	print "AUTORESULT_STRING=$result\n";
	return;
    }	

    # If the driver was called with a specific userid, then submit
    # the autoresult string to the Autolab server over the Internet.
    if ($userid) {
	my $status = submitr($Driverhdrs::SERVER_NAME, 
			     $Driverhdrs::SERVER_PORT, 
			     $Driverhdrs::COURSE_NAME, 
			     $userid, 
			     $Driverhdrs::LAB, 
			     $result);
	
	# Print the status of the transfer
	if (!($status =~ /OK/)) {
	    print "$status\n";
	    print "Did not send autoresult string to the Autolab server.\n";
	    exit(1);
	}
	print "Success: Sent autoresult string for $userid to the Autolab server.\n";
    }	
}

#
# driver_submitfile - This is the routine that a driver calls when 
#    it needs to submit a file Autolab.
#
sub driver_submitfile ($$) {
    my $userid = shift;       # User id for this submission
    my $file = shift;	      # filename for this submission
    # never submitted by autograder
    # my $autograded = shift;   # Set if called by an autograder

    # submissions may be anonymous
    if ($userid eq "") {
      $userid = "anonymous";
    }

    # If the driver was called with a specific userid, then submit
    # the autoresult string to the Autolab server over the Internet.
    my $status = submitf($Driverhdrs::SERVER_NAME, 
			 $Driverhdrs::SERVER_PORT, 
			 $Driverhdrs::COURSE_NAME, 
			 $userid, 
			 $Driverhdrs::LAB, 
			 $file);

    # Print the status of the transfer
    print "$status\n";
  }

#####
# Private functions
#

#
# submitr - Sends an autoresult string to the submitr.pl CGI program
#           on the Autolab server.
#
sub submitr ($$$$$$) {
    my $hostname = shift;
    my $port = shift;
    my $course = shift;
    my $userid = shift;
    my $lab = shift;
    my $result = shift;

    my $internet_addr;
    my $enc_result;
    my $paddr;
    my $line;
    my $http_version;
    my $errcode;
    my $errmsg;

    # Establish the connection to the server
    socket(SERVER, PF_INET, SOCK_STREAM, getprotobyname('tcp'));
    $internet_addr = inet_aton($hostname)
	or die "Could not convert $hostname to an internet address: $!\n";
    $paddr = sockaddr_in($port, $internet_addr);
    connect(SERVER, $paddr)
	or die "Could not connect to $hostname:$port:$!\n";

    select((select(SERVER), $| = 1)[0]); # enable command buffering

    # Send HTTP request to server
    $enc_result = url_encode($result);
    print SERVER  "GET /$course/submitr.pl/?userid=$userid&lab=$lab&result=$enc_result&submit=submit HTTP/1.0\r\n\r\n";

    # Get first HTTP response line
    $line = <SERVER>;
    chomp($line);
    ($http_version, $errcode, $errmsg) = split(/\s+/, $line);
    if ($errcode != 200) {
	return "Error: HTTP request failed with error $errcode: $errmsg";
    }

    # Read the remaining HTTP response header lines
    while ($line = <SERVER>) {
	if ($line =~ /^\r\n/) {
	    last;
	}
    }

    # Read and return the response from the Autolab server
    $line = <SERVER>;
    chomp($line);

    close SERVER;
    return $line;
    
}
    
#
# submitf - Sends a file to be tested to the submitf.pl CGI program
#           on the Autolab server.
#
sub submitf ($$$$$$) {
    my $hostname = shift;
    my $port = shift;
    my $course = shift;
    my $userid = shift;
    my $lab = shift;
    my $file = shift;
    my $filebase = basename($file);

    my $file_size;
    my $internet_addr;
    my $enc_file;
    my $paddr;
    my $line;
    my $http_version;
    my $errcode;
    my $errmsg;

    # Establish the connection to the server
    socket(SERVER, PF_INET, SOCK_STREAM, getprotobyname('tcp'));
    $internet_addr = inet_aton($hostname)
	or die "Could not convert $hostname to an internet address: $!\n";
    $paddr = sockaddr_in($port, $internet_addr);
    connect(SERVER, $paddr)
	or die "Could not connect to $hostname:$port:$!\n";

    select((select(SERVER), $| = 1)[0]); # enable command buffering

    open(TESTFILE, "< $file") or die "Could not open $file\n";
    $file_size = (stat(TESTFILE))[7];

    # Send HTTP request to server
    $enc_file = url_encode($filebase);
    print SERVER  "POST /$course/submitf.pl/?userid=$userid&lab=$lab&file=$enc_file&submit=submit HTTP/1.0\r\n";
    print SERVER "Content-Type: text/plain\r\n";
    print SERVER "Content-Length: $file_size\r\n";
    print SERVER "\r\n";
    while ($line = <TESTFILE>) {
      print SERVER $line;
    }
    close TESTFILE;

    # Get first HTTP response line
    $line = <SERVER>;
    chomp($line);
    ($http_version, $errcode, $errmsg) = split(/\s+/, $line);
    if ($errcode != 200) {
	return "Error: HTTP request failed with error $errcode: $errmsg";
    }

    # Read the remaining HTTP response header lines
    while ($line = <SERVER>) {
	if ($line =~ /^\r\n/) {
	    last;
	}
    }

    # Read and return the response from the Autolab server
    while ($line = <SERVER>) {
      print $line;
    }
    chomp($line);		# chomp last line?

    close SERVER;
    return $line;
    
}

#
# url_encode - Encode text string so it can be included in URI of GET request
#
sub url_encode ($) {
    my $value = shift;

    $value =~s/([^a-zA-Z0-9_\-.])/uc sprintf("%%%02x",ord($1))/eg;
    return $value;
}

###
# system_with_timeout ($secs, $command)
# executes system($command) and returns its value
# or raises SIGALRM if it takes more than $secs
# taken from Perl 5.8.8 documentation of "alarm" and perlipc
#
# possible alternative:
# $result = system("ulimit -t <timeout>; prog> /tmp/$$_temp_file 2>/dev/null");

sub system_with_timeout {
  my $secs = shift;
  my $command = shift;
  my $result;
  my $kid;

  printd(2, "%% executing $command\n");
  my $p = fork();
  if ( $p == 0 ) { 
    # child
    setpgid(0,0);
    exec("${command}");
    # we need to do this because somehow sml manages to 
    # fork a second internal process that ignores direct
    # kills of the child.  instead we create a new process
    # group and then send an INT process to the direct child
    # which then passes a KILL onto every process in its new group.
  } else { 
    # parent
    printd(2, "%% child process is $p, parent is $$\n");

    eval {
      local $SIG{ALRM} = sub { die "alarm\n" };
      alarm $secs;		# turn on alarm timer
      $kid = waitpid($p,0);	# blocking wait for child
      $result = $?;		# record result, if terminated normally
      alarm 0;			# turn off timer
    };
    if ($@) {			# check whether "eval" died
      if ($@ eq "alarm\n") {
	## We had a timeout
	printd(0, "%% Timeout after $secs seconds\n");
	kill('KILL', -$p);	# kill all members of process group
	$kid = 0;
	while ($kid >= 0) {     # 0 means some children are still running
	  $kid = waitpid(-1, WNOHANG); # reap all members of process group
	  if ($kid > 0) {
	    printd(2, "%% reaped child $kid\n");
	  }
	}
	printd(2, "%% reaped all children\n");
	return 14;		# return SIGALRM to signal timeout
      } else {
	die;			# propage exception if not "alarm"
      }
    } else {			# no error: eval returned normally
      printd(2, "%% process $p returned normally\n");
      return $result;
    }
  }
}

###
# printd($level, $msg)
# print message if debug level >= $level
#
sub printd {
  my $level = shift;
  my $msg = shift;
  if ($Driverlib::DEBUG >= $level) {
    printf($msg);
  }
}

###
# read_file($file)
# returns file contents as a single string
#
sub read_file {
  my $file = shift;
  my $contents;
  open(INPUT, "<$file") or return undef;
  read(INPUT, $contents, -s INPUT);
  return $contents;
}

###
# read_testdirective($file) = ($directive, $value)
# parses the first line of a test file
# #test <directive> <value>
# where value is optional
# $directive may be undefined (if line is not well-formed),
# as may be $value
#
sub read_testdirective {
  my $file = shift;
  my $line;
  my ($test, $directive, $value);
  open(MYFILE, "<$file") or die "can't open test file: $file\n";
  $line = <MYFILE>;
  ($test, $directive, $value) = split(/\s+/, $line, 3);
  if ($test eq "#test") {
    return ($directive, $value);
  } else {
    return (undef, undef);
  }
}

# Always end a module with a 1 so that it returns TRUE
1;

